// Filename:	message.C
// Contents:	the methods for the message object
// Author:	Greg Shaw
// Created:	2/15/96

#ifndef _MESSAGE_C_
#define _MESSAGE_C_

#include "bbshdr.h"

#ifdef USE_DATABASE

extern User   user;             // alias and other information
extern Chat chatobj;    // check for chat requests



/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */



// Function:	constructor
// Purpose:	initialize the message object
// Input:	none
// Output:	none
// Author:	Greg Shaw
// Created:	3/6/96

Message::Message()
{
	cur_hdr = NULL;    // current header
	cur_msg = NULL;       // current message (being displayed)
	glist = NULL;		// no group list

	read_groupfile();	// read the user's group file
}


// Function:	destructor
// Purpose:	clean up the message object
// Input:	none
// Output:	none
// Author:	Greg Shaw
// Created:	3/6/96

Message::~Message()
{
	clear_grouplist();
}

// Function:	add_to_grouplist
// Purpose:	add a new record to the group list for the user
// Input:	gptr - a pointer to the data for the new record
// Output:	non zero for error
// Author:	Greg Shaw
// Created:	3/23/96

int Message::add_to_grouplist(char *section, int group, int high_message, int selected)// add to the grouplist
{
	Group	*gptr;	// iterator
	Group	*gnew;	// new record

	if (gnew=(Group *)malloc(sizeof(Group)), gnew == NULL)
	{
		ap_log("Unable to malloc new group.");
		return(1);
	}
	strcpy(gnew->section,section);
	gnew->group = group;
	gnew->high_message = high_message;
	gnew->selected = selected;	// did they select this group?
	gnew->next = NULL;
	if (glist == NULL)	// empty list?
	{
		glist = gnew;
	}
	else
	{
		for (gptr=glist; gptr->next != NULL; gptr=gptr->next);
		gptr->next = gnew;
	}
	return(0);
}


// Function:	can_view
// Purpose:	can a user view this message?
// Input:	section - the section to post to
//		del - is it a delete or view check?
// Output:	1 for yes, 0 for no
// Author:	Greg Shaw
// Created:	3/17/96

int Message::can_view(char *section, int del, char *from)        // can a user delete this message?
{
	CardRec	*card;			// user's card information
	char	fm=0;		// flags modifier
	char	am='>';		// acl modifier
	SectionHdr	shdr;	// section information

	if (msql.get_header(section,&shdr,1))	// header?
		return(1); // failed, give access
	// check against the moderator list
	if (from == NULL)	// no 'from' yet 
	{
		if (strstr(shdr.moderator,username()) != NULL)
			return(1);	// moderator can delete anything
	}
	else
	{
		if (strstr(shdr.moderator,from) != NULL)
			return(1);	// moderator can delete anything
		return(1);	// no way to check other access at this point
	}
	// check for sysop card access
	card = user.usercard();
	if (strcmp(card->colr,gstrl("BLACK"))==0)
		return(1);
	if (del)	// remaining access doesn't matter for delete
		return(0);
	// now check access
	if (shdr.modifier & ACLGREATER)
		am = '>';
	else if (shdr.modifier & ACLEQUAL)
		am = '=';
	else if (shdr.modifier & ACLLESSER)
		am = '<';
	if (shdr.modifier & FLAGSMATCH)
		fm = '=';
	else if (shdr.modifier & FLAGSNOTMATCH)
		fm = '!';
	else
		ap_log("Unknown modifier in flags");
	return(user.can_access(shdr.acl,shdr.flags,am,fm));
}

// Function:	check_grouplist
// Purpose:	check for a section in the grouplist
// Input:	section - the section to look for
// Output:	group if found, NULL if not
// Author:	Greg Shaw
// Created:	4/10/96

Group *Message::check_grouplist(char *section)
{

	Group	*gptr;

	for (gptr=glist; gptr != NULL && strcmp(gptr->section,section) !=
		0; gptr = gptr->next);
	return(gptr);
}

// Function:	clear_grouplist
// Purpose:	free the groupfile storage
// Input:	none
// Output:	0
// Author:	Greg Shaw
// Created:	3/22/96

int Message::clear_grouplist(void)
{
	Group	*gptr1;

	while (glist != NULL)
	{
		gptr1 = glist;
		glist = glist->next;
		free(gptr1);
	}
	return(0);
}

// Function:	delete_mail
// Purpose:	delete all mail from mailbox 
// Input:	section - section to delete it from
// Output:	non-zero for error
// Author:	Greg Shaw
// Created:	8/12/96

int Message::delete_mail(char *section)
{
	clear_scr();
	sstrl("MSDELMAIL");
	if (yesno())
	{
		if (msql.delete_mail(section))
		{
			sstrcrl("MSMAILDELETED");
		}
		else
		{
			sstrcrl("MSMAILDELFAILED");
		}
	}
	waitcr();
	return(0);
}

// Function:	delete_message
// Purpose:	delete a message from the system
// Input:	section - section to delete it from
//		msg - the message to delete
//		has_thread - should a thread be deleted (updated) too?
// Output:	non-zero for error
// Author:	Greg Shaw
// Created:	3/22/96

int Message::delete_message(char *section, Msg *msg, int has_thread)
{
	clear_scr();
	if (can_view(section,1,NULL) || strstr(msg->from_,username()) != NULL ||
		strstr(msg->to,username()) != NULL)
	{
		if (msql.delete_message(section,msg, has_thread))
		{
			sstrcrl("MSDELFAIL");
			return(1);
		}
		else
		{
			sstrcrl("MSDELSUC");
		}
	}
	else
	{
		// error
		sstrcrl("MSNODEL");
		cr();
		sstrcrl("MSSORRY");
		return(1);
	}
	waitcr();
	return(0);
}

// Function:	forward_to_sendmail
// Purpose:	forward an email to the sendmail daemon
// Input:	section - the section to post to
//		newmsg - the message to send
// Output:	non zero for error
// Author:	Greg Shaw
// Created:	4/14/96

int Message::forward_to_sendmail(Msg *newmsg)
{
	FILE	*msgfile;
	char	fname[30];
	char	tmpstr[255];
	char	to[80];
	char	from[80];

	strcpy(fname,"/tmp/rmsgXXXXXX");
	mktemp(fname);
	if (msgfile = bopen(fname,"w"), msgfile == NULL)
	{
		ap_log("Unable to open sendmail temp file for write.");
		return(-1);
	}
	// build a message within the file
	sscanf(newmsg->to,"%s%*s",to);
	if (strchr(to,'@') == NULL)
	{
		domainname(from);
		sprintf(tmpstr,"%s@%s.%s",to,mailhost(),from);
		strcpy(to,tmpstr);
	}
	fprintf(msgfile,"%s\n",newmsg->text);
	bclose(msgfile);
	sscanf(newmsg->from_,"%s%*s",from);
	sprintf(tmpstr,"%s -s \"%s\" %s < %s",MAIL_PATH,newmsg->subject,to,fname);
	if (sysint(tmpstr,0,1) != 0)
	{
		unlink(fname);
		return(-1);
	}
	unlink(fname);
	return(0);
}

// Function:	high_message
// Purpose:	(set or) return the high_message field for a particular 
//		section from the group list
// Input:	section - the section to post to
//		message - -1 for query, otherwise to set
// Output:	highest message for that group
// Author:	Greg Shaw
// Created:	3/23/96

int Message::high_message(char *section, int message)       // set (or get) highest message in a section
{
	Group *gptr;
	int	hmsg;

	hmsg = 0;
	gptr = check_grouplist(section);
	if (gptr == NULL)
	{
		if (message != -1)	// query only?
			add_to_grouplist(section,0,message,0);	// add to grouplist
	}
	else
	{
		if (message != -1)	// query?
			gptr->high_message = message;
		else
			hmsg = gptr->high_message;
	}
	return(hmsg);
}


// Function:	list_messages
// Purpose:	list the messages in a section addressed to the user
// Input:	section - the section to search
// Output:	the list of messages addressed to the user (if any)
// Author:	Greg Shaw
// Created:	4/25/96

int Message::list_messages(char *section)
{
	int pos = 0;
	char row[3][255];
        int     x=0;
	char 	*c;
        time_t  msgtime;
        struct tm *msgtm;
        char    tmpstr[255];    // selection string
        char    timstr[80];


	if (msql.list_messages(section,-1,row) == 0)
	{
		sstrcrl("MSNOMES");
	}
	else
	{
		sstrcrl("MSLISTTITLE");
		pos = 0;
		while (msql.list_messages(section,pos,row) != 0)
		{
			sscanf(row[1],"%ld",&msgtime);
			msgtm = localtime(&msgtime);
			strftime(timstr,80,"%x",msgtm);
			if (c = strchr(row[0],'('), c != NULL)
				*c = 0;	// delete the mailing address from list
			sprintf(tmpstr,gstrl("MSLISTITEM"),row[0],timstr,row[2]);
			sstrcr_c(tmpstr);
			if (x%user.lin() == 0 && x != 0)
			{
				waitcr();
			}
			pos++;
		}
	}	
	waitcr();
	return(0);
}

// Function:	list_threads
// Purpose:	list the threads selected in a tabular format, similar to
//		the files object file listing
// Input:	groups - the number of groups selected
//		lines - the number of lines on the user's terminal
//		glist - the list of groups that the thread list came from
// Output:	the number of messages posted (if any)
// Author:	Greg Shaw
// Created:	3/23/96

int Message::list_threads(int groups, int lines, Group *glist)
{
	int	done;		// loop 
	Thread	*tpos;		// current record
	Thread	*screen[MAX_FILES_ON_SCREEN];	// currently on screen
	Thread	*searchpos=NULL;	// current search target
	Msg	fakereply;	// fake a 'reply' for posting to a thread
	time_t	then;		// inactivity checking
	SectionHdr	shdr;		// section header information
	char	tmpstr[255];	// output string
	char	*bcast;		// a broadcast message (from broadcast object)
	char	c;		// input char
	int	searching;	// are we (still) searching?
	int	direction;	// direction of movement
	int	screenitems=0;	// number of items on screen
	int	x;
	int	posted = 0;

	const int screensize = lines-3;	// number of items per screen
	const char validkeys[] = "-rpsfbq\n\r";	// valid selection keys
	char   bsstr[] =            // backspace
		{
			0x8,0x20,0x8,0x0
		};


	searching = 0;	// not searching (yet)
	direction = 1;	// positive direction
	tlist.head();	// start at the top of the list
	for (done=0; done<1;)
	{
		clear_scr();
		switch(direction)
		{
		case -1:	// backwards
			screenitems = 0;
			// rewind 2 screens	
			tlist.rewind(screensize*2);
			for (x = 0; x<(screensize); x++)
			{
				tpos = tlist.next();
				if (tpos != NULL)
				{
					screen[screenitems] = tpos;
					tpos=tpos->next;
					screenitems++;
				}
			}
			break;
		case 0:	// no movement - redisplay current screen
			break;
		case 1:	// forwards - next screen
			// build list of screen items
			screenitems = 0;
			for (x = 0; x<(screensize); x++)
			{
				tpos = tlist.next();
				if (tpos != NULL)
				{
					screen[screenitems] = tpos;
					tpos=tpos->next;
					screenitems++;
				}
			}
			if (screenitems == 0)	// no items?
			{	// then redraw previous screen
				direction = -1;
				continue;
			}
			break;
		}
		// draw header
		sprintf(tmpstr,gstrl("MSTHRHEAD"),groups,groups>1?gstrl("MSS"):"",
			tlist.numrecs(),tlist.numrecs() >
			1?gstrl("MSS"):"", (((float)tlist.position()+1)/(float)tlist.numrecs())*100.0);
		sstrcr_c(tmpstr);
		sstrcrl("MSTHRHEAD2");
		// draw items
		for (x = 0; x < screenitems; x++)
		{
			if (msql.get_header(screen[x]->section,&shdr,1) == 0)
			{
				sprintf(tmpstr,gstrl("MSTHRLINE"),x+1,
					shdr.long_name,
					screen[x]->messages,screen[x]->subject);
			}
			else
			{
				sprintf(tmpstr,gstrl("MSTHRLINE"),x+1,
					screen[x]->section,
					screen[x]->messages,screen[x]->subject);
			}
			sstrcr_c(tmpstr);

		}
		// fill in the rest of the space (if empty)
		for (x=0; x<(screensize-screenitems); x++)
		{
			cr();
		}
		// display footer
		sstrl("MSTHRPROMPT");
		// look for chars
		time(&then);
		while (c = tolower(gch(2)), c == 0 || strchr(validkeys,c) == NULL)
		{
                        if (c != 0)         // invalid char?
                                sstr_c(bsstr);  // then erase it
                        if ((time(NULL) - then)/60 > inactivity_timeout())
                        {
                                return(0);
                        }
                        if (bcast = chatobj.check_broadcast(), bcast != NULL)
                        {
                                        // private request?
                                if (!chatobj.check_private(bcast,0,1))
                                {                   // nope, regular message
                                        clear_scr();
                                        sstrcrl("MESSAGERECEIVED");
                                        cr();
                                        sstrcrl(bcast);
                                        cr();
                                        cr();
                                        waitcr();
					continue;
                                }
                        }
		}
		cr();
		direction = 0;	// no movement by default
		switch(c)
		{
		case 'r':	// read thread
		{
			sstrl("MSREADWHICH");
			tmpstr[0] = 0;
			gstr(tmpstr,3);
			if (sscanf(tmpstr,"%d",&x) != 1)
				continue;
			x--;	// correct for 1..x rather than 0..x
			if (x < 0 || x >= screenitems)
				continue;
			sstrl("MSREADNEW");
			while (c = tolower(gch(2)), c == 0 || !(c == 'n' || c == 'a'));
			cr();
			if (c == 'n')	// new messages
				view(screen[x]->section,screen[x]->thread_number,high_message(screen[x]->section,-1));
			else // all messages
				view(screen[x]->section,screen[x]->thread_number,0);
			break;
		}
		case 'p':	// post message to thread
		{
			cr();
			sstrl("MSPOSTQUEST");
			while (c = tolower(gch(2)), c == 0 || !(c == 'e' || c == 'n' || c == 'q'));
			cr();
			cr();
			switch (c)
			{
			case 'e':
			{	// existing thread
				tmpstr[0] = 0;
				sstrl("MSPOSTTHREAD");
				gstr(tmpstr,3);
				if (sscanf(tmpstr,"%d",&x) != 1)
					continue;
				x--;	// convert from 1..x to 0..x
				// create a fake 'message' to allow subject to 
				// remain the same
				strcpy(fakereply.subject,screen[x]->subject);
				fakereply.from_[0] = 0;
				strcpy(fakereply.to,gstrl("MSALL"));
				fakereply.thread_number = screen[x]->thread_number;
				tmpstr[0] = 0;
				fakereply.text = tmpstr;
				posted += post(screen[x]->section,&fakereply,0,NULL);
				break;
			}
			case 'q':	// quit
			{
				continue;
				break;
			}
			case 'n':	// create a new thread
			{
				// list the groups that the user may post to
				int numgroups = 0;
				Group *gptr;
				SectionHdr hdr;	// for name lookups
				// find number of groups
				for (gptr = glist; gptr != NULL; gptr=gptr->next)
					numgroups++;
				// show the user a list of groups
				sstrcrl("MSPOSTGROUPS");
				x = 0;
				for (gptr = glist; gptr != NULL; gptr=gptr->next)
				{
					if (msql.get_header(gptr->section,&hdr,1))
						return(0);
					if (x!=0 && x%(lines -2) == 0)
						waitcr();
					sprintf(tmpstr,gstrl("MSGROUPLIST"),x+1,hdr.long_name);
					sstrcr_c(tmpstr);
					x++;
				}	
				// now ask which group
				cr();
				sstrl("MSPOSTGROUP");
				tmpstr[0] = 0;
				gstr(tmpstr,3);
				if (sscanf(tmpstr,"%d",&x) != 1)
					continue;
				if (x<1 || x > numgroups)
					continue;
				gptr = glist;
				for (int t = 0; t != x-1; t++)
					gptr = gptr->next;
				posted += post(gptr->section,NULL,0,NULL);
				break;
			}
			}
			break;
		}
		case 's':	// search
		{
			int ask = 0;
			if (searching)
			{
				sstrl("CONTINUESEARCH");
				if (!yesno())
					ask = 1;
			}
			else 
			{
				searchpos = NULL;
				ask = 1;
			}
			if (ask)
			{
				sstrcrl("MSSEARCHNOTE");
				sstrcrl("MSSEARCHCASE");
				cr();
				sstrl("MSSEARCHSTR");
				tmpstr[0] = 0;
				gstr(tmpstr,MAX_SUBJECT-1);
				if (strlen(tmpstr) == 0)	// don't bother on an empty string
					continue;
			}
			searchpos = tlist.search(searchpos,tmpstr);
			if (searchpos == NULL)	// none found
			{
				sstrcrl("MSSTRINGNOT");
				waitcr();
				searching = 0;
			}
			else
			{	// go to that record
				tlist.setpos(searchpos);
				direction = 1;	// go forward from that point
			}
			break;
		}
		case 'f':	// forward
		case '\r':
		case '\n':
			direction = 1;
			break;
		case 'b':	// back
			direction = -1;
			break;
		case '-':	
		case 'q':	// quit
			done++;
		}
	}
	return(posted);
}

// Function:	mailavail
// Purpose:	check for new private mail to the user
// Input:	none
// Output:	1 if new mail has arrived, 0 otherwise
// Author:	Greg Shaw
// Created:	5/24/96

int Message::mailavail(void)
{
	static time_t then;	// don't check too often
	static int prev_messages;	// number of messages at last check
	int	num_messages;

	// attempt checkpoint
	if (abs(time(NULL)-then) > 30000)
		prev_messages = msql.get_private_messages("private",0);

	if (abs(time(NULL)-then) > mailchecktime())
	{
		time(&then);
		num_messages = msql.get_private_messages("private",0);
		if (num_messages > prev_messages)
		{	// new mail
			return(1);
		}
		prev_messages = num_messages;
	}
	return(0);
}

// Function:	move_message
// Purpose:	move a message from one section to another
// Input:	section - the section to post to
//		reply - post a reply to this message (NULL for no reply)
// Output:	1 for success, 0 for failure
// Author:	Greg Shaw
// Created:	3/6/96

int Message::move_message(char *section, Msg *msg)      // move message to another section
{
	SectionHdr	shdr;		// section header information
	char	tmpstr[255];

	sstrl("MSMOVE");
	gstr(tmpstr,MAX_SECTION);
	if (msql.get_header(section,&shdr,1) == 0)
	{
		// delete old message
		if (!delete_message(section,msg,1))
		{
			sstrcrl("MSDELFAIL");
		}
		else
		{
			// now add to new section
			if (!post_msg(tmpstr, msg, 1, 0))
			{
				sstrcrl("MSMESMOVED");
			}
			else
			{
				sstrcrl("MSADDFAILED");
			}
		}
	}
	else
		sstrcrl("MSSECNOTFOUND");
	waitcr();
	return(0);
}


// Function:	post
// Purpose:	post a message to the system
// Input:	section - the section to post to
//		reply - post a reply to this message (NULL for no reply)
//		priv - is it a private email post?
//		to - prompt for a private email address or hard code?
// Output:	1 for success, 0 for failure
// Author:	Greg Shaw
// Created:	3/6/96


int Message::post(char *section, Msg *reply, int priv, char *to)        // post public message
{
	SectionHdr	shdr;		// section header information
	CardRec	*card;			// user's card information
	Msg	newmsg;			// new message
	int	quotes;			// number of single quotes in msg
	int	textsize;		// size of text
	char	tmpstr[255];		
	char	fname[14];		// message filename (for editing)
	char	tagline[MAX_TAGLINE];	// tagline for message
	char	*c;
	char	y;			// key
	struct stat	fistat;		// file status
	FILE	*msgfile;		// message file
	const char validkeys[] = "123q";	// valid menu selection keys

	// get section
	if (msql.get_header(section,&shdr,1))
		return(0);
	card = user.usercard();
	// check for read only section
	if (shdr.read_only && !can_view(section,0,NULL))
	{	// only sysop (or moderator) may post to read-only sections
		sstrcrl("MSSECREADONLY");
		waitcr();
		return(0);
	}
	// check for non access
	if (!can_view(section,0,NULL))
	{	// acl not high enough
		sstrcrl("MSSECNOACCESS");
		waitcr();
		return(0);
	}
	if (priv)
	{	// prompt for 'to:'
		if (to != NULL)
			strcpy(newmsg.to,to);
		else
		{
			sstrl("MSADDRESS");
			if (reply != NULL)
				sscanf(reply->from_,"%s",tmpstr);
			else
				tmpstr[0] = 0;
			gstr(tmpstr,MAX_FROM_TO-1);

			if (strlen(tmpstr) == 0)
				return(0);
			strcpy(newmsg.to,tmpstr);
		}
	}
	// put up prompt with subject
	if (reply != NULL)
		sprintf(tmpstr,"%s",reply->subject);
	else
		strcpy(tmpstr,"");
	sstrl("MSSUBJECT");
	gstr(tmpstr,MAX_SUBJECT-1);
	if (strlen(tmpstr) == 0)
		return(0);
	if (!priv)
	{
		c = dequote(tmpstr);
		if (strlen(c) > MAX_SUBJECT)	// too long to store?
			c[MAX_SUBJECT-1] = 0;
		strcpy(tmpstr,c);
		ap_log(tmpstr);
		free(c);	// avoid memory leak
	}
	strcpy(newmsg.subject,tmpstr);
	domainname(tmpstr);
	sprintf(newmsg.from_,gstrl("MSEMAIL"),username(),mailhost(),tmpstr,user.fname,user.lname);
	if (shdr.anonymous)
	{
		sstrl("MSANONPOST");
		if (yesno())
			strcpy(newmsg.from_,gstrl("MSANONYMOUS"));
		// ask if to post anonymously
	}
	if (strlen(shdr.moderator) != 0)	// is there a moderator?
	{
		// copy moderator's address into 'to'.
		strcpy(newmsg.to,shdr.moderator);
		// make sure it goes into email
		priv = 1;
	}
	strcpy(fname,"/tmp/rmsgXXXXXX");
	mktemp(fname);
	// create file
	creat(fname,0700);	// NOTE: I am ignoring any error 
	if (reply != NULL)
	{
		sstrl("MSINCLUDE");
		if (yesno())
		{
			// generate quote file from message
			if (msgfile = bopen(fname,"w"), msgfile == NULL)
			{
				sprintf(tmpstr,"Unable to open message file for editing %s",fname);
				ap_log(tmpstr);
				return(0);
			}
			// Ok.  Add quotes to message
			fputc(QUOTE_CHAR,msgfile);	// add 1st line quote
			c = reply->text;
			while (*c != 0)
			{
				fputc(*c,msgfile);
				if (*c == '\n')	// end of line?
					fputc(QUOTE_CHAR,msgfile);
				c++;
			}
			bclose(msgfile);
		}
		// ask to quote reply
		if (!priv)
			strcpy(newmsg.to,reply->from_);	// copy 'from' to 'to'
	}
	else
	{
		if (!priv)
			strcpy(newmsg.to,gstrl("MSALL"));	// public message area -- 
	}
	// edit file
	sprintf(tmpstr,gstrl("MSMSGLEN"),shdr.maxlines*80);
	sstrcr_c(tmpstr);
	waitcr();
	y = '1';
	while (y != '2' && y != 'q')
	{
		if (y == '1')
		{
			sprintf(tmpstr,"%s %s",user.editorname(),fname);
			sysint(tmpstr,0,0);
			stat(fname,&fistat);
			if (fistat.st_size+1 > shdr.maxlines*80)
			{
				sstrcrl("MSTOOLARGE");  
				sstrcrl("MSREMOVE");
				cr();
				waitcr();
				// note: this could be a 'bad thing', as
				// it doesn't give the user any option but
				// to edit the message
				continue;
			}
		}
		sstrl("MSMESSAGEEDITMENU");
		while (y = gch(1),  y == 0 || strchr(validkeys,y) == NULL);
		cr();
		// if ( y == '3' ) // add later
			// spell check message
	}
	switch(y)
	{
	case '2':	// post message
		cr();
		sstrcrl("MSPOSTING");
		tagline[0] = 0;
		if (shdr.tagline)	// include a tagline?
		{
			msql.get_tagline(shdr.tagline,tagline);
		}
		// get size of file for malloc
		stat(fname,&fistat);
		textsize = fistat.st_size+1+MAX_QUOTE+MAX_TAGLINE;
		if (newmsg.text = (char *)malloc(textsize), newmsg.text == NULL)
		{
			sprintf(tmpstr,"Unable to malloc memory for new message text of size %ld",fistat.st_size+1);
			ap_log(tmpstr);
			return(0);
		}
		// open edited file
		if (msgfile = bopen(fname,"r"), msgfile == NULL)
		{
			sprintf(tmpstr,"Unable to open message file for reading %s",fname);
			ap_log(tmpstr);
			return(0);
		}
		// read file
		y = 0;
		c = newmsg.text;
		quotes = 0;
		while (y = fgetc(msgfile), y != EOF)
		{
			if (!priv)
			{
				if (y == '\'')	// must backtick quotes
				{
					*c++ = '\\';
					quotes++;
					if (quotes > MAX_QUOTE)
					{
						ap_log("too many quotes in message.  Post aborded.");
						sstrcrl("MSPOSTFAIL");
						waitcr();
						return(0);
					}
				}
			}
			*c++ = y;
		}
		*c = 0;
		bclose(msgfile);
		// tack on the tagline, if any
		if (strlen(tagline) > 0)
		{
			// add separator line
			strcat(newmsg.text,"------\n");
			strcat(newmsg.text,tagline);
			strcat(newmsg.text,"\n");
		}
		if (unlink(fname)<0)
		{
			sprintf(tmpstr,"Unable to unlink %s",fname);
			ap_log(tmpstr);
		}
		newmsg.local_origin = 1;
		// chop off any size violations added through taglines or such
		newmsg.text[textsize-1] = 0;
		if (priv)	// forward to sendmail for private
		{
			if (forward_to_sendmail(&newmsg) == -1)
			{	// mail failed.  tell the user
				sstrcrl("MSPOSTUNKNOWN");
				cr();
				sstrcrl("MSPOSTCHECK");
				waitcr();
				return(0);
			}
		}
		else
		{
			if (msql.post(section, &newmsg,0))
			{
				sstrcrl("MSPOSTFAIL");
				waitcr();
				return(0);
			}
		}
		break;
	case 'q':	// forget message
		return(0);
	}
	if (strlen(shdr.moderator) != 0)	// moderated?
		sstrcrl("MSPOSTMODERATOR");
	else
		sstrcrl("MSPOSTSUCC");
	waitcr();
	return(1);
}

// Function:	post_msg
// Purpose:	post a message to the system non-interactively
// Input:	section - the section to post to
//		newmsg  - post message
//		local_origin -- is the message local?
// Output:	0 for success
//		1 for permission denied
//		2 for other error
// Author:	Greg Shaw
// Created:	5/19/96


int Message::post_msg(char *section, Msg *newmsg, int local_origin, int priv)// post message
{
	SectionHdr	shdr;		// section header information
	char	tmpstr[255];		
	char	*c;
	char	*t;

	// get section
	if (msql.get_header(section,&shdr,1))
		return(2);	// section not found
	// readonly?
	if (shdr.read_only && !can_view(section,0,newmsg->from_))
	{	// only sysop (or moderator) may post to read-only sections
		return(4);
	}
	// check acl
	if (!can_view(section,0,newmsg->from_))
	{
		return(5);
	}
	// backtick single quotes
	c = dequote(newmsg->subject);
	if (strlen(c) > MAX_SUBJECT) 
		c[MAX_SUBJECT-1] = 0;
	strcpy(newmsg->subject,c);
	newmsg->local_origin = local_origin;
	free(c);
	domainname(tmpstr);
	// post message
	// get size of file for malloc
	c = dequote(newmsg->text);
	t = newmsg->text;	// save the pointer address for later	
	if (strlen(t) > ((unsigned)80*shdr.maxlines)-1024)
	{
		newmsg->text[80*shdr.maxlines-1024] = 0;
		ap_log("Message truncated, message too long.");
	}
	newmsg->text = c;
	newmsg->local_origin = 0;
	if (msql.post(section, newmsg, priv))
	{
		return(2);
	}
	newmsg->text = t;
	free(c);
	return(0);
}

// Function:	read_groupfile
// Purpose:	read the group file from the user's home directory
// Input:	none
// Output:	the group file is read
// Author:	Greg Shaw
// Created:	3/17/96

int Message::read_groupfile(void)       // read user's $HOME/.rgroups file
{
	char 	*home;		// home directory name
	char	*line=NULL;		// line of input from file
	char	*c;
	char	tmpstr[255];	
	FILE	*infile;	// input file
	SectionHdr	sec;	// section header
	char	section[MAX_SECTION_LENGTH];	// section name
	int	high_message;	// highest message in section
	int	selected;	// group selected?

	if (home=getenv("HOME"), home == NULL)
	{
		// ap_log("read_groupfile: No $HOME environment variable.");
			return(-1);	// not much to do at this point
	}
	sprintf(tmpstr,"%s/%s",home,GROUPFILENAME);
	if (infile = bopen(tmpstr,"r"), infile == NULL)
		return(0);	// not an error
	// file exists.  read it.
	// format: 
	// groupname  highmessage
	while (!feof(infile))
	{
		line = getline(infile);
		// chop any comments
		if (c = strchr(line,'#'), c != NULL)
			c = 0;	// ignore from # on 
		if (sscanf(line,"%s%d%d",section, &selected, &high_message) == 3)
		{
			// get section group id (for check)
			if (msql.get_header(section,&sec,1))
			{
				bclose(infile);
				ap_log("get of section header failed.");
				return(1);
			}
			add_to_grouplist(section,0,high_message,selected);	// add to grouplist
		}
	}
	bclose(infile);	// all done!
	return(0);
}

// Function:	read_private
// Purpose:	read private messages
// Input:	section - the section to read
//		highest - the highest last message read (for delete)
// Output:	the number of private messages posted, or,
//		-1 for a request to exit and re-enter (due to a message delete)
// Author:	Greg Shaw
// Created:	4/14/96
// Notes:	1. This is almost identical to the view() function.
//		2. The highest parameter tracks where the user was when a 
//		delete was done.  Without that parameter, deleting a
//		message in the middle of the mailbox meant that you had to
//		go through the same messages, in order, rather than
//		starting at the point just beyond the deleted message

int Message::read_private(char *section, int *highest)
{
	int	dir; 		// direction of read
	char	c;		// input char
	Msg	*cur;		// current message
	FILE	*mfile;		// file for viewing message
	time_t	then;		// last 'move'
	int	num_messages;	// number of messages in thread
	int	cur_message;	// current message (ordinal)
	int 	done;
	int	posted = 0;	// number of posted messages
	char	*bcast;		// a broadcast message (from broadcast object)
	char	fname[MAXPATHLEN];	// name of output file
	char	tmpstr[255];
	SectionHdr	sec;	// current section header
	const char	validchars[] = "bfdrq";
	char	bsstr[] =            // backspace
		{
			0x8,0x20,0x8,0x0
		};


	msql.get_header(section,&sec,1);	// get header information
	cur_message = -1;	// before first record
	// select messages in thread greater than high_message
	num_messages = msql.get_private_messages(section,1);
	if (num_messages == 0)
	{
		sstrcrl("MSNOMES");
		waitcr();
		return(0);
	}
	dir = 1;	// move forward to start
	cur = NULL;	// no current message (yet)
	if (*highest != 0)	// re-entry?
	{
		// now look for a message higher than the previous higher messsage
		cur = msql.head();
		cur_message = 0;
		dir = 0;	// redisplay current message
		while (cur != NULL && cur->message_number <= *highest)
		{
			msql.clear_message(&cur);
			cur = msql.next();
			cur_message++;
		}
		if (cur == NULL)	// no higher messages
		{
			// then start at the top
			cur = msql.head();
			cur_message = 0;
		}
	}
	done = 0;
	while (!done)
	{
		switch(dir)
		{
		case -1:	// move backwards
			if (cur_message > 0)
				cur_message--;
			else
			{
				sstrcrl("MSNOPRIORMES");
				waitcr();
				break;	// can't move beyond first record
			}
			if (cur != NULL)
				msql.clear_message(&cur);
			cur = msql.previous();
			break;
		case 0:	// redisplay current message
			break;
		case 1:	// move	forward 
			if (cur_message < num_messages-1)
				cur_message++;
			else
			{
				sstrcrl("MSNOMOREMES");
				waitcr();
				break;	// can't move beyond last record
			}
			if (cur != NULL)
				msql.clear_message(&cur);
			cur = msql.next();
			break;
		}
		strcpy(fname,"/tmp/msgXXXXXX");
		mktemp(fname);
		// build file with message
		if (mfile = bopen(fname,"w"), mfile == NULL)
		{
			ap_log("Unable to write message view file.");
			return(0);
		}
		fprintf(mfile,"%s\n",process(QTOQUIT));
		fprintf(mfile,process(gstrl("MSPRIVVIEWTITLE")),cur->message_number,sec.long_name,cur_message+1,num_messages,'\n');
		fprintf(mfile,process(gstrl("MSFROM")),cur->from_,'\n');
		fprintf(mfile,process(gstrl("MSTO")),cur->to,'\n');
		fprintf(mfile,process(gstrl("MSMESSUBJECT")),cur->subject,'\n');
		fprintf(mfile,process(gstrl("MSDATE")),ctime(&cur->date),'\n');
		fprintf(mfile,process(gstrl("MSTEXT")),cur->text,'\n');
		bclose(mfile);
		// display file using system pager
		sprintf(tmpstr,"%s %s",sys_pager(),fname);
		clear_scr();
		sysint(tmpstr,0,0);
		if (unlink(fname) < 0)
		{
			sprintf(tmpstr,"Unable to unlink %s",fname);
			ap_log(tmpstr);
		}
		// put up prompt
		sstrl("MSVIEWPROMPT");
		time(&then);
		while (c = tolower(gch(1)), c == 0 || strchr(validchars,c) == NULL)
		{
                        if (c != 0)         // invalid char?
                                sstr_c(bsstr);  // then erase it
                        if ((time(NULL) - then)/60 > inactivity_timeout())
                        {
                                return(0);
                        }
                        if (bcast = chatobj.check_broadcast(), bcast != NULL)
                        {
                                        // private request?
                                if (!chatobj.check_private(bcast,0,1))
                                {                   // nope, regular message
                                        clear_scr();
                                        sstrcrl("MESSAGERECEIVED");
                                        cr();
                                        sstrcrl(bcast);
                                        cr();
                                        cr();
                                        waitcr();
					continue;
                                }
                        }
		}
		cr();
		cr();
		switch(c)
		{

		case 'f':	// forward (next)
			dir = 1;
			break;
		case 'b':	// backward (previous)
			dir = -1;
			break;
		case 'r':	// reply to message
			posted += post(section,cur,1,NULL);
			break;
		case 'd':	// delete message
			*highest = cur->message_number;
			if (!delete_message(section,cur,0))
			{
				return(-1);
			}
			waitcr();
			break;
		case '-':
		case 'q':	// quit
			done++;
			continue;
		}
	}
	return(posted);
}

// Function:	read_public
// Purpose:	read public messages
// Input:	section - the section to read
//		group - the section grouping (-1 for user-selected groups)
// Output:	the number of message posted (normally 0)
// Author:	Greg Shaw
// Created:	3/17/96

int Message::read_public(char *section, int group, int lines)// read messages
{
	Group	*lglist;
	Group	*gptr;
	Group 	*gend=NULL;
	Group	*newg;
	Group	grp;
	int	groups;
	int	posted = 0;

	lglist = NULL;
	groups = 0;
	// bbs grouping read?
	if (group > 0)
	{
		// get group ids from info table
		groups = msql.get_bbs_group(&lglist, group);
		// add the 'high_message' to the group
		// by comparing to the existing group list
		for (gptr = lglist; gptr != NULL; gptr=gptr->next)
		{
			// get highest message in section
			gptr->high_message = high_message(gptr->section,-1);
		}
	}
	// user's grouping read?
	else if (group == -1)
	{
		for (gptr = glist; gptr != NULL; gptr=gptr->next)
		{
			if (gptr->selected)
			{
				// build a list of the groups the user has selected
				groups++;
				if (newg = (Group *)malloc(sizeof(Group)), newg == NULL)
				{
					ap_log("new group - out of memory!");
					return(0);
				}
				strcpy(newg->section,gptr->section);
				newg->group = gptr->group;
				newg->high_message = gptr->high_message;
				newg->selected = 1;
				newg->next = NULL;
				if (lglist == NULL)	// first record?
				{
					lglist = newg;
					gend = newg;
				}
				else
				{
					gend->next = newg;
					gend = newg;
				}
			}
		}
		if (lglist == NULL)	// empty list?
		{
			sstrcrl("MSNOSECTIONS");
			waitcr();
			return(0);
		}
	}
	// individual section read?
	else if (section != NULL)	
	{
		// build group selection list
		strcpy(grp.section,section);
		grp.group = group;
		grp.high_message = high_message(section,-1);
		grp.next = NULL;
		lglist = &grp;
		groups = 1;
	}
	// build thread query
	if (msql.build_thread_list(lglist,&tlist)==0)	// get associated threads
	{
		sstrcrl("MSNOMES");
		waitcr();
		return(0);
	}
	posted = list_threads(groups, lines, lglist);	// view thread list
	// blow away the local group list
	if (lglist != NULL && lglist != &grp)
	{ 	
		while (lglist != NULL)
		{
			gend = lglist;
			lglist = lglist->next;
			free(gend);
		}
	}
	// nuke the thread list
	tlist.clear_list();
	return(posted);
}

// Function:	save_groupfile
// Purpose:	save the contents of the glist to the user's group file
// Input:	none
// Output:	0 for normal exit, non zero for failure
// Author:	Greg Shaw
// Created:	3/22/96

int Message::save_groupfile(void)
{
	FILE	*outfile;
	char	tmpstr[255];
	Group	*gptr;

	
	sprintf(tmpstr,"%s/%s",getenv("HOME"),GROUPFILENAME);
	if (outfile = bopen(tmpstr,"w"), outfile == NULL)
	{
		ap_log("Unable to save group file.");
		return(1);
	}
	for (gptr = glist; gptr != NULL; gptr=gptr->next)
	{
		fprintf(outfile,"%s %d %d\n",gptr->section,gptr->selected,gptr->high_message);
	}
	bclose(outfile);
	return(1);
}

// Function:	scan
// Purpose:	scan the database for messages to the user
// Input:	none
// Output:	user is shown the messages addressed to them
// Author:	Greg Shaw
// Created:	4/12/96

int Message::scan(time_t last_logon)
{
	SectionHdr	sec;
	int		recnum;
	int		mesfound;
	char		tmpstr[255];

	mesfound = 0;
	// go through the groups, scanning each message base for the user's name
	recnum = 0;
	while (!msql.iterate_header(&sec,recnum,1))
	{
		sprintf(tmpstr,gstrl("MSSCANNING"),sec.long_name);
		sstr_c(tmpstr);
		mesfound = msql.scan(&sec,last_logon);
		if (mesfound > 0)
			sprintf(tmpstr,gstrl("MSSCANFOUND"),mesfound,mesfound>1?"s":"");
		else
			sprintf(tmpstr,"	%s",gstrl("MSNONEWMES"));
		sstrcr_c(tmpstr);
		recnum++;
	}
	return(0);
}

// Function:	scan_section
// Purpose:	scan the database for messages to the user and dump out the 
//		headers of the messages
// Input:	last_logon - time of last logon
//		section - the section to look through
// Output:	user is shown the headers of the messages addressed to them
// Author:	Greg Shaw
// Created:	4/12/96


int Message::scan_section(time_t last_logon, char *section) // scan for message to user in section 
{
}

// Function:	section_information
// Purpose:	display the section information for a section
// Input:	section - the section to post to
// Output:	0 for normal exit, non zero for failure
// Author:	Greg Shaw
// Created:	3/17/96


int Message::section_information(char *section) // show section information
{
	char	tmpstr[255];
	char	stype[50];	// type of section
	SectionHdr	shdr;	// section information
	struct tm *update;	// time of last update
	char	timestr[80];

	msql.get_header(section,&shdr,1);	// get header information
	clear_scr();
	switch(shdr.section_type)
	{
	case PRIVATE_MESSAGE:
		strcpy(stype,gstrl("MSPRIVATE"));
		break;
	case PUBLIC_MESSAGE:
		strcpy(stype,gstrl("MSPUBLIC"));
		break;
	case FIDO_MESSAGE:
		strcpy(stype,gstrl("MSFIDO"));
		break;
	case USENET_MESSAGE:
		strcpy(stype,gstrl("MSUSENET"));
		break;
	}
	sprintf(tmpstr,gstrl("MSSECTION"),stype,shdr.long_name,shdr.table_name);
	sstrcr_c(tmpstr);
	sprintf(tmpstr,gstrl("MSSECTIONTITLE"),msql.num_messages(section),shdr.high_message);
	sstrcr_c(tmpstr);
	sprintf(tmpstr,gstrl("MSTHREADS"),msql.num_threads(section),shdr.high_thread);
	sstrcr_c(tmpstr);
	update = localtime(&shdr.date);
	strftime(timestr,79,"%c",update);
	sprintf(tmpstr,gstrl("MSLASTPOST"),timestr);
	sstrcr_c(tmpstr);
	sprintf(tmpstr,gstrl("MSACL"),shdr.acl,shdr.flags);
	sstrcr_c(tmpstr);
	sprintf(tmpstr,gstrl("MSANONYPOST"),shdr.anonymous==1?gstrl("YES"):gstrl("NO"));
	sstrcr_c(tmpstr);
	sprintf(tmpstr,gstrl("MSREADONLY"),shdr.read_only==1?gstrl("YES"):gstrl("NO"));
	sstrcr_c(tmpstr);
	sprintf(tmpstr,gstrl("MSGROUPING"),shdr.group);
	sstrcr_c(tmpstr);
	sprintf(tmpstr,gstrl("MSMAXLINES"),shdr.maxlines);
	sstrcr_c(tmpstr);
	sprintf(tmpstr,gstrl("MSDAYSTOKEEP"),shdr.days);
	sstrcr_c(tmpstr);
	waitcr();
	return(0);
}

// Function:	select_sections
// Purpose:	bring up a list of available groups for the user to select
// Input:	none
// Output:	glist is updated
// Author:	Greg Shaw
// Created:	4/10/96

int Message::select_sections(void)      // select sections for grouping
{
	SectionHdr	sec;
	int		recnum;
	Group		*found;

	sstrcrl("MSSELECTSEC");
	// go through the groups, prompting whether to add
	recnum = 0;
	while (!msql.iterate_header(&sec,recnum,0))
	{
		found = check_grouplist(sec.table_name);
		sstr_c(sec.long_name);
		sstr_c(" ");
		if (yesno())
		{
			if (found == NULL)	// not already in list?
				add_to_grouplist(sec.table_name,0,0,1);	// add to grouplist
			else
				found->selected = 1;
		}
		else
		{
			if (found != NULL)
				found->selected = 0;
		}
		recnum++;
	}
	return(0);
}

// Function:	show_groups
// Purpose:	show the groups that the user has selected for group read
// Input:	none
// Output:	the list of sections
// Author:	Greg Shaw
// Created:	4/10/96

int Message::show_groups(void)  // show groups currently selected
{
	Group *gptr;
	int	found;	// any selected groups found?
	SectionHdr sec;

	found = 0;
	clear_scr();
	sstrcrl("MSCURSELECTED");
	for (gptr = glist; gptr != NULL; gptr = gptr->next)
	{
		if (gptr->selected)
		{
			found++;
			msql.get_header(gptr->section,&sec,1);	// get header information
			sstrcr(sec.long_name);
		}
	}
	cr();
	if (!found)
		sstrcrl("MSNOSECTIONS");
	waitcr();
	return(0);
}

// Function:	view
// Purpose:	view a thread of messages
// Input:	section - the section the thread is in
//		thread_number - the thread number to view
//		high_message - the highest message that the user has read
// Output:	the number of messages posted
// Author:	Greg Shaw
// Created:	3/17/96

int Message::view(char *section, int thread_number, int high_msg)
{
	int	dir; 		// direction of read
	char	c;		// input char
	Msg	*cur;		// current message
	FILE	*mfile;		// file for viewing message
	time_t	then;		// last 'move'
	int	num_messages;	// number of messages in thread
	int	cur_message;	// current message (ordinal)
	int 	done;
	int	posted = 0;	// number of posted messages
	char	*bcast;		// a broadcast message (from broadcast object)
	char	fname[MAXPATHLEN];	// name of output file
	char	tmpstr[255];
	char	validchars[] = "bfderq";
	SectionHdr	sec;	// current section header
	char   bsstr[] =            // backspace
		{
			0x8,0x20,0x8,0x0
		};


	msql.get_header(section,&sec,1);	// get header information
	cur_message = -1;	// before first record
	// select messages in thread greater than high_message
	num_messages = msql.get_messages(section,thread_number,high_msg);
	if (num_messages == 0)
	{
		sstrcrl("MSNOMES");
		waitcr();
		return(0);
	}
	dir = 1;	// move forward to start
	cur = NULL;	// no current message (yet)
	for (done = 0; !done;)
	{
		switch(dir)
		{
		case -1:	// move backwards
			if (cur_message > 0)
				cur_message--;
			else
			{
				sstrcrl("MSNOMOREMES");
				waitcr();
				break;	// can't move beyond first record
			}
			if (cur != NULL)
				msql.clear_message(&cur);
			cur = msql.previous();
			break;
		case 0:	// redisplay current message
			break;
		case 1:	// move	forward 
			if (cur_message < num_messages-1)
				cur_message++;
			else
			{
				sstrcrl("MSNOMOREMES");
				waitcr();
				break;	// can't move beyond last record
			}
			if (cur != NULL)
				msql.clear_message(&cur);
			cur = msql.next();
			if (high_msg < cur->message_number)
				high_msg = cur->message_number;
			break;
		}
		strcpy(fname,"/tmp/msgXXXXXX");
		mktemp(fname);
		// build file with message
		if (mfile = bopen(fname,"w"), mfile == NULL)
		{
			ap_log("Unable to write message view file.");
			return(0);
		}
		fprintf(mfile,"%s\n",process(QTOQUIT));
		if (num_messages - cur_message-1 == 0)	// last message?
			fprintf(mfile,process(gstrl("MSVIEWLASTTITLE")),cur->message_number,sec.long_name,'\n');
		else
			fprintf(mfile,process(gstrl("MSVIEWTITLE")),cur->message_number,sec.long_name,num_messages - cur_message-1,'\n');
		fprintf(mfile,process(gstrl("MSFROM")),cur->from_,'\n');
		fprintf(mfile,process(gstrl("MSTO")),cur->to,'\n');
		fprintf(mfile,process(gstrl("MSMESSUBJECT")),cur->subject,'\n');
		fprintf(mfile,process(gstrl("MSDATE")),ctime(&cur->date),'\n');
		fprintf(mfile,process(gstrl("MSTEXT")),cur->text,'\n');
		bclose(mfile);
		// display file using system pager
		sprintf(tmpstr,"%s %s",sys_pager(),fname);
		clear_scr();
		sysint(tmpstr,0,0);
		if (unlink(fname) < 0)
		{
			sprintf(tmpstr,"Unable to unlink %s",fname);
			ap_log(tmpstr);
		}
		// put up prompt
		sstrl("MSVIEWPROMPT");
		time(&then);
		while (c = tolower(gch(1)), c == 0 || strchr(validchars,c) == NULL)
		{
                        if (c != 0)         // invalid char?
                                sstr_c(bsstr);  // then erase it
                        if ((time(NULL) - then)/60 > inactivity_timeout())
                        {
                                return(0);
                        }
                        if (bcast = chatobj.check_broadcast(), bcast != NULL)
                        {
                                        // private request?
                                if (!chatobj.check_private(bcast,0,1))
                                {                   // nope, regular message
                                        clear_scr();
                                        sstrcrl("MESSAGERECEIVED");
                                        cr();
                                        sstrcrl(bcast);
                                        cr();
                                        cr();
                                        waitcr();
					continue;
                                }
                        }
		}
		cr();
		cr();
		switch(c)
		{

		case 'f':	// forward (next)
			dir = 1;
			break;
		case 'b':	// backward (previous)
			dir = -1;
			break;
		case 'r':	// reply to message
			posted += post(section,cur,0,NULL);
			break;
		case 'e':	// reply through email
			posted += post(PRIVATE_SECTION,cur,1,NULL);
			break;
		case 'd':	// delete message
			if (!delete_message(section,cur,1))
				done++;
			break;
		case '-':
		case 'q':	// quit
			done++;
			continue;
		}
	}
	high_message(section,high_msg);
	return(posted);
}

#endif // USE_DATABASE

#endif // _MESSAGE_C_

