/*͸
   OS/2 Video API Version 1.00  13-01-96    (C) 1996 by CodeLand Australia 
   WIN.C Window routines                               All Rights Reserved 
  ;*/

#include <malloc.h>
#include <string.h>

#define INCL_DOS
#define INCL_VIO
#include <os2.h>

#include "vid.h"
#include "win.h"

/* For WinHLine() and WinVLine() */
#define HORZ  0
#define VERT  1
#define ULC   bt[0]    /* upper left corner         */
#define UHL   bt[1]    /* upper horizontal line     */
#define URC   bt[2]    /* upper right corner        */
#define LVL   bt[3]    /* left vertical line        */
#define RVL   bt[4]    /* right vertical line       */
#define LLC   bt[5]    /* lower left corner         */
#define LHL   bt[6]    /* lower horizontal line     */
#define LRC   bt[7]    /* lower right corner        */
#define MJ    bt[8]    /* middle junction           */
#define LVJ   bt[9]    /* left vertical junction    */
#define RVJ   bt[10]   /* right vertical junction   */
#define UHJ   bt[11]   /* upper horizontal junction */
#define LHJ   bt[12]   /* lower horizontal junction */

/**/

/* Global window information record */
struct _winfo_t _WinInfo = {
    NULL,       /* Pointer to active window         */
    NULL,       /* Pointer to last hidden window    */
    NULL,       /* Pointer to head menu record      */
    NULL,       /* Pointer to current menu record   */
    NULL,       /* Pointer to help info record      */
    0,          /* Last handle given to a window    */
    500,        /* Max files allowed in wpickfile() */
    0,          /* Pointer to current help category */
    W_NOERROR,  /* Error num from last window func  */
    0,          /* Total number of open windows     */
    0,0,        /* System variables used in menus   */
    1,          /* Check for Esc in input funcions? */
    8,          /* Window TTY output tab width      */
    ' '         /* Character to fill windows with   */
};

static struct _wrec_t *ActCurr, *ActFound;
static USHORT ActCrow, ActCcol;
static struct _wrec_t *curr;

/**/

/* For WinActiv() */
static SHORT pascal bshadow_blocking (VOID);
static PSHORT pascal calc_bshadow (struct _wrec_t *wrec);
static PSHORT pascal calc_rshadow (struct _wrec_t *wrec);
static PSHORT pascal calc_window (struct _wrec_t *wrec);
static SHORT pascal rshadow_blocking (VOID);
static VOID pascal swap_contents (PSHORT pfound, PSHORT pcurr, SHORT shadow);
static SHORT pascal window_blocking (VOID);
/* For WinPutS() */
static PCHAR pascal process_esc (PCHAR str);
/* For WinHLine() and WinVLine() */
static SHORT pascal disp_char (SHORT wrow, SHORT wcol, SHORT attr, SHORT btype,
       SHORT ch, SHORT direc);
static SHORT pascal isupvert (SHORT Btype, SHORT C);
static SHORT pascal isdownvert (SHORT Btype, SHORT C);
static SHORT pascal islefthorz (SHORT Btype, SHORT C);
static SHORT pascal isrighthorz (SHORT Btype, SHORT C);
static SHORT pascal read_char (SHORT Row, SHORT Col);
static PUSHORT pascal read_line (USHORT row, USHORT col, USHORT count);

/**/

/* Checks validity of given window row coordinate */
SHORT WinChkRow (SHORT Row)
{
    return((Row<0||
        (Row>(((SHORT)_WinInfo.active->erow-(SHORT)_WinInfo.active->border)-
        ((SHORT)_WinInfo.active->srow+(SHORT)_WinInfo.active->border))))?1:0);
}

/**/

/* Checks validity of given window column coordinate */
SHORT WinChkCol (SHORT Col)
{
    return((Col<0||
        (Col>(((SHORT)_WinInfo.active->ecol-(SHORT)_WinInfo.active->border)-
        ((SHORT)_WinInfo.active->scol+(SHORT)_WinInfo.active->border))))?1:0);
}

/**/

/* Checks validity of given window coordinates */
SHORT WinChkCoord (SHORT Row, SHORT Col)
{
    return ((WinChkRow(Row)||WinChkCol(Col))?1:0);
}

/**/

/* Sets window cursor coordinates */
SHORT WinGotoXY (SHORT Wrow, SHORT Wcol)
{
    SHORT row, col;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for valid cursor coordinates */
    if(WinChkCoord(Wrow,Wcol)) return (_WinInfo.errno=W_INVCOORD);

    /* Calculate effective cursor coordinates */
    row=(SHORT)_WinInfo.active->srow+Wrow+_WinInfo.active->border;
    col=(SHORT)_WinInfo.active->scol+Wcol+_WinInfo.active->border;

    /* Update window record */
    _WinInfo.active->row=(UCHAR)row;
    _WinInfo.active->column=(UCHAR)col;

    /* Set cursor location */
    VidGotoXY(row,col);

    /* Return with no error */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Gives active window a shadow  */
SHORT WinShadow (SHORT Attr)
{
    SHORT *wsbuf,*q;
    SHORT ccol, crow, srow, scol, erow, ecol, stop, chat1, chat2;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* See if window already has a shadow */
    if(_WinInfo.active->wsbuf!=NULL) return (_WinInfo.errno=W_NOERROR);

    /* get window coordinates from the window's record */
    srow=(SHORT)_WinInfo.active->srow;
    scol=(SHORT)_WinInfo.active->scol;
    erow=(SHORT)_WinInfo.active->erow;
    ecol=(SHORT)_WinInfo.active->ecol;

    /* allocate buffer to hold shadow's contents */
    if((wsbuf=malloc((((erow-srow)*sizeof(SHORT))+ecol-scol+1)*sizeof(SHORT)))
        ==NULL) return(_WinInfo.errno=W_ALLOCERR);

    /* Check for/adjust to monochrome attributes */
    Attr=VidMapAttr(Attr);

    /* Start at upper right corner of shadow and work down */
    crow=srow+1; ccol=ecol+1; q=wsbuf;

    /* Draw shadow to right of window */
    while(crow<=erow) {
        /* Read current screen characters/attributes */
        chat1=VidReadChAt(crow,ccol);
        chat2=VidReadChAt(crow,ccol+1);

        /* Save in shadow's buffer */
        *q++=chat1; *q++=chat2;

        /* Write characters back to screen using shadow's attribute */
        VidPrintC(crow,ccol,(chat1&0x8000)?(Attr|BLINK):Attr,(CHAR)chat1);
        VidPrintC(crow++,ccol+1,(chat2&0x8000)?(Attr|BLINK):Attr,(CHAR)chat2);
    }

    /* Start at lower left corner of shadow and work right */
    crow=erow+1; ccol=scol+2; stop=ecol+2;

    /* Draw bottom shadow */
    while(ccol<=stop) {
        /* Read current screen character/attribute string */
        chat1=VidReadChAt(crow,ccol);

        /* Save in shadow's buffer */
        *q++=chat1;

        /* Write character back to screen using shadow's attribute */
        VidPrintC(crow,ccol++,(chat1&0x8000)?(Attr|BLINK):Attr,(CHAR)chat1);
    }

    /* Save info in window's record */
    _WinInfo.active->wsbuf=wsbuf;
    _WinInfo.active->wsattr=(UCHAR)Attr;

    /* Return with no error */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Removes shadow from active window */
SHORT WinShadOff (VOID)
{
    SHORT *q, ccol, crow, srow, scol, erow, ecol, stop, chat;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* If window doesn't have a shadow, ignore request */
    if(_WinInfo.active->wsbuf==NULL) return (_WinInfo.errno=W_NOERROR);

    /* Get window coordinates from the window's record */
    srow=(SHORT)_WinInfo.active->srow;
    scol=(SHORT)_WinInfo.active->scol;
    erow=(SHORT)_WinInfo.active->erow;
    ecol=(SHORT)_WinInfo.active->ecol;

    /* Start at upper right corner of shadow and work down */
    crow=srow+1; ccol=ecol+1; q=_WinInfo.active->wsbuf;

    /* Delete shadow to right of window */
    while(crow<=erow) {
        chat=*q++; VidPrintC(crow,ccol,chat>>8,(CHAR)chat);
        chat=*q++; VidPrintC(crow++,ccol+1,chat>>8,(CHAR)chat);
    }

    /* Start at lower left corner of shadow and work right */
    crow=erow+1; ccol=scol+2; stop=ecol+2;

    /* Delete bottom shadow */
    while(ccol<=stop) {
        chat=*q++; VidPrintC(crow,ccol++,chat>>8,(CHAR)chat);
    }

    /* Free memory held by shadow */
    free(_WinInfo.active->wsbuf);

    /* Update window's record */
    _WinInfo.active->wsbuf=NULL;
    _WinInfo.active->wsattr=0xff;

    /* Return with no error */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Opens a window and makes it active */
WINDOW WinOpen (SHORT Srow, SHORT Scol, SHORT Erow, SHORT Ecol,
                SHORT Btype, SHORT Battr,SHORT Wattr)
{
    PSHORT wbuf;
    struct _wrec_t *wrec;
    SHORT border;

    /* Check for valid box type */
    if(Btype<0 || Btype>5) {
        _WinInfo.errno=W_INVBTYPE;
        return 0;
    }

    /* See if window is to have a border */
    border=((Btype==5)?0:1);

    /* Check for valid coordinates */
    if(Srow>(Erow-border) || Scol>(Ecol-border)) {
        _WinInfo.errno=W_INVCOORD;
        return 0;
    }

    /* Check for monochrome adapter, adjust attributes */
    Battr=VidMapAttr(Battr); Wattr=VidMapAttr(Wattr);

    /* Allocate memory for new record */
    wrec=malloc(sizeof(struct _wrec_t));
    if(wrec==NULL) {
        _WinInfo.errno=W_ALLOCERR;
        return 0;
    }

    /* Save affected area of screen */
    if((wbuf=WinSave(Srow,Scol,Erow,Ecol))==NULL) {
        free(wrec);
        _WinInfo.errno=W_ALLOCERR;
        return 0;
    }

    /* Add new record to linked list */
    if(_WinInfo.active!=NULL) _WinInfo.active->next=wrec;
    wrec->prev=_WinInfo.active;
    wrec->next=NULL;
    _WinInfo.active=wrec;

    /* Draw and fill text box on screen */
    if(border) VidBox(Srow,Scol,Erow,Ecol,Btype,Battr);
    VidFill(Srow+border,Scol+border,Erow-border,Ecol-border,_WinInfo.fillch,
        Wattr);

    /* Increment window handle counter */
    _WinInfo.handle++;

    /* Save window info in window record */
    _WinInfo.active->wbuf=wbuf;
    _WinInfo.active->whandle=_WinInfo.handle;
    _WinInfo.active->srow=(UCHAR)Srow;
    _WinInfo.active->scol=(UCHAR)Scol;
    _WinInfo.active->erow=(UCHAR)Erow;
    _WinInfo.active->ecol=(UCHAR)Ecol;
    _WinInfo.active->btype=(UCHAR)Btype;
    _WinInfo.active->wattr=(UCHAR)Wattr;
    _WinInfo.active->battr=(UCHAR)Battr;
    _WinInfo.active->border=(UCHAR)border;
    _WinInfo.active->row=(UCHAR)Srow+(UCHAR)border;
    _WinInfo.active->column=(UCHAR)Scol+(UCHAR)border;
    _WinInfo.active->attr=(UCHAR)Wattr;
    _WinInfo.active->title=NULL;
    _WinInfo.active->tpos='\0';
    _WinInfo.active->help=0;
    _WinInfo.active->form=NULL;
    _WinInfo.active->wsbuf=NULL;

    /* Increment total number of open windows */
    _WinInfo.total++;

    /* initialize cursor location to window row 0 column 0 */
    WinGotoXY(0,0);

    /* Return normally */
    _WinInfo.errno=W_NOERROR;
    return _WinInfo.handle;
}

/**/

/* Closes a window */
SHORT WinClose (VOID)
{
    struct _wrec_t *wrec;

    /* Check for active window */
    if(!_WinInfo.total) return(_WinInfo.errno=W_NOACTIVE);

    /* If window has a shadow, close shadow first */
    if(_WinInfo.active->wsbuf!=NULL) WinShadOff();

    /* Restore contents of and free memory held by window */
    WinRestore(_WinInfo.active->wbuf);

    /* Decrement total number of open windows */
    _WinInfo.total--;

    /* Free memory held by window's record and update linked list */
    wrec=_WinInfo.active->prev;
    free(_WinInfo.active);
    _WinInfo.active=wrec;
    if(_WinInfo.active!=NULL) _WinInfo.active->next=NULL;

    /* Update cursor location and help category */
    if(_WinInfo.active!=NULL) {
        VidGotoXY(_WinInfo.active->row,_WinInfo.active->column);
        if(_WinInfo.active->help) _WinInfo.help=_WinInfo.active->help;
    }
    else {
        /* Restore original cursor location */
        VidGotoXY(_VidInfo.CurRow,_VidInfo.CurCol);
        /* Restore cursor size */
        VidSetCurSz(_VidInfo.CurSSt,_VidInfo.CurSEd);
    }

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Closes all open windows */
SHORT WinCloseAll (VOID)
{
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    while(_WinInfo.total) if(WinClose()) return(_WinInfo.errno);

    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Gives active window a title */
SHORT WinTitle (PCHAR Str, SHORT Tpos, SHORT Tattr)
{
    SHORT left, right, start, len, width, offs;
    PCHAR p;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for monochrome adapter, adjust title attribute */
    Tattr=VidMapAttr(Tattr);

    /* Redraw box if deleting or moving title */
    if(Str==NULL||_WinInfo.active->title!=NULL) {
        if(_WinInfo.active->border)
            VidBox((SHORT)_WinInfo.active->srow,(SHORT)_WinInfo.active->scol,
                (SHORT)_WinInfo.active->erow,(SHORT)_WinInfo.active->ecol,
                (SHORT)_WinInfo.active->btype,(SHORT)_WinInfo.active->battr);
    }

    /* If not deleting the title, calculate position and display it */
    if(Str!=NULL) {
        left=(SHORT)_WinInfo.active->scol+1;
        right=(SHORT)_WinInfo.active->ecol-1;
        width=right-left+1;
        len=strlen(Str);

        /* Don't display title if window is borderless */
        if(_WinInfo.active->border) {

            switch(Tpos) {
                case TLEFT:
                    start=(len>(width-3))?left:(left+1);
                    break;
                case TCENTER:
                    start=(len>(width-2))?left:(((width/2)+left)-(len/2));
                    break;
                default:        /* default is TRIGHT */
                    offs=width-len;
                    if(offs>2) offs--;
                    start=(len>width)?left:(left+offs);
            }

            /* Allocate space for window title string, and copy it there */
            if((p=malloc(((width>len)?width:len)+1))==NULL)
                return (_WinInfo.errno=W_ALLOCERR);
            strcpy(p,Str);
            *(p+width)='\0';

            /* display title string */
            VidPrintS(_WinInfo.active->srow,start,Tattr,p);

            free(p); /* Free allocated space */
        }
    }

    /* Update window's record */
    _WinInfo.active->title=Str;
    _WinInfo.active->tpos=(UCHAR)Tpos;
    _WinInfo.active->tattr=(UCHAR)Tattr;

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Checks for valid window box coordinates */
SHORT WinChkBox (SHORT Srow, SHORT Scol, SHORT Erow, SHORT Ecol)
{
    return((WinChkCoord(Srow,Scol)||WinChkCoord(Erow,Ecol)||
        (Srow>Erow)||(Scol>Ecol))?1:0);
}

/**/

/* Displays a text box inside active window */
SHORT WinBox (SHORT Srow, SHORT Scol, SHORT Erow, SHORT Ecol, SHORT Btype, SHORT Attr)
{
    SHORT border;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for valid coordinates */
    if(WinChkBox(Srow,Scol,Erow,Ecol)) return(_WinInfo.errno=W_INVCOORD);

    /* Check for window border */
    border=(SHORT)_WinInfo.active->border;

    /* Display the text box */
    VidBox((SHORT)_WinInfo.active->srow+Srow+border,
        (SHORT)_WinInfo.active->scol+Scol+border,
        (SHORT)_WinInfo.active->srow+Erow+border,
        (SHORT)_WinInfo.active->scol+Ecol+border,
         Btype,Attr);

    /* Return with no error */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Fills a region of active window w/specified char/attribute */
SHORT WinFill (SHORT Srow, SHORT Scol, SHORT Erow, SHORT Ecol, SHORT Ch, SHORT Attr)
{
    SHORT border;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for valid coordinates */
    if(WinChkBox(Srow,Scol,Erow,Ecol)) return (_WinInfo.errno=W_INVCOORD);

    /* Check for window border */
    border=(SHORT)_WinInfo.active->border;

    /* Fill in specified region */
    VidFill((SHORT)_WinInfo.active->srow+Srow+border,
        (SHORT)_WinInfo.active->scol+Scol+border,
        (SHORT)_WinInfo.active->srow+Erow+border,
        (SHORT)_WinInfo.active->scol+Ecol+border,
        Ch,Attr);

    /* Return with no error */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/*  Restores a previously saved window to the screen */
VOID WinRestore (PUSHORT wbuf)
{
    USHORT crow;
    USHORT *q, srow, scol, erow, ecol, bytes, width;

    /* Get coordinates from window's buffer */
    q=wbuf; srow=*q++; scol=*q++; erow=*q++; ecol=*q++;

    /* Calculate info on window */
    width=ecol-scol+1; bytes=width*2;

    /* Do for all rows in window */
    for(crow=srow;crow<=erow;crow++) {
        VioWrtCellStr((PCH)q,bytes,crow,scol,0);
        q+=width;
    }

    free(wbuf); /* Free window's buffer */
}

/**/

/*  Saves a window of the screen */
PUSHORT WinSave (USHORT srow, USHORT scol, USHORT erow, USHORT ecol)
{
    USHORT *q, *wbuf, crow, bytes, width;

    /* Allocate buffer to hold window's contents */
    if((wbuf=malloc(((erow-srow+1)*(ecol-scol+1)+4)*sizeof(USHORT)))!=NULL) {
        /* Calculate info on window */
        width=ecol-scol+1; bytes=width*2;

        /* Write window's coordinates to allocated buffer */
        q=wbuf; *q++=srow; *q++=scol; *q++=erow; *q++=ecol;

        /* Do for all rows in window */
        for(crow=srow;crow<=erow;crow++) {
            VioReadCellStr((PCH)q,&bytes,crow,scol,0);
            q+=width;
        }
    }

    return wbuf; /* Return address of window's buffer */
}

/**/

/* Displays a character inside active window */
SHORT WinPrintC (SHORT Row, SHORT Col, SHORT Attr, SHORT Ch)
{
    SHORT row, col;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for valid coordinates */
    if(WinChkCoord(Row,Col)) return (_WinInfo.errno=W_INVCOORD);

    /* Calculate effective coordinates */
    row=(SHORT)Row+_WinInfo.active->srow+_WinInfo.active->border;
    col=(SHORT)Col+_WinInfo.active->scol+_WinInfo.active->border;

    /* Check for monochrome adapter, adjust attribute */
    Attr=VidMapAttr(Attr);

    /* Display the character */
    VidPrintC(row,col,Attr,(CHAR)Ch);

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}
/**/

/* Displays a string inside active window */
SHORT WinPrintS (SHORT Row, SHORT Col, SHORT Attr, PCHAR Str)
{
    SHORT row, col, max, border;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for valid coordinates */
    if(WinChkCoord(Row,Col)) return (_WinInfo.errno=W_INVCOORD);

    /* See if window has border */
    border=(SHORT)_WinInfo.active->border;

    /* Calculate effective coordinates */
    row=(SHORT)_WinInfo.active->srow+Row+border;
    col=(SHORT)_WinInfo.active->scol+Col+border;
    max=(((SHORT)_WinInfo.active->ecol-border)-col+1);

    /* See if wraparound is needed - if not, use faster prints() function */
    if(strlen(Str)<=(USHORT)max) {
        VidPrintS(row,col,Attr,Str);
        return (_WinInfo.errno=W_NOERROR);
    }

    /* Display as much of string as possible */
    while(*Str&&max--) VidPrintC(row,col++,Attr,*Str++);

    return (_WinInfo.errno=W_STRLONG);
}

/**/

/* Draws a horizontal text line in active window */
SHORT WinHLine (SHORT wsrow, SHORT wscol, SHORT count, SHORT btype, SHORT attr)
{
    PCHAR bt;
    SHORT row, col, up, down;
    CHAR ch;
    SHORT lrow, lcol, lcount;
    SHORT *line1, *line2;
    CHAR cell[2];

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for valid box type */
    if(btype<0||btype>5) return(_WinInfo.errno=W_INVBTYPE);
    row=wsrow; col=wscol; lcount=count;

    lrow=(SHORT)_WinInfo.active->srow+row+(SHORT)_WinInfo.active->border;
    lcol=(SHORT)_WinInfo.active->scol+col+(SHORT)_WinInfo.active->border;

    /* Abbreviate pointer */
    bt=_box_table[btype];

    /* Predraw the line */
    cell[1]=(CHAR)VidMapAttr(attr);
    cell[0]=(CHAR)UHL;
    VioWrtNCell(cell,count,lrow,lcol,0);

    if(col>0) { lcol--; lcount++; }
    line1=read_line(lrow-1,lcol,lcount);
    line2=read_line(lrow+1,lcol,lcount);
    if(line1==NULL||line2==NULL) return (_WinInfo.errno=W_ALLOCERR);

    if(count) {

        /* See if a left junction or corner is needed */
        up=isupvert(btype,line1[col]);
        down=isdownvert(btype,line2[col]);
        if(up&&down) ch=LVJ;
        else if(up) ch=LLC;
        else if(down) ch=ULC;
        else ch=UHL;

        /* Display leftmost character */
        if(disp_char(row,col,attr,btype,ch,HORZ)) {
            free(line1); free(line2);
            return (_WinInfo.errno);
        }
        col++;
        count--;
    }

    /* Do while not last character */
    while(count>1) {

        /* See if a middle junction is needed */
        up=isupvert(btype,line1[col]);
        down=isdownvert(btype,line2[col]);
        if(up&&down) ch=MJ;
        else if(up) ch=LHJ;
        else if(down) ch=UHJ;
        else ch=UHL;

        /* Display middle character */
        if(disp_char(row,col,attr,btype,ch,HORZ)) {
            free(line1); free(line2);
            return (_WinInfo.errno);
        }
        col++;
        count--;
    }

    if(count) {

        /* See if a right junction or corner is needed */
        up=isupvert(btype,line1[col]);
        down=isdownvert(btype,line2[col]);
        if(up&&down) ch=RVJ;
        else if(up) ch=LRC;
        else if(down) ch=URC;
        else ch=UHL;

        /* Display rightmost character */
        if(disp_char(row,col,attr,btype,ch,HORZ)) {
            free(line1); free(line2);
            return (_WinInfo.errno);
        }
    }

    free(line1); free(line2);

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Draws a vertical text line in active window */
SHORT WinVLine (SHORT wsrow, SHORT wscol, SHORT count, SHORT btype, SHORT attr)
{
    PCHAR bt;
    SHORT col;
    SHORT row, left, right;
    CHAR ch;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for valid box type */
    if(btype<0||btype>5) return (_WinInfo.errno=W_INVBTYPE);
    row=wsrow; col=wscol;

    /* Abbreviate pointer */
    bt=_box_table[btype];

    if(count) {

        /* See if a top junction or corner is needed */
        left=islefthorz(btype,read_char(row,col-1));
        right=isrighthorz(btype,read_char(row,col+1));
        if(left&&right) ch=UHJ;
        else if(left) ch=URC;
        else if(right) ch=ULC;
        else ch=LVL;

        /* Display uppermost character */
        if(disp_char(row,col,attr,btype,ch,VERT)) return (_WinInfo.errno);
        row++;
        count--;
    }

    /* Do while not last character */
    while(count>1) {
        left=islefthorz(btype,read_char(row,col-1));
        right=isrighthorz(btype,read_char(row,col+1));
        if(left&&right) ch=MJ;
        else if(left) ch=RVJ;
        else if(right) ch=LVJ;
        else ch=LVL;

        /* Display middle character */
        if(disp_char(row,col,attr,btype,ch,VERT)) return (_WinInfo.errno);
        row++;
        count--;
    }

    if(count) {

        /* see if a bottom junction or corner is needed */
        left=islefthorz(btype,read_char(row,col-1));
        right=isrighthorz(btype,read_char(row,col+1));
        if(left&&right) ch=LHJ;
        else if(left) ch=LRC;
        else if(right) ch=LLC;
        else ch=LVL;

        /* Display bottommost character */
        if(disp_char(row,col,attr,btype,ch,VERT)) return (_WinInfo.errno);
    }

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Displays a string centered in active window */
SHORT WinCenters (SHORT wrow, SHORT attr, PCHAR str)
{
    SHORT window_width, string_length;
    SHORT start_column, border;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for valid row */
    if(WinChkCoord(wrow,0)) return (_WinInfo.errno=W_INVCOORD);

    /* Check for window border */
    border=(SHORT)_WinInfo.active->border;

    /* Calculate start column & window width */
    start_column=(SHORT)_WinInfo.active->scol+border;
    window_width=((SHORT)_WinInfo.active->ecol-border)-start_column+1;

    /* Check length of input string */
    string_length=strlen(str);
    if(string_length>window_width) return (_WinInfo.errno=W_STRLONG);

    /* Display the string */
    VidPrintS(_WinInfo.active->srow+wrow+border,
          ((window_width/2)+start_column)-(string_length/2),attr,str);

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Returns the handle of the active window */
WINDOW WinHandle (VOID)
{
    /* Test for active window */
    if(!_WinInfo.total) {
        _WinInfo.errno=W_NOACTIVE;
        return 0;
    }

    /* Return normally */
    _WinInfo.errno=W_NOERROR;
    return (_WinInfo.active->whandle);
}

/**/

SHORT WinTextAttr (SHORT Attr)
{
    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Update window's record */
    _WinInfo.active->attr=(UCHAR)VidMapAttr(Attr);

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Clears from cursor postion to end of window's line  */
SHORT WinClrEol (VOID)
{
    SHORT col, stopcol;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Calculate stop column */
    stopcol=_WinInfo.active->ecol-_WinInfo.active->border;

    /* Clear to end of window's line */
    for(col=_WinInfo.active->column;col<=stopcol;col++)
        VidPrintC(_WinInfo.active->row,col,_WinInfo.active->attr,
            _WinInfo.fillch);

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Clears from cursor postion to end of window */
SHORT WinClrEos (VOID)
{
    SHORT wrow, werow;
    SHORT wr, wc;

    /* Check for active window */
    if(!_WinInfo.total) return(_WinInfo.errno=W_NOACTIVE);

    /* Save current window row and column */
    WinReadCur(&wr,&wc);

    wrow=wr;
    werow=_WinInfo.active->erow - _WinInfo.active->srow -
        _WinInfo.active->border;

    WinClrEol(); wrow++;

    while(wrow<=werow) {
        WinGotoXY(wrow,0);
        WinClrEol();
        wrow++;
    }

    /* Restore window row and column */
    WinGotoXY(wr,wc);

    return (_WinInfo.errno=W_NOERROR); /* Return normally */
}

/**/

/* Scrolls active window up or down  */
SHORT WinScroll (SHORT count, SHORT direction)
{
    SHORT numrows, border;
    BYTE bCell[2];

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for window border */
    border=_WinInfo.active->border;

    /* Check scroll count against number of rows in scroll boundary */
    numrows= _WinInfo.active->erow-border - _WinInfo.active->srow+border + 1;
    if(count > numrows) count=numrows;

    bCell[0]=' '; bCell[1]=(UCHAR)_WinInfo.active->wattr; /* Attribute */

    if(direction==SDOWN) {
        VioScrollDn(
            _WinInfo.active->srow+border,   /* Top row               */
            _WinInfo.active->scol+border,   /* Left column           */
            _WinInfo.active->erow-border,   /* Bottom row            */
            _WinInfo.active->ecol-border,   /* Right column          */
            count,                          /* Number of lines       */
            bCell,                          /* Cell to write         */
            0);                             /* AVIO video handle     */
    }
    else {
        VioScrollUp(
            _WinInfo.active->srow+border,   /* Top row               */
            _WinInfo.active->scol+border,   /* Left column           */
            _WinInfo.active->erow-border,   /* Bottom row            */
            _WinInfo.active->ecol-border,   /* Right column          */
            count,                          /* Number of lines       */
            bCell,                          /* Cell to write         */
            0);                             /* AVIO video handle     */
    }

    /* Return with no error */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Reads current cursor location inside window */
SHORT WinReadCur (PSHORT wrow, PSHORT wcol)
{
    SHORT row, col;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Read effective cursor coordinates */
    VidReadCur(&row,&col);

    /* Calculate window cursor coordinates */
    *wrow=row-_WinInfo.active->srow-_WinInfo.active->border;
    *wcol=col-_WinInfo.active->scol-_WinInfo.active->border;

    return (_WinInfo.errno=W_NOERROR); /* Return normally */
}

/**/

/* Clears the active window in specified attribute */
SHORT WinCClear (SHORT attr)
{
    SHORT border;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Check for window border */
    border=_WinInfo.active->border;

    VidFill(_WinInfo.active->srow+border,_WinInfo.active->scol+border,
            _WinInfo.active->erow-border,_WinInfo.active->ecol-border,
            _WinInfo.fillch,attr);

    WinGotoXY(0,0); /* Home the cursor */

    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Returns address of record of given window handle  */
struct _wrec_t * WinFindRec (SHORT whandle)
{
    struct _wrec_t *wrec;

    /* Scan through linked list for record belonging to requested handle */
    wrec=_WinInfo.active;
    while(wrec!=NULL) {
        if(whandle==wrec->whandle) break;
        wrec=wrec->prev;
    }

    /* Return address of found record */
    return wrec;
}

/**/

/* Display a character specified number of times */
SHORT WinDupC (SHORT ch, SHORT count)
{
    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Display ch for count times */
    while(count--) WinPutC(ch);

    /* Return with _WinInfo.errno set by wputc() */
    return (_WinInfo.errno);
}

/**/

/* Displays a character inside active window */
SHORT WinPutC (SHORT ch)
{
    SHORT ccol, crow, border, cwcol, scol;

    /* Check for active window */
    if(!_WinInfo.total) return(_WinInfo.errno=W_NOACTIVE);

    /* Get coordinates from window's record */
    crow=_WinInfo.active->row;
    ccol=_WinInfo.active->column;
    scol=_WinInfo.active->scol;
    border=_WinInfo.active->border;

    /* Test the input character for control characters */
    switch(ch) {
        case LF:
            crow++;
        case CR:
            ccol=scol+border;
            break;
        case BS:
            if(ccol==(scol+border)) {
                ccol=_WinInfo.active->ecol-border;
                crow--;
                if(crow<(_WinInfo.active->srow+border)) crow++;
            }
            else {
                ccol--;
            }
            break;
        case HT:
            cwcol=ccol-border-scol;
            ccol+=(VidTabStop(cwcol,_WinInfo.tabwidth)-cwcol);
            break;
        case BEL:
            VidBeep();
            break;
        default:
            VidGotoXY(crow,ccol);
            VidPrintC(crow,ccol,_WinInfo.active->attr,(CHAR)ch);
            ccol++;
    }

    /* See if wrap-around is needed */
    if(ccol>(_WinInfo.active->ecol-border)) {
        ccol=scol+border;
        crow++;
    }

    /* See if scroll is needed */
    if(crow>(_WinInfo.active->erow-border)) {
        WinScroll(1,SUP);
        crow--;
    }

    /* Update window's record */
    _WinInfo.active->row=(UCHAR)crow;
    _WinInfo.active->column=(UCHAR)ccol;

    /* Reset cursor position */
    VidGotoXY(crow,ccol);

    /* return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Displays a string inside active window */
SHORT WinPutS (PCHAR str)
{
    PUCHAR ccol;
    PUCHAR crow;
    UCHAR cwcol, border, scol;
    PCHAR q;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* Get effective coordinates from window's record */
    crow  =&(_WinInfo.active->row);
    ccol  =&(_WinInfo.active->column);
    scol  =_WinInfo.active->scol;
    border=_WinInfo.active->border;

    /* Do while not end of string */
    for(q=str;*q;q++) {

        /* Test the input character for control characters */
        switch(*q) {
            case LF:
                (*crow)++;
            case CR:
                *ccol=scol+border;
                break;
            case BS:
                if(*ccol==(scol+border)) {
                    *ccol=_WinInfo.active->ecol-border;
                    (*crow)--;
                    if(*crow<(_WinInfo.active->srow+border)) (*crow)++;
                }
                else {
                    (*ccol)--;
                }
                break;
            case HT:
                cwcol=(*ccol)-border-scol;
                (*ccol)+=(VidTabStop(cwcol,_WinInfo.tabwidth)-cwcol);
                break;
            case BEL:
                VidBeep();
                break;
            case ESC:
                q=process_esc(q);
                break;
            default:
                VidGotoXY(*crow,*ccol);
                VidPrintC(*crow,*ccol,_WinInfo.active->attr,*q);
                (*ccol)++;
        }

        /* See if wrap-around is needed */
        if( (*ccol) > (_WinInfo.active->ecol-border) ) {
            *ccol=scol+border;
            (*crow)++;
        }

        /* See if scroll is needed */
        if( (*crow) > (_WinInfo.active->erow-border) ) {
            WinScroll(1,SUP);
            (*crow)--;
        }
    }

    /* Reset cursor position */
    VidGotoXY(*crow,*ccol);

    /* Return normally */
    return (_WinInfo.errno=W_NOERROR);
}

/**/

/* Activates a window */
SHORT WinActiv (WINDOW whandle)
{
    USHORT startcol, stopcol;
    struct _wrec_t *prev, *next;

    /* Check for active window */
    if(!_WinInfo.total) return (_WinInfo.errno=W_NOACTIVE);

    /* If window is already active, ignore request */
    if(whandle==_WinInfo.active->whandle) return (_WinInfo.errno=W_NOERROR);

    /* Find address of window's record */
    ActFound=WinFindRec(whandle);
    if(ActFound==NULL) return (_WinInfo.errno=W_NOTFOUND);

    /* Check every character position in window to activate */
    for(ActCrow=ActFound->srow;ActCrow<=ActFound->erow;ActCrow++) {
        for(ActCcol=ActFound->scol;ActCcol<=ActFound->ecol;ActCcol++) {

            /* Check all window records "above" window to activate */
            for(ActCurr=ActFound->next;ActCurr!=NULL;ActCurr=ActCurr->next) {

                /* See if current position in window to activate */
                /* is blocked by same position in test window    */
                if(window_blocking()) {

                    /* Calculate buffer addresses and swap contents */
                    swap_contents(calc_window(ActFound),calc_window(ActCurr),0);
                    break;
                }

                /* See if test window has a shadow */
                if(ActCurr->wsbuf!=NULL) {

                    /* See if shadow to the right of test window is        */
                    /* Blocking the current position of window to activate */
                    if(rshadow_blocking()) {
                        swap_contents(calc_window(ActFound),calc_rshadow(ActCurr),1);
                        break;
                    }

                    /* See if shadow to the bottom of test window is       */
                    /* blocking the current position of window to activate */
                    if(bshadow_blocking()) {
                        swap_contents(calc_window(ActFound),calc_bshadow(ActCurr),1);
                        break;
                    }
                }
            }
        }
    }

    /* If window to activate has a shadow, then check      */
    /* every character position in the shadow to see       */
    /* if it is blocked by another window or window shadow */
    if(ActFound->wsbuf!=NULL) {

        /* Search the right shadow of window to activiate */
        startcol=ActFound->ecol+1;
        stopcol=startcol+1;
        for(ActCrow=ActFound->srow+1;ActCrow<=ActFound->erow;ActCrow++) {
            for(ActCcol=startcol;ActCcol<=stopcol;ActCcol++) {

                /* Check all window records "above" shadow to activate */
                for(ActCurr=ActFound->next;ActCurr!=NULL;ActCurr=ActCurr->next) {

                    /* See if current position in shadow to activate */
                    /* is blocked by same position in current window */
                    if(window_blocking()) {

                        /* Calculate buffer addresses and swap contents */
                        swap_contents(calc_rshadow(ActFound),calc_window(ActCurr),2);
                        break;
                    }

                    /* See if test window has a shadow */
                    if(ActCurr->wsbuf!=NULL) {

                        /* See if current position of window to activate is */
                        /* blocked by the right shadow of the test window   */
                        if(rshadow_blocking()) {
                            swap_contents(calc_rshadow(ActFound),
                                          calc_rshadow(ActCurr),3);
                            break;
                        }

                        /* See if current position of window to activate is */
                        /* blocked by the bottom shadow of the test window  */
                        if(bshadow_blocking()) {

                            swap_contents(calc_rshadow(ActFound),
                                          calc_bshadow(ActCurr),3);
                            break;
                        }
                    }
                }
            }
        }

        /* Search bottom shadow */
        startcol=ActFound->scol+2;
        stopcol=ActFound->ecol+2;
        ActCrow=ActFound->erow+1;
        for(ActCcol=startcol;ActCcol<=stopcol;ActCcol++) {

            /* Check all window records "above" shadow to activate */
            for(ActCurr=ActFound->next;ActCurr!=NULL;ActCurr=ActCurr->next) {

                /* See if current position in shadow to activate */
                /* is blocked by same position in test window    */
                if(window_blocking()) {

                    /* Calculate buffer addresses and swap contents */
                    swap_contents(calc_bshadow(ActFound),calc_window(ActCurr),2);
                    break;
                }

                /* See if test window has a shadow */
                if(ActCurr->wsbuf!=NULL) {
                    if(rshadow_blocking()) {
                        swap_contents(calc_bshadow(ActFound),calc_rshadow(ActCurr),
                                      3);
                        break;
                    }
                    if(bshadow_blocking()) {
                        swap_contents(calc_bshadow(ActFound),calc_bshadow(ActCurr),
                                      3);
                        break;
                    }
                }
            }
        }
    }

    /* Re-link pointer to window record to be activated */
    prev=ActFound->prev;
    next=ActFound->next;
    if(prev!=NULL) prev->next=next;
    next->prev=prev;
    _WinInfo.active->next=ActFound;
    ActFound->prev=_WinInfo.active;
    ActFound->next=NULL;
    _WinInfo.active=ActFound;

    /* Update help category */
    if(_WinInfo.active->help) _WinInfo.help=_WinInfo.active->help;

    /* Reset cursor position */
    VidGotoXY(_WinInfo.active->row,_WinInfo.active->column);

    return (_WinInfo.errno=W_NOERROR); /* Return normally */
}

/**/

/* Detects if a given window's bottom shadow is blocking */
/* the current window or its shadow at specified coordinates */
static SHORT pascal bshadow_blocking (VOID)
{
    SHORT isblocking=0;

    if(ActCrow==(USHORT)(ActCurr->erow+1))
        if((ActCcol>=(USHORT)(ActCurr->scol+2)) &&
            (ActCcol<=(USHORT)(ActCurr->ecol+2)))
            isblocking=1;

    return isblocking;
}

/**/

static PSHORT pascal calc_bshadow (struct _wrec_t *wrec)
{
    return (wrec->wsbuf+((((ActCrow-wrec->srow-1)*2)+(ActCcol-wrec->scol-2))));
}

/**/

static PSHORT pascal calc_rshadow (struct _wrec_t *wrec)
{
    return (wrec->wsbuf+((((ActCrow-wrec->srow-1)*2)+(ActCcol-wrec->ecol-1))));
}

/**/

static PSHORT pascal calc_window (struct _wrec_t *wrec)
{
    return (wrec->wbuf+4+((ActCrow-wrec->srow)*(wrec->ecol-wrec->scol+1))+
           (ActCcol-wrec->scol));
}

/**/

/* Detects if a given window's right shadow is blocking */
/* the current window or its shadow at specified coordinates */
static SHORT pascal rshadow_blocking (VOID)
{
    SHORT isblocking=0;

    if(ActCcol==(USHORT)(ActCurr->ecol+1)||ActCcol==(USHORT)(ActCurr->ecol+2))
        if((ActCrow>=(USHORT)(ActCurr->srow+1))&&(ActCrow<=ActCurr->erow))
            isblocking=1;

    return isblocking;
}

/**/

/* Exchange the contents of the applicable buffers */
static VOID pascal swap_contents (PSHORT pfound, PSHORT pcurr, SHORT shadow)
{
    struct _wrec_t *wptr;
    SHORT temp;
    SHORT chat,attr;

    /* Display character from current position in window to */
    /* activate on the screen.  if character is part of a   */
    /* shadow, reflect the character on the screen.         */
    VidGotoXY(ActCrow,ActCcol);
    temp=VidReadChAt(ActCrow,ActCcol);
    if(shadow&2) *pcurr=(*pcurr&0xff00)|(temp&0x00ff);
    attr=*pcurr>>8;
    attr=((temp&0x8000&&shadow)?(attr|BLINK):attr);
    VidPrintC(ActCrow,ActCcol,attr,(CHAR)*pcurr);

    /* Let window position directly above position  */
    /* to activate have the character that it holds */
    *pcurr=*pfound;

    /* If current character position to activate will */
    /* activate over a shadow in another window       */
    if(shadow&1) {

        /* Resolve all shadows upwards */
        wptr=ActCurr;
        chat=(ActCurr->wsattr<<8)|(*pfound&0x00ff);
        for(ActCurr=ActCurr->next;ActCurr!=NULL;ActCurr=ActCurr->next) {
            if(window_blocking()) {
                *(calc_window(ActCurr))=chat;
                chat=temp;
                break;
            }
            else {
                if(bshadow_blocking())
                    *(calc_bshadow(ActCurr))=chat;
                else
                    if(rshadow_blocking())
                        *(calc_rshadow(ActCurr))=chat;
            }
        }
        temp=chat;
        ActCurr=wptr;
    }

    /* Let character position activated hold character */
    /* that was on the screen in the same position     */
    *pfound=temp;
}

/**/

/* Detects if given window is blocking the */
/* current window or its shadow at specified coordinates */
static SHORT pascal window_blocking (VOID)
{
    SHORT isblocking=0;

    if(ActCrow>=ActCurr->srow &&
       ActCrow<=ActCurr->erow &&
       ActCcol>=ActCurr->scol &&
       ActCcol<=ActCurr->ecol
    ) isblocking=1;

    return isblocking;
}

/**/

/* Process an Escape sequence when encountered - Used by WinPutS() */
static PCHAR pascal process_esc (PCHAR str)
{
    SHORT attr;
    PCHAR p;
    SHORT ch, wrow, wcol;

    attr=_WinInfo.active->attr;
    for(p=str;*p==ESC;p++) {
        switch(*++p) {
            case '+':   /* increase text attribute */
                WinTextAttr(++attr);
                break;
            case '-':   /* decrease text attribute */
                WinTextAttr(--attr);
                break;
            case 'A':   /* change attribute */
                WinTextAttr(*++p);
                break;
            case 'F':   /* change foreground attribute */
                WinTextAttr((*++p&7)|(attr&248));
                break;
            case 'B':   /* change background attribute */
                WinTextAttr((*++p&112)|(attr&143));
                break;
            case 'I':   /* toggle intensity bit */
                WinTextAttr((attr&8)?(attr&247):(attr|8));
                break;
            case 'L':   /* toggle blinking bit */
                WinTextAttr((attr&128)?(attr&127):(attr|128));
                break;
            case 'X':   /* reverse attribute */
                WinTextAttr(VidRevsAttr(attr));
                break;
            case 'R':   /* set cursor row */
                WinReadCur(&wrow,&wcol);
                WinGotoXY(*++p,wcol);
                break;
            case 'C':   /* set cursor column */
                WinReadCur(&wrow,&wcol);
                WinGotoXY(wrow,*++p);
                break;
            case 'E':   /* erase */
                switch(*++p) {
                    case 'W':   /* erase window */
                        WinClear();
                        break;
                    case 'S':   /* erase to end of window */
                        WinClrEos();
                        break;
                    case 'L':   /* erase to end of window's line */
                        WinClrEol();
                        break;
                }
                break;
            case 'D':   /* duplicate character */
                ch=*++p;
                WinDupC(ch,*++p);
                break;
            default:
                p--;
        }
    }

    return (--p);
}

/**/

static SHORT pascal disp_char (SHORT wrow, SHORT wcol, SHORT attr, SHORT btype, SHORT ch, SHORT direc)
{
    PCHAR bt;
    SHORT col, row;

    /* See if next to a border, if so, connect to it */
    if(_WinInfo.active->border) {

        /* Abbreviate pointer */
        bt=_box_table[btype];

        /* Calculate effective row and column */
        row=(SHORT)_WinInfo.active->srow+(SHORT)_WinInfo.active->border+wrow;
        col=(SHORT)_WinInfo.active->scol+(SHORT)_WinInfo.active->border+wcol;

        /* See if this is a horizontal or vertical line */
        if(direc==HORZ) {

            /* Make sure that the box type characters match */
            if(LVL==_box_table[_WinInfo.active->btype][3]) {

                /* Check left border */
                if(col==(_WinInfo.active->scol+1)) {
                    VidPrintC(row,_WinInfo.active->scol,attr,LVJ);
                    ch=UHL;
                }

                /* Check right border */
                if(col==(_WinInfo.active->ecol-1)) {
                    VidPrintC(row,_WinInfo.active->ecol,attr,RVJ);
                    ch=UHL;
                }
            }
        }
        else {

            /* Make sure that the box type characters match */
            if(UHL==_box_table[_WinInfo.active->btype][1]) {

                /* Check top border */
                if(row==(_WinInfo.active->srow+1)) {
                    VidPrintC(_WinInfo.active->srow,col,attr,UHJ);
                    ch=LVL;
                }

                /* Check bottom border */
                if(row==(_WinInfo.active->erow-1)) {
                    VidPrintC(_WinInfo.active->erow,col,attr,LHJ);
                    ch=LVL;
                }
            }
        }
    }

    /* Display character */
    if(ch!=UHL) if(WinPrintC(wrow,wcol,attr,ch)) return (_WinInfo.errno);

    /* Return normally */
    return 0;
}

/**/

static SHORT pascal isupvert (SHORT Btype, SHORT C)
{
    PCHAR bt;
    CHAR ch;

    bt=_box_table[Btype];
    ch=(CHAR)C;
    return((ch==LVL||ch==UHJ||ch==ULC||ch==URC||ch==LVJ||ch==RVJ||ch==MJ)
        ?1:0);
}

/**/

static SHORT pascal isdownvert (SHORT Btype, SHORT C)
{
    PCHAR bt;
    CHAR ch;

    bt=_box_table[Btype];
    ch=(CHAR)C;
    return((ch==LVL||ch==LHJ||ch==LLC||ch==LRC||ch==LVJ||ch==RVJ||ch==MJ)
        ?1:0);
}

/**/

static SHORT pascal islefthorz (SHORT Btype, SHORT C)
{
    PCHAR bt;
    CHAR ch;

    bt=_box_table[Btype];
    ch=(CHAR)C;
    return((ch==UHL||ch==LVJ||ch==LLC||ch==ULC||ch==UHJ||ch==LHJ||ch==MJ)
        ?1:0);
}

/**/

static SHORT pascal isrighthorz (SHORT Btype, SHORT C)
{
    PCHAR bt;
    CHAR ch;

    bt=_box_table[Btype];
    ch=(CHAR)C;
    return((ch==UHL||ch==RVJ||ch==LRC||ch==URC||ch==UHJ||ch==LHJ||ch==MJ)
        ?1:0);
}

/**/

/* Read character at window location */
static SHORT pascal read_char (SHORT Row, SHORT Col)
{
    CHAR achCell[2];
    USHORT cb;
    SHORT row, col;

    row=Row+_WinInfo.active->srow+_WinInfo.active->border;
    col=Col+_WinInfo.active->scol+_WinInfo.active->border;
    cb=2;

    VioReadCellStr(
        achCell,               /* Buffer for string                     */
        &cb,                   /* Points to variable for string length  */
        row,                   /* Starting location (row)               */
        col,                   /* Starting location (column)            */
        0);                    /* Video handle                          */

    return (SHORT)achCell[0];
}

/**/

/* Read a character string at location */
static PUSHORT pascal read_line (USHORT row, USHORT col, USHORT count)
{
    USHORT len=count*2;
    USHORT *buf;

    if((buf=malloc(len))!=NULL) VioReadCellStr((PCHAR)buf,&len,row,col,0);

    return buf;
}

/**/

