/*
 * ChkPro v1.0r4 - ProBoard Configuration Checking Utility
 *
 * Copyright (C) 1995 by Branislav L. Slantchev
 * A product of Silicon Creations, part of the PB-GNU Project
 *
 * For details, see the file "copying".
 *
*/
#include <stdio.h>
#include <string.h>
#include <proutil.h>
#include <fsys.h>
#include <grep.h>
#include <strlib.h>

/* local data structures */
static byte groupDef[32];  /* 32 * 8 = 256 groups (one bit per group) */

/* these functions are in the main module */
extern void _pascal log(const char *format, ...);
extern void _pascal logexit(const char *format, ...);

/* generalized function to check directory existence */
static int _pascal direxists(const char *path);

/*************************************************************************\
 ** Process the FILECFG.PRO file.
\*************************************************************************/
	int _pascal
do_filecfg(bool repair)
{
	FILE *fpSrc;
	int   areaNum, retval = 0;

	/* initialize the groups array, current directory and glob pattern */
	memset( groupDef, 0, sizeof(groupDef) );
	glob_compile( "FA_*.CTL" );

	/* open the area definition file and assign a large buffer */
	if( NULL == (fpSrc = pbfopen(fnFILECFG, "rb")) )
		logexit("Fatal: unable to open file '%s'\n", fnFILECFG);
	setvbuf(fpSrc, NULL, _IOFBF, 0x1000);

	for( areaNum = 1; ; areaNum++ ){
		FILECFG buf;
		int     i;

		/* read record and see if we're at the end of file */
		if( 1 != fread(&buf, sizeof(buf), 1, fpSrc) ){
			if( feof(fpSrc) ) break;
			logexit("fread() failed on '%s'\n", fnFILECFG);
		}

		/* check for empty file area */
		if( EOS == buf.name[0] ) continue;

		/* see if we have FA_???.CTL for the data path */
		if( 0 == glob_match(basename(buf.filepath)) ){
			char newpath[MAXPATH];

			sprintf( newpath, "FA_%d.CTL", areaNum );
			/* check for mismatch of the FA_???.CTL file */
			if( NULL == stristr(buf.filepath, newpath) ){
				log("Area %d: control file incorrect - '%s'\n",
					 areaNum, buf.filepath);
				retval = -1;
			}
		}
		/* if it has the CopyLocal flag on, assume remote/CDR area */
		else if( TRUE == buf.copyLocal ){
        	if( -1 == direxists(buf.filepath) ){
				log("Area %d: path '%s' does not exist...possibly offline\n",
						areaNum, buf.filepath);
			}
		}
		/* if not control file, check the data directory */
		else{
			if( -1 == direxists(buf.filepath) ){
				int stat;

				if( repair ) stat = mkDirTree(buf.filepath);
				log("Area %d: path '%s' does not exist%s\n",
					areaNum, buf.filepath, repair ?
					((-1 == stat) ? "...create failed" : "...created" ) : "" );
				retval = -1;
			}
		}

		/* check the file listing */
		if( -1 == access(buf.listpath, 0) ){
			log("Area %d: listing '%s' does not exist.\n",
					areaNum, buf.listpath);
			retval = -1;
		}

		/* set the group bits (if num == 0, we ignore it) */
		for( i = 0; i < 4; ++i ){
			int n = buf.groups[i] - 1;

			if( 0 != buf.groups[i] ) groupDef[n >> 3] |= 1 << (n & 0x07);
		}
	}
	/* close the file area definition file */
	fclose(fpSrc);

	/* now see is we have any orphaned groups */
	if( NULL == (fpSrc = pbfopen(fnFGROUPS, "rb")) ){
		log("Unable to open group file '%s', groups not checked\n", fnFGROUPS);
		retval = -1;
		goto _endCheck;
	}

	for( areaNum = 0;; areaNum++){
		FILEGROUP buf;

		/* read a group definition */
		if( 1 != fread(&buf, sizeof(buf), 1, fpSrc) ){
			if( feof(fpSrc) ) break;
			logexit("fread() failed on reading '%s'\n", fnFGROUPS);
		}

		if( EOS == buf.name[0] ) continue;  /* empty group */

		/* see if the bit for this group is set */
		if( !(groupDef[areaNum >> 3] & (1 << (areaNum & 0x07))) ){
			log("Group %d is not used for any of the file areas\n",
				areaNum + 1);
			retval = -1;
		}
	}

	/* close the file group definition file */
	fclose(fpSrc);
_endCheck:
	return retval;
}


/*
 * Checks if directory exists, 0 if it does, -1 otherwise
*/
	int _pascal
direxists(const char *path)
{
	static char buf[MAXPATH];

    strcpy(buf, path);
    if( '\\' == buf[strlen(buf) - 1] ) strcat( buf, "." );
    else strcat( buf, "\\." );

    return access(buf, 0);
}

/*************************************************************************\
 ** Packing function
\*************************************************************************/
 	int _pascal
do_packfile( void )
{
	FILE *fpSrc, *fpDest;
    char  oldPath[MAXPATH], newPath[MAXPATH];

    chext(newPath, fnFILECFG, ".CHK");
 	if( NULL == (fpSrc = pbfopen(fnFILECFG, "rb")) )
		logexit("Fatal: could not open filebase file '%s'\n", fnFILECFG);
 	if( NULL == (fpDest = pbfopen(newPath, "w+b")) )
		logexit("Fatal: could not create file '%s'\n", newPath);

    for( ;; ){
		FILECFG buf;

		/* read a file area record */
		if( 1 != fread(&buf, sizeof(buf), 1, fpSrc) ){
			if( feof(fpSrc) ) break;
			logexit("fread() failed on file '%s'\n", fnMESSAGES);
		}

		if( EOS == buf.name[0] ) continue;  /* skip the empty area */

        /* write this one to the new file */
        fwrite(&buf, sizeof(buf), 1, fpDest);
	}

    fclose(fpSrc);
    fclose(fpDest);

    /* now, backup the old file and rename the new one */
    chext(newPath, strcpy(oldPath, _mkSysPath(fnFILECFG)), ".BAK");
    if( 0 == access(newPath, 0) ) rmfile(newPath);
    rename(oldPath, newPath);
    chext(newPath, oldPath, ".CHK");
    rename(newPath, oldPath);

    return 0;
}
