/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.collection;

import org.ckkloverdos.filter.IFilter;
import org.ckkloverdos.function.IFilterFunction;
import org.ckkloverdos.function.IFilterProcedure;
import org.ckkloverdos.function.IFunction;
import org.ckkloverdos.function.IProcedure;
import org.ckkloverdos.string.IToStringAware;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.*;

/**
 * A generalized list with several element-processing methods, mainly influenced by functional-oriented
 * programming.
 *
 * <p/>
 * All the methods that return an <code>IL</code> list always construct a new list
 * unless specifically stated.
 *
 * <p/>
 * Indices start from zero. An <em>improper index</em> is an index beyond the bounds
 * of this list (either negative or greater than {@link #size() size}-1).
 * 
 * @author Christos KK Loverdos
 */
public interface IL extends IToStringAware
{
    /**
     * Returns an iterator over the elements in this list.
     */
    public Iterator iterator();

    /**
     * Returns an iterator over the elements in this list that are
     * {@link org.ckkloverdos.filter.IFilter#accept(Object, Object) accepted}
     * by the given <code>filter</code>.
     * @param filter
     */
    public Iterator iterator(IFilter filter);

    /**
     * Returns an iterator over the elements in this list that are
     * {@link org.ckkloverdos.filter.IFilter#accept(Object,Object) accepted}
     * by the given <code>filter</code>.
     *
     * The <code>hints</code> parameter is passed directly to the <code>filter</code>.
     * @param filter
     * @param hints
     */
    public Iterator iterator(IFilter filter, Object hints);

    /**
     * Returns the number of elements in this list.
     */
    public int size();

    /**
     * Returns true if this list contains the specified element.
     * @param o
     */
    public boolean contains(Object o);

    /**
     * Returns <code>true</code> if this list wraps an <code>Object</code> array.
     */
    public boolean isArray();

    /**
     * Returns <code>true</code> if this list wraps a {@link java.util.Collection}.
     */
    public boolean isCollection();

    /**
     * Returns <code>true</code> if this list wraps a {@link java.util.Collection}
     * and more specifically a {@link java.util.Set}.
     */
    public boolean isSet();

    /**
     * Returns <code>true</code> if this list wraps a {@link java.util.Collection}
     * and more specifically a {@link java.util.List}.
     */
    public boolean isList();

    /**
     * Returns an array containing all of the elements in this collection.
     * If this list is already wrapping an <code>Object</code> array, this array
     * is returned.
     */
    public Object[] toArray();

    /**
     * Returns an array containing all of the elements in this collection;
     * the runtime type of the returned array is that of the specified class.
     * If this list is already wrapping an <code>Object</code> array, this array
     * is returned and it MUST be of the given type.
     */
    public Object[] toArray(Class arrayClass);

    /**
     * Convenience method that returns the strings of this list, assuming of course that
     * this list contains strings.
     */
    public String[] toStringArray();

    /**
     * Returns a {@link java.util.List} containing the elements of this array.
     * If this list is already wrapping a {@link java.util.List}, then it is returned
     * intact, otherwise a new one is created.
     */
    public List toList();

    /**
     * Returns a {@link java.util.Set} containing the unique elements of this array.
     * If this list is already wrapping a {@link java.util.Set}, then it is returned
     * intact, otherwise a new one is created.
     */
    public Set toSet();

    /**
     * Returns a {@link java.util.SortedSet} containing the unique elements of this array.
     * If this list is already wrapping a {@link java.util.SortedSet}, then it is returned
     * intact, otherwise a new one is created.
     */
    public SortedSet toSortedSet();

    /**
     * Returns a {@link java.util.Collection} containing the elements of this array.
     * If this list is already wrapping a {@link java.util.Collection}, then it is returned
     * intact, otherwise a new one is created.
     */
    public Collection toCollection();

    /**
     * Returns the <code>i</code>-th element of this list.
     * @param i
     */
    public Object get(int i);

    /**
     * Convenience method that returns the <code>i</code>-th string of this list,
     * assuming that the list contains strings.
     * @param i
     */
    public String getString(int i);

    /**
     * Returns the first element of this list, retrieved from the {@link #iterator() iterator}.
     * @throws IndexOutOfBoundsException of the list has no elements.
     */
    public Object head();

    /**
     * Returns the first element of this list that is accepted by <code>filter</code>.
     * Elements are retrieved from the {@link #iterator() iterator}.
     * @throws IndexOutOfBoundsException of the list has no elements or no element
     * is accepted by <code>filter</code>.
     */
    public Object head(IFilter filter);

    /**
     * Returns the first element of this list that is accepted by <code>filter</code>.
     * Elements are retrieved from the {@link #iterator() iterator}.
     * The second parameter <code>hints</code> is directly passed to the <code>filter</code>.
     *
     * @throws IndexOutOfBoundsException of the list has no elements or no element
     * is accepted by <code>filter</code>.
     */
    public Object head(IFilter filter, Object hints);

    /**
     * Returns the list constructed from this one if we remove the {@link #head()}.
     * If this list has no elements, an empty one is returned.
     */
    public IL tail();

    /**
     * Returns the first <code>n</code> elements of this list, <code>n</code> starting from one.
     * If the list contains less than <code>n</code> elements, all are returned.
     * If <code>n</code> is less than one, an empty list is returned.
     *
     * <h3>Example:</h3>
     * <pre>
     * IL il = new L("zero", "one", "two", "three", "four");
     * StdLog.log("il = " + il.take(3));
     * </pre>
     * prints:
     * <pre>
     * [0="zero", 1="one", 2="two"]
     * </pre>
     *
     * @param n
     */
    public IL take(int n);

    // + filter, map
    /**
     * Returns a new list with all the elements that are accepted by <code>filter</code>.
     * @param filter
     */
    public IL filter(IFilter filter);

    /**
     * Returns a new list with all the elements that are accepted by <code>filter</code>.
     * The second parameter <code>hints</code> is directly passed to the <code>filter</code>.
     *
     */
    public IL filter(IFilter filter, Object hints);

    /**
     * Returns a new list containing all the elements of this list, mapped by <code>function</code>.
     * @param function
     */
    public IL map(IFunction function);

    /**
     * Returns a new list containing all the elements of this list, mapped by <code>function</code>.
     * The second parameter <code>hints</code> is directly passed to the <code>function</code>.
     */
    public IL map(IFunction function, Object hints);

    /**
     * Combines filtering and mapping in one place.
     * @param ff
     */
    public IL filterMap(IFilterFunction ff);

    /**
     * Combines filtering and mapping in one place.
     * The second parameter <code>hints</code> is directly passed to <code>ff</code>.
     *
     * <p/>
     * If <code>hints</code> is an implementation of {@link org.ckkloverdos.hint.BinaryHint},
     * then {@link org.ckkloverdos.hint.BinaryHint#getHintA()} is passed to {@link IFilterFunction#accept(Object, Object)}
     * and {@link org.ckkloverdos.hint.BinaryHint#getHintB()} is passed to {@link IFilterFunction#evaluate(Object, Object)}.
     *
     * @see org.ckkloverdos.hint.BinaryHint
     * @param ff
     */
    public IL filterMap(IFilterFunction ff, Object hints);
    // - filter, map

    // + projections
    /**
     * Returns a new list with the JavaBean properties of the given <code>name</code>
     * for each element.
     *
     * <p/>
     * If <code>name</code> ends with <code>()</code>, then the name is treated as a method,
     * which is used to give the new elements.
     *
     * <h3>Example</h3>
     * The code:
     * <pre>
     * IL il = new L("zero", "one", "two", "three", "four");
     * System.out.println(il.selectProperty("length()"));
     * </pre>
     * prints:
     * <pre>
     * [0=4, 1=3, 2=3, 3=5, 4=4]
     * </pre>
     * @param name
     */
    public IL selectProperty(String name);
    // - projections

    // + forEach
    /**
     * Processes all alements with <code>procedure</code>. 
     * @param procedure
     */
    public void forEach(IProcedure procedure);

    /**
     * Processes all alements with <code>procedure</code>.
     * The second parameter <code>hints</code> is directly passed to <code>procedure</code>.
     * @param procedure
     * @param hints
     */
    public void forEach(IProcedure procedure, Object hints);

    /**
     * Combines filtering and processing in one place.
     * The elements to be processed are those accepted by the filter.
     */
    public void filterForEach(IFilterProcedure procedure);

    /**
     * Combines filtering and processing in one place.
     * The elements to be processed are those accepted by the filter.
     * The second parameter <code>hints</code> is directly passed to <code>procedure</code>.
     *
     * <p/>
     * If <code>hints</code> is an implementation of {@link org.ckkloverdos.hint.BinaryHint},
     * then {@link org.ckkloverdos.hint.BinaryHint#getHintA()} is passed to {@link org.ckkloverdos.function.IFilterProcedure#accept(Object,Object)}
     * and {@link org.ckkloverdos.hint.BinaryHint#getHintB()} is passed to {@link org.ckkloverdos.function.IFilterProcedure#process(Object, Object)}.
     *
     * @see org.ckkloverdos.hint.BinaryHint
     */
    public void filterForEach(IFilterProcedure procedure, Object hints);
    // - forEach

    /**
     * Returns a copy of this list. The underlying backing store (collection or array)
     * is preserved in the new list.
     */
    public IL copy();

    /**
     * Adds the element to the list. The addition is made directly at the backing store and
     * no new list is created.
     *
     * <p/>
     * If the backing store is an array, then the component type of the array
     * must be assignable from the class of the new element, unless the new
     * element is <code>null</code>.
     *
     * <h3>Example 1</h3>
     * The code:
     * <pre>
     * IL l = new L("zero", "one", "two", "three", "four");
     * System.out.println(l.add(new Integer(1000)));
     * </pre>
     * will print:
     * <pre>
     * [0="zero", 1="one", 2="two", 3="three", 4="four", 5=1000]
     * </pre>
     *
     * <h3>Example 2</h3>
     * The code:
     * <pre>
     * IL l = new L(new String[]{"zero", "one", "two", "three", "four"});
     * System.out.println(l.add(new Integer(1000)));
     * </pre>
     * will throw an exception: <code>java.lang.IllegalArgumentException: array element type mismatch</code>.
     *
     * <h3>Example 3</h3>
     * The code:
     * <pre>
     * IL l = new L(new Number[] {new Long(1), new Float(2.2)});
     * System.out.println(l.add(new Integer(1000)));
     * </pre>
     * will print:
     * <pre>
     * [0=1, 1=2.2, 2=1000]
     * </pre>
     * @param o
     * @return this list with the new element added.
     */
    public IL add(Object o);

    /**
     * Adds all the elements of the collection to this list.
     * The comments of {@link #add(Object)} apply here as well.
     * @param c
     */
    public IL addAll(Collection c);

    /**
     * Adds all the elements of the <code>array</code> to this list.
     * The comments of {@link #add(Object)} apply here as well.
     * @param array
     */
    public IL addAll(Object[] array);
    
    /**
     * Adds all the elements of the given list to this one.
     * The comments of {@link #add(Object)} apply here as well.
     * @param l
     */
    public IL addAll(IL l);

    /**
     * Removes the element from the list (only once).
     * The deletion is made directly at the backing store and
     * no new list is created.
     * @param o
     */
    public IL remove(Object o);

    /**
     * Completely removes the element from the list.
     * The deletion is made directly at the backing store and
     * no new list is created.
     * @param o
     */
    public IL clear(Object o);

    /**
     * Sets the elements of this list from those of the collection <code>proxy</code>.
     * This is lightweight and just updates the underlying backing store
     * reference with that of the <code>proxy</code>.
     * @param proxy
     */
    public IL setFrom(CollectionProxy proxy);

    // + any, all
    /**
     * Returns <code>true</code> iff all the elements can be accepted by the <code>filter</code>.
     * In case the list is empty, <code>false</code> is returned.
     * @param filter
     */
    public boolean all(IFilter filter);

    /**
     * Returns <code>true</code> iff all the elements can be accepted by the <code>filter</code>.
     * In case the list is empty, <code>false</code> is returned.
     * The second parameter <code>hints</code> is directly passed to <code>filter</code>.
     * @param filter
     * @param hints
     */
    public boolean all(IFilter filter, Object hints);

    /**
     * Returns <code>true</code> iff all the elements can be accepted by the <code>filter</code>.
     * In case the list is empty, <code>true</code> is returned.
     * @param filter
     */
    public boolean any(IFilter filter);

    /**
     * Returns <code>true</code> iff all the elements can be accepted by the <code>filter</code>.
     * In case the list is empty, <code>true</code> is returned.
     * The second parameter <code>hints</code> is directly passed to <code>filter</code>.
     * @param filter
     * @param hints
     */
    public boolean any(IFilter filter, Object hints);
    // - any, all

    /**
     * Returns a set containing the unique elements of this list.
     * If the underlying backing store is already a set, then it is returned as is.
     */
    public IL unique();

    /**
     * Implements list zipping. The elements of the new list are instances of
     * {@link org.ckkloverdos.tuple.Pair}. The backing store of the returned
     * list is a {@link List}.
     * @param other
     */
    public IL zip(IL other);

    /**
     * Sorts the elements of the list.
     */
    public IL sort();

    /**
     * Sorts the elements of the list, using the given <code>Comparator</code>.
     */
    public IL sort(Comparator c);
    
    // + Set methods
    /**
     * Returns a new list containg the elements of this list along with the
     * elements of <code>other</code>.
     * @param other
     */
    public IL union(IL other);

    /**
     * Returns a new list containg the elements of this list that also
     * belong to the <code>other</code> list.
     * @param other
     */
    public IL intersect(IL other);

    /**
     * Returns a new list containg the elements of this list that do not
     * belong to the <code>other</code> list.
     * @param other
     */
    public IL minus(IL other);
    // - Set methods

    /// + String methods
    /**
     * Chops the <code>prefix</code> from all elements, which are assumed to be strings.
     * <code>null</code>s pass through.
     * @param prefix
     */
    public IL chopPrefix(String prefix);

    /**
     * Chops the regular expression <code>prefix</code> from all elements, which are assumed to be strings.
     * <code>null</code>s pass through.
     * @param prefix
     */
    public IL chopPrefixRE(String prefix);

    /**
     * Chops the <code>suffix</code> from all elements, which are assumed to be strings.
     * <code>null</code>s pass through.
     */
    public IL chopSuffix(String suffix);

    /**
     * Chops the regula expression <code>suffix</code> from all elements, which are assumed to be strings.
     * <code>null</code>s pass through.
     */
    public IL chopSuffixRE(String suffix);

    /**
     * Filters all elements that start with <code>s</code>, assuming that elements are strings.
     * <code>null</code>s do not pass through.
     * @param s
     */
    public IL filterStartsWith(String s);

    /**
     * Filters all elements that do not start with <code>s</code>, assuming that elements are strings.
     * <code>null</code>s pass through.
     * @param s
     */
    public IL filterNotStartsWith(String s);

    /**
     * Filters all elements that end with <code>s</code>, assuming that elements are strings.
     * <code>null</code>s do not pass through.
     * @param s
     */
    public IL filterEndsWith(String s);

    /**
     * Filters all elements that end with <code>s</code>, assuming that elements are strings.
     * <code>null</code>s pass through.
     * @param s
     */
    public IL filterNotEndsWith(String s);

    /**
     * Uses {@link java.util.regex.Matcher#find()}.
     * <code>null</code>s do not pass through.
     * @param re
     */
    public IL filterFindRE(String re);

    /**
     * Uses {@link java.util.regex.Matcher#find()}.
     * <code>null</code>s pass through.
     * @param re
     */
    public IL filterNotFindRE(String re);

    /**
     * Uses {@link java.util.regex.Matcher#find()}.
     * <code>null</code>s do not pass through.
     * @param re
     */
    public IL filterFindRE(String re, int modifiers);

    /**
     * Uses {@link java.util.regex.Matcher#find()}.
     * <code>null</code>s pass through.
     * @param re
     */
    public IL filterNotFindRE(String re, int modifiers);

    /**
     * Uses {@link java.util.regex.Matcher#matches()}.
     * <code>null</code>s do not pass through.
     * @param re
     */
    public IL filterMatchesRE(String re);

    /**
     * Uses {@link java.util.regex.Matcher#matches()}.
     * <code>null</code>s pass through.
     * @param re
     */
    public IL filterNotMatchesRE(String re);

    /**
     * Uses {@link java.util.regex.Matcher#matches()}.
     * <code>null</code>s do not pass through.
     * @param re
     */
    public IL filterMatchesRE(String re, int modifiers);

    /**
     * Uses {@link java.util.regex.Matcher#matches()}.
     * <code>null</code>s pass through.
     * @param re
     */
    public IL filterNotMatchesRE(String re, int modifiers);

    
    public IL filterStartsWithRE(String re);
    public IL filterNotStartsWithRE(String re);
    public IL filterStartsWithRE(String re, int modifiers);
    public IL filterNotStartsWithRE(String re, int modifiers);
    public IL filterEndsWithRE(String re);
    public IL filterNotEndsWithRE(String re);
    public IL filterEndsWithRE(String re, int modifiers);
    public IL filterNotEndsWithRE(String re, int modifiers);
    /// - String methods

    /// + print methods

    /**
     * Prints the elements to standard output.
     */
    public void print();

    /**
     * Prints the elements to the <code>PrintStream</code>.
     * @param ps
     */
    public void print(PrintStream ps);

    /**
     * Prints the elements to the <code>PrintWriter</code>.
     * @param pw
     */
    public void print(PrintWriter pw);
    /// - print methods
}
