/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.io;

import org.ckkloverdos.log.StdLog;
import org.ckkloverdos.string.StringUtil;

import java.io.*;
import java.net.URL;

/**
 * IO-related utility methods.
 * @author Christos KK Loverdos
 */
public class IOUtil
{
   /**
	 * Given an stream, we read it exhaustively and return the resulting
	 * byte array.
	 *
	 * The stream is closed after reading iff <code>close</code> is <code>true</code>.
    *
	 * @param is
	 *            The stream
	 * @return an array of bytes as the contents of the stream
	 * @throws IOException
	 */
	public static byte[] readBytes(InputStream is, boolean close) throws IOException
	{
		byte[] result = new byte[0];
		byte[] buffer = new byte[8192];
		int count;

		while((count = is.read(buffer)) > -1)
		{
			byte[] newresult = new byte[result.length + count];
			System.arraycopy(result, 0, newresult, 0, result.length);
			System.arraycopy(buffer, 0, newresult, result.length, count);

			result = newresult;
		}

		if(close)
		{
			is.close();
		}
		return result;
	}

    /**
     * Given an stream, we read it exhaustively and return the resulting
     * byte array.
     *
     * @param is The stream
     * @return an array of bytes as the contents of the stream
     * @throws IOException
     */
    public static byte[] readBytes(InputStream is) throws IOException
    {
        return readBytes(is, false);
    }
    
    /**
	 * Given a stream, we read it exhaustively and return the characters
	 * contained in the stream as a string.
	 *
	 * The {@link org.ckkloverdos.string.StringUtil#UTF8} character encoding
     * is used for the translation of bytes to characters.
	 *
	 * The stream is closed after reading iff <code>close</code> is <code>true</code>.
	 *
	 * @param is
	 *            The channel stream
	 * @return the string contents of the stream
	 * @throws IOException
	 */
	public static String readString(InputStream is, boolean close) throws IOException
    {
		return new String(readBytes(is, close), StringUtil.UTF8);
	}

    /**
     * Given a stream, we read it exhaustively and return the characters
     * contained in the stream as a string.
     * <p/>
     * The {@link org.ckkloverdos.string.StringUtil#UTF8} character encoding
     * is used for the translation of bytes to characters.
     *
     * @param is The channel stream
     * @return the string contents of the stream
     * @throws IOException
     */
    public static String readString(InputStream is) throws IOException
    {
        return new String(readBytes(is, false), StringUtil.UTF8);
    }


    /**
	 * Given an stream, we read it exhaustively and return the characters
	 * contained in the stream as a string.
	 *
	 * We use <tt>encoding</tt> for the translation of bytes to characters.
	 *
	 * The stream is closed after reading iff <code>close</code> is <code>true</code>.
	 *
	 * @param is
	 *            The channel stream
	 * @param encoding
	 *            The requested character encoding. Not null
	 * @return the string contents of the channel stream
	 * @throws IOException
	 */
	public static String readString(InputStream is, String encoding, boolean close) throws IOException
	{
		return new String(readBytes(is, close), encoding);
	}

    /**
     * Given an stream, we read it exhaustively and return the characters
     * contained in the stream as a string.
     * <p/>
     * We use <tt>encoding</tt> for the translation of bytes to characters.
     *
     * @param is       The channel stream
     * @param encoding The requested character encoding. Not null
     * @return the string contents of the channel stream
     * @throws IOException
     */
    public static String readString(InputStream is, String encoding) throws IOException
    {
        return new String(readBytes(is, false), encoding);
    }

    /**
     * Copies the bytes from <code>is</code> to <code>os</code>.
     * @param is
     * @param os
     * @throws IOException
     */
    public static void copy(InputStream is, OutputStream os) throws IOException
	{
		byte[] buffer = new byte[1024];
		int count;

		while((count = is.read(buffer)) > -1)
		{
			os.write(buffer, 0, count);
		}
	}

    /**
     * The <code>file</code> is read as a string, using {@link org.ckkloverdos.string.StringUtil#UTF8}.
     * @param file
     * @throws IOException
     */
    public static String readString(File file) throws IOException
    {
		return readString(file, StringUtil.UTF8);
	}

    /**
     * The <code>file</code> is read as a string, using the provided character <code>encoding</code>.
     * @param file
     * @throws IOException
     */
    public static String readString(File file, String encoding) throws IOException
    {
		return readString(new FileInputStream(file), encoding, true);
	}

    /**
     * Safely opens the stream from <code>url</code>. No exceptions are thrown.
     * Returns <code>null</code> on error.
     * @param url
     */
    public static InputStream safeOpenStream(URL url)
	{
		try
		{
			return url.openStream();
		}
		catch (NullPointerException e)
		{
            StdLog.error(e);
        }
		catch (IOException e)
		{
			StdLog.error(e);
		}
		return null;
	}

    /**
     * Safely closes the input stream, even if it is <code>null</code>, without
     * throwing any exceptions.
     * @param is
     */
    public static void safeClose(InputStream is)
    {
        try
        {
            is.close();
        }
        catch(Exception e)
        {
            StdLog.error(e);
        }
    }

    /**
      * Reads the line with the given line number.
      *
      * <p/>
      * <b>Note: Lines start from 1.</b>
      *
      * @param f
      * @param line
      */
     public static String getFileLine(File f, int line)
     {
    	 return getFileLine(f, line, null);
     }

     /**
      * Reads the line with the given line number.
      * If <code>encoding</code> is <code>null</code>, {@link org.ckkloverdos.string.StringUtil#UTF8}
      * is used.
      *
      * <p/>
      * <b>Note: Lines start from 1.</b>
      *
      * @param f
      * @param line
      * @param encoding
      */
     public static String getFileLine(File f, int line, String encoding)
     {
    	 if(line <= 0)
    	 {
    		 return null;
    	 }

    	 if(null == encoding)
    	 {
    		 encoding = StringUtil.UTF8;
    	 }

    	 FileInputStream fis = null;
    	 Reader r = null;
    	 BufferedReader br = null;
    	 try
    	 {
    		 fis = new FileInputStream(f);
    		 r = new InputStreamReader(fis, encoding);
    		 br = new BufferedReader(r);

    		 int n = 1;
    		 for(String s = br.readLine(); null != s; s = br.readLine(), n++)
    		 {
    			 if(n == line)
    			 {
    				 return s;
    			 }
    		 }
    	 }
    	 catch(Exception e)
    	 {
             StdLog.error(e);
         }
    	 finally
    	 {
    		 safeClose(br);
    		 safeClose(r);
    		 safeClose(fis);
    	 }

    	 return null;
     }

    /**
     * Safely closes the reader, even if it is <code>null</code>, without
     * throwing any exceptions.
     * @param r
     */
    public static void safeClose(Reader r)
    {
        try
        {
            r.close();
        }
        catch (Exception e)
        {
            StdLog.error(e);
        }
    }

    /**
     * Creates a {@link org.ckkloverdos.io.IReadLine} implementation
     * from the specified buffered reader.
     * @param br
     */
    public static IReadLine readLineFromReader(BufferedReader br)
    {
        return new ReadLineFromReader(br);
    }

    /**
     * Creates a {@link org.ckkloverdos.io.IReadLine} implementation
     * from the specified reader.
     * @param r
     */
    public static IReadLine readLineFromReader(Reader r)
    {
        return new ReadLineFromReader(r);
    }

    /**
     * Returns the filename extension of the given file.
     * @param file
     * @throws IOException
     */
    public static String getFileExtension(File file) throws IOException
    {
        String ext = "";
        String name = file.getAbsolutePath();
        String pname = new File(name).getParentFile().getAbsolutePath();

        if(!name.equals(pname))
        {
            String fileName = name.substring(name.indexOf(pname) + pname.length());
            int dotPos = fileName.lastIndexOf('.');
            if(-1 != dotPos)
            {
                ext = fileName.substring(dotPos + 1);
            }
        }

        return ext;
    }

    /**
     * Returns the extension of the given filename.
     * @param name
     * @throws IOException
     */
    public static String getFileExtension(String name) throws IOException
    {
        return getFileExtension(new File(name));
    }
}
