/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.resource;

import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

/**
 * A resource obtained from a class loader. The underlying implementation is URL-based.
 * 
 * @author Christos KK Loverdos
 */
public class ClassLoaderResource implements IResource, IToStringAware
{
    private URL url;

    /**
     * Contructs a new resource whose URL is the provided one.
     * Normally, this URL has been obtained by a class loader call, like:
     * {@link ClassLoader#getResource(String)}.
     * 
     * @param url
     */
    public ClassLoaderResource(URL url)
    {
        this.url = url;
    }

    /**
     * Returns the string representation of the underlying resource URL.
     */
    public String getName()
    {
        return url.toString();
    }

    /**
     * Returns the underlying URL.
     */
    public URL getURL()
    {
        return url;
    }

    /**
     * Returns the input stream as obtained from the underlying URL.
     * If the resource doesnot exist, then <code>null</code> is returned.
     * @throws IOException
     */
    public InputStream getInputStream() throws IOException
    {
        if(!exists())
        {
            return null;
        }

        return getURL().openStream();
    }

    /**
     * Returns <code>true</code> iff the underlying URL is not <code>null</code>.
     */
    public boolean exists()
    {
        return null != getURL();
    }

    /**
     * Returns <code>true</code> iff the {@link java.net.URL#getProtocol() protocol}
     * of the underlying URL is <code>"file"</code>.
     */
    public boolean isFile()
    {
        return exists() && getURL().getProtocol().equals("file");
    }

    /**
     * Returns a file object generated from the {@link java.net.URL#getPath() path} of the
     * underlying URL. If this resource doesnot represent a file, then <code>null</code>
     * is returned.
     */
    public File getFile()
    {
        if(!isFile())
        {
            return null;
        }

        return new File(getURL().getPath());
    }

    public String toString()
    {
        ToString ts = new ToString(this);
        toStringAware(ts);
        return ts.toString();
    }

    public void toStringAware(ToString ts)
    {
        ts.add(url);
    }
}
