/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.resource;

import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;

import java.io.File;

/**
 * Filesystem-based resource context.
 * 
 * @author Christos KK Loverdos
 */
public class FileResourceContext extends ResourseContextSkeleton implements IToStringAware
{
    private File[] roots;

    /**
     * Constructs a new context that will try to resolve resources
     * from the provided filesystem locations (roots).
     * @param roots
     * @param parent
     */
    public FileResourceContext(File[] roots, IResourceContext parent)
    {
        super(parent);
        this.roots = roots;
    }

    /**
     * Constructs a new context that will try to resolve resources
     * from the provided filesystem locations (roots).
     * @param roots
     */
    public FileResourceContext(File[] roots)
    {
        this(roots, null);
    }

    /**
     * Constructs a new context that will try to resolve resources
     * from the provided filesystem location (root).
     * @param root
     * @param parent
     */
    public FileResourceContext(File root, IResourceContext parent)
    {
        this(new File[]{root}, parent);
    }

    /**
     * Constructs a new context that will try to resolve resources
     * from the provided filesystem location (root).
     * @param root
     */
    public FileResourceContext(File root)
    {
        this(root, null);
    }


    /**
     * Constructs a new context that will try to resolve resources
     * from the provided filesystem location (root).
     *
     * @param root
     */
    public FileResourceContext(String root)
    {
        this(new File(root), null);
    }
    /**
     * Returns the resource with the given name, using only the underlying
     * file roots to resolve it (the parent context is not consulted).
     * @param name
     * @return the resource if found or {@link org.ckkloverdos.resource.EmptyResource#INSTANCE} otherwise.
     */
    public IResource getLocalResource(String name)
    {
        name = normalize(name);
        for(int i = 0; i < roots.length; i++)
        {
            File file = new File(roots[i], name);
            if(file.exists())
            {
                return new FileResource(file);
            }
        }

        return EmptyResource.INSTANCE;
    }

    public String toString()
    {
        ToString ts = new ToString(this);
        toStringAware(ts);
        return ts.toString();
    }

    public void toStringAware(ToString ts)
    {
        ToString ts2 = ts.save();
        ts
            .add(roots)
            .restore(ts2);
    }
}
