/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.source;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

/**
 * Generic representation of binary or text source.
 * This for example could be java source code or the bytes of a jar file.
 * @author Christos KK Loverdos
 */
public interface ISource
{
    /**
     * Return the name of this source.
     */
    public String getName();

    /**
     * Return the {@link org.ckkloverdos.source.SourceType type} of this source.
     */
    public SourceType getType();

    /**
     * Return the description of this source.
     */
    public String getDescription();

    /**
     * Return <code>true</code> iff any of the {@link #getInputStream()},
     * {@link #getString()}, {@link #getReader()} methods will return
     * a non-<code>null</code> object.
     */
    public boolean isValid();

    /**
     * Return <code>true</code> if the type of this source is <code>text</code>.
     * @return <code>true</code> iff <code>getType().isText()</code> returns <code>true</code>.
     */
    public boolean isText();

    /**
     * Get an input stream for this source.
     * @throws IOException
     */
    public InputStream getInputStream() throws IOException;

    /**
     * Get a string from the contents of this source. The contents
     * are interpeted using the provided <code>encoding</code>.
     * @param encoding
     * @throws IOException
     */
    public String getString(String encoding) throws IOException;

    /**
     * Get a string from the contents of this source. The contents
     * are interpeted using the default {@link org.ckkloverdos.string.StringUtil#UTF8}
     * encoding.
     * @throws IOException
     */
    public String getString() throws IOException;

    /**
     * Get a reader for the contents of this source. The contents
     * are interpeted using the provided <code>encoding</code>.
     * @param encoding
     * @throws IOException
     */
    public Reader getReader(String encoding) throws IOException;

    /**
     * Get a reader for the contents of this source. The contents
     * are interpeted using the default {@link org.ckkloverdos.string.StringUtil#UTF8}
     * encoding.
     * @throws IOException
     */
    public Reader getReader() throws IOException;
}