/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.source;

import java.util.*;

/**
 * A registry for source types.
 *
 * @author Christos KK Loverdos
 *
 * @see org.ckkloverdos.source.SourceType
 */
public class SourceTypeRegistry
{
    private Map name2type;
    private Map ext2type;

    public SourceTypeRegistry()
    {
        this.name2type = new HashMap();
        this.ext2type = new HashMap();
    }

    private String fixExt(String ext)
    {
        while('.' == ext.charAt(0))
        {
            ext = ext.substring(1);
        }
        return ext.toLowerCase();
    }

    /**
     * Registers a type for the given <code>name</code> and <code>extension</code>.
     * If a type with the same <code>name</code> exists, then the <code>extension</code>
     * is registered for the existing type and that one is returned.
     * @param name
     * @param extension
     */
    public SourceType register(String name, String extension)
    {
        SourceType type = getTypeFromName(name);
        if(null == type)
        {
            type = new SourceType(name);
            name2type.put(name, type);
        }
        ext2type.put(fixExt(extension), type);

        return type;
    }

    /**
     * Registers a type for the given <code>name</code> and <code>extensions</code>.
     * If a type with the same <code>name</code> exists, then the <code>extensions</code>
     * are registered for the existing type and that one is returned.
     * @param name
     * @param extensions
     */
    public SourceType register(String name, String[] extensions)
    {
        SourceType type = getTypeFromName(name);
        if(null == type)
        {
            type = new SourceType(name);
            name2type.put(name, type);
        }

        for(int i = 0; i < extensions.length; i++)
        {
            String extension = extensions[i];
            ext2type.put(fixExt(extension), type);
        }

        return type;
    }

    /**
     * Returns the type associated with the given <code>extension</code>.
     * @param extension
     */
    public SourceType getTypeFromExt(String extension)
    {
        return (SourceType) ext2type.get(fixExt(extension));
    }

    /**
     * Returns the type associated with the given <code>name</code>.
     * @param name
     */
    public SourceType getTypeFromName(String name)
    {
        return (SourceType) name2type.get(name);
    }

    /**
     * Returns the extensions associated with the type of the given <code>name</code>.
     * @param name
     * @return the extensions or an empty array if none is found.
     */
    public String[] getExtensionsForType(String name)
    {
        SourceType type = getTypeFromName(name);
        if(null == type)
        {
            return null;
        }

        List retVal = new ArrayList();
        Iterator exts = ext2type.keySet().iterator();
        while(exts.hasNext())
        {
            String ext = (String) exts.next();
            SourceType extType = (SourceType) ext2type.get(ext);
            if(type.is(extType))
            {
                retVal.add(ext);
            }
        }

        return (String[]) retVal.toArray(new String[retVal.size()]);
    }
}