package org.ckkloverdos.type.java;

import org.ckkloverdos.collection.L;
import org.ckkloverdos.filter.IFilter;
import org.ckkloverdos.function.IProcedure;
import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;
import org.ckkloverdos.util.Util;

import java.util.*;

/**
 * A {@link JavaType} registry, able to remember type aliases.
 *
 * Also, in case a primitive type needs to be handled uniformly as its respective
 * object type (e.g. <code>int</code> and <code>java.lang.Integer</code>), then their classes
 * can be registered with the {@link #setRepresentative(Class, Class)} or
 * {@link #setRepresentative(Class, JavaType)} methods. For example:
 * <blockquote><pre>
 * JavaTypeRegistry r = new JavaTypeRegistry();
 * r.setRepresentative(int.class, Integer.class);
 * </pre></blockquote>
 *
 * <p>
 * Representatives must be registered before anything else.
 *
 * <p>
 * The common practice for an application is to have a singleton registry that internally
 * uses an instance of <code>JavaTypeRegistry</code>. {@link ObjectJavaTypes}
 * uses this idiom.
 * 
 * @see JavaType
 * @see ObjectJavaTypes
 * 
 * @author Christos KK Loverdos
 */
public class JavaTypeRegistry implements IToStringAware
{
    private Map class2type;
    private Map alias2class;
    private Map representatives;

    /**
     * The only constructor.
     */
    public JavaTypeRegistry()
    {
        this.class2type = new HashMap();
        this.alias2class = new HashMap();
        this.representatives = new HashMap();
    }

    /**
     * Sets <code>representative</code> to be the representative class of <code>original</code> class.
     * This is usually used for primitive classes and their corresponding object classes.
     *  
     * @param original
     * @param representative
     */
    public void setRepresentative(Class original, Class representative)
    {
        this.representatives.put(original, representative);
    }

    /**
     * Sets the java class wrapped by <code>representativeType</code> to be the representative
     * class of the <code>original</code> class.
     * 
     * @param original
     * @param representativeType
     */
    public void setRepresentative(Class original, JavaType representativeType)
    {
        this.representatives.put(original, representativeType.getJavaClass());
    }

    /**
     * Returns the representative of class <code>what</code> or <code>null</code> if no
     * representative has been set for class <code>what</code>.
     * 
     * @param what
     */
    public Class getRepresentative(Class what)
    {
        Class repr = (Class) representatives.get(what);
        if(null == repr)
        {
            repr = what;
        }
        return repr;
    }

    /**
     * Returns the representatives mapping. The keys are the original classes and the values
     * are their representatives.
     *
     */
    public Map getRepresentatives()
    {
        return representatives;
    }

    /**
     * Returns the <code>JavaType</code> wrapping class <code>c</code> or creates a new one
     * if it doesnot exist in the registry.
     * if the class <code>c</code> has a representative, the <code>JavaType</code> of its representative is
     * returned. In any case, if the <code>JavaType</code> doesnot exist it is created in-place.
     *
     * @param c
     */
    public JavaType register(Class c)
    {
        JavaType type = getTypeByClass(c);
        if(null == type)
        {
            type = setTypeByClass(c, new JavaType(c));
        }
        return type;
    }

    /**
     * Registers <code>javaType</code> for the java class {@link JavaType#getJavaClass() it wraps}.
     * If the wrapped java class has a representative, then <code>javaType</code> is registered
     * for the representative instead.
     * 
     * @param javaType
     */
    public JavaType register(JavaType javaType)
    {
        Class c = javaType.getJavaClass();
        c = getRepresentative(c); // ??
        return setTypeByClass(c, javaType);
    }

    /**
     * Registers the <code>alias</code> for the java class <code>c</code>.
     * If <code>c</code> has a representative, then the <code>alias</code>
     * is registered for the representative instead.
     *
     * <p>
     * If this <code>alias</code> registration
     * is happening before any other registration for class <code>c</code>, then
     * a new <code>JavaType</code> is created for class <code>c</code> and registered
     * as well. So, the following code:
     * <pre>
     * r.register("INTEGER", Integer.class)
     * </pre>
     * is equivalent to:
     * <pre>
     * r.register(Integer.class)
     * r.register("INTEGER", Integer.class)
     * </pre>
     *
     * @param alias
     * @param c
     * @return the <code>JavaType</code> representing the <code>alias</code>.
     */
    public JavaType register(String alias, Class c)
    {
        c = getRepresentative(c);
        setClassByAlias(alias, c);

        JavaType type = getTypeByClass(c);
        if(null == type)
        {
            type = new JavaType(c);
        }

        if(!class2type.containsKey(c))
        {
            setTypeByClass(c, type);
        }

        return type;
    }

    /**
     * Registers the <code>alias</code> for the class wrapped by <code>type</code>.
     * In effect, this is equivalent to <code>register(alias, type.getJavaClass())</code>.
     * @param alias
     * @param type
     * @return the <code>JavaType</code> created for the alias.
     */
    public JavaType register(String alias, JavaType type)
    {
        return register(alias, type.getJavaClass());
    }

    /**
     * Returns the <code>JavaType</code> wrapping class <code>c</code>.
     * If <code>c</code> has a representative, then it returns the <code>JavaType</code>
     * wrapping the representative.
     *
     * @return the registered <code>JavaType</code> or <code>null</code> if none
     * is found in the registry.
     */
    public JavaType getByClass(Class c)
    {
        return getTypeByClass(c);
    }

    /**
     * Returns the <code>JavaType</code> for the alias represented by <code>name</code>.
     * If the alias refers to a java class that has a representative, then the
     * representative's <code>JavaType</code> is returned instead.
     *
     * @return the registered <code>JavaType</code> or <code>null</code> if none
     * is found in the registry.
     */
    public JavaType getByName(String name)
    {
        Class c = getClassByAlias(name);
        if(null != c)
        {
            return getTypeByClass(c);
        }
        return null;
    }

    /**
     * Returns a new instance having exactly the same mappings maintained by this registry.
     */
    public JavaTypeRegistry copy()
    {
        JavaTypeRegistry r = new JavaTypeRegistry();
        r.class2type.putAll(this.class2type);
        r.alias2class.putAll(this.alias2class);
        r.representatives.putAll(this.representatives);

        return r;
    }

    public String toString()
    {
        ToString ts = new ToString(this, true);
        toStringAware(ts);
        return ts.toString();
    }

    public void toStringAware(ToString ts)
    {
        ToString _ts0 = ts.save();
        final ToString _ts = _ts0.setMultiline();
        
        new L(class2type.values())
            .sort(new Comparator()
            {
                public int compare(Object o1, Object o2)
                {
                    JavaType a = (JavaType) o1;
                    JavaType b = (JavaType) o2;
                    return a.getName().compareTo(b.getName());
                }
            })
            .forEach(new IProcedure()
            {
                public void process(Object o, Object hints)
                {
                    JavaType type =(JavaType) o;
                    String typeName = type.getName();
                    String typeFullName = type.getFullName();

                    if(!typeName.equals(typeFullName))
                    {
                        _ts.add(typeName, typeFullName);
                    }
                    else
                    {
                        _ts.add(typeName);
                    }
                    String[] aliases = getAliases(type.getJavaClass());
                    if(0 != aliases.length)
                    {
                        ToString tsa = _ts.save();
                        _ts.setUsingTypeNames(false)
                            .setUsingIndices(false)
                            .setMultiline(false)
                            .setStringQuoted(false);
                        _ts.add(aliases);
                        _ts.restore(tsa);
                    }
                }
            });

        ts.restore(_ts0);
    }

    /**
     * Checks if the two types are equal, taking representatives into account.
     */
    public boolean is(JavaType a, JavaType b)
    {
        return register(a).equals(register(b));
    }

    /**
     * Checks if the two types are equal, taking representatives into account.
     */
    public boolean is(JavaType a, String b)
    {
        return register(a).equals(getByName(b));
    }

    /**
     * Checks if the two types are equal, taking representatives into account.
     */
    public boolean is(Class a, JavaType b)
    {
        return register(a).equals(register(b));
    }

    /**
     * Checks if the two types are equal, taking representatives into account.
     */
    public boolean is(Class a, Class b)
    {
        return register(a).equals(register(b));
    }

    /**
     * Checks if the two types are equal, taking representatives into account.
     */
    public boolean is(Class a, String b)
    {
        return register(a).equals(getByName(b));
    }

    /**
     * Returns the <code>JavaType</code> for the given class. If the given class
     * has a representative, then the <code>JavaType</code> for the representative is
     * returned instead.
     * @param c
     */
    private JavaType getTypeByClass(Class c)
    {
        Class actual = (Class) representatives.get(c);
        if(null == actual)
        {
            actual = c;
        }

        return (JavaType) class2type.get(actual);
    }

    private Class getClassByAlias(String name)
    {
        return (Class) alias2class.get(name);
    }

    private void setClassByAlias(String alias, Class c)
    {
        alias2class.put(alias, c);
    }

    private JavaType setTypeByClass(Class c, JavaType type)
    {
        class2type.put(c, type);
        return type;
    }

    private String[] getAliases(Class c)
    {
        final Class _c = c;
        return new L(alias2class.keySet()).filter(new IFilter()
        {
            public boolean accept(Object object, Object hints)
            {
                return Util.equalSafe(alias2class.get(object), _c);
            }
        })
            .toStringArray();
    }
}
