/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.util;

import org.ckkloverdos.collection.IL;

import java.util.Collection;

/**
 * Utility assertions.
 *
 * @author Christos KK Loverdos
 */

public final class Assert
{

    private Assert()
    {
    }

    private static String getMsg(String msg)
    {
        StackTraceElement[] st = CallUtil.getStackTrace();
        if(null == msg)
        {
            return "Failed " + st[1].getMethodName() + "() assertion at " + st[2].getClassName() + "::" + st[2].getMethodName();
        }
        else
        {
            return "Failed " + st[1].getMethodName() + "() assertion at " + st[2].getClassName() + "::" + st[2].getMethodName() + ": " + msg;
        }
    }

    public static void notNull(Object o)
    {
        if(null == o)
        {
            throw new NullPointerException(getMsg(null));
        }
    }

    public static void notNull(Object o, String msg)
    {
        if(null == o)
        {
            throw new NullPointerException(getMsg(msg));
        }
    }

    public static void notNullObject(Object o, String objectName)
    {
        if(null == o)
        {
            throw new NullPointerException(getMsg(objectName + " must not be null"));
        }
    }

    public static void holds(boolean condition)
    {
        if(!condition)
        {
            throw new UnsatisfiedConditionException(getMsg(null));
        }
    }

    public static void holds(boolean condition, String msg)
    {
        if(!condition)
        {
            throw new UnsatisfiedConditionException(getMsg(msg));
        }
    }

    public static void notEmpty(String s)
    {
        if(Util.emptySafe(s))
        {
            throw new UnsatisfiedConditionException(getMsg(null));
        }
    }

    public static void notEmpty(String s, String msg)
    {
        if(Util.emptySafe(s))
        {
            throw new UnsatisfiedConditionException(getMsg(msg));
        }
    }

    public static void notEmpty(Object[] array)
    {
        if(Util.emptySafe(array))
        {
            throw new UnsatisfiedConditionException(getMsg(null));
        }
    }

    public static void notEmpty(Object[] array, String msg)
    {
        if(Util.emptySafe(array))
        {
            throw new UnsatisfiedConditionException(getMsg(msg));
        }
    }

    public static void notEmpty(Collection collection)
    {
        if(Util.emptySafe(collection))
        {
            throw new UnsatisfiedConditionException(getMsg(null));
        }
    }

    public static void notEmpty(Collection collection, String msg)
    {
        if(Util.emptySafe(collection))
        {
            throw new UnsatisfiedConditionException(getMsg(msg));
        }
    }

    public static void notEmpty(IL list)
    {
        if(Util.emptySafe(list))
        {
            throw new UnsatisfiedConditionException(getMsg(null));
        }
    }

    public static void notEmpty(IL list, String msg)
    {
        if(Util.emptySafe(list))
        {
            throw new UnsatisfiedConditionException(getMsg(msg));
        }
    }

    public static void isSublass(Class subClass, Class ofClass)
    {
        if(!ofClass.isAssignableFrom(subClass))
        {
            throw new UnsatisfiedConditionException(getMsg(null));
        }
    }

    public static void isSublass(Class subClass, Class ofClass, String msg)
    {
        if(!ofClass.isAssignableFrom(subClass))
        {
            throw new UnsatisfiedConditionException(getMsg(msg));
        }
    }

    public static void isSublassSafe(Object o, Class ofClass)
    {
        if(null != o && !ofClass.isAssignableFrom(o.getClass()))
        {
            throw new UnsatisfiedConditionException(getMsg(null));
        }
    }

    public static void isSublassSafe(Object o, Class ofClass, String msg)
    {
        if(null != o && !ofClass.isAssignableFrom(o.getClass()))
        {
            throw new UnsatisfiedConditionException(getMsg(msg));
        }
    }

    public static void main(String[] args)
    {
        Assert.holds(true == false);
    }
}
