/*
 * $Id:MandelbrotFractalConfigElement.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.mandelbrot.fractal;

import net.sf.jame.core.config.AbstractConfigElement;
import net.sf.jame.core.config.ConfigContext;
import net.sf.jame.core.config.ListConfigElement;
import net.sf.jame.core.config.SingleConfigElement;
import net.sf.jame.mandelbrot.fractal.incolouring.IncolouringFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.outcolouring.OutcolouringFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.rendering.RenderingFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.transforming.TransformingFormulaConfigElement;

/**
 * @author Andrea Medeghini
 */
public class MandelbrotFractalConfigElement extends AbstractConfigElement {
	private static final long serialVersionUID = 1L;
	public static final String CLASS_ID = "MandelbrotFractal";
	private final SingleConfigElement<RenderingFormulaConfigElement> renderingFormulaSingleElement = new SingleConfigElement<RenderingFormulaConfigElement>("RenderingFormulaSingleElement");
	private final SingleConfigElement<TransformingFormulaConfigElement> transformingFormulaSingleElement = new SingleConfigElement<TransformingFormulaConfigElement>("TransformingFormulaSingleElement");
	private final ListConfigElement<IncolouringFormulaConfigElement> incolouringFormulaListElement = new ListConfigElement<IncolouringFormulaConfigElement>("IncolouringFormulaListElement", 1);
	private final ListConfigElement<OutcolouringFormulaConfigElement> outcolouringFormulaListElement = new ListConfigElement<OutcolouringFormulaConfigElement>("OutcolouringFormulaListElement", 1);

	/**
	 * Constructs a new element.
	 */
	public MandelbrotFractalConfigElement() {
		super(MandelbrotFractalConfigElement.CLASS_ID);
	}

	/**
	 * Returns the renderingFormulaElement.
	 * 
	 * @return the renderingFormulaElement.
	 */
	public RenderingFormulaConfigElement getRenderingFormulaConfigElement() {
		return renderingFormulaSingleElement.getValue();
	}

	/**
	 * Sets the renderingFormulaElement.
	 * 
	 * @param renderingFormulaElement the renderingFormulaElement to set.
	 */
	public void setRenderingFormulaConfigElement(final RenderingFormulaConfigElement renderingFormulaElement) {
		renderingFormulaSingleElement.setValue(renderingFormulaElement);
	}

	/**
	 * Returns the transformingFormulaElement.
	 * 
	 * @return the transformingFormulaElement.
	 */
	public TransformingFormulaConfigElement getTransformingFormulaConfigElement() {
		return transformingFormulaSingleElement.getValue();
	}

	/**
	 * Sets the transformingFormulaElement.
	 * 
	 * @param transformingFormulaElement the transformingFormulaElement to set.
	 */
	public void setTransformingFormulaConfigElement(final TransformingFormulaConfigElement transformingFormulaElement) {
		transformingFormulaSingleElement.setValue(transformingFormulaElement);
	}

	/**
	 * Returns an incolouring formula element.
	 * 
	 * @param index the formula index.
	 * @return the incolouring formula.
	 */
	public IncolouringFormulaConfigElement getIncolouringFormulaConfigElement(final int index) {
		return incolouringFormulaListElement.getElement(index);
	}

	/**
	 * Returns an incolouring formula element index.
	 * 
	 * @param formulaElement the incolouring formula element.
	 * @return the index.
	 */
	public int indexOfIncolouringFormulaConfigElement(final IncolouringFormulaConfigElement formulaElement) {
		return incolouringFormulaListElement.indexOfElement(formulaElement);
	}

	/**
	 * Returns the number of incolouring formula elements.
	 * 
	 * @return the number of incolouring formula elements.
	 */
	public int getIncolouringFormulaConfigElementCount() {
		return incolouringFormulaListElement.getElementCount();
	}

	/**
	 * Adds a incolouring formula element.
	 * 
	 * @param formulaElement the incolouring formula element.
	 */
	public void appendIncolouringFormulaConfigElement(final IncolouringFormulaConfigElement formulaElement) {
		incolouringFormulaListElement.appendElement(formulaElement);
	}

	/**
	 * Adds a incolouring formula element.
	 * 
	 * @param index the index.
	 * @param formulaElement the incolouring formula element.
	 */
	public void insertIncolouringFormulaConfigElementAfter(final int index, final IncolouringFormulaConfigElement formulaElement) {
		incolouringFormulaListElement.insertElementAfter(index, formulaElement);
	}

	/**
	 * Adds a incolouring formula element.
	 * 
	 * @param index the index.
	 * @param formulaElement the incolouring formula element.
	 */
	public void insertIncolouringFormulaConfigElementBefore(final int index, final IncolouringFormulaConfigElement formulaElement) {
		incolouringFormulaListElement.insertElementBefore(index, formulaElement);
	}

	/**
	 * Removes a incolouring formula element.
	 * 
	 * @param index the element index.
	 */
	public void removeIncolouringFormulaConfigElement(final int index) {
		incolouringFormulaListElement.removeElement(index);
	}

	/**
	 * Removes a incolouring formula element.
	 * 
	 * @param formulaElement the incolouring formula element to remove.
	 */
	public void removeIncolouringFormulaConfigElement(final IncolouringFormulaConfigElement formulaElement) {
		incolouringFormulaListElement.removeElement(formulaElement);
	}

	/**
	 * Returns an outcolouring formula element.
	 * 
	 * @param index the formula index.
	 * @return the outcolouring formula.
	 */
	public OutcolouringFormulaConfigElement getOutcolouringFormulaConfigElement(final int index) {
		return outcolouringFormulaListElement.getElement(index);
	}

	/**
	 * Returns an outcolouring formula element index.
	 * 
	 * @param formulaElement the outcolouring formula element.
	 * @return the index.
	 */
	public int indexOfOutcolouringFormulaConfigElement(final OutcolouringFormulaConfigElement formulaElement) {
		return outcolouringFormulaListElement.indexOfElement(formulaElement);
	}

	/**
	 * Returns the number of outcolouring formula elements.
	 * 
	 * @return the number of outcolouring formula elements.
	 */
	public int getOutcolouringFormulaConfigElementCount() {
		return outcolouringFormulaListElement.getElementCount();
	}

	/**
	 * Adds a outcolouring formula element.
	 * 
	 * @param formulaElement the outcolouring formula element.
	 */
	public void appendOutcolouringFormulaConfigElement(final OutcolouringFormulaConfigElement formulaElement) {
		outcolouringFormulaListElement.appendElement(formulaElement);
	}

	/**
	 * Adds a outcolouring formula element.
	 * 
	 * @param index the index.
	 * @param formulaElement the outcolouring formula element.
	 */
	public void insertOutcolouringFormulaConfigElementAfter(final int index, final OutcolouringFormulaConfigElement formulaElement) {
		outcolouringFormulaListElement.insertElementAfter(index, formulaElement);
	}

	/**
	 * Adds a outcolouring formula element.
	 * 
	 * @param index the index.
	 * @param formulaElement the outcolouring formula element.
	 */
	public void insertOutcolouringFormulaConfigElementBefore(final int index, final OutcolouringFormulaConfigElement formulaElement) {
		outcolouringFormulaListElement.insertElementBefore(index, formulaElement);
	}

	/**
	 * Removes a outcolouring formula element.
	 * 
	 * @param index the element index to remove.
	 */
	public void removeOutcolouringFormulaConfigElement(final int index) {
		outcolouringFormulaListElement.removeElement(index);
	}

	/**
	 * Removes a outcolouring formula element.
	 * 
	 * @param formulaElement the outcolouring formula element to remove.
	 */
	public void removeOutcolouringFormulaConfigElement(final OutcolouringFormulaConfigElement formulaElement) {
		outcolouringFormulaListElement.removeElement(formulaElement);
	}

	/**
	 * @return
	 */
	@Override
	public MandelbrotFractalConfigElement clone() {
		final MandelbrotFractalConfigElement element = new MandelbrotFractalConfigElement();
		element.setRenderingFormulaConfigElement(getRenderingFormulaConfigElement().clone());
		element.setTransformingFormulaConfigElement(getTransformingFormulaConfigElement().clone());
		for (int i = 0; i < incolouringFormulaListElement.getElementCount(); i++) {
			element.appendIncolouringFormulaConfigElement(incolouringFormulaListElement.getElement(i).clone());
		}
		for (int i = 0; i < outcolouringFormulaListElement.getElementCount(); i++) {
			element.appendOutcolouringFormulaConfigElement(outcolouringFormulaListElement.getElement(i).clone());
		}
		return element;
	}

	/**
	 * @see net.sf.jame.core.config.AbstractConfigElement#setContext(net.sf.jame.core.config.ConfigContext)
	 */
	@Override
	public void setContext(final ConfigContext context) {
		super.setContext(context);
		renderingFormulaSingleElement.setContext(getContext());
		transformingFormulaSingleElement.setContext(getContext());
		incolouringFormulaListElement.setContext(context);
		outcolouringFormulaListElement.setContext(context);
	}

	/**
	 * @return
	 */
	public SingleConfigElement<RenderingFormulaConfigElement> getRenderingFormulaSingleElement() {
		return renderingFormulaSingleElement;
	}

	/**
	 * @return
	 */
	public SingleConfigElement<TransformingFormulaConfigElement> getTransformingFormulaSingleElement() {
		return transformingFormulaSingleElement;
	}

	/**
	 * @return the incolouringFormulaListElement
	 */
	public ListConfigElement<IncolouringFormulaConfigElement> getIncolouringFormulaListElement() {
		return incolouringFormulaListElement;
	}

	/**
	 * @return the outcolouringFormulaListElement
	 */
	public ListConfigElement<OutcolouringFormulaConfigElement> getOutcolouringFormulaListElement() {
		return outcolouringFormulaListElement;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		final MandelbrotFractalConfigElement other = (MandelbrotFractalConfigElement) obj;
		if (incolouringFormulaListElement == null) {
			if (other.incolouringFormulaListElement != null) {
				return false;
			}
		}
		else if (!incolouringFormulaListElement.equals(other.incolouringFormulaListElement)) {
			return false;
		}
		if (outcolouringFormulaListElement == null) {
			if (other.outcolouringFormulaListElement != null) {
				return false;
			}
		}
		else if (!outcolouringFormulaListElement.equals(other.outcolouringFormulaListElement)) {
			return false;
		}
		if (renderingFormulaSingleElement == null) {
			if (other.renderingFormulaSingleElement != null) {
				return false;
			}
		}
		else if (!renderingFormulaSingleElement.equals(other.renderingFormulaSingleElement)) {
			return false;
		}
		if (transformingFormulaSingleElement == null) {
			if (other.transformingFormulaSingleElement != null) {
				return false;
			}
		}
		else if (!transformingFormulaSingleElement.equals(other.transformingFormulaSingleElement)) {
			return false;
		}
		return true;
	}

	/**
	 * @see net.sf.jame.core.config.AbstractConfigElement#dispose()
	 */
	@Override
	public void dispose() {
		renderingFormulaSingleElement.dispose();
		transformingFormulaSingleElement.dispose();
		incolouringFormulaListElement.dispose();
		outcolouringFormulaListElement.dispose();
		super.dispose();
	}
}
