/*
 * $Id:AbstractFractalRenderer.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.mandelbrot.renderer;

import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;
import java.util.concurrent.ThreadFactory;

import net.sf.jame.core.math.Complex;
import net.sf.jame.core.util.Colors;
import net.sf.jame.mandelbrot.fractal.MandelbrotFractalRuntimeElement;
import net.sf.jame.mandelbrot.fractal.incolouring.IncolouringFormulaRuntimeElement;
import net.sf.jame.mandelbrot.fractal.outcolouring.OutcolouringFormulaRuntimeFormula;
import net.sf.jame.twister.DefaultThreadFactory;
import net.sf.jame.twister.DoubleVector2D;
import net.sf.jame.twister.DoubleVector4D;
import net.sf.jame.twister.ImageTile;
import net.sf.jame.twister.IntegerVector2D;
import net.sf.jame.twister.IntegerVector4D;
import net.sf.jame.twister.View;
import net.sf.jame.twister.renderer.Surface;
import net.sf.jame.twister.renderer.TwisterRenderer;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public abstract class AbstractFractalRenderer implements FractalRenderer {
	protected static final Logger logger = Logger.getLogger(AbstractFractalRenderer.class);
	private Graphics2D newBuffer;
	private Graphics2D oldBuffer;
	private BufferedImage newImage;
	private BufferedImage oldImage;
	private IntegerVector2D bufferSize;
	private ImageTile newTile;
	private ImageTile oldTile;
	private int imageDim;
	private int tileDim;
	protected int newShiftValue;
	protected int oldShiftValue;
	protected double rotationValue;
	protected DoubleVector2D newConstant;
	protected DoubleVector2D oldConstant;
	protected int renderMode = FractalRenderer.MODE_CALCULATE;
	protected int newImageMode = 0;
	protected int oldImageMode = 0;
	private boolean isDynamic = false;
	private boolean isZoomDynamic = false;
	private boolean isInterrupted = false;
	private AffineTransform transform = new AffineTransform();
	protected RenderedArea area = new RenderedArea();
	protected RenderingStrategy renderingStrategy;
	protected MandelbrotFractalRuntimeElement fractal;
	private final Complex center = new Complex();
	private final Complex scale = new Complex();
	protected View newView = new View(new IntegerVector4D(0, 0, 0, 0), new DoubleVector4D(0, 0, 1, 0), new DoubleVector4D(0, 0, 0, 0));
	protected View oldView = new View(new IntegerVector4D(0, 0, 0, 0), new DoubleVector4D(0, 0, 1, 0), new DoubleVector4D(0, 0, 0, 0));
	protected int percent = 100;
	protected int status = TwisterRenderer.STATUS_TERMINATED;
	private final RenderTask renderTask = new RenderTask();
	protected final ThreadFactory factory;
	private boolean isViewChanged;

	/**
	 * @param threadPriority
	 */
	public AbstractFractalRenderer(final int threadPriority) {
		factory = new DefaultThreadFactory("FractalRenderer", true, threadPriority);
		renderTask.start();
	}

	private class RenderTask implements Runnable {
		private final Object lock = new Object();
		private Thread thread;
		private boolean dirty;
		private boolean running;

		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				while (running) {
					synchronized (lock) {
						if (!dirty) {
							lock.wait();
						}
						try {
							render();
						}
						catch (final Exception e) {
							e.printStackTrace();
						}
						dirty = false;
						lock.notify();
					}
				}
			}
			catch (final InterruptedException e) {
			}
			synchronized (lock) {
				dirty = false;
				lock.notify();
			}
		}

		/**
		 * 
		 */
		public void stop() {
			try {
				if (thread != null) {
					isInterrupted = true;
					running = false;
					thread.interrupt();
					thread.join();
				}
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}

		/**
		 * 
		 */
		public void start() {
			if (thread == null) {
				thread = factory.newThread(this);
				thread.setName(thread.getName() + " RenderTask");
				isInterrupted = false;
				running = true;
				thread.start();
			}
		}

		/**
		 * 
		 */
		public void executeTask() {
			synchronized (lock) {
				isInterrupted = false;
				dirty = true;
				lock.notify();
			}
		}

		/**
		 * @throws InterruptedException
		 */
		public void waitTask() throws InterruptedException {
			synchronized (lock) {
				while (dirty) {
					lock.wait();
				}
			}
		}

		/**
		 * 
		 */
		public void abortTask() {
			isInterrupted = true;
		}
	}

	/**
	 * @see java.lang.Object#finalize()
	 */
	@Override
	public void finalize() throws Throwable {
		dispose();
		area = null;
		transform = null;
		renderingStrategy = null;
		super.finalize();
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#dispose()
	 */
	public void dispose() {
		renderTask.stop();
		free();
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#getFractal()
	 */
	public MandelbrotFractalRuntimeElement getFractal() {
		return fractal;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#setFractal(net.sf.jame.mandelbrot.fractal.MandelbrotFractalRuntimeElement)
	 */
	public void setFractal(final MandelbrotFractalRuntimeElement fractal) {
		this.fractal = fractal;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#setView(net.sf.jame.twister.View, net.sf.jame.twister.DoubleVector2D, int)
	 */
	public void setView(final View view, final DoubleVector2D constant, final int imageMode) {
		synchronized (this) {
			newView = view;
			newConstant = constant;
			newImageMode = imageMode;
		}
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#setView(net.sf.jame.twister.View)
	 */
	public void setView(final View view) {
		synchronized (this) {
			newView = view;
		}
	}

	private void updateView(final View view) {
		rotationValue = view.getRotation().getZ();
		newShiftValue = (int) Math.rint(view.getRotation().getW());
		isDynamic = (view.getStatus().getZ() == 1) || (view.getStatus().getW() == 1);
		isZoomDynamic = isDynamic;
		if (view.getStatus().getZ() == 2) {
			setMode(FractalRenderer.MODE_CALCULATE);
		}
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#isDynamic()
	 */
	public boolean isDynamic() {
		final boolean value = isDynamic;
		isDynamic = false;
		return value;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#isViewChanged()
	 */
	public boolean isViewChanged() {
		final boolean value = isViewChanged;
		isViewChanged = false;
		return value;
	}

	/**
	 * 
	 */
	protected void free() {
		if (newImage != null) {
			newImage.flush();
			newBuffer.dispose();
		}
		newImage = null;
		newBuffer = null;
		if (oldImage != null) {
			oldImage.flush();
			oldBuffer.dispose();
		}
		oldImage = null;
		oldBuffer = null;
	}

	/**
	 * 
	 */
	protected void init() {
		imageDim = (int) Math.sqrt(((oldTile.getImageSize().getX() + oldTile.getTileBorder().getX() * 2) * (oldTile.getImageSize().getX() + oldTile.getTileBorder().getX() * 2)) + ((oldTile.getImageSize().getY() + oldTile.getTileBorder().getY() * 2) * (oldTile.getImageSize().getY() + oldTile.getTileBorder().getY() * 2)));
		tileDim = (int) Math.sqrt(((oldTile.getTileSize().getX() + oldTile.getTileBorder().getX() * 2) * (oldTile.getTileSize().getX() + oldTile.getTileBorder().getX() * 2)) + ((oldTile.getTileSize().getY() + oldTile.getTileBorder().getY() * 2) * (oldTile.getTileSize().getY() + oldTile.getTileBorder().getY() * 2)));
		bufferSize = new IntegerVector2D(tileDim, tileDim);
		newImage = new BufferedImage(bufferSize.getX(), bufferSize.getY(), Surface.DEFAULT_TYPE);
		oldImage = new BufferedImage(bufferSize.getX(), bufferSize.getY(), Surface.DEFAULT_TYPE);
		newBuffer = newImage.createGraphics();
		oldBuffer = oldImage.createGraphics();
	}

	/**
	 * 
	 */
	protected final void swapImages() {
		final BufferedImage tmpImage = oldImage;
		oldImage = newImage;
		newImage = tmpImage;
		final Graphics2D tmpBuffer = oldBuffer;
		oldBuffer = newBuffer;
		newBuffer = tmpBuffer;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#setMode(int)
	 */
	public void setMode(final int renderMode) {
		this.renderMode |= renderMode;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#getMode()
	 */
	public int getMode() {
		return renderMode;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#startRenderer()
	 */
	public void startRenderer() {
		renderTask.executeTask();
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#abortRenderer()
	 */
	public void abortRenderer() {
		renderTask.abortTask();
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#joinRenderer()
	 */
	public void joinRenderer() throws InterruptedException {
		renderTask.waitTask();
	}

	private void render() {
		try {
			synchronized (this) {
				if (oldView != newView) {
					isViewChanged = true;
					updateView(newView);
				}
				if (newShiftValue != oldShiftValue) {
					setMode(FractalRenderer.MODE_REFRESH);
				}
				if (newImageMode != oldImageMode) {
					setMode(FractalRenderer.MODE_CALCULATE);
				}
				if ((newConstant != oldConstant) && (newImageMode != 0)) {
					setMode(FractalRenderer.MODE_CALCULATE);
				}
				if (fractal.isRenderingFormulaChanged()) {
					setMode(FractalRenderer.MODE_CALCULATE);
				}
				if (fractal.isTransformingFormulaChanged()) {
					setMode(FractalRenderer.MODE_CALCULATE);
				}
				if (fractal.isIncolouringFormulaChanged()) {
					setMode(FractalRenderer.MODE_REFRESH);
				}
				if (fractal.isOutcolouringFormulaChanged()) {
					setMode(FractalRenderer.MODE_REFRESH);
				}
				// if (!isDynamic) {
				// setMode(FractalRenderer.MODE_REFRESH);
				// }
				oldView = newView;
				oldConstant = newConstant;
				oldImageMode = newImageMode;
				oldShiftValue = newShiftValue;
				if (newTile != oldTile) {
					setMode(FractalRenderer.MODE_CALCULATE);
					oldTile = newTile;
					free();
					init();
				}
				if (oldImageMode == 0) {
					renderingStrategy = getMandelbrotRenderingStrategy();
				}
				else {
					renderingStrategy = getJuliaRenderingStrategy();
				}
				percent = 0;
				status = TwisterRenderer.STATUS_RENDERING;
			}
			doFractal(isZoomDynamic);
			if (percent == 100) {
				status = TwisterRenderer.STATUS_TERMINATED;
			}
			else {
				status = TwisterRenderer.STATUS_ABORTED;
			}
			isZoomDynamic = false;
		}
		catch (final OutOfMemoryError e) {
			e.printStackTrace();
		}
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#getRenderingStatus()
	 */
	public int getRenderingStatus() {
		return status;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#setTile(net.sf.jame.twister.ImageTile)
	 */
	public void setTile(final ImageTile tile) {
		synchronized (this) {
			newTile = tile;
		}
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#getTile()
	 */
	public ImageTile getTile() {
		return oldTile;
	}

	/**
	 * @return
	 */
	public int getBufferWidth() {
		return bufferSize.getX();
	}

	/**
	 * @return
	 */
	public int getBufferHeight() {
		return bufferSize.getY();
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#setMandelbrotMode(boolean)
	 */
	public void setMandelbrotMode(final Integer mode) {
		synchronized (this) {
			newImageMode = mode;
		}
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#setConstant(net.sf.jame.twister.DoubleVector2D)
	 */
	public void setConstant(final DoubleVector2D constant) {
		synchronized (this) {
			newConstant = constant;
		}
	}

	/**
	 * @return
	 */
	protected abstract boolean isTileSupported();

	/**
	 * 
	 */
	protected void updateShift() {
	}

	/**
	 * 
	 */
	protected void updateRegion() {
		if (fractal.getRenderingFormula().getFormulaRuntime() != null) {
			final DoubleVector2D s = fractal.getRenderingFormula().getFormulaRuntime().getScale();
			final double x = oldView.getPosition().getX();
			final double y = oldView.getPosition().getY();
			final double z = oldView.getPosition().getZ();
			scale.r = s.getX() * z;
			scale.i = s.getY() * z;
			center.r = fractal.getRenderingFormula().getFormulaRuntime().getCenter().getX() + x;
			center.i = fractal.getRenderingFormula().getFormulaRuntime().getCenter().getY() + y;
			final double imageOffsetX = (imageDim - oldTile.getImageSize().getX() - oldTile.getTileBorder().getX() * 2) / 2;
			final double imageOffsetY = (imageDim - oldTile.getImageSize().getY() - oldTile.getTileBorder().getY() * 2) / 2;
			double sx = (scale.r * 0.5d * imageDim) / oldTile.getImageSize().getX();
			double sy = (scale.i * 0.5d * imageDim) / oldTile.getImageSize().getX();
			Complex p0 = new Complex(center.r - sx, center.i - sy);
			Complex p1 = new Complex(center.r + sx, center.i + sy);
			final Complex t0 = new Complex();
			final Complex t1 = new Complex();
			final double dr = p1.r - p0.r;
			final double di = p1.i - p0.i;
			t0.r = p0.r + dr * (imageOffsetX + oldTile.getTileOffset().getX() + oldTile.getTileSize().getX() / 2d) / imageDim;
			t0.i = p0.i + di * (imageOffsetY + oldTile.getTileOffset().getY() + oldTile.getTileSize().getY() / 2d) / imageDim;
			t1.r = p0.r + dr * (imageOffsetX + oldTile.getImageSize().getX() / 2d) / imageDim;
			t1.i = p0.i + di * (imageOffsetY + oldTile.getImageSize().getY() / 2d) / imageDim;
			final AffineTransform t = new AffineTransform();
			t.rotate(-rotationValue, t1.r / dr, t1.i / di);
			Point2D.Double p = new Point2D.Double(t0.r / dr, t0.i / di);
			p = (Point2D.Double) t.transform(p, p);
			p.setLocation(p.getX() * dr, p.getY() * di);
			sx = dr * (bufferSize.getX() / 2d) / imageDim;
			sy = di * (bufferSize.getY() / 2d) / imageDim;
			p0 = new Complex(p.getX() - sx, p.getY() - sy);
			p1 = new Complex(p.getX() + sx, p.getY() + sy);
			area.points[0] = p0;
			area.points[1] = p1;
		}
		else {
			final Complex p0 = new Complex(-0.5, +0.5);
			final Complex p1 = new Complex(-0.5, +0.5);
			area.points[0] = p0;
			area.points[1] = p1;
		}
	}

	/**
	 * 
	 */
	protected void updateTransform() {
		final int offsetX = (getBufferWidth() - oldTile.getTileSize().getX()) / 2;
		final int offsetY = (getBufferHeight() - oldTile.getTileSize().getY()) / 2;
		transform = AffineTransform.getTranslateInstance(-offsetX, -offsetY);
		final int centerX = getBufferWidth() / 2 + oldTile.getTileBorder().getX();
		final int centerY = getBufferHeight() / 2 + oldTile.getTileBorder().getY();
		transform.rotate(rotationValue, centerX, centerY);
	}

	/**
	 * @param p
	 * @return the color.
	 */
	protected int renderPoint(final RenderedPoint cp) {
		fractal.getRenderingFormula().getFormulaRuntime().renderPoint(cp);
		return renderColor(cp);
	}

	/**
	 * @param p
	 * @return the color.
	 */
	protected int renderColor(final RenderedPoint cp) {
		int newRGB = 0;
		int tmpRGB = 0;
		if (cp.time > 0) {
			if (fractal.getOutcolouringFormulaCount() == 1) {
				final OutcolouringFormulaRuntimeFormula outcolouringFormula = fractal.getOutcolouringFormula(0);
				if (outcolouringFormula.getFormulaRuntime() != null && outcolouringFormula.isEnabled()) {
					if (newShiftValue != 0) {
						newRGB = outcolouringFormula.getFormulaRuntime().renderColor(cp, newShiftValue);
					}
					else {
						newRGB = outcolouringFormula.getFormulaRuntime().renderColor(cp);
					}
				}
			}
			else {
				for (int i = 0; i < fractal.getOutcolouringFormulaCount(); i++) {
					final OutcolouringFormulaRuntimeFormula outcolouringFormula = fractal.getOutcolouringFormula(i);
					if (outcolouringFormula.getFormulaRuntime() != null && outcolouringFormula.isEnabled()) {
						if (newShiftValue != 0) {
							tmpRGB = outcolouringFormula.getFormulaRuntime().renderColor(cp, newShiftValue);
						}
						else {
							tmpRGB = outcolouringFormula.getFormulaRuntime().renderColor(cp);
						}
						newRGB = (newRGB != 0) ? Colors.mixColors(newRGB, tmpRGB, outcolouringFormula.getOpacity()) : tmpRGB;
					}
				}
			}
			return newRGB;
		}
		else {
			if (fractal.getIncolouringFormulaCount() == 1) {
				final IncolouringFormulaRuntimeElement incolouringFormula = fractal.getIncolouringFormula(0);
				if (incolouringFormula.getFormulaRuntime() != null && incolouringFormula.isEnabled()) {
					if (newShiftValue != 0) {
						newRGB = incolouringFormula.getFormulaRuntime().renderColor(cp, newShiftValue);
					}
					else {
						newRGB = incolouringFormula.getFormulaRuntime().renderColor(cp);
					}
				}
			}
			else {
				for (int i = 0; i < fractal.getIncolouringFormulaCount(); i++) {
					final IncolouringFormulaRuntimeElement incolouringFormula = fractal.getIncolouringFormula(i);
					if (incolouringFormula.getFormulaRuntime() != null && incolouringFormula.isEnabled()) {
						if (newShiftValue != 0) {
							tmpRGB = incolouringFormula.getFormulaRuntime().renderColor(cp, newShiftValue);
						}
						else {
							tmpRGB = incolouringFormula.getFormulaRuntime().renderColor(cp);
						}
						newRGB = (newRGB != 0) ? Colors.mixColors(newRGB, tmpRGB, incolouringFormula.getOpacity()) : tmpRGB;
					}
				}
			}
			return newRGB;
		}
	}

	/**
	 * @param dynamic
	 */
	protected abstract void doFractal(boolean dynamic);

	/**
	 * @return true if solidguess is supported.
	 */
	public boolean isSolidGuessSupported() {
		for (int i = 0; i < fractal.getOutcolouringFormulaCount(); i++) {
			final OutcolouringFormulaRuntimeFormula outcolouringFormula = fractal.getOutcolouringFormula(i);
			if (outcolouringFormula.getFormulaRuntime() != null && !outcolouringFormula.getFormulaRuntime().isSolidGuessAllowed() && outcolouringFormula.isEnabled()) {
				return false;
			}
		}
		for (int i = 0; i < fractal.getIncolouringFormulaCount(); i++) {
			final IncolouringFormulaRuntimeElement incolouringFormula = fractal.getIncolouringFormula(i);
			if (incolouringFormula.getFormulaRuntime() != null && !incolouringFormula.getFormulaRuntime().isSolidGuessAllowed() && incolouringFormula.isEnabled()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * @return true if symetry is supported.
	 */
	public boolean isVerticalSymetrySupported() {
		return renderingStrategy.isVerticalSymetrySupported();
	}

	/**
	 * @return true if symetry is supported.
	 */
	public boolean isHorizontalSymetrySupported() {
		return renderingStrategy.isHorizontalSymetrySupported();
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#drawImage(java.awt.Graphics2D)
	 */
	public void drawImage(final Graphics2D g) {
		final AffineTransform t = g.getTransform();
		if (oldTile != null) {
			// g.setClip(oldTile.getTileBorder().getX(), oldTile.getTileBorder().getY(), oldTile.getTileSize().getX(), oldTile.getTileSize().getY());
			// g.setClip(0, 0, oldTile.getTileSize().getX() + oldTile.getTileBorder().getX() + 2, oldTile.getTileSize().getY() + oldTile.getTileBorder().getY() + 2);
			g.setTransform(transform);
			g.drawImage(newImage, 0, 0, null);
			g.setTransform(t);
			g.setClip(null);
			// g.dispose();
		}
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#drawImage(java.awt.Graphics2D, int, int, int, int)
	 */
	public void drawImage(final Graphics2D g, final int x, final int y, final int w, final int h) {
		final AffineTransform t = g.getTransform();
		if (oldTile != null) {
			g.setClip(x, y, w, h);
			g.setTransform(transform);
			final double sx = w / (double) getTile().getTileSize().getX();
			final double sy = h / (double) getTile().getTileSize().getY();
			final int dw = (int) Math.rint(bufferSize.getX() * sx);
			final int dh = (int) Math.rint(bufferSize.getY() * sy);
			g.drawImage(newImage, x, y, dw, dh, null);
			g.setTransform(t);
			g.setClip(null);
			// g.dispose();
		}
	}

	/**
	 * @return
	 */
	protected Graphics2D getGraphics() {
		swapImages();
		return newBuffer;
	}

	/**
	 * @see net.sf.jame.mandelbrot.renderer.FractalRenderer#isInterrupted()
	 */
	public boolean isInterrupted() {
		return isInterrupted;
	}

	/**
	 * @return the strategy.
	 */
	protected abstract RenderingStrategy getMandelbrotRenderingStrategy();

	/**
	 * @return the strategy.
	 */
	protected abstract RenderingStrategy getJuliaRenderingStrategy();

	/**
	 * @author Andrea Medeghini
	 */
	protected interface RenderingStrategy {
		/**
		 * @param p
		 * @param pw
		 * @param px
		 * @return the time
		 */
		public int renderPoint(RenderedPoint p, Complex px, Complex pw);

		/**
		 * @return true if vertical symetry is supported.
		 */
		public boolean isVerticalSymetrySupported();

		/**
		 * @return true if horizontal symetry is supported.
		 */
		public boolean isHorizontalSymetrySupported();

		/**
		 * 
		 */
		public void updateParameters();
	}
}
