/*
 * $Id:MandelbrotConfigPanel.java 482 2008-01-24 21:49:17Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.mandelbrot.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.LayoutManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import net.sf.jame.core.config.ListConfigElementEvents;
import net.sf.jame.core.config.ValueChangeEvent;
import net.sf.jame.core.config.ValueChangeListener;
import net.sf.jame.core.config.ValueConfigElementEvents;
import net.sf.jame.core.extension.ConfigurableExtension;
import net.sf.jame.core.extension.Extension;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.core.extension.NullConfigurableExtension;
import net.sf.jame.core.swing.extension.ConfigurableExtensionComboBoxModel;
import net.sf.jame.core.swing.extension.ExtensionListCellRenderer;
import net.sf.jame.core.swing.util.GUIFactory;
import net.sf.jame.core.tree.NodeSession;
import net.sf.jame.mandelbrot.MandelbrotConfig;
import net.sf.jame.mandelbrot.MandelbrotRegistry;
import net.sf.jame.mandelbrot.extensions.image.MandelbrotImageConfig;
import net.sf.jame.mandelbrot.fractal.MandelbrotFractalConfigElement;
import net.sf.jame.mandelbrot.fractal.incolouring.IncolouringFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.incolouring.extension.IncolouringFormulaExtensionConfig;
import net.sf.jame.mandelbrot.fractal.incolouring.extension.IncolouringFormulaExtensionRuntime;
import net.sf.jame.mandelbrot.fractal.outcolouring.OutcolouringFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.outcolouring.extension.OutcolouringFormulaExtensionConfig;
import net.sf.jame.mandelbrot.fractal.outcolouring.extension.OutcolouringFormulaExtensionRuntime;
import net.sf.jame.mandelbrot.fractal.rendering.RenderingFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.rendering.extension.RenderingFormulaExtensionConfig;
import net.sf.jame.mandelbrot.fractal.rendering.extension.RenderingFormulaExtensionRuntime;
import net.sf.jame.mandelbrot.fractal.transforming.TransformingFormulaConfigElement;
import net.sf.jame.mandelbrot.fractal.transforming.extension.TransformingFormulaExtensionConfig;
import net.sf.jame.mandelbrot.fractal.transforming.extension.TransformingFormulaExtensionRuntime;
import net.sf.jame.twister.DoubleVector4D;
import net.sf.jame.twister.Speed;
import net.sf.jame.twister.renderer.RenderContext;
import net.sf.jame.twister.swing.StackLayout;
import net.sf.jame.twister.swing.TwisterSwingRegistry;
import net.sf.jame.twister.swing.TwisterSwingResources;
import net.sf.jame.twister.swing.ViewContext;
import net.sf.jame.twister.swing.view.NavigatorViewRuntime;
import net.sf.jame.twister.swing.view.extension.ViewExtensionRuntime;
import net.sf.jame.twister.util.ExtensionConfigElementEvents;

/**
 * @author Andrea Medeghini
 */
public class MandelbrotConfigPanel extends JPanel {
	private static final long serialVersionUID = 1L;
	private static final Color oddColor = new Color(0xFFDDDDDD);
	private static final Color evenColor = new Color(0xFFD0D0D0);
	private MandelbrotFractalPanel fractalPanel;
	private final ViewContext viewContext;
	private final RenderContext context;
	private final NodeSession session;

	/**
	 * @param config
	 * @param viewContext
	 * @param context
	 * @param session
	 */
	public MandelbrotConfigPanel(final MandelbrotConfig config, final ViewContext viewContext, final RenderContext context, final NodeSession session) {
		this.viewContext = viewContext;
		this.context = context;
		this.session = session;
		final MandelbrotImagePanel imagePanel = new MandelbrotImagePanel(config);
		fractalPanel = new MandelbrotFractalPanel(config, config.getMandelbrotFractalConfigElement());
		setLayout(new BorderLayout());
		add(fractalPanel, BorderLayout.CENTER);
		add(imagePanel, BorderLayout.SOUTH);
		setBorder(BorderFactory.createMatteBorder(1, 0, 1, 0, Color.DARK_GRAY));
		final ValueChangeListener configListener = new ValueChangeListener() {
			public void valueChanged(ValueChangeEvent e) {
				switch (e.getEventType()) {
					case ValueConfigElementEvents.VALUE_CHANGED: {
						remove(fractalPanel);
						fractalPanel = new MandelbrotFractalPanel(config, config.getMandelbrotFractalConfigElement());
						add(fractalPanel, BorderLayout.CENTER);
						viewContext.setComponent(MandelbrotConfigPanel.this);
						break;
					}
					default: {
						break;
					}
				}
			}
		};
		config.getFractalSingleElement().addChangeListener(configListener);
	}

	private static JCheckBox createIconCheckBox(final String key, final String iconKey, final int width, final int height) {
		final JCheckBox checkbox = GUIFactory.createCheckBox((String) null, MandelbrotSwingResources.getInstance().getString("tooltip." + key));
		try {
			checkbox.setIcon(new ImageIcon(ImageIO.read(MandelbrotConfigPanel.class.getResourceAsStream("/icons/" + iconKey + "-icon.png"))));
			checkbox.setSelectedIcon(new ImageIcon(ImageIO.read(MandelbrotConfigPanel.class.getResourceAsStream("/icons/" + iconKey + "-selected-icon.png"))));
		}
		catch (final Exception e) {
			System.out.println("key = " + key + ", iconKey = " + iconKey);
			e.printStackTrace();
		}
		checkbox.setOpaque(false);
		checkbox.setPreferredSize(new Dimension(width, height));
		checkbox.setMinimumSize(new Dimension(width, height));
		checkbox.setMaximumSize(new Dimension(width, height));
		return checkbox;
	}

	private static JButton createIconButton(final String key, final String iconKey, final int width, final int height) {
		final JButton button = GUIFactory.createButton((String) null, MandelbrotSwingResources.getInstance().getString("tooltip." + key));
		try {
			button.setIcon(new ImageIcon(ImageIO.read(MandelbrotConfigPanel.class.getResourceAsStream("/icons/" + iconKey + "-icon.png"))));
			button.setPressedIcon(new ImageIcon(ImageIO.read(MandelbrotConfigPanel.class.getResourceAsStream("/icons/" + iconKey + "-pressed-icon.png"))));
		}
		catch (final Exception e) {
			System.out.println("key = " + key + ", iconKey = " + iconKey);
			e.printStackTrace();
		}
		button.setOpaque(false);
		button.setPreferredSize(new Dimension(width, height));
		button.setMinimumSize(new Dimension(width, height));
		button.setMaximumSize(new Dimension(width, height));
		return button;
	}

	private static JCheckBox createSelectionCheckBox() {
		final JCheckBox checkbox = GUIFactory.createSmallCheckBox((String) null, (String) null);
		checkbox.setOpaque(false);
		checkbox.setPreferredSize(new Dimension(20, 20));
		checkbox.setMinimumSize(new Dimension(20, 20));
		checkbox.setMaximumSize(new Dimension(20, 20));
		return checkbox;
	}

	private static JCheckBox createCheckBox() {
		final JCheckBox checkbox = GUIFactory.createSmallCheckBox((String) null, (String) null);
		checkbox.setOpaque(false);
		return checkbox;
	}

	private static JCheckBox createTextCheckBox(final String key, final int width, final int height) {
		final JCheckBox checkbox = GUIFactory.createCheckBox(MandelbrotSwingResources.getInstance().getString("label." + key), MandelbrotSwingResources.getInstance().getString("tooltip." + key));
		// final FontMetrics fm = checkbox.getFontMetrics(checkbox.getFont());
		// int width = fm.stringWidth(checkbox.getText()) + 20;
		checkbox.setPreferredSize(new Dimension(width, height));
		checkbox.setMinimumSize(new Dimension(width, height));
		checkbox.setMaximumSize(new Dimension(width, height));
		checkbox.setOpaque(false);
		return checkbox;
	}

	private static JButton createTextButton(final int width, final int height) {
		final JButton button = GUIFactory.createSmallButton((String) null, (String) null);
		button.setPreferredSize(new Dimension(width, height));
		button.setMinimumSize(new Dimension(width, height));
		button.setMaximumSize(new Dimension(width, height));
		button.setOpaque(false);
		return button;
	}

	private static JButton createTextButton(final String key, final int width, final int height) {
		final JButton button = GUIFactory.createSmallButton(MandelbrotSwingResources.getInstance().getString("label." + key), MandelbrotSwingResources.getInstance().getString("tooltip." + key));
		// final FontMetrics fm = button.getFontMetrics(button.getFont());
		// int width = fm.stringWidth(button.getText());
		button.setPreferredSize(new Dimension(width, height));
		button.setMinimumSize(new Dimension(width, height));
		button.setMaximumSize(new Dimension(width, height));
		button.setOpaque(false);
		return button;
	}

	private static JSpinner createSpinner(final int min, final int max, final int step) {
		final JSpinner spinner = GUIFactory.createSpinner(new SpinnerNumberModel(0, min, max, step), (String) null);
		// spinner.setPreferredSize(new Dimension(60, GUIFactory.DEFAULT_HEIGHT));
		spinner.setMaximumSize(new Dimension(60, GUIFactory.DEFAULT_HEIGHT));
		spinner.setMinimumSize(new Dimension(60, GUIFactory.DEFAULT_HEIGHT));
		// if (!"Mac OS X".equals(System.getProperty("os.name")) || !UIManager.getLookAndFeel().isNativeLookAndFeel()) {
		// spinner.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED, Color.LIGHT_GRAY, Color.DARK_GRAY));
		// }
		return spinner;
	}

	private static JComboBox createExtensionComboBox(final ComboBoxModel model, final int width, final int height) {
		final JComboBox extensionComboBox = GUIFactory.createSmallComboBox(model, (String) null);
		extensionComboBox.setRenderer(new ExtensionListCellRenderer());
		extensionComboBox.setPreferredSize(new Dimension(width, height));
		extensionComboBox.setMaximumSize(new Dimension(width, height));
		extensionComboBox.setMinimumSize(new Dimension(width, height));
		extensionComboBox.setOpaque(false);
		return extensionComboBox;
	}

	private static JLabel createTextLabel(final String key, final int alignment, final int width, final int height) {
		final JLabel label = GUIFactory.createSmallLabel(MandelbrotSwingResources.getInstance().getString("label." + key), SwingConstants.LEFT);
		label.setBorder(BorderFactory.createEmptyBorder(0, 8, 0, 8));
		label.setPreferredSize(new Dimension(width, height));
		label.setMinimumSize(new Dimension(width, height));
		label.setMaximumSize(new Dimension(width, height));
		return label;
	}

	private static JLabel createLabel(final String text) {
		final JLabel label = GUIFactory.createSmallLabel(text, SwingConstants.LEFT);
		label.setBorder(BorderFactory.createEmptyBorder(0, 8, 0, 8));
		return label;
	}

	private static JTextField createTextField(final String text, final int width, final int height) {
		final JTextField textfield = GUIFactory.createTextField(text, null);
		textfield.setPreferredSize(new Dimension(width, height));
		textfield.setMinimumSize(new Dimension(width, height));
		textfield.setMaximumSize(new Dimension(width, height));
		// if (!"Mac OS X".equals(System.getProperty("os.name")) || !UIManager.getLookAndFeel().isNativeLookAndFeel()) {
		// textfield.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED, Color.LIGHT_GRAY, Color.DARK_GRAY));
		// }
		return textfield;
	}

	private static JPanel createPanel(final LayoutManager layoutManager, final boolean opaque) {
		final JPanel panel = new JPanel(layoutManager);
		panel.setOpaque(opaque);
		return panel;
	}

	private static Box createHorizontalBox(final boolean opaque) {
		final Box box = Box.createHorizontalBox();
		box.setOpaque(opaque);
		return box;
	}

	private static Box createVerticalBox(final boolean opaque) {
		final Box box = Box.createVerticalBox();
		box.setOpaque(opaque);
		return box;
	}

	private static Component createSpace() {
		final Component box = Box.createHorizontalStrut(4);
		return box;
	}

	private class MandelbrotImagePanel extends JPanel {
		private static final long serialVersionUID = 1L;

		public MandelbrotImagePanel(final MandelbrotConfig config) {
			final JCheckBox showPreviewCheckBox = createCheckBox();
			final JCheckBox showOrbitCheckBox = createCheckBox();
			final JLabel showPreviewLabel = createTextLabel("showPreview", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel showOrbitLabel = createTextLabel("showOrbit", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel imageModeLabel = createTextLabel("imageMode", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel inputModeLabel = createTextLabel("inputMode", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel zoomSpeedLabel = createTextLabel("zoomSpeed", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel shiftSpeedLabel = createTextLabel("shiftSpeed", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel rotationSpeedLabel = createTextLabel("rotationSpeed", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final JTextField zoomSpeedTextfield = createTextField(String.valueOf(config.getSpeed().getPosition().getZ()), 200, GUIFactory.DEFAULT_HEIGHT);
			final JTextField shiftSpeedTextfield = createTextField(String.valueOf(config.getSpeed().getPosition().getW()), 200, GUIFactory.DEFAULT_HEIGHT);
			final JTextField rotationSpeedTextfield = createTextField(String.valueOf(config.getSpeed().getRotation().getZ()), 200, GUIFactory.DEFAULT_HEIGHT);
			final JComboBox imageModeComboBox = GUIFactory.createSmallComboBox(null);
			imageModeComboBox.setOpaque(false);
			imageModeComboBox.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			imageModeComboBox.setMaximumSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			imageModeComboBox.setMinimumSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			imageModeComboBox.addItem(new Object[] { "Mandelbrot", MandelbrotImageConfig.IMAGE_MODE_MANDELBROT });
			imageModeComboBox.addItem(new Object[] { "Julia", MandelbrotImageConfig.IMAGE_MODE_JULIA });
			imageModeComboBox.setRenderer(new DefaultListCellRenderer() {
				private static final long serialVersionUID = 1L;

				@Override
				public Component getListCellRendererComponent(final JList list, final Object value, final int index, final boolean isSelected, final boolean cellHasFocus) {
					return super.getListCellRendererComponent(list, ((Object[]) value)[0], index, isSelected, cellHasFocus);
				}
			});
			final JComboBox inputModeComboBox = GUIFactory.createSmallComboBox(null);
			inputModeComboBox.setOpaque(false);
			inputModeComboBox.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			inputModeComboBox.setMaximumSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			inputModeComboBox.setMinimumSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			inputModeComboBox.addItem(new Object[] { TwisterSwingResources.getInstance().getString("label.zoom"), MandelbrotImageConfig.INPUT_MODE_ZOOM });
			inputModeComboBox.addItem(new Object[] { TwisterSwingResources.getInstance().getString("label.select"), MandelbrotImageConfig.INPUT_MODE_SELECT });
			inputModeComboBox.setRenderer(new DefaultListCellRenderer() {
				private static final long serialVersionUID = 1L;

				@Override
				public Component getListCellRendererComponent(final JList list, final Object value, final int index, final boolean isSelected, final boolean cellHasFocus) {
					return super.getListCellRendererComponent(list, ((Object[]) value)[0], index, isSelected, cellHasFocus);
				}
			});
			final Box tmpPanel0 = createHorizontalBox(false);
			tmpPanel0.add(imageModeLabel);
			tmpPanel0.add(createSpace());
			tmpPanel0.add(imageModeComboBox);
			tmpPanel0.add(Box.createHorizontalGlue());
			final Box tmpPanel1 = createHorizontalBox(false);
			tmpPanel1.add(showOrbitLabel);
			tmpPanel1.add(createSpace());
			tmpPanel1.add(showOrbitCheckBox);
			tmpPanel1.add(Box.createHorizontalGlue());
			final Box tmpPanel2 = createHorizontalBox(false);
			tmpPanel2.add(showPreviewLabel);
			tmpPanel2.add(createSpace());
			tmpPanel2.add(showPreviewCheckBox);
			tmpPanel2.add(Box.createHorizontalGlue());
			final Box tmpPanel3 = createHorizontalBox(false);
			tmpPanel3.add(inputModeLabel);
			tmpPanel3.add(createSpace());
			tmpPanel3.add(inputModeComboBox);
			tmpPanel3.add(Box.createHorizontalGlue());
			final Box tmpPanel4 = createHorizontalBox(false);
			tmpPanel4.add(zoomSpeedLabel);
			tmpPanel4.add(createSpace());
			tmpPanel4.add(zoomSpeedTextfield);
			tmpPanel4.add(Box.createHorizontalGlue());
			final Box tmpPanel5 = createHorizontalBox(false);
			tmpPanel5.add(rotationSpeedLabel);
			tmpPanel5.add(createSpace());
			tmpPanel5.add(rotationSpeedTextfield);
			tmpPanel5.add(Box.createHorizontalGlue());
			final Box tmpPanel6 = createHorizontalBox(false);
			tmpPanel6.add(shiftSpeedLabel);
			tmpPanel6.add(createSpace());
			tmpPanel6.add(shiftSpeedTextfield);
			tmpPanel6.add(Box.createHorizontalGlue());
			final Box tmpPanel7 = createVerticalBox(false);
			tmpPanel7.add(tmpPanel0);
			tmpPanel7.add(Box.createVerticalStrut(8));
			tmpPanel7.add(tmpPanel3);
			tmpPanel7.add(Box.createVerticalStrut(8));
			tmpPanel7.add(tmpPanel2);
			tmpPanel7.add(Box.createVerticalStrut(8));
			tmpPanel7.add(tmpPanel1);
			tmpPanel7.add(Box.createVerticalStrut(8));
			tmpPanel7.add(tmpPanel4);
			tmpPanel7.add(Box.createVerticalStrut(8));
			tmpPanel7.add(tmpPanel6);
			tmpPanel7.add(Box.createVerticalStrut(8));
			tmpPanel7.add(tmpPanel5);
			tmpPanel7.add(Box.createVerticalGlue());
			setLayout(new BorderLayout());
			add(tmpPanel7, BorderLayout.CENTER);
			setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			setOpaque(false);
			showPreviewCheckBox.setSelected(config.getShowPreview());
			showOrbitCheckBox.setSelected(config.getShowOrbit());
			updateImageMode(config, imageModeComboBox);
			updateInputMode(config, inputModeComboBox);
			final ValueChangeListener showPreviewListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					showPreviewCheckBox.setSelected(config.getShowPreview());
				}
			};
			final ValueChangeListener showOrbitListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					showOrbitCheckBox.setSelected(config.getShowOrbit());
				}
			};
			final ValueChangeListener imageModeListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					updateImageMode(config, imageModeComboBox);
				}
			};
			final ValueChangeListener inputModeListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					updateInputMode(config, inputModeComboBox);
				}
			};
			final ValueChangeListener speedListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					zoomSpeedTextfield.setText(String.valueOf(config.getSpeed().getPosition().getZ()));
					shiftSpeedTextfield.setText(String.valueOf(config.getSpeed().getPosition().getW()));
					rotationSpeedTextfield.setText(String.valueOf(config.getSpeed().getRotation().getZ()));
				}
			};
			config.getShowPreviewElement().addChangeListener(showPreviewListener);
			config.getShowOrbitElement().addChangeListener(showOrbitListener);
			config.getImageModeElement().addChangeListener(imageModeListener);
			config.getInputModeElement().addChangeListener(inputModeListener);
			config.getSpeedElement().addChangeListener(speedListener);
			showPreviewCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					config.getContext().updateTimestamp();
					config.setShowPreview(showPreviewCheckBox.isSelected());
					context.refresh();
				}
			});
			showOrbitCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					config.getContext().updateTimestamp();
					config.setShowOrbit(showOrbitCheckBox.isSelected());
					context.refresh();
				}
			});
			imageModeComboBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					// config.getImageModeElement().removeChangeListener(imageModeListener);
					config.getContext().updateTimestamp();
					config.setImageMode((Integer) ((Object[]) imageModeComboBox.getSelectedItem())[1]);
					// config.getImageModeElement().addChangeListener(imageModeListener);
					context.refresh();
				}
			});
			inputModeComboBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					// config.getInputModeElement().removeChangeListener(inputModeListener);
					config.getContext().updateTimestamp();
					config.setInputMode((Integer) ((Object[]) inputModeComboBox.getSelectedItem())[1]);
					// config.getInputModeElement().addChangeListener(inputModeListener);
					context.refresh();
				}
			});
			zoomSpeedTextfield.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					config.getContext().updateTimestamp();
					Speed speed = config.getSpeed();
					try {
						Speed newSpeed = new Speed(new DoubleVector4D(0, 0, Double.valueOf(zoomSpeedTextfield.getText()), speed.getPosition().getW()), speed.getRotation());
						config.setSpeed(newSpeed);
						context.refresh();
					}
					catch (Exception x) {
					}
				}
			});
			shiftSpeedTextfield.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					config.getContext().updateTimestamp();
					Speed speed = config.getSpeed();
					try {
						Speed newSpeed = new Speed(new DoubleVector4D(0, 0, speed.getPosition().getZ(), Double.valueOf(shiftSpeedTextfield.getText())), speed.getRotation());
						config.setSpeed(newSpeed);
						context.refresh();
					}
					catch (Exception x) {
					}
				}
			});
			rotationSpeedTextfield.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					config.getContext().updateTimestamp();
					Speed speed = config.getSpeed();
					try {
						Speed newSpeed = new Speed(speed.getPosition(), new DoubleVector4D(0, 0, Double.valueOf(rotationSpeedTextfield.getText()), 0));
						config.setSpeed(newSpeed);
						context.refresh();
					}
					catch (Exception x) {
					}
				}
			});
		}

		private void updateImageMode(final MandelbrotConfig config, final JComboBox imageModeComboBox) {
			switch (config.getImageMode()) {
				case MandelbrotImageConfig.IMAGE_MODE_MANDELBROT: {
					imageModeComboBox.setSelectedIndex(0);
					break;
				}
				case MandelbrotImageConfig.IMAGE_MODE_JULIA: {
					imageModeComboBox.setSelectedIndex(1);
					break;
				}
				default: {
					break;
				}
			}
		}

		private void updateInputMode(final MandelbrotConfig config, final JComboBox inputModeComboBox) {
			switch (config.getInputMode()) {
				case MandelbrotImageConfig.INPUT_MODE_ZOOM: {
					inputModeComboBox.setSelectedIndex(0);
					break;
				}
				case MandelbrotImageConfig.INPUT_MODE_SELECT: {
					inputModeComboBox.setSelectedIndex(1);
					break;
				}
				default: {
					break;
				}
			}
		}
	}

	private class MandelbrotFractalPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private RenderingFormulaPanel renderingFormulaPanel;
		private TransformingFormulaPanel transformingFormulaPanel;

		/**
		 * @param config
		 * @param fractalElement
		 */
		public MandelbrotFractalPanel(final MandelbrotConfig config, final MandelbrotFractalConfigElement fractalElement) {
			final JPanel incolouringFormulasPanel = createPanel(new StackLayout(), false);
			final JPanel outcolouringFormulasPanel = createPanel(new StackLayout(), false);
			final JButton editIncolouringFormulasButton = createTextButton(200, GUIFactory.DEFAULT_HEIGHT);
			final JButton editOutcolouringFormulasButton = createTextButton(200, GUIFactory.DEFAULT_HEIGHT);
			final JButton selectIncolouringFormulaButton = createIconButton("selectIncolouringFormulas", "select", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton appendIncolouringFormulaButton = createIconButton("appendIncolouringFormula", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton removeIncolouringFormulaButton = createIconButton("removeIncolouringFormulas", "remove", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton selectOutcolouringFormulaButton = createIconButton("selectOutcolouringFormulas", "select", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton appendOutcolouringFormulaButton = createIconButton("appendOutcolouringFormula", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton removeOutcolouringFormulaButton = createIconButton("removeOutcolouringFormulas", "remove", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JLabel incolouringFormulasLabel2 = createTextLabel("incolouringFormulas", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel outcolouringFormulasLabel2 = createTextLabel("outcolouringFormulas", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final Box outcolouringFormulasPanel4 = createHorizontalBox(false);
			outcolouringFormulasPanel4.add(Box.createHorizontalGlue());
			outcolouringFormulasPanel4.add(selectOutcolouringFormulaButton);
			outcolouringFormulasPanel4.add(createSpace());
			outcolouringFormulasPanel4.add(appendOutcolouringFormulaButton);
			outcolouringFormulasPanel4.add(createSpace());
			outcolouringFormulasPanel4.add(removeOutcolouringFormulaButton);
			outcolouringFormulasPanel4.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY), BorderFactory.createEmptyBorder(4, 4, 4, 4)));
			final JLabel outcolouringFormulasLabel = createTextLabel("noOutcolouringFormula", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			outcolouringFormulasLabel.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			final Box outcolouringFormulasPanel3 = createHorizontalBox(false);
			outcolouringFormulasPanel3.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			outcolouringFormulasPanel3.add(outcolouringFormulasLabel);
			outcolouringFormulasPanel3.add(Box.createHorizontalGlue());
			final JPanel outcolouringFormulasPanel2 = createPanel(new BorderLayout(), true);
			outcolouringFormulasPanel2.setName(createOutcolouringFormulasPanelName());
			outcolouringFormulasPanel2.add(outcolouringFormulasPanel4, BorderLayout.NORTH);
			outcolouringFormulasPanel2.add(outcolouringFormulasPanel, BorderLayout.CENTER);
			outcolouringFormulasPanel2.add(outcolouringFormulasPanel3, BorderLayout.SOUTH);
			outcolouringFormulasPanel2.setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY));
			final Box incolouringFormulasPanel4 = createHorizontalBox(false);
			incolouringFormulasPanel4.add(Box.createHorizontalGlue());
			incolouringFormulasPanel4.add(selectIncolouringFormulaButton);
			incolouringFormulasPanel4.add(createSpace());
			incolouringFormulasPanel4.add(appendIncolouringFormulaButton);
			incolouringFormulasPanel4.add(createSpace());
			incolouringFormulasPanel4.add(removeIncolouringFormulaButton);
			incolouringFormulasPanel4.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY), BorderFactory.createEmptyBorder(4, 4, 4, 4)));
			final JLabel incolouringFormulasLabel = createTextLabel("noIncolouringFormula", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			incolouringFormulasLabel.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			final Box incolouringFormulasPanel3 = createHorizontalBox(false);
			incolouringFormulasPanel3.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			incolouringFormulasPanel3.add(incolouringFormulasLabel);
			incolouringFormulasPanel3.add(Box.createHorizontalGlue());
			final JPanel incolouringFormulasPanel2 = createPanel(new BorderLayout(), true);
			incolouringFormulasPanel2.setName(createIncolouringFormulasPanelName());
			incolouringFormulasPanel2.add(incolouringFormulasPanel4, BorderLayout.NORTH);
			incolouringFormulasPanel2.add(incolouringFormulasPanel, BorderLayout.CENTER);
			incolouringFormulasPanel2.add(incolouringFormulasPanel3, BorderLayout.SOUTH);
			incolouringFormulasPanel2.setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY));
			renderingFormulaPanel = new RenderingFormulaPanel(fractalElement, fractalElement.getRenderingFormulaConfigElement());
			transformingFormulaPanel = new TransformingFormulaPanel(fractalElement, fractalElement.getTransformingFormulaConfigElement());
			final Box outcolouringFormulasPanel5 = createHorizontalBox(false);
			outcolouringFormulasPanel5.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			outcolouringFormulasPanel5.add(outcolouringFormulasLabel2);
			outcolouringFormulasPanel5.add(createSpace());
			outcolouringFormulasPanel5.add(editOutcolouringFormulasButton);
			outcolouringFormulasPanel5.add(Box.createHorizontalGlue());
			final Box incolouringFormulasPanel5 = createHorizontalBox(false);
			incolouringFormulasPanel5.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			incolouringFormulasPanel5.add(incolouringFormulasLabel2);
			incolouringFormulasPanel5.add(createSpace());
			incolouringFormulasPanel5.add(editIncolouringFormulasButton);
			incolouringFormulasPanel5.add(Box.createHorizontalGlue());
			final Box subpanelContainer = createVerticalBox(false);
			subpanelContainer.add(Box.createVerticalStrut(8));
			subpanelContainer.add(renderingFormulaPanel);
			subpanelContainer.add(Box.createVerticalStrut(8));
			subpanelContainer.add(transformingFormulaPanel);
			subpanelContainer.add(Box.createVerticalStrut(8));
			subpanelContainer.add(incolouringFormulasPanel5);
			subpanelContainer.add(Box.createVerticalStrut(8));
			subpanelContainer.add(outcolouringFormulasPanel5);
			subpanelContainer.add(Box.createVerticalStrut(8));
			// Box formulasPanel = createHorizontalBox(false);
			// formulasPanel.add(Box.createHorizontalGlue());
			// formulasPanel.add(appendIncolouringFormulaButton);
			// formulasPanel.add(removeIncolouringFormulaButton);
			// formulasPanel.add(appendOutcolouringFormulaButton);
			// formulasPanel.add(removeOutcolouringFormulaButton);
			// formulasPanel.setBorder(BorderFactory.createMatteBorder(1, 0, 0, 0, Color.DARK_GRAY));
			setLayout(new BorderLayout());
			add(subpanelContainer, BorderLayout.CENTER);
			// add(formulasPanel, BorderLayout.SOUTH);
			editIncolouringFormulasButton.setText(createEditIncolouringFormulasText(fractalElement));
			editOutcolouringFormulasButton.setText(createEditOutcolouringFormulasText(fractalElement));
			incolouringFormulasPanel3.setVisible(fractalElement.getIncolouringFormulaConfigElementCount() == 0);
			outcolouringFormulasPanel3.setVisible(fractalElement.getOutcolouringFormulaConfigElementCount() == 0);
			for (int i = 0; i < fractalElement.getIncolouringFormulaConfigElementCount(); i++) {
				final IncolouringFormulaConfigElement formulaElement = fractalElement.getIncolouringFormulaConfigElement(i);
				final IncolouringFormulaPanel formulaPanel = new IncolouringFormulaPanel(fractalElement, formulaElement);
				incolouringFormulasPanel.add(formulaPanel);
			}
			for (int i = 0; i < fractalElement.getOutcolouringFormulaConfigElementCount(); i++) {
				final OutcolouringFormulaConfigElement formulaElement = fractalElement.getOutcolouringFormulaConfigElement(i);
				final OutcolouringFormulaPanel formulaPanel = new OutcolouringFormulaPanel(fractalElement, formulaElement);
				outcolouringFormulasPanel.add(formulaPanel);
			}
			for (int i = 0; i < incolouringFormulasPanel.getComponentCount(); i++) {
				incolouringFormulasPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
				((JComponent) incolouringFormulasPanel.getComponent(i)).setOpaque(i % 2 == 0);
			}
			for (int i = 0; i < outcolouringFormulasPanel.getComponentCount(); i++) {
				outcolouringFormulasPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
				((JComponent) outcolouringFormulasPanel.getComponent(i)).setOpaque(i % 2 == 0);
			}
			final ValueChangeListener incolouringFormulasListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ListConfigElementEvents.ELEMENT_ADDED: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							incolouringFormulasPanel.add(new IncolouringFormulaPanel(fractalElement, (IncolouringFormulaConfigElement) e.getParams()[0]));
							incolouringFormulasPanel3.setVisible(fractalElement.getIncolouringFormulaConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_AFTER: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							incolouringFormulasPanel.add(new IncolouringFormulaPanel(fractalElement, (IncolouringFormulaConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue() + 1);
							incolouringFormulasPanel3.setVisible(fractalElement.getIncolouringFormulaConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_BEFORE: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							incolouringFormulasPanel.add(new IncolouringFormulaPanel(fractalElement, (IncolouringFormulaConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue());
							incolouringFormulasPanel3.setVisible(fractalElement.getIncolouringFormulaConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_REMOVED: {
							incolouringFormulasPanel.remove(((Integer) e.getParams()[1]).intValue());
							incolouringFormulasPanel3.setVisible(fractalElement.getIncolouringFormulaConfigElementCount() == 0);
							viewContext.resize();
							break;
						}
						default: {
							break;
						}
					}
					for (int i = 0; i < incolouringFormulasPanel.getComponentCount(); i++) {
						incolouringFormulasPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
						((JComponent) incolouringFormulasPanel.getComponent(i)).setOpaque(i % 2 == 0);
					}
					editIncolouringFormulasButton.setText(createEditIncolouringFormulasText(fractalElement));
				}
			};
			final ValueChangeListener outcolouringFormulasListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ListConfigElementEvents.ELEMENT_ADDED: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							outcolouringFormulasPanel.add(new OutcolouringFormulaPanel(fractalElement, (OutcolouringFormulaConfigElement) e.getParams()[0]));
							outcolouringFormulasPanel3.setVisible(fractalElement.getOutcolouringFormulaConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_AFTER: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							outcolouringFormulasPanel.add(new OutcolouringFormulaPanel(fractalElement, (OutcolouringFormulaConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue() + 1);
							outcolouringFormulasPanel3.setVisible(fractalElement.getOutcolouringFormulaConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_BEFORE: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							outcolouringFormulasPanel.add(new OutcolouringFormulaPanel(fractalElement, (OutcolouringFormulaConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue());
							outcolouringFormulasPanel3.setVisible(fractalElement.getOutcolouringFormulaConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_REMOVED: {
							outcolouringFormulasPanel.remove(((Integer) e.getParams()[1]).intValue());
							outcolouringFormulasPanel3.setVisible(fractalElement.getOutcolouringFormulaConfigElementCount() == 0);
							viewContext.resize();
							break;
						}
						default: {
							break;
						}
					}
					for (int i = 0; i < outcolouringFormulasPanel.getComponentCount(); i++) {
						outcolouringFormulasPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
						((JComponent) outcolouringFormulasPanel.getComponent(i)).setOpaque(i % 2 == 0);
					}
					editOutcolouringFormulasButton.setText(createEditOutcolouringFormulasText(fractalElement));
				}
			};
			final ValueChangeListener renderingFormulaListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ValueConfigElementEvents.VALUE_CHANGED: {
							subpanelContainer.remove(renderingFormulaPanel);
							renderingFormulaPanel = new RenderingFormulaPanel(fractalElement, fractalElement.getRenderingFormulaConfigElement());
							subpanelContainer.add(renderingFormulaPanel);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			final ValueChangeListener transformingListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ValueConfigElementEvents.VALUE_CHANGED: {
							subpanelContainer.remove(transformingFormulaPanel);
							transformingFormulaPanel = new TransformingFormulaPanel(fractalElement, fractalElement.getTransformingFormulaConfigElement());
							subpanelContainer.add(transformingFormulaPanel);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			fractalElement.getIncolouringFormulaListElement().addChangeListener(incolouringFormulasListener);
			fractalElement.getOutcolouringFormulaListElement().addChangeListener(outcolouringFormulasListener);
			fractalElement.getRenderingFormulaSingleElement().addChangeListener(renderingFormulaListener);
			fractalElement.getTransformingFormulaSingleElement().addChangeListener(transformingListener);
			appendIncolouringFormulaButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					fractalElement.getContext().updateTimestamp();
					for (int i = fractalElement.getIncolouringFormulaConfigElementCount() - 1; i >= 0; i--) {
						final IncolouringFormulaConfigElement formulaElement = fractalElement.getIncolouringFormulaConfigElement(i);
						if (formulaElement.getUserData() != null) {
							fractalElement.insertIncolouringFormulaConfigElementBefore(i, new IncolouringFormulaConfigElement());
							return;
						}
					}
					fractalElement.appendIncolouringFormulaConfigElement(new IncolouringFormulaConfigElement());
					context.refresh();
				}
			});
			removeIncolouringFormulaButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					fractalElement.getContext().updateTimestamp();
					for (int i = fractalElement.getIncolouringFormulaConfigElementCount() - 1; i >= 0; i--) {
						final IncolouringFormulaConfigElement formulaElement = fractalElement.getIncolouringFormulaConfigElement(i);
						if (formulaElement.getUserData() != null) {
							fractalElement.removeIncolouringFormulaConfigElement(i);
						}
					}
					context.refresh();
				}
			});
			// insertIncolouringFormulaAfterButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = 0; i < fractalElement.getIncolouringFormulaConfigElementCount(); i++) {
			// IncolouringFormulaConfigElement formulaElement = fractalElement.getIncolouringFormulaConfigElement(i);
			// if (formulaElement.getUserData() != null) {
			// fractalElement.insertIncolouringFormulaConfigElementAfter(i, new IncolouringFormulaConfigElement());
			// }
			// }
			// }
			// });
			// insertIncolouringFormulaBeforeButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = fractalElement.getIncolouringFormulaConfigElementCount() - 1; i >= 0; i--) {
			// IncolouringFormulaConfigElement formulaElement = fractalElement.getIncolouringFormulaConfigElement(i);
			// if (formulaElement.getUserData() != null) {
			// fractalElement.insertIncolouringFormulaConfigElementBefore(i, new IncolouringFormulaConfigElement());
			// }
			// }
			// }
			// });
			appendOutcolouringFormulaButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					fractalElement.getContext().updateTimestamp();
					for (int i = fractalElement.getOutcolouringFormulaConfigElementCount() - 1; i >= 0; i--) {
						final OutcolouringFormulaConfigElement formulaElement = fractalElement.getOutcolouringFormulaConfigElement(i);
						if (formulaElement.getUserData() != null) {
							fractalElement.insertOutcolouringFormulaConfigElementBefore(i, new OutcolouringFormulaConfigElement());
							return;
						}
					}
					fractalElement.appendOutcolouringFormulaConfigElement(new OutcolouringFormulaConfigElement());
					context.refresh();
				}
			});
			removeOutcolouringFormulaButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					fractalElement.getContext().updateTimestamp();
					for (int i = fractalElement.getOutcolouringFormulaConfigElementCount() - 1; i >= 0; i--) {
						final OutcolouringFormulaConfigElement formulaElement = fractalElement.getOutcolouringFormulaConfigElement(i);
						if (formulaElement.getUserData() != null) {
							fractalElement.removeOutcolouringFormulaConfigElement(i);
						}
					}
					context.refresh();
				}
			});
			// insertOutcolouringFormulaAfterButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = 0; i < fractalElement.getOutcolouringFormulaConfigElementCount(); i++) {
			// OutcolouringFormulaConfigElement formulaElement = fractalElement.getOutcolouringFormulaConfigElement(i);
			// if (formulaElement.getUserData() != null) {
			// fractalElement.insertOutcolouringFormulaConfigElementAfter(i, new OutcolouringFormulaConfigElement());
			// }
			// }
			// }
			// });
			// insertOutcolouringFormulaBeforeButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = fractalElement.getOutcolouringFormulaConfigElementCount() - 1; i >= 0; i--) {
			// OutcolouringFormulaConfigElement formulaElement = fractalElement.getOutcolouringFormulaConfigElement(i);
			// if (formulaElement.getUserData() != null) {
			// fractalElement.insertOutcolouringFormulaConfigElementBefore(i, new OutcolouringFormulaConfigElement());
			// }
			// }
			// }
			// });
			selectIncolouringFormulaButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					boolean allSelected = true;
					for (int i = 0; i < incolouringFormulasPanel.getComponentCount(); i++) {
						if (!((IncolouringFormulaPanel) incolouringFormulasPanel.getComponent(i)).isSelected()) {
							allSelected = false;
						}
					}
					if (allSelected) {
						for (int i = 0; i < incolouringFormulasPanel.getComponentCount(); i++) {
							((IncolouringFormulaPanel) incolouringFormulasPanel.getComponent(i)).setSelected(false);
						}
					}
					else {
						for (int i = 0; i < incolouringFormulasPanel.getComponentCount(); i++) {
							((IncolouringFormulaPanel) incolouringFormulasPanel.getComponent(i)).setSelected(true);
						}
					}
				}
			});
			selectOutcolouringFormulaButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					boolean allSelected = true;
					for (int i = 0; i < outcolouringFormulasPanel.getComponentCount(); i++) {
						if (!((OutcolouringFormulaPanel) outcolouringFormulasPanel.getComponent(i)).isSelected()) {
							allSelected = false;
						}
					}
					if (allSelected) {
						for (int i = 0; i < outcolouringFormulasPanel.getComponentCount(); i++) {
							((OutcolouringFormulaPanel) outcolouringFormulasPanel.getComponent(i)).setSelected(false);
						}
					}
					else {
						for (int i = 0; i < outcolouringFormulasPanel.getComponentCount(); i++) {
							((OutcolouringFormulaPanel) outcolouringFormulasPanel.getComponent(i)).setSelected(true);
						}
					}
				}
			});
			editIncolouringFormulasButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					viewContext.setComponent(incolouringFormulasPanel2);
				}
			});
			editOutcolouringFormulasButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					viewContext.setComponent(outcolouringFormulasPanel2);
				}
			});
		}

		private String createEditIncolouringFormulasText(final MandelbrotFractalConfigElement fractalElement) {
			if (fractalElement.getIncolouringFormulaConfigElementCount() == 1) {
				return fractalElement.getIncolouringFormulaConfigElementCount() + " " + MandelbrotSwingResources.getInstance().getString("label.incolouringFormula");
			}
			else {
				return fractalElement.getIncolouringFormulaConfigElementCount() + " " + MandelbrotSwingResources.getInstance().getString("label.incolouringFormulas");
			}
		}

		private String createEditOutcolouringFormulasText(final MandelbrotFractalConfigElement fractalElement) {
			if (fractalElement.getOutcolouringFormulaConfigElementCount() == 1) {
				return fractalElement.getOutcolouringFormulaConfigElementCount() + " " + MandelbrotSwingResources.getInstance().getString("label.outcolouringFormula");
			}
			else {
				return fractalElement.getOutcolouringFormulaConfigElementCount() + " " + MandelbrotSwingResources.getInstance().getString("label.outcolouringFormulas");
			}
		}

		private String createIncolouringFormulasPanelName() {
			return MandelbrotSwingResources.getInstance().getString("name.incolouringFormulas");
		}

		private String createOutcolouringFormulasPanelName() {
			return MandelbrotSwingResources.getInstance().getString("name.outcolouringFormulas");
		}
	}

	private class RenderingFormulaPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private JComponent configView;

		/**
		 * @param fractalElement
		 * @param formulaElement
		 */
		public RenderingFormulaPanel(final MandelbrotFractalConfigElement fractalElement, final RenderingFormulaConfigElement formulaElement) {
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(MandelbrotRegistry.getInstance().getRenderingFormulaRegistry(), true);
			final JButton editOptionsButton = createTextButton("editConfig", 100, GUIFactory.DEFAULT_HEIGHT);
			final JComboBox extensionComboBox = createExtensionComboBox(model, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel label = createTextLabel("renderingFormula", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final Box formulaPanel = createHorizontalBox(false);
			formulaPanel.add(label);
			formulaPanel.add(createSpace());
			formulaPanel.add(extensionComboBox);
			formulaPanel.add(createSpace());
			formulaPanel.add(editOptionsButton);
			formulaPanel.add(Box.createHorizontalGlue());
			formulaPanel.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 8));
			setLayout(new BorderLayout());
			add(formulaPanel, BorderLayout.NORTH);
			setOpaque(false);
			setBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0));
			editOptionsButton.setEnabled((formulaElement != null) && (formulaElement.getReference() != null));
			if (formulaElement.getReference() != null) {
				model.setSelectedItemByExtensionId(formulaElement.getReference().getExtensionId());
			}
			final ActionListener comboListener = new ActionListener() {
				@SuppressWarnings("unchecked")
				public void actionPerformed(ActionEvent e) {
					ConfigurableExtension<RenderingFormulaExtensionRuntime, RenderingFormulaExtensionConfig> extension = (ConfigurableExtension<RenderingFormulaExtensionRuntime, RenderingFormulaExtensionConfig>) extensionComboBox.getSelectedItem();
					try {
						context.stopRenderers();
						formulaElement.getContext().updateTimestamp();
						if (extension instanceof NullConfigurableExtension) {
							formulaElement.setReference(null);
						}
						else {
							formulaElement.setReference(extension.createConfigurableExtensionReference());
						}
						context.startRenderers();
						context.refresh();
						if (configView != null) {
							viewContext.removeComponent(configView);
							configView = null;
						}
					}
					catch (ExtensionException x) {
						x.printStackTrace();
					}
				}
			};
			final ValueChangeListener effectListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ExtensionConfigElementEvents.EXTENSION_REFERENCE_CHANGED: {
							editOptionsButton.setEnabled(formulaElement.getReference() != null);
							extensionComboBox.removeActionListener(comboListener);
							if (formulaElement.getReference() != null) {
								model.setSelectedItemByExtensionId(formulaElement.getReference().getExtensionId());
							}
							else {
								model.setSelectedItem(0);
							}
							if (configView != null) {
								viewContext.removeComponent(configView);
								configView = null;
							}
							extensionComboBox.addActionListener(comboListener);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			formulaElement.getExtensionElement().addChangeListener(effectListener);
			extensionComboBox.addActionListener(comboListener);
			editOptionsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					if (formulaElement.getReference() != null) {
						if (configView == null) {
							try {
								final Extension<ViewExtensionRuntime> extension = TwisterSwingRegistry.getInstance().getViewExtension(formulaElement.getReference().getExtensionId());
								configView = extension.createExtensionRuntime().createView(formulaElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
								configView = new NavigatorViewRuntime().createView(formulaElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
						}
						configView.setName(createRenderingFormulaPanelName());
						viewContext.setComponent(configView);
					}
				}
			});
		}

		private String createRenderingFormulaPanelName() {
			return MandelbrotSwingResources.getInstance().getString("name.renderingFormula");
		}
	}

	private class TransformingFormulaPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private JComponent configView;

		/**
		 * @param fractalElement
		 * @param formulaElement
		 */
		public TransformingFormulaPanel(final MandelbrotFractalConfigElement fractalElement, final TransformingFormulaConfigElement formulaElement) {
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(MandelbrotRegistry.getInstance().getTransformingFormulaRegistry(), true);
			final JButton editOptionsButton = createTextButton("editConfig", 100, GUIFactory.DEFAULT_HEIGHT);
			final JComboBox extensionComboBox = createExtensionComboBox(model, 200, GUIFactory.DEFAULT_HEIGHT);
			final JLabel label = createTextLabel("transformingFormula", SwingConstants.LEFT, 200, GUIFactory.DEFAULT_HEIGHT);
			final Box formulaPanel = createHorizontalBox(false);
			formulaPanel.add(label);
			formulaPanel.add(createSpace());
			formulaPanel.add(extensionComboBox);
			formulaPanel.add(createSpace());
			formulaPanel.add(editOptionsButton);
			formulaPanel.add(Box.createHorizontalGlue());
			formulaPanel.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 8));
			setLayout(new BorderLayout());
			add(formulaPanel, BorderLayout.NORTH);
			setOpaque(false);
			setBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0));
			editOptionsButton.setEnabled((formulaElement != null) && (formulaElement.getReference() != null));
			if (formulaElement.getReference() != null) {
				model.setSelectedItemByExtensionId(formulaElement.getReference().getExtensionId());
			}
			final ActionListener comboListener = new ActionListener() {
				@SuppressWarnings("unchecked")
				public void actionPerformed(ActionEvent e) {
					ConfigurableExtension<TransformingFormulaExtensionRuntime, TransformingFormulaExtensionConfig> extension = (ConfigurableExtension<TransformingFormulaExtensionRuntime, TransformingFormulaExtensionConfig>) extensionComboBox.getSelectedItem();
					try {
						context.stopRenderers();
						formulaElement.getContext().updateTimestamp();
						if (extension instanceof NullConfigurableExtension) {
							formulaElement.setReference(null);
						}
						else {
							formulaElement.setReference(extension.createConfigurableExtensionReference());
						}
						context.startRenderers();
						context.refresh();
						if (configView != null) {
							viewContext.removeComponent(configView);
							configView = null;
						}
					}
					catch (ExtensionException x) {
						x.printStackTrace();
					}
				}
			};
			final ValueChangeListener effectListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ExtensionConfigElementEvents.EXTENSION_REFERENCE_CHANGED: {
							editOptionsButton.setEnabled(formulaElement.getReference() != null);
							extensionComboBox.removeActionListener(comboListener);
							if (formulaElement.getReference() != null) {
								model.setSelectedItemByExtensionId(formulaElement.getReference().getExtensionId());
							}
							else {
								model.setSelectedItem(0);
							}
							if (configView != null) {
								viewContext.removeComponent(configView);
								configView = null;
							}
							extensionComboBox.addActionListener(comboListener);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			formulaElement.getExtensionElement().addChangeListener(effectListener);
			extensionComboBox.addActionListener(comboListener);
			editOptionsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					if (formulaElement.getReference() != null) {
						if (configView == null) {
							try {
								final Extension<ViewExtensionRuntime> extension = TwisterSwingRegistry.getInstance().getViewExtension(formulaElement.getReference().getExtensionId());
								configView = extension.createExtensionRuntime().createView(formulaElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
								configView = new NavigatorViewRuntime().createView(formulaElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
						}
						configView.setName(createTransformingFormulaPanelName());
						viewContext.setComponent(configView);
					}
				}
			});
		}

		private String createTransformingFormulaPanelName() {
			return MandelbrotSwingResources.getInstance().getString("name.transformingFormula");
		}
	}

	private class IncolouringFormulaPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private final JCheckBox selectedCheckBox = createSelectionCheckBox();
		private JComponent configView;

		/**
		 * @param fractalElement
		 * @param formulaElement
		 */
		public IncolouringFormulaPanel(final MandelbrotFractalConfigElement fractalElement, final IncolouringFormulaConfigElement formulaElement) {
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(MandelbrotRegistry.getInstance().getIncolouringFormulaRegistry(), true);
			final JButton editOptionsButton = createTextButton("editConfig", 100, GUIFactory.DEFAULT_HEIGHT);
			final JCheckBox lockedCheckBox = createIconCheckBox("locked", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox enabledCheckBox = createIconCheckBox("enabled", "visible", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox autoIterationsCheckBox = createIconCheckBox("autoIterations", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JSpinner opacitySpinner = createSpinner(0, 100, 1);
			final JSpinner iterationsSpinner = createSpinner(0, 5000, 1);
			final JComboBox extensionComboBox = createExtensionComboBox(model, 140, GUIFactory.DEFAULT_HEIGHT);
			final JTextField label = createTextField(formulaElement.getLabel(), 200, GUIFactory.DEFAULT_HEIGHT);
			final Box formulaPanel = createHorizontalBox(false);
			formulaPanel.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 8));
			formulaPanel.add(createSpace());
			formulaPanel.add(selectedCheckBox);
			formulaPanel.add(label);
			formulaPanel.add(createSpace());
			formulaPanel.add(opacitySpinner);
			formulaPanel.add(createSpace());
			formulaPanel.add(lockedCheckBox);
			formulaPanel.add(createSpace());
			formulaPanel.add(enabledCheckBox);
			formulaPanel.add(createSpace());
			formulaPanel.add(iterationsSpinner);
			formulaPanel.add(createSpace());
			formulaPanel.add(autoIterationsCheckBox);
			formulaPanel.add(Box.createHorizontalGlue());
			formulaPanel.add(extensionComboBox);
			formulaPanel.add(createSpace());
			formulaPanel.add(editOptionsButton);
			setLayout(new BorderLayout());
			add(formulaPanel, BorderLayout.CENTER);
			setOpaque(false);
			setBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0));
			lockedCheckBox.setSelected(formulaElement.isLocked());
			enabledCheckBox.setSelected(formulaElement.isEnabled());
			editOptionsButton.setEnabled((formulaElement != null) && (formulaElement.getReference() != null));
			opacitySpinner.setValue(formulaElement.getOpacity().intValue());
			opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
			iterationsSpinner.setValue(formulaElement.getIterations());
			iterationsSpinner.setToolTipText(createIterationsTooltip(iterationsSpinner));
			autoIterationsCheckBox.setSelected(formulaElement.getAutoIterations());
			iterationsSpinner.setEnabled(!autoIterationsCheckBox.isSelected());
			if (formulaElement.getReference() != null) {
				model.setSelectedItemByExtensionId(formulaElement.getReference().getExtensionId());
			}
			final ActionListener comboListener = new ActionListener() {
				@SuppressWarnings("unchecked")
				public void actionPerformed(ActionEvent e) {
					ConfigurableExtension<IncolouringFormulaExtensionRuntime, IncolouringFormulaExtensionConfig> extension = (ConfigurableExtension<IncolouringFormulaExtensionRuntime, IncolouringFormulaExtensionConfig>) extensionComboBox.getSelectedItem();
					try {
						context.stopRenderers();
						formulaElement.getContext().updateTimestamp();
						if (extension instanceof NullConfigurableExtension) {
							formulaElement.setReference(null);
						}
						else {
							formulaElement.setReference(extension.createConfigurableExtensionReference());
						}
						context.startRenderers();
						context.refresh();
						if (configView != null) {
							viewContext.removeComponent(configView);
							configView = null;
						}
					}
					catch (ExtensionException x) {
						x.printStackTrace();
					}
				}
			};
			final ValueChangeListener formulaListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ExtensionConfigElementEvents.EXTENSION_REFERENCE_CHANGED: {
							editOptionsButton.setEnabled(formulaElement.getReference() != null);
							extensionComboBox.removeActionListener(comboListener);
							if (formulaElement.getReference() != null) {
								model.setSelectedItemByExtensionId(formulaElement.getReference().getExtensionId());
							}
							else {
								model.setSelectedItem(0);
							}
							if (configView != null) {
								viewContext.removeComponent(configView);
								configView = null;
							}
							extensionComboBox.addActionListener(comboListener);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			final ValueChangeListener lockedListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					lockedCheckBox.setSelected(formulaElement.isLocked());
				}
			};
			final ValueChangeListener enabledListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					enabledCheckBox.setSelected(formulaElement.isEnabled());
				}
			};
			final ValueChangeListener labelListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					label.setText(formulaElement.getLabel());
					if (configView != null) {
						configView.setName(createIncolouringFormulaPanelName(formulaElement));
					}
				}
			};
			final ValueChangeListener autoIterationsListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					autoIterationsCheckBox.setSelected(formulaElement.getAutoIterations());
					iterationsSpinner.setEnabled(!autoIterationsCheckBox.isSelected());
				}
			};
			final ValueChangeListener iterationsListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					iterationsSpinner.setValue(formulaElement.getIterations());
				}
			};
			final ValueChangeListener opacityListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					opacitySpinner.setValue(formulaElement.getOpacity().intValue());
				}
			};
			formulaElement.getLabelElement().addChangeListener(labelListener);
			formulaElement.getLockedElement().addChangeListener(lockedListener);
			formulaElement.getEnabledElement().addChangeListener(enabledListener);
			formulaElement.getExtensionElement().addChangeListener(formulaListener);
			formulaElement.getAutoIterationsElement().addChangeListener(autoIterationsListener);
			formulaElement.getIterationsElement().addChangeListener(iterationsListener);
			formulaElement.getOpacityElement().addChangeListener(opacityListener);
			extensionComboBox.addActionListener(comboListener);
			editOptionsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					if (formulaElement.getReference() != null) {
						if (configView == null) {
							try {
								final Extension<ViewExtensionRuntime> extension = TwisterSwingRegistry.getInstance().getViewExtension(formulaElement.getReference().getExtensionId());
								configView = extension.createExtensionRuntime().createView(formulaElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
								configView = new NavigatorViewRuntime().createView(formulaElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
						}
						if (configView != null) {
							configView.setName(createIncolouringFormulaPanelName(formulaElement));
						}
						viewContext.setComponent(configView);
					}
				}
			});
			label.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					// formulaElement.getLabelElement().removeChangeListener(labelListener);
					formulaElement.getContext().updateTimestamp();
					formulaElement.setLabel(label.getText());
					if (configView != null) {
						configView.setName(createIncolouringFormulaPanelName(formulaElement));
					}
					// formulaElement.getLabelElement().addChangeListener(labelListener);
				}
			});
			autoIterationsCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setAutoIterations(autoIterationsCheckBox.isSelected());
					iterationsSpinner.setEnabled(!autoIterationsCheckBox.isSelected());
					context.refresh();
				}
			});
			iterationsSpinner.addChangeListener(new ChangeListener() {
				public void stateChanged(final ChangeEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setIterations(((Number) iterationsSpinner.getValue()).intValue());
					iterationsSpinner.setToolTipText(createIterationsTooltip(iterationsSpinner));
					context.refresh();
				}
			});
			opacitySpinner.addChangeListener(new ChangeListener() {
				public void stateChanged(final ChangeEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setOpacity(((Number) opacitySpinner.getValue()).intValue());
					opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
					context.refresh();
				}
			});
			lockedCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setLocked(lockedCheckBox.isSelected());
				}
			});
			enabledCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setEnabled(enabledCheckBox.isSelected());
					context.refresh();
				}
			});
			selectedCheckBox.addItemListener(new ItemListener() {
				public void itemStateChanged(final ItemEvent e) {
					if (selectedCheckBox.isSelected()) {
						formulaElement.setUserData(Boolean.TRUE);
					}
					else {
						formulaElement.setUserData(null);
					}
				}
			});
		}

		private String createOpacityTooltip(final JSpinner opacitySpinner) {
			return MandelbrotSwingResources.getInstance().getString("label.opacity") + " " + opacitySpinner.getValue() + "%";
		}

		private String createIterationsTooltip(final JSpinner iterationsSpinner) {
			return MandelbrotSwingResources.getInstance().getString("label.iterations") + " " + iterationsSpinner.getValue();
		}

		private String createIncolouringFormulaPanelName(final IncolouringFormulaConfigElement formulaElement) {
			return formulaElement.getLabel();
		}

		/**
		 * @param selected
		 */
		public void setSelected(final boolean selected) {
			selectedCheckBox.setSelected(selected);
		}

		/**
		 * @return
		 */
		public boolean isSelected() {
			return selectedCheckBox.isSelected();
		}
	}

	private class OutcolouringFormulaPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private final JCheckBox selectedCheckBox = createSelectionCheckBox();
		private JComponent configView;

		/**
		 * @param fractalElement
		 * @param formulaElement
		 */
		public OutcolouringFormulaPanel(final MandelbrotFractalConfigElement fractalElement, final OutcolouringFormulaConfigElement formulaElement) {
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(MandelbrotRegistry.getInstance().getOutcolouringFormulaRegistry(), true);
			final JButton editOptionsButton = createTextButton("editConfig", 100, GUIFactory.DEFAULT_HEIGHT);
			final JCheckBox lockedCheckBox = createIconCheckBox("locked", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox enabledCheckBox = createIconCheckBox("enabled", "visible", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox autoIterationsCheckBox = createIconCheckBox("autoIterations", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JSpinner opacitySpinner = createSpinner(0, 100, 1);
			final JSpinner iterationsSpinner = createSpinner(0, 5000, 1);
			final JComboBox extensionComboBox = createExtensionComboBox(model, 140, GUIFactory.DEFAULT_HEIGHT);
			final JTextField label = createTextField(formulaElement.getLabel(), 200, GUIFactory.DEFAULT_HEIGHT);
			final Box formulaPanel = createHorizontalBox(false);
			formulaPanel.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 8));
			formulaPanel.add(createSpace());
			formulaPanel.add(selectedCheckBox);
			formulaPanel.add(label);
			formulaPanel.add(createSpace());
			formulaPanel.add(opacitySpinner);
			formulaPanel.add(createSpace());
			formulaPanel.add(lockedCheckBox);
			formulaPanel.add(createSpace());
			formulaPanel.add(enabledCheckBox);
			formulaPanel.add(createSpace());
			formulaPanel.add(iterationsSpinner);
			formulaPanel.add(createSpace());
			formulaPanel.add(autoIterationsCheckBox);
			formulaPanel.add(Box.createHorizontalGlue());
			formulaPanel.add(extensionComboBox);
			formulaPanel.add(createSpace());
			formulaPanel.add(editOptionsButton);
			setLayout(new BorderLayout());
			add(formulaPanel, BorderLayout.CENTER);
			setOpaque(false);
			setBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0));
			lockedCheckBox.setSelected(formulaElement.isLocked());
			enabledCheckBox.setSelected(formulaElement.isEnabled());
			editOptionsButton.setEnabled((formulaElement != null) && (formulaElement.getReference() != null));
			opacitySpinner.setValue(formulaElement.getOpacity().intValue());
			opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
			iterationsSpinner.setValue(formulaElement.getIterations());
			iterationsSpinner.setToolTipText(createIterationsTooltip(iterationsSpinner));
			autoIterationsCheckBox.setSelected(formulaElement.getAutoIterations());
			iterationsSpinner.setEnabled(!autoIterationsCheckBox.isSelected());
			if (formulaElement.getReference() != null) {
				model.setSelectedItemByExtensionId(formulaElement.getReference().getExtensionId());
			}
			final ActionListener comboListener = new ActionListener() {
				@SuppressWarnings("unchecked")
				public void actionPerformed(ActionEvent e) {
					ConfigurableExtension<OutcolouringFormulaExtensionRuntime, OutcolouringFormulaExtensionConfig> extension = (ConfigurableExtension<OutcolouringFormulaExtensionRuntime, OutcolouringFormulaExtensionConfig>) extensionComboBox.getSelectedItem();
					try {
						context.stopRenderers();
						formulaElement.getContext().updateTimestamp();
						if (extension instanceof NullConfigurableExtension) {
							formulaElement.setReference(null);
						}
						else {
							formulaElement.setReference(extension.createConfigurableExtensionReference());
						}
						context.startRenderers();
						context.refresh();
						if (configView != null) {
							viewContext.removeComponent(configView);
							configView = null;
						}
					}
					catch (ExtensionException x) {
						x.printStackTrace();
					}
				}
			};
			final ValueChangeListener formulaListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ExtensionConfigElementEvents.EXTENSION_REFERENCE_CHANGED: {
							editOptionsButton.setEnabled(formulaElement.getReference() != null);
							extensionComboBox.removeActionListener(comboListener);
							if (formulaElement.getReference() != null) {
								model.setSelectedItemByExtensionId(formulaElement.getReference().getExtensionId());
							}
							else {
								model.setSelectedItem(0);
							}
							if (configView != null) {
								viewContext.removeComponent(configView);
								configView = null;
							}
							extensionComboBox.addActionListener(comboListener);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			final ValueChangeListener lockedListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					lockedCheckBox.setSelected(formulaElement.isLocked());
				}
			};
			final ValueChangeListener enabledListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					enabledCheckBox.setSelected(formulaElement.isEnabled());
				}
			};
			final ValueChangeListener labelListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					label.setText(formulaElement.getLabel());
					if (configView != null) {
						configView.setName(createOutcolouringFormulaPanelName(formulaElement));
					}
				}
			};
			final ValueChangeListener autoIterationsListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					autoIterationsCheckBox.setSelected(formulaElement.getAutoIterations());
					iterationsSpinner.setEnabled(!autoIterationsCheckBox.isSelected());
				}
			};
			final ValueChangeListener iterationsListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					iterationsSpinner.setValue(formulaElement.getIterations());
				}
			};
			final ValueChangeListener opacityListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					opacitySpinner.setValue(formulaElement.getOpacity().intValue());
				}
			};
			formulaElement.getLabelElement().addChangeListener(labelListener);
			formulaElement.getLockedElement().addChangeListener(lockedListener);
			formulaElement.getEnabledElement().addChangeListener(enabledListener);
			formulaElement.getExtensionElement().addChangeListener(formulaListener);
			formulaElement.getAutoIterationsElement().addChangeListener(autoIterationsListener);
			formulaElement.getIterationsElement().addChangeListener(iterationsListener);
			formulaElement.getOpacityElement().addChangeListener(opacityListener);
			extensionComboBox.addActionListener(comboListener);
			editOptionsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					if (formulaElement.getReference() != null) {
						if (configView == null) {
							try {
								final Extension<ViewExtensionRuntime> extension = TwisterSwingRegistry.getInstance().getViewExtension(formulaElement.getReference().getExtensionId());
								configView = extension.createExtensionRuntime().createView(formulaElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
								configView = new NavigatorViewRuntime().createView(formulaElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
						}
						if (configView != null) {
							configView.setName(createOutcolouringFormulaPanelName(formulaElement));
						}
						viewContext.setComponent(configView);
					}
				}
			});
			label.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					// formulaElement.getLabelElement().removeChangeListener(labelListener);
					formulaElement.getContext().updateTimestamp();
					formulaElement.setLabel(label.getText());
					if (configView != null) {
						configView.setName(createOutcolouringFormulaPanelName(formulaElement));
					}
					// formulaElement.getLabelElement().addChangeListener(labelListener);
				}
			});
			autoIterationsCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setAutoIterations(autoIterationsCheckBox.isSelected());
					iterationsSpinner.setEnabled(!autoIterationsCheckBox.isSelected());
					context.refresh();
				}
			});
			iterationsSpinner.addChangeListener(new ChangeListener() {
				public void stateChanged(final ChangeEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setIterations(((Number) iterationsSpinner.getValue()).intValue());
					iterationsSpinner.setToolTipText(createIterationsTooltip(iterationsSpinner));
					context.refresh();
				}
			});
			opacitySpinner.addChangeListener(new ChangeListener() {
				public void stateChanged(final ChangeEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setOpacity(((Number) opacitySpinner.getValue()).intValue());
					opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
					context.refresh();
				}
			});
			lockedCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setLocked(lockedCheckBox.isSelected());
				}
			});
			enabledCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					formulaElement.getContext().updateTimestamp();
					formulaElement.setEnabled(enabledCheckBox.isSelected());
					context.refresh();
				}
			});
			selectedCheckBox.addItemListener(new ItemListener() {
				public void itemStateChanged(final ItemEvent e) {
					if (selectedCheckBox.isSelected()) {
						formulaElement.setUserData(Boolean.TRUE);
					}
					else {
						formulaElement.setUserData(null);
					}
				}
			});
		}

		private String createOpacityTooltip(final JSpinner opacitySpinner) {
			return MandelbrotSwingResources.getInstance().getString("label.opacity") + " " + opacitySpinner.getValue() + "%";
		}

		private String createIterationsTooltip(final JSpinner iterationsSpinner) {
			return MandelbrotSwingResources.getInstance().getString("label.iterations") + " " + iterationsSpinner.getValue();
		}

		private String createOutcolouringFormulaPanelName(final OutcolouringFormulaConfigElement formulaElement) {
			return formulaElement.getLabel();
		}

		/**
		 * @param selected
		 */
		public void setSelected(final boolean selected) {
			selectedCheckBox.setSelected(selected);
		}

		/**
		 * @return
		 */
		public boolean isSelected() {
			return selectedCheckBox.isSelected();
		}
	}
}
