/*
 * $Id:MovieWindow.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.media.swing;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.DisplayMode;
import java.awt.Frame;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import net.sf.jame.media.Context;
import net.sf.jame.media.EngineException;
import net.sf.jame.media.Movie;

public final class MovieWindow extends Frame implements Context {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	private GraphicsEnvironment environment;
	private GraphicsDevice device;
	private Engine engine;
	private Movie movie;
	private Color color = Color.white;
	private boolean debug = false;
	private boolean loop = false;

	public MovieWindow(final Movie movie, final String title) {
		try {
			this.movie = movie;
			environment = GraphicsEnvironment.getLocalGraphicsEnvironment();
			if (GraphicsEnvironment.isHeadless()) {
				throw new EngineException("Graphics environment is headless!");
			}
			device = environment.getDefaultScreenDevice();
			if (device.getType() != GraphicsDevice.TYPE_RASTER_SCREEN) {
				throw new EngineException("Graphics device not found!");
			}
			loadProperties();
			setTitle(title);
			setBackground(color);
			setForeground(color);
			setUndecorated(true);
			setIgnoreRepaint(false);
			this.setCursor(new Cursor(Cursor.HAND_CURSOR));
			if (!device.isFullScreenSupported()) {
				throw new EngineException("Fullscreen not supported!");
			}
			device.setFullScreenWindow(this);
			if (!device.isDisplayChangeSupported()) {
				throw new EngineException("Displaychange not supported!");
			}
			device.setDisplayMode(new DisplayMode(640, 480, 32, DisplayMode.REFRESH_RATE_UNKNOWN));
			engine = new Engine(this, this);
			addWindowListener(new windowAdapter());
			addKeyListener(new keyAdapter());
			this.requestFocus();
			engine.start();
		}
		catch (final EngineException e) {
			e.printStackTrace();
		}
	}

	private void loadProperties() {
		if (System.getProperty("debug", "false").toLowerCase().equals("true")) {
			debug = true;
		}
		if (System.getProperty("loop", "false").toLowerCase().equals("true")) {
			loop = true;
		}
		try {
			color = new Color(Integer.parseInt(System.getProperty("color", "#FFFFFF").substring(1), 16));
		}
		catch (final NumberFormatException e) {
		}
	}

	void kill() {
		engine.stop();
		device.setFullScreenWindow(null);
	}

	public Movie getMovie() {
		return movie;
	}

	public Color getColor() {
		return getBackground();
	}

	public boolean debug() {
		return debug;
	}

	public boolean loop() {
		return loop;
	}

	public void println(final String s) {
		System.out.println(s);
	}

	public void print(final String s) {
		System.out.print(s);
	}

	public void exit(final int code) {
		System.exit(code);
	}

	private class keyAdapter extends KeyAdapter {
		public keyAdapter() {
		}

		@Override
		public void keyPressed(final KeyEvent e) {
			switch (e.getKeyCode()) {
				case KeyEvent.VK_ESCAPE: {
					kill();
					break;
				}
				default:
					break;
			}
		}
	}

	private class windowAdapter extends WindowAdapter {
		public windowAdapter() {
		}

		@Override
		public void windowClosing(final WindowEvent e) {
			kill();
		}
	}
}
