/*
 * $Id:Application.java 491 2008-01-28 21:59:31Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.runtime.swing;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.swing.JFrame;

import net.sf.jame.launcher.Launcher;
import net.sf.jame.runtime.LauncherContextListener;
import net.sf.jame.runtime.LauncherThreadFactory;
import net.sf.jame.twister.swing.TwisterContext;
import net.sf.jame.twister.swing.TwisterLauncherThread;

import org.apache.log4j.PropertyConfigurator;
import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;

/**
 * Application implementation.
 * 
 * @author Andrea Medeghini
 */
public class Application implements IApplication {
	private static final List<JFrame> frames = new ArrayList<JFrame>(1);
	private final Launcher<DefaultTwisterContext> launcher = new Launcher<DefaultTwisterContext>(new DefaultTwisterContext(), new DefaultLauncherThreadFactory());
	private LauncherContextListener listener;
	
	/**
	 * @see org.eclipse.equinox.app.IApplication#start(org.eclipse.equinox.app.IApplicationContext)
	 */
	public Object start(final IApplicationContext context) throws Exception {
		Properties log4jProperties = new Properties();
		log4jProperties.put("log4j.rootLogger", "INFO, console");
		log4jProperties.put("log4j.appender.console", "org.apache.log4j.ConsoleAppender");
		log4jProperties.put("log4j.appender.console.layout", "org.apache.log4j.PatternLayout");
		log4jProperties.put("log4j.appender.console.layout.ConversionPattern", "%d{HH:mm:ss,SSS} %-5p %c - %m%n");
		log4jProperties.put("log4j.logger.net.sf.jame", "INFO");
		log4jProperties.put("log4j.logger.org.apache.derby", "INFO");
		PropertyConfigurator.configure(log4jProperties);
		boolean restart = false;
		try {
			launcher.init();
			launcher.start();
			if (listener != null) {
				listener.started();
			}
			context.applicationRunning();
			restart = launcher.dispatch();
			if (listener != null) {
				listener.stopped();
			}
			launcher.dispose();
		}
		catch (final Exception e) {
			e.printStackTrace();
		}
		if (restart) {
			return IApplication.EXIT_RESTART;
		}
		return IApplication.EXIT_OK;
	}

	/**
	 * @see org.eclipse.equinox.app.IApplication#stop()
	 */
	public void stop() {
		launcher.stop();
	}

	private class DefaultTwisterContext implements TwisterContext {
		/**
		 * @see net.sf.jame.twister.swing.TwisterContext#addFrame(javax.swing.JFrame)
		 */
		public void addFrame(final JFrame frame) {
			Application.frames.add(frame);
		}

		/**
		 * @see net.sf.jame.twister.swing.TwisterContext#removeFrame(javax.swing.JFrame)
		 */
		public void removeFrame(final JFrame frame) {
			Application.frames.remove(frame);
		}

		/**
		 * @see net.sf.jame.twister.swing.TwisterContext#getFrameCount()
		 */
		public int getFrameCount() {
			return Application.frames.size();
		}

		/**
		 * @see net.sf.jame.runtime.LauncherContext#exit()
		 */
		public void exit() {
			launcher.stop();
		}

		/**
		 * @see net.sf.jame.runtime.LauncherContext#restart()
		 */
		public void restart() {
			launcher.restart();
		}

		/**
		 * @see net.sf.jame.runtime.LauncherContext#setContextListener(net.sf.jame.runtime.LauncherContextListener)
		 */
		public void setContextListener(LauncherContextListener listener) {
			Application.this.listener = listener;
		}
	}

	private class DefaultLauncherThreadFactory implements LauncherThreadFactory<DefaultTwisterContext> {
		/**
		 * @see net.sf.jame.runtime.LauncherThreadFactory#createThread(net.sf.jame.runtime.LauncherContext)
		 */
		public Thread createThread(final DefaultTwisterContext context) {
			return new TwisterLauncherThread(context);
		}
	}
}
