/*
 * $Id:DistributedSpoolJob.java 491 2008-01-28 21:59:31Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.spool.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;

import net.sf.jame.core.xml.XML;
import net.sf.jame.service.Service;
import net.sf.jame.service.job.RenderJobDataRow;
import net.sf.jame.service.spool.DistributedSpoolJobInterface;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.twister.TwisterClip;
import net.sf.jame.twister.TwisterClipXMLImporter;

import org.w3c.dom.Document;

/**
 * @author Andrea Medeghini
 */
public class DistributedSpoolJob implements DistributedSpoolJobInterface {
	private final JobListener listener;
	private final String jobId;
	private String remoteJobId;
	private long lastUpdate;
	private boolean started;
	private boolean aborted;
	private boolean terminated;
	private final Service service;
	private RenderJobDataRow jobDataRow;

	/**
	 * @param service
	 * @param jobId
	 * @param listener
	 */
	public DistributedSpoolJob(final Service service, final String jobId, final JobListener listener) {
		lastUpdate = System.currentTimeMillis();
		this.listener = listener;
		this.service = service;
		this.jobId = jobId;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getJobId()
	 */
	public String getJobId() {
		return jobId;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getFrameNumber()
	 */
	public int getFrameNumber() {
		return jobDataRow.getFrameNumber();
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedSpoolJobInterface#setFrameNumber(int)
	 */
	public void setFrameNumber(final int frameNumber) {
		jobDataRow.setFrameNumber(frameNumber);
		lastUpdate = System.currentTimeMillis();
		listener.stateChanged(this);
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedSpoolJobInterface#getRAF()
	 */
	public RandomAccessFile getRAF() throws IOException {
		return service.getJobRandomAccessFile(jobDataRow.getJobId());
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedSpoolJobInterface#getStream()
	 */
	public OutputStream getStream() throws IOException {
		return service.getJobOutputStream(jobDataRow.getJobId());
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedSpoolJobInterface#getStream(boolean)
	 */
	public OutputStream getStream(final boolean append) throws IOException {
		return service.getJobOutputStream(jobDataRow.getJobId(), append);
	}

	/**
	 * @see net.sf.jame.service.spool.SpoolJobInterface#getClip()
	 */
	public TwisterClip getClip() throws IOException {
		try {
			final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
			final InputStream is = service.getClipInputStream(jobDataRow.getClipId());
			final Document doc = XML.loadDocument(is, "twister-clip.xml");
			return importer.importFromElement(doc.getDocumentElement());
		}
		catch (final Exception e) {
			throw new IOException(e.getMessage());
		}
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final StringBuilder builder = new StringBuilder();
		builder.append("id = ");
		builder.append(jobId);
		builder.append(", frameNumber = ");
		builder.append(getFrameNumber());
		return builder.toString();
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getLastUpdate()
	 */
	public long getLastUpdate() {
		return lastUpdate;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#reset()
	 */
	public void reset() {
		started = false;
		aborted = false;
		terminated = false;
		lastUpdate = System.currentTimeMillis();
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#start()
	 */
	public void start() {
		terminated = false;
		aborted = false;
		started = true;
		listener.started(this);
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#stop()
	 */
	public void stop() {
		started = false;
		terminated = true;
		listener.stopped(this);
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#abort()
	 */
	public void abort() {
		aborted = true;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#dispose()
	 */
	public void dispose() {
		// jobDataRow = null;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isStarted()
	 */
	public boolean isStarted() {
		return started;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isAborted()
	 */
	public boolean isAborted() {
		return aborted;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isTerminated()
	 */
	public boolean isTerminated() {
		return terminated;
	}

	private void terminate() {
		listener.terminated(this);
	}

	/**
	 * @see net.sf.jame.service.spool.SpoolJobInterface#getJobDataRow()
	 */
	public RenderJobDataRow getJobDataRow() {
		return jobDataRow;
	}

	/**
	 * @see net.sf.jame.service.spool.SpoolJobInterface#setJobDataRow(net.sf.jame.service.job.RenderJobDataRow)
	 */
	public void setJobDataRow(final RenderJobDataRow jobRowData) {
		jobDataRow = jobRowData;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedSpoolJobInterface#getRemoteJobId()
	 */
	public String getRemoteJobId() {
		return remoteJobId;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedSpoolJobInterface#setRemoteJobId(java.lang.String)
	 */
	public void setRemoteJobId(final String remoteJobId) {
		this.remoteJobId = remoteJobId;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedSpoolJobInterface#getTotalFrames()
	 */
	public int getTotalFrames() {
		return (jobDataRow.getFrameRate() > 0) ? ((jobDataRow.getStopTime() - jobDataRow.getStartTime()) * jobDataRow.getFrameRate()) : 1;
	}

	/**
	 * @see net.sf.jame.service.spool.DistributedSpoolJobInterface#setTerminated(boolean)
	 */
	public void setTerminated(final boolean terminated) {
		this.terminated = terminated;
		if (terminated) {
			terminate();
		}
	}
}
