/*
 * $Id:ConfigElementListNode.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister.util;

import java.io.Serializable;

import net.sf.jame.core.config.ConfigElement;
import net.sf.jame.core.config.ListConfigElement;
import net.sf.jame.core.config.ListConfigElementEvents;
import net.sf.jame.core.config.ValueChangeEvent;
import net.sf.jame.core.config.ValueChangeListener;
import net.sf.jame.core.tree.MutableNode;
import net.sf.jame.core.tree.Node;
import net.sf.jame.core.tree.NodeAction;
import net.sf.jame.core.tree.NodeEditor;
import net.sf.jame.core.tree.NodeSession;
import net.sf.jame.core.tree.NodeValue;

/**
 * @author Andrea Medeghini
 */
public abstract class ConfigElementListNode<T extends ConfigElement> extends MutableNode {
	protected ConfigElementListener listListener;
	protected ListConfigElement<T> listElement;

	/**
	 * Constructs a new list node.
	 * 
	 * @param listElement the frame element.
	 */
	public ConfigElementListNode(final String nodeId, final ListConfigElement<T> listElement) {
		super(nodeId);
		if (listElement == null) {
			throw new IllegalArgumentException("listElement is null");
		}
		this.listElement = listElement;
		listListener = new ConfigElementListener();
	}

	/**
	 * @param value
	 * @return
	 */
	protected abstract ConfigElementNode<T> createChildNode(T value);

	/**
	 * @see net.sf.jame.core.tree.Node#setSession(net.sf.jame.core.tree.NodeSession)
	 */
	@Override
	public void setSession(final NodeSession session) {
		if (session != null) {
			listElement.addChangeListener(listListener);
		}
		else {
			listElement.removeChangeListener(listListener);
		}
		super.setSession(session);
	}

	/**
	 * @see net.sf.jame.core.tree.Node#nodeAdded()
	 */
	@Override
	protected void nodeAdded() {
		createChildNodes();
	}

	/**
	 * @see net.sf.jame.core.tree.Node#nodeRemoved()
	 */
	@Override
	protected void nodeRemoved() {
	}

	/**
	 * @see net.sf.jame.core.tree.Node#addLabel(java.lang.StringBuilder)
	 */
	@Override
	protected void addLabel(final StringBuilder builder) {
		super.addLabel(builder);
		builder.append(" (Total count " + listElement.getElementCount() + ")");
	}

	protected void createChildNodes() {
		createConfigElementNodes(listElement);
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean equals(final Object o) {
		if (o instanceof ConfigElementListNode) {
			return (listElement == ((ConfigElementListNode) o).listElement);
		}
		return false;
	}

	/**
	 * Creates the nodes.
	 * 
	 * @param listElement the frame element.
	 */
	protected void createConfigElementNodes(final ListConfigElement<T> listElement) {
		for (int i = 0; i < listElement.getElementCount(); i++) {
			final T configElement = listElement.getElement(i);
			final Node configElementNode = createChildNode(configElement);
			appendChildNode(configElementNode);
		}
	}

	/**
	 * @see net.sf.jame.core.tree.MutableNode#getChildValueType()
	 */
	@Override
	public abstract Class<?> getChildValueType();

	/**
	 * @see net.sf.jame.core.tree.DefaultNode#createNodeEditor()
	 */
	@Override
	protected NodeEditor createNodeEditor() {
		return new ListNodeEditor();
	}

	/**
	 * @param value
	 * @return
	 */
	public abstract NodeValue<?> createNodeValue(final Serializable value);

	protected class ListNodeEditor extends NodeEditor {
		/**
		 * 
		 */
		public ListNodeEditor() {
			super(ConfigElementListNode.this);
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#createChildNode(net.sf.jame.core.tree.NodeValue)
		 */
		@SuppressWarnings("unchecked")
		@Override
		protected Node createChildNode(final NodeValue value) {
			final T configElement = ((NodeValue<T>) value).getValue();
			configElement.setContext(getContext());
			return ConfigElementListNode.this.createChildNode(configElement);
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#doAppendNode(net.sf.jame.core.tree.Node)
		 */
		@SuppressWarnings("unchecked")
		@Override
		protected void doAppendNode(final Node node) {
			listElement.removeChangeListener(listListener);
			listElement.appendElement(((ConfigElementNode<T>) node).getConfigElement());
			listElement.addChangeListener(listListener);
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#doInsertNodeAfter(int, net.sf.jame.core.tree.Node)
		 */
		@SuppressWarnings("unchecked")
		@Override
		protected void doInsertNodeAfter(final int index, final Node node) {
			listElement.removeChangeListener(listListener);
			listElement.insertElementAfter(index, ((ConfigElementNode<T>) node).getConfigElement());
			listElement.addChangeListener(listListener);
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#doInsertNodeBefore(int, net.sf.jame.core.tree.Node)
		 */
		@SuppressWarnings("unchecked")
		@Override
		protected void doInsertNodeBefore(final int index, final Node node) {
			listElement.removeChangeListener(listListener);
			listElement.insertElementBefore(index, ((ConfigElementNode<T>) node).getConfigElement());
			listElement.addChangeListener(listListener);
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#doRemoveNode(int)
		 */
		@Override
		protected void doRemoveNode(final int nodeIndex) {
			listElement.removeChangeListener(listListener);
			listElement.removeElement(nodeIndex);
			listElement.addChangeListener(listListener);
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#getNodeValueType()
		 */
		@Override
		public Class<?> getNodeValueType() {
			return ConfigElementListNodeValue.class;
		}

		/**
		 * @see net.sf.jame.core.tree.NodeEditor#createNodeValue(java.io.Serializable)
		 */
		@SuppressWarnings("unchecked")
		@Override
		public NodeValue createNodeValue(final Serializable value) {
			return ConfigElementListNode.this.createNodeValue(value);
		}
	}

	protected class ConfigElementListener implements ValueChangeListener, ListConfigElementEvents {
		@SuppressWarnings("unchecked")
		public void valueChanged(final ValueChangeEvent e) {
			cancel();
			switch (e.getEventType()) {
				case ELEMENT_ADDED: {
					final ConfigElementNode<T> configElementNode = createChildNode((T) e.getParams()[0]);
					appendChildNode(configElementNode);
					getSession().appendAction(new NodeAction(getNodeClass(), NodeAction.ACTION_APPEND_NODE, e.getTimestamp(), getNodePath(), e.getParams()[1], configElementNode.getConfigElement().clone()));
					break;
				}
				case ELEMENT_INSERTED_AFTER: {
					final ConfigElementNode<T> configElementNode = createChildNode((T) e.getParams()[0]);
					insertNodeAfter((Integer) e.getParams()[1], configElementNode);
					getSession().appendAction(new NodeAction(getNodeClass(), NodeAction.ACTION_INSERT_NODE_AFTER, e.getTimestamp(), getNodePath(), e.getParams()[1], configElementNode.getConfigElement().clone()));
					break;
				}
				case ELEMENT_INSERTED_BEFORE: {
					final ConfigElementNode<T> configElementNode = createChildNode((T) e.getParams()[0]);
					insertNodeBefore((Integer) e.getParams()[1], configElementNode);
					getSession().appendAction(new NodeAction(getNodeClass(), NodeAction.ACTION_INSERT_NODE_BEFORE, e.getTimestamp(), getNodePath(), e.getParams()[1], configElementNode.getConfigElement().clone()));
					break;
				}
				case ELEMENT_REMOVED: {
					final ConfigElementNode<T> configElementNode = createChildNode((T) e.getParams()[0]);
					removeChildNode((Integer) e.getParams()[1]);
					getSession().appendAction(new NodeAction(getNodeClass(), NodeAction.ACTION_REMOVE_NODE, e.getTimestamp(), getNodePath(), e.getParams()[1], configElementNode.getConfigElement().clone()));
					break;
				}
				default: {
					break;
				}
			}
		}
	}
}
