/*
 * $Id:TwisterConfigPanel.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.LayoutManager;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.HashMap;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ComboBoxModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import net.sf.jame.core.config.ConfigContext;
import net.sf.jame.core.config.DefaultConfigContext;
import net.sf.jame.core.config.ListConfigElementEvents;
import net.sf.jame.core.config.ValueChangeEvent;
import net.sf.jame.core.config.ValueChangeListener;
import net.sf.jame.core.config.ValueConfigElementEvents;
import net.sf.jame.core.extension.ConfigurableExtension;
import net.sf.jame.core.extension.Extension;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.core.extension.NullConfigurableExtension;
import net.sf.jame.core.swing.color.ColorChangeEvent;
import net.sf.jame.core.swing.color.ColorChangeListener;
import net.sf.jame.core.swing.color.ColorChooser;
import net.sf.jame.core.swing.color.ColorField;
import net.sf.jame.core.swing.extension.ConfigurableExtensionComboBoxModel;
import net.sf.jame.core.swing.extension.ExtensionListCellRenderer;
import net.sf.jame.core.swing.util.GUIFactory;
import net.sf.jame.core.tree.NodeSession;
import net.sf.jame.twister.Color32bit;
import net.sf.jame.twister.DefaultThreadFactory;
import net.sf.jame.twister.ImageTile;
import net.sf.jame.twister.IntegerVector2D;
import net.sf.jame.twister.TwisterConfig;
import net.sf.jame.twister.TwisterRegistry;
import net.sf.jame.twister.TwisterRuntime;
import net.sf.jame.twister.effect.EffectConfigElement;
import net.sf.jame.twister.effect.extension.EffectExtensionConfig;
import net.sf.jame.twister.effect.extension.EffectExtensionRuntime;
import net.sf.jame.twister.frame.FrameConfigElement;
import net.sf.jame.twister.frame.filter.FrameFilterConfigElement;
import net.sf.jame.twister.frame.filter.extension.FrameFilterExtensionConfig;
import net.sf.jame.twister.frame.filter.extension.FrameFilterExtensionRuntime;
import net.sf.jame.twister.frame.layer.GroupLayerConfigElement;
import net.sf.jame.twister.frame.layer.ImageLayerConfigElement;
import net.sf.jame.twister.frame.layer.LayerConfigElement;
import net.sf.jame.twister.frame.layer.filter.LayerFilterConfigElement;
import net.sf.jame.twister.frame.layer.filter.extension.LayerFilterExtensionConfig;
import net.sf.jame.twister.frame.layer.filter.extension.LayerFilterExtensionRuntime;
import net.sf.jame.twister.frame.layer.image.ImageConfigElement;
import net.sf.jame.twister.frame.layer.image.extension.ImageExtensionConfig;
import net.sf.jame.twister.frame.layer.image.extension.ImageExtensionRuntime;
import net.sf.jame.twister.renderer.DefaultTwisterRenderer;
import net.sf.jame.twister.renderer.RenderContext;
import net.sf.jame.twister.renderer.Surface;
import net.sf.jame.twister.renderer.TwisterRenderer;
import net.sf.jame.twister.renderer.TwisterRenderingHints;
import net.sf.jame.twister.swing.view.NavigatorViewRuntime;
import net.sf.jame.twister.swing.view.extension.ViewExtensionRuntime;
import net.sf.jame.twister.util.ExtensionConfigElementEvents;

/**
 * @author Andrea Medeghini
 */
public class TwisterConfigPanel extends JPanel {
	private static final long serialVersionUID = 1L;
	private static final Color oddColor = new Color(0xFFDDDDDD);
	private static final Color evenColor = new Color(0xFFD0D0D0);
	private final ViewContext viewContext;
	private final RenderContext context;
	private final NodeSession session;
	private final TwisterConfig config;

	/**
	 * @param config
	 * @param viewContext
	 * @param context
	 * @param session
	 */
	public TwisterConfigPanel(final TwisterConfig config, final ViewContext viewContext, final RenderContext context, final NodeSession session) {
		this.viewContext = viewContext;
		this.context = context;
		this.session = session;
		this.config = config;
		setLayout(new BorderLayout());
		add(new ConfigPanel(config), BorderLayout.CENTER);
	}

	private static JCheckBox createIconCheckBox(final String key, final String iconKey, final int width, final int height) {
		final JCheckBox checkbox = GUIFactory.createCheckBox((String) null, TwisterSwingResources.getInstance().getString("tooltip." + key));
		try {
			checkbox.setIcon(new ImageIcon(ImageIO.read(TwisterConfigPanel.class.getResourceAsStream("/icons/" + iconKey + "-icon.png"))));
			checkbox.setSelectedIcon(new ImageIcon(ImageIO.read(TwisterConfigPanel.class.getResourceAsStream("/icons/" + iconKey + "-selected-icon.png"))));
		}
		catch (final Exception e) {
			System.out.println("key = " + key + ", iconKey = " + iconKey);
			e.printStackTrace();
		}
		checkbox.setOpaque(false);
		checkbox.setPreferredSize(new Dimension(width, height));
		checkbox.setMinimumSize(new Dimension(width, height));
		checkbox.setMaximumSize(new Dimension(width, height));
		return checkbox;
	}

	private static JButton createIconButton(final String key, final String iconKey, final int width, final int height) {
		final JButton button = GUIFactory.createButton((String) null, TwisterSwingResources.getInstance().getString("tooltip." + key));
		try {
			button.setIcon(new ImageIcon(ImageIO.read(TwisterConfigPanel.class.getResourceAsStream("/icons/" + iconKey + "-icon.png"))));
			button.setPressedIcon(new ImageIcon(ImageIO.read(TwisterConfigPanel.class.getResourceAsStream("/icons/" + iconKey + "-pressed-icon.png"))));
		}
		catch (final Exception e) {
			System.out.println("key = " + key + ", iconKey = " + iconKey);
			e.printStackTrace();
		}
		button.setOpaque(false);
		button.setPreferredSize(new Dimension(width, height));
		button.setMinimumSize(new Dimension(width, height));
		button.setMaximumSize(new Dimension(width, height));
		return button;
	}

	private static JCheckBox createSelectionCheckBox() {
		final JCheckBox checkbox = GUIFactory.createSmallCheckBox((String) null, (String) null);
		checkbox.setOpaque(false);
		checkbox.setPreferredSize(new Dimension(20, 20));
		checkbox.setMinimumSize(new Dimension(20, 20));
		checkbox.setMaximumSize(new Dimension(20, 20));
		return checkbox;
	}

	private static JCheckBox createCheckBox() {
		final JCheckBox checkbox = GUIFactory.createSmallCheckBox((String) null, (String) null);
		checkbox.setOpaque(false);
		return checkbox;
	}

	private static JCheckBox createTextCheckBox(final String key, final int width, final int height) {
		final JCheckBox checkbox = GUIFactory.createCheckBox(TwisterSwingResources.getInstance().getString("label." + key), TwisterSwingResources.getInstance().getString("tooltip." + key));
		// final FontMetrics fm = checkbox.getFontMetrics(checkbox.getFont());
		// int width = fm.stringWidth(checkbox.getText()) + 20;
		checkbox.setPreferredSize(new Dimension(width, height));
		checkbox.setMinimumSize(new Dimension(width, height));
		checkbox.setMaximumSize(new Dimension(width, height));
		checkbox.setOpaque(false);
		return checkbox;
	}

	private static JButton createTextButton(final int width, final int height) {
		final JButton button = GUIFactory.createSmallButton((String) null, (String) null);
		button.setPreferredSize(new Dimension(width, height));
		button.setMinimumSize(new Dimension(width, height));
		button.setMaximumSize(new Dimension(width, height));
		button.setOpaque(false);
		return button;
	}

	private static JButton createTextButton(final String key, final int width, final int height) {
		final JButton button = GUIFactory.createSmallButton(TwisterSwingResources.getInstance().getString("label." + key), TwisterSwingResources.getInstance().getString("tooltip." + key));
		// final FontMetrics fm = button.getFontMetrics(button.getFont());
		// int width = fm.stringWidth(button.getText());
		button.setPreferredSize(new Dimension(width, height));
		button.setMinimumSize(new Dimension(width, height));
		button.setMaximumSize(new Dimension(width, height));
		button.setOpaque(false);
		return button;
	}

	private static JSpinner createSpinner(final int min, final int max, final int step) {
		final JSpinner spinner = GUIFactory.createSpinner(new SpinnerNumberModel(0, min, max, step), (String) null);
		// spinner.setPreferredSize(new Dimension(60, GUIFactory.DEFAULT_HEIGHT));
		spinner.setMaximumSize(new Dimension(60, GUIFactory.DEFAULT_HEIGHT));
		spinner.setMinimumSize(new Dimension(60, GUIFactory.DEFAULT_HEIGHT));
		// if (!"Mac OS X".equals(System.getProperty("os.name")) || !UIManager.getLookAndFeel().isNativeLookAndFeel()) {
		// spinner.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED, Color.LIGHT_GRAY, Color.DARK_GRAY));
		// }
		return spinner;
	}

	private static JComboBox createExtensionComboBox(final ComboBoxModel model, final int width, final int height) {
		final JComboBox extensionComboBox = GUIFactory.createSmallComboBox(model, (String) null);
		extensionComboBox.setRenderer(new ExtensionListCellRenderer());
		extensionComboBox.setPreferredSize(new Dimension(width, height));
		extensionComboBox.setMaximumSize(new Dimension(width, height));
		extensionComboBox.setMinimumSize(new Dimension(width, height));
		extensionComboBox.setOpaque(false);
		return extensionComboBox;
	}

	private static JLabel createTextLabel(final String key, final int alignment, final int width, final int height) {
		final JLabel label = GUIFactory.createSmallLabel(TwisterSwingResources.getInstance().getString("label." + key), SwingConstants.LEFT);
		label.setBorder(BorderFactory.createEmptyBorder(0, 8, 0, 8));
		label.setPreferredSize(new Dimension(width, height));
		label.setMinimumSize(new Dimension(width, height));
		label.setMaximumSize(new Dimension(width, height));
		return label;
	}

	private static JLabel createLabel(final String text) {
		final JLabel label = GUIFactory.createSmallLabel(text, SwingConstants.LEFT);
		label.setBorder(BorderFactory.createEmptyBorder(0, 8, 0, 8));
		return label;
	}

	private static JTextField createTextField(final String text, final int width, final int height) {
		final JTextField textfield = GUIFactory.createTextField(text, null);
		textfield.setPreferredSize(new Dimension(width, height));
		textfield.setMinimumSize(new Dimension(width, height));
		textfield.setMaximumSize(new Dimension(width, height));
		// if (!"Mac OS X".equals(System.getProperty("os.name")) || !UIManager.getLookAndFeel().isNativeLookAndFeel()) {
		// textfield.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED, Color.LIGHT_GRAY, Color.DARK_GRAY));
		// }
		return textfield;
	}

	private static JPanel createPanel(final LayoutManager layoutManager, final boolean opaque) {
		final JPanel panel = new JPanel(layoutManager);
		panel.setOpaque(opaque);
		return panel;
	}

	private static Box createHorizontalBox(final boolean opaque) {
		final Box box = Box.createHorizontalBox();
		box.setOpaque(opaque);
		return box;
	}

	private static Box createVerticalBox(final boolean opaque) {
		final Box box = Box.createVerticalBox();
		box.setOpaque(opaque);
		return box;
	}

	private static Component createSpace() {
		final Component box = Box.createHorizontalStrut(4);
		return box;
	}

	// private static JSlider createSlider(final String key, final int min, final int max, final int value, final int spacing, final String suffix) {
	// final JSlider slider = new JSlider(min, max);
	// final int mid = (min + max) / 2;
	// final Dictionary<Integer, JLabel> labels = new Hashtable<Integer, JLabel>();
	// labels.put(new Integer(min), GUIFactory.createLabel(String.valueOf(min) + suffix));
	// labels.put(new Integer(mid), GUIFactory.createLabel(String.valueOf(mid) + suffix));
	// labels.put(new Integer(max), GUIFactory.createLabel(String.valueOf(max) + suffix));
	// slider.setLabelTable(labels);
	// slider.setMajorTickSpacing(spacing);
	// slider.setPaintTicks(true);
	// slider.setPaintTrack(true);
	// slider.setPaintLabels(true);
	// slider.setValue(value);
	// slider.setToolTipText(MandelbrotSwingResources.getInstance().getString("tooltip." + key) + " " + slider.getValue() + suffix);
	// slider.addChangeListener(new SliderChangeListener(slider, key));
	// slider.setOpaque(false);
	// slider.setFont(SMALL_FONT);
	// return slider;
	// }
	private class RenderTask implements Runnable {
		private final Object lock = new Object();
		private Thread thread;
		private boolean dirty;
		private boolean running;
		private JPanel layersPanel;
		private final DefaultThreadFactory factory = new DefaultThreadFactory("ConfigPanel", true, Thread.MIN_PRIORITY);

		/**
		 * @param layersPanel
		 */
		public RenderTask(JPanel layersPanel) {
			this.layersPanel = layersPanel;
		}

		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				while (running) {
					synchronized (lock) {
						if (dirty) {
							lock.wait();
							dirty = false;
						}
					}
					try {
						Thread.sleep(500);
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							final GroupLayerPanel layerPanel = (GroupLayerPanel) layersPanel.getComponent(i);
							layerPanel.abortPreviewRenderer();
						}
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							final GroupLayerPanel layerPanel = (GroupLayerPanel) layersPanel.getComponent(i);
							layerPanel.joinPreviewRenderer();
						}
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							final GroupLayerPanel layerPanel = (GroupLayerPanel) layersPanel.getComponent(i);
							layerPanel.prepareLayerPreview();
						}
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							final GroupLayerPanel layerPanel = (GroupLayerPanel) layersPanel.getComponent(i);
							layerPanel.startPreviewRenderer();
						}
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							final GroupLayerPanel layerPanel = (GroupLayerPanel) layersPanel.getComponent(i);
							layerPanel.joinPreviewRenderer();
						}
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							final GroupLayerPanel layerPanel = (GroupLayerPanel) layersPanel.getComponent(i);
							SwingUtilities.invokeAndWait(new Runnable() {
								public void run() {
									layerPanel.drawLayerPreview();
								}
							});
						}
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
				}
			}
			catch (final InterruptedException e) {
			}
		}

		/**
		 * 
		 */
		public void stop() {
			try {
				if (thread != null) {
					running = false;
					thread.interrupt();
					thread.join();
				}
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}

		/**
		 * 
		 */
		public void start() {
			if (thread == null) {
				thread = factory.newThread(this);
				thread.setName(thread.getName() + " RenderTask");
				running = true;
				thread.start();
			}
		}

		/**
		 * 
		 */
		public void executeTask() {
			synchronized (lock) {
				dirty = true;
				lock.notify();
			}
		}
	}

	private class ConfigPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private EffectPanel effectPanel;
		private FrameConfigElement frameElement;

		/**
		 * @param config
		 */
		public ConfigPanel(final TwisterConfig config) {
			frameElement = config.getFrameConfigElement();
			final JButton selectGroupButton = createIconButton("selectGroups", "select", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton appendGroupButton = createIconButton("appendGroup", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton removeGroupButton = createIconButton("removeGroups", "remove", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertGroupAfterButton = createIconButton("insertGroupAfter", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertGroupBeforeButton = createIconButton("insertGroupBefore", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton selectFilterButton = createIconButton("selectFilters", "select", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton appendFilterButton = createIconButton("appendFilter", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton removeFilterButton = createIconButton("removeFilters", "remove", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertFilterAfterButton = createIconButton("insertFilterAfter", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertFilterBeforeButton = createIconButton("insertFilterBefore", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton editLayersButton = createTextButton("editLayers", 120, GUIFactory.DEFAULT_HEIGHT);
			final JButton editFiltersButton = createTextButton("editFilters", 120, GUIFactory.DEFAULT_HEIGHT);
			final JButton editEffectsButton = createTextButton("editEffects", 120, GUIFactory.DEFAULT_HEIGHT);
			final JButton editBackgroundButton = createTextButton("editBackground", 120, GUIFactory.DEFAULT_HEIGHT);
			final JButton colorChooseButton = createTextButton("editColor", 80, GUIFactory.DEFAULT_HEIGHT);
			final JPanel layersPanel = createPanel(new StackLayout(), false);
			final JPanel filtersPanel = createPanel(new StackLayout(), false);
			effectPanel = new EffectPanel(config, config.getEffectConfigElement());
			effectPanel.setName(createEffectPanelName(config.getEffectConfigElement()));
			effectPanel.setOpaque(true);
			final JLabel colorLabel = createTextLabel("background", SwingConstants.LEFT, 120, GUIFactory.DEFAULT_HEIGHT);
			colorLabel.setFont(GUIFactory.NORMAL_FONT);
			colorLabel.setHorizontalAlignment(SwingConstants.CENTER);
			final ColorField colorField = new ColorField();
			colorField.setPreferredSize(new Dimension(GUIFactory.DEFAULT_HEIGHT * 3, GUIFactory.DEFAULT_HEIGHT * 3));
			colorField.setMinimumSize(new Dimension(GUIFactory.DEFAULT_HEIGHT * 3, GUIFactory.DEFAULT_HEIGHT * 3));
			colorField.setMaximumSize(new Dimension(GUIFactory.DEFAULT_HEIGHT * 3, GUIFactory.DEFAULT_HEIGHT * 3));
			colorField.setBorder(BorderFactory.createLineBorder(Color.BLACK));
			colorField.setOpaque(false);
			final Box colorPanel = createHorizontalBox(true);
			colorPanel.setName(TwisterSwingResources.getInstance().getString("name.background"));
			colorPanel.add(Box.createHorizontalGlue());
			colorPanel.add(colorLabel);
			colorPanel.add(colorField);
			colorPanel.add(Box.createHorizontalStrut(8));
			colorPanel.add(colorChooseButton);
			colorPanel.add(Box.createHorizontalGlue());
			colorPanel.setBorder(BorderFactory.createMatteBorder(1, 0, 0, 0, Color.DARK_GRAY));
			colorField.setColor(new Color(config.getBackground().getARGB(), true));
			final Box layersPanel4 = createHorizontalBox(false);
			layersPanel4.add(Box.createHorizontalGlue());
			layersPanel4.add(selectGroupButton);
			layersPanel4.add(createSpace());
			layersPanel4.add(appendGroupButton);
			layersPanel4.add(createSpace());
			layersPanel4.add(removeGroupButton);
			// layersPanel4.add(insertGroupAfterButton);
			// layersPanel4.add(insertGroupBeforeButton);
			layersPanel4.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY), BorderFactory.createEmptyBorder(4, 4, 4, 4)));
			final JLabel layersLabel = createTextLabel("noGroupLayer", SwingConstants.LEFT, 120, GUIFactory.DEFAULT_HEIGHT);
			layersLabel.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			final Box layersPanel3 = createHorizontalBox(false);
			layersPanel3.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			layersPanel3.add(layersLabel);
			layersPanel3.add(Box.createHorizontalGlue());
			final JPanel layersPanel2 = createPanel(new BorderLayout(), true);
			layersPanel2.setName(createLayersPanelName());
			layersPanel2.add(layersPanel4, BorderLayout.NORTH);
			layersPanel2.add(layersPanel, BorderLayout.CENTER);
			layersPanel2.add(layersPanel3, BorderLayout.SOUTH);
			layersPanel2.setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY));
			final Box filtersPanel4 = createHorizontalBox(false);
			filtersPanel4.add(Box.createHorizontalGlue());
			filtersPanel4.add(selectFilterButton);
			filtersPanel4.add(createSpace());
			filtersPanel4.add(appendFilterButton);
			filtersPanel4.add(createSpace());
			filtersPanel4.add(removeFilterButton);
			// filtersPanel4.add(insertFilterAfterButton);
			// filtersPanel4.add(insertFilterBeforeButton);
			filtersPanel4.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY), BorderFactory.createEmptyBorder(4, 4, 4, 4)));
			final JLabel filtersLabel = createTextLabel("noFrameFilter", SwingConstants.LEFT, 120, GUIFactory.DEFAULT_HEIGHT);
			filtersLabel.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			final Box filtersPanel3 = createHorizontalBox(false);
			filtersPanel3.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			filtersPanel3.add(filtersLabel);
			filtersPanel3.add(Box.createHorizontalGlue());
			final JPanel filtersPanel2 = createPanel(new BorderLayout(), true);
			filtersPanel2.setName(createFiltersPanelName());
			filtersPanel2.add(filtersPanel4, BorderLayout.NORTH);
			filtersPanel2.add(filtersPanel, BorderLayout.CENTER);
			filtersPanel2.add(filtersPanel3, BorderLayout.SOUTH);
			filtersPanel2.setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY));
			// JPanel b1 = createPanel(new FlowLayout(FlowLayout.RIGHT), false);
			// b1.setPreferredSize(new Dimension(200, 40));
			// b1.add(editLayersButton);
			// JPanel p1 = createPanel(new FlowLayout(FlowLayout.LEFT), false);
			// p1.add(b1);
			// JTextArea a1 = new JTextArea(TwisterSwingResources.getInstance().getString("text.editLayers"));
			// a1.setOpaque(false);
			// a1.setEditable(false);
			// a1.setFont(SMALL_FONT);
			// p1.add(a1);
			// JPanel b2 = createPanel(new FlowLayout(FlowLayout.RIGHT), false);
			// b2.setPreferredSize(new Dimension(200, 40));
			// b2.add(editFiltersButton);
			// JPanel p2 = createPanel(new FlowLayout(FlowLayout.LEFT), false);
			// p2.add(b2);
			// JTextArea a2 = new JTextArea(TwisterSwingResources.getInstance().getString("text.editFilters"));
			// a2.setOpaque(false);
			// a2.setEditable(false);
			// a2.setFont(SMALL_FONT);
			// p2.add(a2);
			// JPanel b3 = createPanel(new FlowLayout(FlowLayout.RIGHT), false);
			// b3.setPreferredSize(new Dimension(200, 40));
			// b3.add(editEffectsButton);
			// JPanel p3 = createPanel(new FlowLayout(FlowLayout.LEFT), false);
			// p3.add(b3);
			// JTextArea a3 = new JTextArea(TwisterSwingResources.getInstance().getString("text.editEffects"));
			// a3.setOpaque(false);
			// a3.setEditable(false);
			// a3.setFont(SMALL_FONT);
			// p3.add(a3);
			// JPanel b4 = createPanel(new FlowLayout(FlowLayout.RIGHT), false);
			// b4.setPreferredSize(new Dimension(200, 40));
			// b4.add(editBackgroundButton);
			// JPanel p4 = createPanel(new FlowLayout(FlowLayout.LEFT), false);
			// p4.add(b4);
			// JTextArea a4 = new JTextArea(TwisterSwingResources.getInstance().getString("text.editBackground"));
			// a4.setOpaque(false);
			// a4.setEditable(false);
			// a4.setFont(SMALL_FONT);
			// p4.add(a4);
			// final JPanel subpanelContainer = createPanel(new GridLayout(4, 1), false);
			// subpanelContainer.add(p1);
			// subpanelContainer.add(p2);
			// subpanelContainer.add(p3);
			// subpanelContainer.add(p4);
			// setLayout(new BorderLayout());
			// add(subpanelContainer, BorderLayout.CENTER);
			// subpanelContainer.setBorder(BorderFactory.createMatteBorder(1, 0, 1, 0, Color.DARK_GRAY));
			final Box subpanelContainer = createHorizontalBox(false);
			subpanelContainer.add(Box.createHorizontalGlue());
			subpanelContainer.add(editLayersButton);
			subpanelContainer.add(Box.createHorizontalStrut(8));
			subpanelContainer.add(editFiltersButton);
			subpanelContainer.add(Box.createHorizontalStrut(8));
			subpanelContainer.add(editEffectsButton);
			subpanelContainer.add(Box.createHorizontalStrut(8));
			subpanelContainer.add(editBackgroundButton);
			subpanelContainer.add(Box.createHorizontalGlue());
			setLayout(new BorderLayout());
			add(subpanelContainer, BorderLayout.CENTER);
			setBorder(BorderFactory.createMatteBorder(1, 0, 1, 0, Color.DARK_GRAY));
			final RenderTask task = new RenderTask(layersPanel);
			task.start();
			final ConfigContext configContext = new DefaultConfigContext() {
				@Override
				public void updateTimestamp() {
					task.executeTask();
//					SwingUtilities.invokeLater(new Runnable() {
//						public void run() {
//							task.executeTask();
//						}
//					});
				}
			};
			config.getContext().setParentConfigContext(configContext);
			final ValueChangeListener backgroundListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					colorField.setColor(new Color(config.getBackground().getARGB(), true));
				}
			};
			layersPanel3.setVisible(frameElement.getLayerConfigElementCount() == 0);
			filtersPanel3.setVisible(frameElement.getFilterConfigElementCount() == 0);
			for (int i = 0; i < frameElement.getLayerConfigElementCount(); i++) {
				final GroupLayerConfigElement groupLayerElement = frameElement.getLayerConfigElement(i);
				final GroupLayerPanel groupLayerPanel = new GroupLayerPanel(frameElement, groupLayerElement, task);
				layersPanel.add(groupLayerPanel);
			}
			for (int i = 0; i < frameElement.getFilterConfigElementCount(); i++) {
				final FrameFilterConfigElement frameFilterElement = frameElement.getFilterConfigElement(i);
				final FrameFilterPanel frameFilterPanel = new FrameFilterPanel(frameElement, frameFilterElement);
				filtersPanel.add(frameFilterPanel);
			}
			for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
				filtersPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
			}
			final ValueChangeListener filtersListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ListConfigElementEvents.ELEMENT_ADDED: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new FrameFilterPanel(frameElement, (FrameFilterConfigElement) e.getParams()[0]));
							filtersPanel3.setVisible(frameElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_AFTER: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new FrameFilterPanel(frameElement, (FrameFilterConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue() + 1);
							filtersPanel3.setVisible(frameElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_BEFORE: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new FrameFilterPanel(frameElement, (FrameFilterConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue());
							filtersPanel3.setVisible(frameElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_REMOVED: {
							filtersPanel.remove(((Integer) e.getParams()[1]).intValue());
							filtersPanel3.setVisible(frameElement.getFilterConfigElementCount() == 0);
							viewContext.resize();
							break;
						}
						default: {
							break;
						}
					}
					for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
						filtersPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
					}
				}
			};
			final ValueChangeListener layersListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					synchronized (task) {
						switch (e.getEventType()) {
							case ListConfigElementEvents.ELEMENT_ADDED: {
								viewContext.resize(GUIFactory.DEFAULT_HEIGHT * 2 + 8);
								layersPanel.add(new GroupLayerPanel(frameElement, (GroupLayerConfigElement) e.getParams()[0], task));
								layersPanel3.setVisible(frameElement.getLayerConfigElementCount() == 0);
								break;
							}
							case ListConfigElementEvents.ELEMENT_INSERTED_AFTER: {
								viewContext.resize(GUIFactory.DEFAULT_HEIGHT * 2 + 8);
								layersPanel.add(new GroupLayerPanel(frameElement, (GroupLayerConfigElement) e.getParams()[0], task), ((Integer) e.getParams()[1]).intValue() + 1);
								layersPanel3.setVisible(frameElement.getLayerConfigElementCount() == 0);
								break;
							}
							case ListConfigElementEvents.ELEMENT_INSERTED_BEFORE: {
								viewContext.resize(GUIFactory.DEFAULT_HEIGHT * 2 + 8);
								layersPanel.add(new GroupLayerPanel(frameElement, (GroupLayerConfigElement) e.getParams()[0], task), ((Integer) e.getParams()[1]).intValue());
								layersPanel3.setVisible(frameElement.getLayerConfigElementCount() == 0);
								break;
							}
							case ListConfigElementEvents.ELEMENT_REMOVED: {
								layersPanel.remove(((Integer) e.getParams()[1]).intValue());
								layersPanel3.setVisible(frameElement.getLayerConfigElementCount() == 0);
								viewContext.resize();
								break;
							}
							default: {
								break;
							}
						}
					}
				}
			};
			final ValueChangeListener frameListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ValueConfigElementEvents.VALUE_CHANGED: {
							synchronized (task) {
								viewContext.setComponent(TwisterConfigPanel.this);
								frameElement.getFilterListElement().removeChangeListener(filtersListener);
								frameElement.getLayerListElement().removeChangeListener(layersListener);
								frameElement = config.getFrameConfigElement();
								layersPanel.removeAll();
								filtersPanel.removeAll();
								layersPanel3.setVisible(frameElement.getLayerConfigElementCount() == 0);
								filtersPanel3.setVisible(frameElement.getFilterConfigElementCount() == 0);
								for (int i = 0; i < frameElement.getLayerConfigElementCount(); i++) {
									final GroupLayerConfigElement groupLayerElement = frameElement.getLayerConfigElement(i);
									final GroupLayerPanel groupLayerPanel = new GroupLayerPanel(frameElement, groupLayerElement, task);
									layersPanel.add(groupLayerPanel);
								}
								for (int i = 0; i < frameElement.getFilterConfigElementCount(); i++) {
									final FrameFilterConfigElement frameFilterElement = frameElement.getFilterConfigElement(i);
									final FrameFilterPanel frameFilterPanel = new FrameFilterPanel(frameElement, frameFilterElement);
									filtersPanel.add(frameFilterPanel);
								}
								frameElement.getFilterListElement().addChangeListener(filtersListener);
								frameElement.getLayerListElement().addChangeListener(layersListener);
							}
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			final ValueChangeListener effectListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ValueConfigElementEvents.VALUE_CHANGED: {
							viewContext.setComponent(TwisterConfigPanel.this);
							effectPanel = new EffectPanel(config, config.getEffectConfigElement());
							effectPanel.setName(createEffectPanelName(config.getEffectConfigElement()));
							effectPanel.setOpaque(true);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			config.getBackgroundElement().addChangeListener(backgroundListener);
			config.getFrameSingleElement().addChangeListener(frameListener);
			config.getEffectSingleElement().addChangeListener(effectListener);
			frameElement.getFilterListElement().addChangeListener(filtersListener);
			frameElement.getLayerListElement().addChangeListener(layersListener);
			appendGroupButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					frameElement.getContext().updateTimestamp();
					for (int i = frameElement.getLayerConfigElementCount() - 1; i >= 0; i--) {
						final GroupLayerConfigElement groupLayerElement = frameElement.getLayerConfigElement(i);
						if (groupLayerElement.getUserData() != null) {
							frameElement.insertLayerConfigElementBefore(i, new GroupLayerConfigElement());
							return;
						}
					}
					frameElement.appendLayerConfigElement(new GroupLayerConfigElement());
					context.refresh();
				}
			});
			removeGroupButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					frameElement.getContext().updateTimestamp();
					for (int i = frameElement.getLayerConfigElementCount() - 1; i >= 0; i--) {
						final GroupLayerConfigElement groupLayerElement = frameElement.getLayerConfigElement(i);
						if (groupLayerElement.getUserData() != null) {
							frameElement.removeLayerConfigElement(i);
						}
					}
					context.refresh();
				}
			});
			// insertGroupAfterButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = 0; i < frameElement.getLayerConfigElementCount(); i++) {
			// GroupLayerConfigElement groupLayerElement = frameElement.getLayerConfigElement(i);
			// if (groupLayerElement.getUserData() != null) {
			// frameElement.insertLayerConfigElementAfter(i, new GroupLayerConfigElement());
			// }
			// }
			// }
			// });
			// insertGroupBeforeButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = frameElement.getLayerConfigElementCount() - 1; i >= 0; i--) {
			// GroupLayerConfigElement groupLayerElement = frameElement.getLayerConfigElement(i);
			// if (groupLayerElement.getUserData() != null) {
			// frameElement.insertLayerConfigElementBefore(i, new GroupLayerConfigElement());
			// }
			// }
			// }
			// });
			appendFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					frameElement.getContext().updateTimestamp();
					for (int i = frameElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
						final FrameFilterConfigElement frameFilterElement = frameElement.getFilterConfigElement(i);
						if (frameFilterElement.getUserData() != null) {
							frameElement.insertFilterConfigElementBefore(i, new FrameFilterConfigElement());
							return;
						}
					}
					frameElement.appendFilterConfigElement(new FrameFilterConfigElement());
					context.refresh();
				}
			});
			removeFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					frameElement.getContext().updateTimestamp();
					for (int i = frameElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
						final FrameFilterConfigElement frameFilterElement = frameElement.getFilterConfigElement(i);
						if (frameFilterElement.getUserData() != null) {
							frameElement.removeFilterConfigElement(i);
						}
					}
					context.refresh();
				}
			});
			// insertFilterAfterButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = 0; i < frameElement.getFilterConfigElementCount(); i++) {
			// FrameFilterConfigElement frameFilterElement = frameElement.getFilterConfigElement(i);
			// if (frameFilterElement.getUserData() != null) {
			// frameElement.insertFilterConfigElementAfter(i, new FrameFilterConfigElement());
			// }
			// }
			// }
			// });
			// insertFilterBeforeButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = frameElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
			// FrameFilterConfigElement frameFilterElement = frameElement.getFilterConfigElement(i);
			// if (frameFilterElement.getUserData() != null) {
			// frameElement.insertFilterConfigElementBefore(i, new FrameFilterConfigElement());
			// }
			// }
			// }
			// });
			selectFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					boolean allSelected = true;
					for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
						if (!((FrameFilterPanel) filtersPanel.getComponent(i)).isSelected()) {
							allSelected = false;
						}
					}
					if (allSelected) {
						for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
							((FrameFilterPanel) filtersPanel.getComponent(i)).setSelected(false);
						}
					}
					else {
						for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
							((FrameFilterPanel) filtersPanel.getComponent(i)).setSelected(true);
						}
					}
				}
			});
			selectGroupButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					boolean allSelected = true;
					for (int i = 0; i < layersPanel.getComponentCount(); i++) {
						if (!((GroupLayerPanel) layersPanel.getComponent(i)).isSelected()) {
							allSelected = false;
						}
					}
					if (allSelected) {
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							((GroupLayerPanel) layersPanel.getComponent(i)).setSelected(false);
						}
					}
					else {
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							((GroupLayerPanel) layersPanel.getComponent(i)).setSelected(true);
						}
					}
				}
			});
			editEffectsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					viewContext.setComponent(effectPanel);
				}
			});
			editBackgroundButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					viewContext.setComponent(colorPanel);
				}
			});
			colorChooseButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					final Color color = ColorChooser.showColorChooser(colorField, TwisterSwingResources.getInstance().getString("label.background"), colorField.getColor());
					if (color != null) {
						colorField.setColor(color);
					}
				}
			});
			colorField.addColorChangeListener(new ColorChangeListener() {
				public void colorChanged(final ColorChangeEvent e) {
					config.getContext().updateTimestamp();
					// config.getBackgroundElement().removeChangeListener(backgroundListener);
					config.setBackground(new Color32bit(colorField.getColor().getRGB()));
					// config.getBackgroundElement().addChangeListener(backgroundListener);
					context.refresh();
				}
			});
			colorField.addMouseListener(new MouseAdapter() {
				@Override
				public void mouseClicked(final MouseEvent e) {
					final Color color = ColorChooser.showColorChooser(colorField, TwisterSwingResources.getInstance().getString("label.background"), colorField.getColor());
					if (color != null) {
						colorField.setColor(color);
					}
				}
			});
			editLayersButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					viewContext.setComponent(layersPanel2);
				}
			});
			editFiltersButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					viewContext.setComponent(filtersPanel2);
				}
			});
		}

		private String createFiltersPanelName() {
			return TwisterSwingResources.getInstance().getString("name.frameFilters");
		}

		private String createLayersPanelName() {
			return TwisterSwingResources.getInstance().getString("name.groupLayers");
		}

		private String createEffectPanelName(final EffectConfigElement effectElement) {
			return TwisterSwingResources.getInstance().getString("name.effects");
		}
	}

	private class GroupLayerPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private final JCheckBox selectedCheckBox = createCheckBox();
		private final TwisterRenderer renderer;
		private final PreviewPanel preview;
		private final Surface surface;
		private final JPanel layersPanel;

		/**
		 * @param frameElement
		 * @param groupLayerElement
		 */
		public GroupLayerPanel(final FrameConfigElement frameElement, final GroupLayerConfigElement groupLayerElement, final RenderTask task) {
			final JTextField label = createTextField(groupLayerElement.getLabel(), 120, GUIFactory.DEFAULT_HEIGHT);
			final JCheckBox lockedCheckBox = createIconCheckBox("locked", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox visibleCheckBox = createIconCheckBox("visible", "visible", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox showLayersButton = createIconCheckBox("showLayers", "group", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton editFiltersButton = createTextButton(80, GUIFactory.DEFAULT_HEIGHT);
			final JButton selectFilterButton = createIconButton("selectFilters", "select", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton appendFilterButton = createIconButton("appendFilter", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton removeFilterButton = createIconButton("removeFilters", "remove", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertFilterAfterButton = createIconButton("insertFilterAfter", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertFilterBeforeButton = createIconButton("insertFilterBefore", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton selectLayerButton = createIconButton("selectLayers", "select", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton appendLayerButton = createIconButton("appendLayer", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton removeLayerButton = createIconButton("removeLayers", "remove", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertLayerAfterButton = createIconButton("insertLayerAfter", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertLayerBeforeButton = createIconButton("insertLayerBefore", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JSpinner opacitySpinner = createSpinner(0, 100, 1);
			final JLabel layersLabel = createTextLabel("noImageLayer", SwingConstants.LEFT, 120, GUIFactory.DEFAULT_HEIGHT);
			final JLabel filtersLabel = createTextLabel("noLayerFilter", SwingConstants.LEFT, 120, GUIFactory.DEFAULT_HEIGHT);
			final JPanel filtersPanel = createPanel(new StackLayout(), false);
			layersPanel = createPanel(new StackLayout(), false);
			layersLabel.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			filtersLabel.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			final Box layersPanel3 = createHorizontalBox(false);
			layersPanel3.add(selectLayerButton);
			layersPanel3.add(createSpace());
			layersPanel3.add(appendLayerButton);
			layersPanel3.add(createSpace());
			layersPanel3.add(removeLayerButton);
			// layersPanel3.add(insertLayerAfterButton);
			// layersPanel3.add(insertLayerBeforeButton);
			final Box layersPanel4 = createHorizontalBox(false);
			layersPanel4.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			layersPanel4.add(layersLabel);
			layersPanel4.add(Box.createHorizontalGlue());
			final JPanel layersPanel2 = createPanel(new BorderLayout(), false);
			layersPanel2.setName(createLayersPanelName(groupLayerElement));
			layersPanel2.add(layersPanel4, BorderLayout.NORTH);
			layersPanel2.add(layersPanel, BorderLayout.CENTER);
			layersPanel2.setBorder(BorderFactory.createMatteBorder(1, 0, 0, 0, Color.DARK_GRAY));
			final Box filtersPanel3 = createHorizontalBox(false);
			filtersPanel3.add(Box.createHorizontalGlue());
			filtersPanel3.add(selectFilterButton);
			filtersPanel3.add(createSpace());
			filtersPanel3.add(appendFilterButton);
			filtersPanel3.add(createSpace());
			filtersPanel3.add(removeFilterButton);
			// filtersPanel3.add(insertFilterAfterButton);
			// filtersPanel3.add(insertFilterBeforeButton);
			filtersPanel3.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY), BorderFactory.createEmptyBorder(4, 4, 4, 4)));
			final Box filtersPanel4 = createHorizontalBox(false);
			filtersPanel4.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			filtersPanel4.add(filtersLabel);
			filtersPanel4.add(Box.createHorizontalGlue());
			final JPanel filtersPanel2 = createPanel(new BorderLayout(), true);
			filtersPanel2.setName(createFiltersPanelName(groupLayerElement));
			filtersPanel2.add(filtersPanel3, BorderLayout.NORTH);
			filtersPanel2.add(filtersPanel, BorderLayout.CENTER);
			filtersPanel2.add(filtersPanel4, BorderLayout.SOUTH);
			filtersPanel2.setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY));
			layersPanel2.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(4, 24, 4, 0), BorderFactory.createMatteBorder(0, 0, 0, 0, Color.DARK_GRAY)));
			surface = new Surface(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT);
			preview = new PreviewPanel(surface);
			preview.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2), BorderFactory.createLineBorder(Color.DARK_GRAY)));
			preview.setPreferredSize(new Dimension(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT));
			preview.setMinimumSize(new Dimension(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT));
			preview.setMaximumSize(new Dimension(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT));
			preview.setSize(new Dimension(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT));
			final TwisterConfig previewConfig = new TwisterConfig();
			buildPreviewConfig(groupLayerElement, previewConfig);
			final TwisterRuntime runtime = new TwisterRuntime(previewConfig);
			final Map<Object, Object> hints = new HashMap<Object, Object>();
			hints.put(TwisterRenderingHints.KEY_MEMORY, TwisterRenderingHints.MEMORY_LOW);
			renderer = new DefaultTwisterRenderer(runtime);
			renderer.setRenderingHints(hints);
			renderer.setTile(new ImageTile(new IntegerVector2D(GUIFactory.DEFAULT_HEIGHT - 4, GUIFactory.DEFAULT_HEIGHT - 4), new IntegerVector2D(GUIFactory.DEFAULT_HEIGHT - 4, GUIFactory.DEFAULT_HEIGHT - 4), new IntegerVector2D(0, 0), new IntegerVector2D(0, 0)));
			startPreviewRenderer();
			joinPreviewRenderer();
			prepareLayerPreview();
			drawLayerPreview();
			final Box layerPanel = createHorizontalBox(false);
			layerPanel.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 0));
			layerPanel.add(createSpace());
			layerPanel.add(showLayersButton);
			layerPanel.add(selectedCheckBox);
			layerPanel.add(preview);
			layerPanel.add(createSpace());
			layerPanel.add(label);
			layerPanel.add(createSpace());
			layerPanel.add(opacitySpinner);
			layerPanel.add(createSpace());
			layerPanel.add(lockedCheckBox);
			layerPanel.add(createSpace());
			layerPanel.add(visibleCheckBox);
			layerPanel.add(createSpace());
			layerPanel.add(editFiltersButton);
			layerPanel.add(createSpace());
			layerPanel.add(Box.createHorizontalGlue());
			layerPanel.add(layersPanel3);
			layersPanel3.setBorder(BorderFactory.createEmptyBorder(4, 4, 4, 4));
			final JPanel subpanelContainer = createPanel(new StackLayout(), false);
			subpanelContainer.add(layersPanel2);
			setLayout(new BorderLayout());
			add(layerPanel, BorderLayout.NORTH);
			add(subpanelContainer, BorderLayout.CENTER);
			showLayersButton.setSelected(true);
			showLayersButton.setPreferredSize(new Dimension(24, 24));
			editFiltersButton.setText(createEditFiltersText(groupLayerElement));
			opacitySpinner.setValue(groupLayerElement.getOpacity().intValue());
			opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
			lockedCheckBox.setSelected(groupLayerElement.isLocked());
			visibleCheckBox.setSelected(groupLayerElement.isVisible());
			layersPanel4.setVisible(groupLayerElement.getLayerConfigElementCount() == 0);
			filtersPanel4.setVisible(groupLayerElement.getFilterConfigElementCount() == 0);
			for (int i = 0; i < groupLayerElement.getLayerConfigElementCount(); i++) {
				final ImageLayerConfigElement imageLayerElement = groupLayerElement.getLayerConfigElement(i);
				layersPanel.add(new ImageLayerPanel(groupLayerElement, imageLayerElement));
			}
			for (int i = 0; i < groupLayerElement.getFilterConfigElementCount(); i++) {
				final LayerFilterConfigElement imageFilterElement = groupLayerElement.getFilterConfigElement(i);
				filtersPanel.add(new LayerFilterPanel(groupLayerElement, imageFilterElement));
			}
			for (int i = 0; i < layersPanel.getComponentCount(); i++) {
				layersPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
			}
			for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
				filtersPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
			}
			final ValueChangeListener lockedListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					lockedCheckBox.setSelected(groupLayerElement.isLocked());
				}
			};
			final ValueChangeListener visibleListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					visibleCheckBox.setSelected(groupLayerElement.isVisible());
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							task.executeTask();
						}
					});
				}
			};
			final ValueChangeListener opacityListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					opacitySpinner.setValue(groupLayerElement.getOpacity().intValue());
					opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
					try {
						task.executeTask();
					}
					catch (Exception x) {
					}
				}
			};
			final ValueChangeListener labelListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					label.setText(groupLayerElement.getLabel());
					layersPanel2.setName(createLayersPanelName(groupLayerElement));
					filtersPanel2.setName(createFiltersPanelName(groupLayerElement));
				}
			};
			final ValueChangeListener filtersListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ListConfigElementEvents.ELEMENT_ADDED: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new LayerFilterPanel(groupLayerElement, (LayerFilterConfigElement) e.getParams()[0]));
							filtersPanel4.setVisible(groupLayerElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_AFTER: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new LayerFilterPanel(groupLayerElement, (LayerFilterConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue() + 1);
							filtersPanel4.setVisible(groupLayerElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_BEFORE: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new LayerFilterPanel(groupLayerElement, (LayerFilterConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue());
							filtersPanel4.setVisible(groupLayerElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_REMOVED: {
							filtersPanel.remove(((Integer) e.getParams()[1]).intValue());
							filtersPanel4.setVisible(groupLayerElement.getFilterConfigElementCount() == 0);
							viewContext.resize();
							break;
						}
						default: {
							break;
						}
					}
					for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
						filtersPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
					}
					editFiltersButton.setText(createEditFiltersText(groupLayerElement));
					try {
						task.executeTask();
					}
					catch (Exception x) {
					}
				}
			};
			final ValueChangeListener layersListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ListConfigElementEvents.ELEMENT_ADDED: {
							if (groupLayerElement.getLayerConfigElementCount() > 1) {
								viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							}
							layersPanel.add(new ImageLayerPanel(groupLayerElement, (ImageLayerConfigElement) e.getParams()[0]));
							layersPanel4.setVisible(groupLayerElement.getLayerConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_AFTER: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							layersPanel.add(new ImageLayerPanel(groupLayerElement, (ImageLayerConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue() + 1);
							layersPanel4.setVisible(groupLayerElement.getLayerConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_BEFORE: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							layersPanel.add(new ImageLayerPanel(groupLayerElement, (ImageLayerConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue());
							layersPanel4.setVisible(groupLayerElement.getLayerConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_REMOVED: {
							layersPanel.remove(((Integer) e.getParams()[1]).intValue());
							layersPanel4.setVisible(groupLayerElement.getLayerConfigElementCount() == 0);
							viewContext.resize();
							break;
						}
						default: {
							break;
						}
					}
					for (int i = 0; i < layersPanel.getComponentCount(); i++) {
						layersPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
					}
					try {
						task.executeTask();
					}
					catch (Exception x) {
					}
				}
			};
			groupLayerElement.getLabelElement().addChangeListener(labelListener);
			groupLayerElement.getLockedElement().addChangeListener(lockedListener);
			groupLayerElement.getVisibleElement().addChangeListener(visibleListener);
			groupLayerElement.getOpacityElement().addChangeListener(opacityListener);
			groupLayerElement.getLayerListElement().addChangeListener(layersListener);
			groupLayerElement.getFilterListElement().addChangeListener(filtersListener);
			appendLayerButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					groupLayerElement.getContext().updateTimestamp();
					for (int i = groupLayerElement.getLayerConfigElementCount() - 1; i >= 0; i--) {
						final ImageLayerConfigElement imageLayerElement = groupLayerElement.getLayerConfigElement(i);
						if (imageLayerElement.getUserData() != null) {
							final ImageLayerConfigElement element = new ImageLayerConfigElement();
							element.setImageConfigElement(new ImageConfigElement());
							groupLayerElement.insertLayerConfigElementBefore(i, element);
							return;
						}
					}
					final ImageLayerConfigElement element = new ImageLayerConfigElement();
					element.setImageConfigElement(new ImageConfigElement());
					groupLayerElement.appendLayerConfigElement(element);
					context.refresh();
				}
			});
			removeLayerButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					groupLayerElement.getContext().updateTimestamp();
					for (int i = groupLayerElement.getLayerConfigElementCount() - 1; i >= 0; i--) {
						final ImageLayerConfigElement imageLayerElement = groupLayerElement.getLayerConfigElement(i);
						if (imageLayerElement.getUserData() != null) {
							groupLayerElement.removeLayerConfigElement(i);
						}
					}
					context.refresh();
				}
			});
			// insertLayerAfterButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = 0; i < groupLayerElement.getLayerConfigElementCount(); i++) {
			// ImageLayerConfigElement imageLayerElement = groupLayerElement.getLayerConfigElement(i);
			// if (imageLayerElement.getUserData() != null) {
			// groupLayerElement.insertLayerConfigElementAfter(i, new ImageLayerConfigElement());
			// }
			// }
			// }
			// });
			// insertLayerBeforeButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = groupLayerElement.getLayerConfigElementCount() - 1; i >= 0; i--) {
			// ImageLayerConfigElement imageLayerElement = groupLayerElement.getLayerConfigElement(i);
			// if (imageLayerElement.getUserData() != null) {
			// groupLayerElement.insertLayerConfigElementBefore(i, new ImageLayerConfigElement());
			// }
			// }
			// }
			// });
			appendFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					groupLayerElement.getContext().updateTimestamp();
					for (int i = groupLayerElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
						final LayerFilterConfigElement layerFilterElement = groupLayerElement.getFilterConfigElement(i);
						if (layerFilterElement.getUserData() != null) {
							groupLayerElement.insertFilterConfigElementBefore(i, new LayerFilterConfigElement());
							return;
						}
					}
					groupLayerElement.appendFilterConfigElement(new LayerFilterConfigElement());
					context.refresh();
				}
			});
			removeFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					groupLayerElement.getContext().updateTimestamp();
					for (int i = groupLayerElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
						final LayerFilterConfigElement layerFilterElement = groupLayerElement.getFilterConfigElement(i);
						if (layerFilterElement.getUserData() != null) {
							groupLayerElement.removeFilterConfigElement(i);
						}
					}
					context.refresh();
				}
			});
			// insertFilterAfterButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = 0; i < groupLayerElement.getFilterConfigElementCount(); i++) {
			// LayerFilterConfigElement layerFilterElement = groupLayerElement.getFilterConfigElement(i);
			// if (layerFilterElement.getUserData() != null) {
			// groupLayerElement.insertFilterConfigElementAfter(i, new LayerFilterConfigElement());
			// }
			// }
			// }
			// });
			// insertFilterBeforeButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = groupLayerElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
			// LayerFilterConfigElement layerFilterElement = groupLayerElement.getFilterConfigElement(i);
			// if (layerFilterElement.getUserData() != null) {
			// groupLayerElement.insertFilterConfigElementBefore(i, new LayerFilterConfigElement());
			// }
			// }
			// }
			// });
			selectLayerButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					boolean allSelected = true;
					for (int i = 0; i < layersPanel.getComponentCount(); i++) {
						if (!((ImageLayerPanel) layersPanel.getComponent(i)).isSelected()) {
							allSelected = false;
						}
					}
					if (allSelected) {
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							((ImageLayerPanel) layersPanel.getComponent(i)).setSelected(false);
						}
					}
					else {
						for (int i = 0; i < layersPanel.getComponentCount(); i++) {
							((ImageLayerPanel) layersPanel.getComponent(i)).setSelected(true);
						}
					}
				}
			});
			selectFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					boolean allSelected = true;
					for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
						if (!((LayerFilterPanel) filtersPanel.getComponent(i)).isSelected()) {
							allSelected = false;
						}
					}
					if (allSelected) {
						for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
							((LayerFilterPanel) filtersPanel.getComponent(i)).setSelected(false);
						}
					}
					else {
						for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
							((LayerFilterPanel) filtersPanel.getComponent(i)).setSelected(true);
						}
					}
				}
			});
			selectedCheckBox.addItemListener(new ItemListener() {
				public void itemStateChanged(final ItemEvent e) {
					if (selectedCheckBox.isSelected()) {
						groupLayerElement.setUserData(Boolean.TRUE);
					}
					else {
						groupLayerElement.setUserData(null);
					}
				}
			});
			lockedCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					groupLayerElement.getContext().updateTimestamp();
					groupLayerElement.setLocked(lockedCheckBox.isSelected());
				}
			});
			visibleCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					groupLayerElement.getContext().updateTimestamp();
					groupLayerElement.setVisible(visibleCheckBox.isSelected());
					context.refresh();
				}
			});
			opacitySpinner.addChangeListener(new ChangeListener() {
				public void stateChanged(final ChangeEvent e) {
					groupLayerElement.getContext().updateTimestamp();
					groupLayerElement.setOpacity(((Number) opacitySpinner.getValue()).intValue());
					opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
					context.refresh();
				}
			});
			label.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					// groupLayerElement.getLabelElement().removeChangeListener(labelListener);
					groupLayerElement.getContext().updateTimestamp();
					groupLayerElement.setLabel(label.getText());
					layersPanel2.setName(createLayersPanelName(groupLayerElement));
					filtersPanel2.setName(createFiltersPanelName(groupLayerElement));
					// groupLayerElement.getLabelElement().addChangeListener(labelListener);
				}
			});
			showLayersButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					layersPanel2.setVisible(showLayersButton.isSelected());
					viewContext.resize();
				}
			});
			editFiltersButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					viewContext.setComponent(filtersPanel2);
				}
			});
		}

		private void buildPreviewConfig(final GroupLayerConfigElement groupLayerElement, final TwisterConfig previewConfig) {
			final FrameConfigElement frameElement = new FrameConfigElement();
			previewConfig.setFrameConfigElement(frameElement);
			final EffectConfigElement effectElement = new EffectConfigElement();
			previewConfig.setEffectConfigElement(effectElement);
			frameElement.appendLayerConfigElement(groupLayerElement);
		}

		/**
		 * 
		 */
		public void abortPreviewRenderer() {
			renderer.abortRenderer();
			for (int i = 0; i < layersPanel.getComponentCount(); i++) {
				final ImageLayerPanel layerPanel = (ImageLayerPanel) layersPanel.getComponent(i);
				layerPanel.abortPreviewRenderer();
			}
		}

		/**
		 * 
		 */
		public void joinPreviewRenderer() {
			renderer.joinRenderer();
			for (int i = 0; i < layersPanel.getComponentCount(); i++) {
				final ImageLayerPanel layerPanel = (ImageLayerPanel) layersPanel.getComponent(i);
				layerPanel.joinPreviewRenderer();
			}
		}

		/**
		 * 
		 */
		public void startPreviewRenderer() {
			renderer.startRenderer();
			for (int i = 0; i < layersPanel.getComponentCount(); i++) {
				final ImageLayerPanel layerPanel = (ImageLayerPanel) layersPanel.getComponent(i);
				layerPanel.startPreviewRenderer();
			}
		}

		/**
		 * 
		 */
		public void prepareLayerPreview() {
			renderer.prepareImage(false);
			for (int i = 0; i < layersPanel.getComponentCount(); i++) {
				final ImageLayerPanel layerPanel = (ImageLayerPanel) layersPanel.getComponent(i);
				layerPanel.prepareLayerPreview();
			}
		}

		/**
		 * 
		 */
		public void drawLayerPreview() {
			renderer.drawImage(surface.getGraphics2D(), 2, 2, GUIFactory.DEFAULT_HEIGHT - 4, GUIFactory.DEFAULT_HEIGHT - 4);
			preview.repaint();
			for (int i = 0; i < layersPanel.getComponentCount(); i++) {
				final ImageLayerPanel layerPanel = (ImageLayerPanel) layersPanel.getComponent(i);
				layerPanel.drawLayerPreview();
			}
		}

		private String createEditFiltersText(final GroupLayerConfigElement groupLayerElement) {
			if (groupLayerElement.getFilterConfigElementCount() == 1) {
				return groupLayerElement.getFilterConfigElementCount() + " " + TwisterSwingResources.getInstance().getString("label.filter");
			}
			else {
				return groupLayerElement.getFilterConfigElementCount() + " " + TwisterSwingResources.getInstance().getString("label.filters");
			}
		}

		private String createFiltersPanelName(final GroupLayerConfigElement groupLayerElement) {
			return groupLayerElement.getLabel();
		}

		private String createLayersPanelName(final GroupLayerConfigElement groupLayerElement) {
			return groupLayerElement.getLabel();
		}

		private String createOpacityTooltip(final JSpinner opacitySpinner) {
			return TwisterSwingResources.getInstance().getString("label.opacity") + " " + opacitySpinner.getValue() + "%";
		}

		/**
		 * @param selected
		 */
		public void setSelected(final boolean selected) {
			selectedCheckBox.setSelected(selected);
		}

		/**
		 * @return
		 */
		public boolean isSelected() {
			return selectedCheckBox.isSelected();
		}
	}

	private class ImageLayerPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private final JCheckBox selectedCheckBox = createCheckBox();
		private final TwisterRenderer renderer;
		private final PreviewPanel preview;
		private final Surface surface;

		/**
		 * @param groupLayerElement
		 * @param imageLayerElement
		 */
		public ImageLayerPanel(final GroupLayerConfigElement groupLayerElement, final ImageLayerConfigElement imageLayerElement) {
			final JTextField label = createTextField(imageLayerElement.getLabel(), 120, GUIFactory.DEFAULT_HEIGHT);
			final JCheckBox lockedCheckBox = createIconCheckBox("locked", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox visibleCheckBox = createIconCheckBox("visible", "visible", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton showFiltersButton = createTextButton(80, GUIFactory.DEFAULT_HEIGHT);
			final JButton selectFilterButton = createIconButton("selectFilters", "select", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton appendFilterButton = createIconButton("appendFilter", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton removeFilterButton = createIconButton("removeFilters", "remove", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertFilterAfterButton = createIconButton("insertFilterAfter", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			// final JButton insertFilterBeforeButton = createIconButton("insertFilterBefore", "add", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JSpinner opacitySpinner = createSpinner(0, 100, 1);
			final JLabel filtersLabel = createTextLabel("noLayerFilter", SwingConstants.LEFT, 120, GUIFactory.DEFAULT_HEIGHT);
			final JPanel filtersPanel = createPanel(new StackLayout(), false);
			final ImagePanel imagePanel = new ImagePanel(imageLayerElement, imageLayerElement.getImageConfigElement());
			imagePanel.setOpaque(false);
			filtersLabel.setPreferredSize(new Dimension(200, GUIFactory.DEFAULT_HEIGHT));
			final Box filtersPanel3 = createHorizontalBox(false);
			filtersPanel3.add(Box.createHorizontalGlue());
			filtersPanel3.add(selectFilterButton);
			filtersPanel3.add(createSpace());
			filtersPanel3.add(appendFilterButton);
			filtersPanel3.add(createSpace());
			filtersPanel3.add(removeFilterButton);
			// filtersPanel3.add(insertFilterAfterButton);
			// filtersPanel3.add(insertFilterBeforeButton);
			filtersPanel3.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY), BorderFactory.createEmptyBorder(4, 4, 4, 4)));
			final Box filtersPanel4 = createHorizontalBox(false);
			filtersPanel4.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 8));
			filtersPanel4.add(filtersLabel);
			filtersPanel4.add(Box.createHorizontalGlue());
			final JPanel filtersPanel2 = createPanel(new BorderLayout(), true);
			filtersPanel2.setName(createFiltersPanelName(imageLayerElement));
			filtersPanel2.add(filtersPanel3, BorderLayout.NORTH);
			filtersPanel2.add(filtersPanel, BorderLayout.CENTER);
			filtersPanel2.add(filtersPanel4, BorderLayout.SOUTH);
			filtersPanel2.setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0, Color.DARK_GRAY));
			surface = new Surface(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT);
			preview = new PreviewPanel(surface);
			preview.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2), BorderFactory.createLineBorder(Color.DARK_GRAY)));
			preview.setPreferredSize(new Dimension(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT));
			preview.setMinimumSize(new Dimension(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT));
			preview.setMaximumSize(new Dimension(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT));
			preview.setSize(new Dimension(GUIFactory.DEFAULT_HEIGHT, GUIFactory.DEFAULT_HEIGHT));
			final TwisterConfig previewConfig = new TwisterConfig();
			buildPreviewConfig(imageLayerElement, previewConfig);
			final TwisterRuntime runtime = new TwisterRuntime(previewConfig);
			final Map<Object, Object> hints = new HashMap<Object, Object>();
			hints.put(TwisterRenderingHints.KEY_MEMORY, TwisterRenderingHints.MEMORY_LOW);
			renderer = new DefaultTwisterRenderer(runtime);
			renderer.setRenderingHints(hints);
			renderer.setTile(new ImageTile(new IntegerVector2D(GUIFactory.DEFAULT_HEIGHT - 4, GUIFactory.DEFAULT_HEIGHT - 4), new IntegerVector2D(GUIFactory.DEFAULT_HEIGHT - 4, GUIFactory.DEFAULT_HEIGHT - 4), new IntegerVector2D(0, 0), new IntegerVector2D(0, 0)));
			startPreviewRenderer();
			joinPreviewRenderer();
			prepareLayerPreview();
			drawLayerPreview();
			final Box layerPanel = createHorizontalBox(false);
			layerPanel.add(createSpace());
			layerPanel.add(selectedCheckBox);
			layerPanel.add(preview);
			layerPanel.add(createSpace());
			layerPanel.add(label);
			layerPanel.add(createSpace());
			layerPanel.add(opacitySpinner);
			layerPanel.add(createSpace());
			layerPanel.add(lockedCheckBox);
			layerPanel.add(createSpace());
			layerPanel.add(visibleCheckBox);
			layerPanel.add(createSpace());
			layerPanel.add(showFiltersButton);
			layerPanel.add(Box.createHorizontalGlue());
			layerPanel.add(imagePanel);
			setLayout(new BorderLayout());
			add(layerPanel, BorderLayout.CENTER);
			setBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0));
			layerPanel.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 4));
			showFiltersButton.setText(createEditFiltersText(imageLayerElement));
			opacitySpinner.setValue(imageLayerElement.getOpacity().intValue());
			opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
			lockedCheckBox.setSelected(imageLayerElement.isLocked());
			visibleCheckBox.setSelected(imageLayerElement.isVisible());
			filtersPanel4.setVisible(imageLayerElement.getFilterConfigElementCount() == 0);
			for (int i = 0; i < imageLayerElement.getFilterConfigElementCount(); i++) {
				final LayerFilterConfigElement imageFilterElement = imageLayerElement.getFilterConfigElement(i);
				filtersPanel.add(new LayerFilterPanel(imageLayerElement, imageFilterElement));
			}
			for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
				filtersPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
			}
			final ValueChangeListener lockedListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					lockedCheckBox.setSelected(imageLayerElement.isLocked());
				}
			};
			final ValueChangeListener visibleListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					visibleCheckBox.setSelected(imageLayerElement.isVisible());
				}
			};
			final ValueChangeListener opacityListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					opacitySpinner.setValue(imageLayerElement.getOpacity().intValue());
					opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
				}
			};
			final ValueChangeListener filtersListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ListConfigElementEvents.ELEMENT_ADDED: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new LayerFilterPanel(imageLayerElement, (LayerFilterConfigElement) e.getParams()[0]));
							filtersPanel4.setVisible(imageLayerElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_AFTER: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new LayerFilterPanel(imageLayerElement, (LayerFilterConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue() + 1);
							filtersPanel4.setVisible(imageLayerElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_INSERTED_BEFORE: {
							viewContext.resize(GUIFactory.DEFAULT_HEIGHT);
							filtersPanel.add(new LayerFilterPanel(imageLayerElement, (LayerFilterConfigElement) e.getParams()[0]), ((Integer) e.getParams()[1]).intValue());
							filtersPanel4.setVisible(imageLayerElement.getFilterConfigElementCount() == 0);
							break;
						}
						case ListConfigElementEvents.ELEMENT_REMOVED: {
							filtersPanel.remove(((Integer) e.getParams()[1]).intValue());
							filtersPanel4.setVisible(imageLayerElement.getFilterConfigElementCount() == 0);
							viewContext.resize();
							break;
						}
						default: {
							break;
						}
					}
					for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
						filtersPanel.getComponent(i).setBackground((i % 2 == 0) ? evenColor : oddColor);
					}
					showFiltersButton.setText(createEditFiltersText(imageLayerElement));
				}
			};
			final ValueChangeListener labelListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					label.setText(imageLayerElement.getLabel());
					filtersPanel2.setName(createFiltersPanelName(imageLayerElement));
				}
			};
			imageLayerElement.getLabelElement().addChangeListener(labelListener);
			imageLayerElement.getLockedElement().addChangeListener(lockedListener);
			imageLayerElement.getVisibleElement().addChangeListener(visibleListener);
			imageLayerElement.getOpacityElement().addChangeListener(opacityListener);
			imageLayerElement.getFilterListElement().addChangeListener(filtersListener);
			appendFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					imageLayerElement.getContext().updateTimestamp();
					for (int i = imageLayerElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
						final LayerFilterConfigElement layerFilterElement = imageLayerElement.getFilterConfigElement(i);
						if (layerFilterElement.getUserData() != null) {
							imageLayerElement.insertFilterConfigElementBefore(i, new LayerFilterConfigElement());
							return;
						}
					}
					imageLayerElement.appendFilterConfigElement(new LayerFilterConfigElement());
					context.refresh();
				}
			});
			removeFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					imageLayerElement.getContext().updateTimestamp();
					for (int i = imageLayerElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
						final LayerFilterConfigElement layerFilterElement = imageLayerElement.getFilterConfigElement(i);
						if (layerFilterElement.getUserData() != null) {
							imageLayerElement.removeFilterConfigElement(i);
						}
					}
					context.refresh();
				}
			});
			// insertFilterAfterButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = 0; i < imageLayerElement.getFilterConfigElementCount(); i++) {
			// LayerFilterConfigElement layerFilterElement = imageLayerElement.getFilterConfigElement(i);
			// if (layerFilterElement.getUserData() != null) {
			// imageLayerElement.insertFilterConfigElementAfter(i, new LayerFilterConfigElement());
			// }
			// }
			// }
			// });
			// insertFilterBeforeButton.addActionListener(new ActionListener() {
			// public void actionPerformed(ActionEvent e) {
			// for (int i = imageLayerElement.getFilterConfigElementCount() - 1; i >= 0; i--) {
			// LayerFilterConfigElement layerFilterElement = imageLayerElement.getFilterConfigElement(i);
			// if (layerFilterElement.getUserData() != null) {
			// imageLayerElement.insertFilterConfigElementBefore(i, new LayerFilterConfigElement());
			// }
			// }
			// }
			// });
			selectFilterButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					boolean allSelected = true;
					for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
						if (!((LayerFilterPanel) filtersPanel.getComponent(i)).isSelected()) {
							allSelected = false;
						}
					}
					if (allSelected) {
						for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
							((LayerFilterPanel) filtersPanel.getComponent(i)).setSelected(false);
						}
					}
					else {
						for (int i = 0; i < filtersPanel.getComponentCount(); i++) {
							((LayerFilterPanel) filtersPanel.getComponent(i)).setSelected(true);
						}
					}
				}
			});
			selectedCheckBox.addItemListener(new ItemListener() {
				public void itemStateChanged(final ItemEvent e) {
					if (selectedCheckBox.isSelected()) {
						imageLayerElement.setUserData(Boolean.TRUE);
					}
					else {
						imageLayerElement.setUserData(null);
					}
				}
			});
			lockedCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					imageLayerElement.getContext().updateTimestamp();
					imageLayerElement.setLocked(lockedCheckBox.isSelected());
				}
			});
			visibleCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					imageLayerElement.getContext().updateTimestamp();
					imageLayerElement.setVisible(visibleCheckBox.isSelected());
					context.refresh();
				}
			});
			opacitySpinner.addChangeListener(new ChangeListener() {
				public void stateChanged(final ChangeEvent e) {
					imageLayerElement.getContext().updateTimestamp();
					imageLayerElement.setOpacity(((Number) opacitySpinner.getValue()).intValue());
					opacitySpinner.setToolTipText(createOpacityTooltip(opacitySpinner));
					context.refresh();
				}
			});
			label.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					// imageLayerElement.getLabelElement().removeChangeListener(labelListener);
					imageLayerElement.getContext().updateTimestamp();
					imageLayerElement.setLabel(label.getText());
					filtersPanel2.setName(createFiltersPanelName(imageLayerElement));
					// imageLayerElement.getLabelElement().addChangeListener(labelListener);
				}
			});
			showFiltersButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					viewContext.setComponent(filtersPanel2);
				}
			});
		}

		private void buildPreviewConfig(final ImageLayerConfigElement imageLayerElement, final TwisterConfig previewConfig) {
			final FrameConfigElement frameElement = new FrameConfigElement();
			previewConfig.setFrameConfigElement(frameElement);
			final EffectConfigElement effectElement = new EffectConfigElement();
			previewConfig.setEffectConfigElement(effectElement);
			final GroupLayerConfigElement groupLayerElement = new GroupLayerConfigElement();
			frameElement.appendLayerConfigElement(groupLayerElement);
			groupLayerElement.appendLayerConfigElement(imageLayerElement);
		}

		/**
		 * 
		 */
		public void abortPreviewRenderer() {
			renderer.abortRenderer();
		}

		/**
		 * 
		 */
		public void joinPreviewRenderer() {
			renderer.joinRenderer();
		}

		/**
		 * 
		 */
		public void startPreviewRenderer() {
			renderer.startRenderer();
		}

		/**
		 * 
		 */
		public void prepareLayerPreview() {
			renderer.prepareImage(false);
		}

		/**
		 * 
		 */
		public void drawLayerPreview() {
			renderer.drawImage(surface.getGraphics2D(), 2, 2, GUIFactory.DEFAULT_HEIGHT - 4, GUIFactory.DEFAULT_HEIGHT - 4);
			preview.repaint();
		}

		private String createEditFiltersText(final ImageLayerConfigElement imageLayerElement) {
			if (imageLayerElement.getFilterConfigElementCount() == 1) {
				return imageLayerElement.getFilterConfigElementCount() + " " + TwisterSwingResources.getInstance().getString("label.filter");
			}
			else {
				return imageLayerElement.getFilterConfigElementCount() + " " + TwisterSwingResources.getInstance().getString("label.filters");
			}
		}

		private String createFiltersPanelName(final ImageLayerConfigElement imageLayerElement) {
			return imageLayerElement.getLabel();
		}

		private String createOpacityTooltip(final JSpinner opacitySpinner) {
			return TwisterSwingResources.getInstance().getString("label.opacity") + " " + opacitySpinner.getValue() + "%";
		}

		/**
		 * @param selected
		 */
		public void setSelected(final boolean selected) {
			selectedCheckBox.setSelected(selected);
		}

		/**
		 * @return
		 */
		public boolean isSelected() {
			return selectedCheckBox.isSelected();
		}
	}

	private class LayerFilterPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private final JCheckBox selectedCheckBox = createCheckBox();
		private JComponent configView;

		/**
		 * @param layerElement
		 * @param imageFilterElement
		 */
		public LayerFilterPanel(final LayerConfigElement layerElement, final LayerFilterConfigElement imageFilterElement) {
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(TwisterRegistry.getInstance().getLayerFilterRegistry(), true);
			final JTextField label = createTextField(imageFilterElement.getLabel(), 120, GUIFactory.DEFAULT_HEIGHT);
			final JCheckBox lockedCheckBox = createIconCheckBox("locked", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox enabledCheckBox = createIconCheckBox("enabled", "visible", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton editOptionsButton = createTextButton("editConfig", 100, GUIFactory.DEFAULT_HEIGHT);
			final JComboBox extensionComboBox = createExtensionComboBox(model, 140, GUIFactory.DEFAULT_HEIGHT);
			final Box filterPanel = createHorizontalBox(false);
			filterPanel.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 8));
			filterPanel.add(createSpace());
			filterPanel.add(selectedCheckBox);
			filterPanel.add(label);
			filterPanel.add(createSpace());
			filterPanel.add(lockedCheckBox);
			filterPanel.add(createSpace());
			filterPanel.add(enabledCheckBox);
			filterPanel.add(createSpace());
			filterPanel.add(Box.createHorizontalGlue());
			filterPanel.add(extensionComboBox);
			filterPanel.add(createSpace());
			filterPanel.add(editOptionsButton);
			setLayout(new BorderLayout());
			setBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0));
			add(filterPanel, BorderLayout.CENTER);
			lockedCheckBox.setSelected(imageFilterElement.isLocked());
			enabledCheckBox.setSelected(imageFilterElement.isEnabled());
			editOptionsButton.setEnabled((imageFilterElement != null) && (imageFilterElement.getReference() != null));
			if (imageFilterElement.getReference() != null) {
				model.setSelectedItemByExtensionId(imageFilterElement.getReference().getExtensionId());
			}
			final ValueChangeListener lockedListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					lockedCheckBox.setSelected(imageFilterElement.isLocked());
				}
			};
			final ValueChangeListener enabledListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					enabledCheckBox.setSelected(imageFilterElement.isEnabled());
				}
			};
			final ValueChangeListener labelListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					label.setText(imageFilterElement.getLabel());
					if (configView != null) {
						configView.setName(createFiltersPanelName(imageFilterElement));
					}
				}
			};
			final ActionListener comboListener = new ActionListener() {
				@SuppressWarnings("unchecked")
				public void actionPerformed(ActionEvent e) {
					ConfigurableExtension<LayerFilterExtensionRuntime, LayerFilterExtensionConfig> extension = (ConfigurableExtension<LayerFilterExtensionRuntime, LayerFilterExtensionConfig>) extensionComboBox.getSelectedItem();
					try {
						context.stopRenderers();
						imageFilterElement.getContext().updateTimestamp();
						if (extension instanceof NullConfigurableExtension) {
							imageFilterElement.setReference(null);
						}
						else {
							imageFilterElement.setReference(extension.createConfigurableExtensionReference());
						}
						context.startRenderers();
						context.refresh();
						if (configView != null) {
							viewContext.removeComponent(configView);
							configView = null;
						}
					}
					catch (ExtensionException x) {
						x.printStackTrace();
					}
				}
			};
			final ValueChangeListener filterListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ExtensionConfigElementEvents.EXTENSION_REFERENCE_CHANGED: {
							editOptionsButton.setEnabled(imageFilterElement.getReference() != null);
							extensionComboBox.removeActionListener(comboListener);
							if (imageFilterElement.getReference() != null) {
								model.setSelectedItemByExtensionId(imageFilterElement.getReference().getExtensionId());
							}
							else {
								model.setSelectedItem(0);
							}
							if (configView != null) {
								viewContext.removeComponent(configView);
								configView = null;
							}
							extensionComboBox.addActionListener(comboListener);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			imageFilterElement.getExtensionElement().addChangeListener(filterListener);
			imageFilterElement.getLabelElement().addChangeListener(labelListener);
			imageFilterElement.getLockedElement().addChangeListener(lockedListener);
			imageFilterElement.getEnabledElement().addChangeListener(enabledListener);
			extensionComboBox.addActionListener(comboListener);
			editOptionsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					if (imageFilterElement.getReference() != null) {
						if (configView == null) {
							try {
								final Extension<ViewExtensionRuntime> extension = TwisterSwingRegistry.getInstance().getViewExtension(imageFilterElement.getReference().getExtensionId());
								configView = extension.createExtensionRuntime().createView(imageFilterElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
								configView = new NavigatorViewRuntime().createView(imageFilterElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
						}
						if (configView != null) {
							configView.setName(createFiltersPanelName(imageFilterElement));
						}
						viewContext.setComponent(configView);
					}
				}
			});
			selectedCheckBox.addItemListener(new ItemListener() {
				public void itemStateChanged(final ItemEvent e) {
					if (selectedCheckBox.isSelected()) {
						imageFilterElement.setUserData(Boolean.TRUE);
					}
					else {
						imageFilterElement.setUserData(null);
					}
				}
			});
			label.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					// imageFilterElement.getLabelElement().removeChangeListener(labelListener);
					imageFilterElement.getContext().updateTimestamp();
					imageFilterElement.setLabel(label.getText());
					if (configView != null) {
						configView.setName(createFiltersPanelName(imageFilterElement));
					}
					// imageFilterElement.getLabelElement().addChangeListener(labelListener);
				}
			});
			lockedCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					imageFilterElement.getContext().updateTimestamp();
					imageFilterElement.setLocked(lockedCheckBox.isSelected());
				}
			});
			enabledCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					imageFilterElement.getContext().updateTimestamp();
					imageFilterElement.setEnabled(enabledCheckBox.isSelected());
					context.refresh();
				}
			});
		}

		private String createFiltersPanelName(final LayerFilterConfigElement imageFilterElement) {
			return imageFilterElement.getLabel();
		}

		/**
		 * @param selected
		 */
		public void setSelected(final boolean selected) {
			selectedCheckBox.setSelected(selected);
		}

		/**
		 * @return
		 */
		public boolean isSelected() {
			return selectedCheckBox.isSelected();
		}
	}

	private class FrameFilterPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private final JCheckBox selectedCheckBox = createCheckBox();
		private JComponent configView;

		/**
		 * @param frameElement
		 * @param frameFilterElement
		 */
		public FrameFilterPanel(final FrameConfigElement frameElement, final FrameFilterConfigElement frameFilterElement) {
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(TwisterRegistry.getInstance().getFrameFilterRegistry(), true);
			final JTextField label = createTextField(frameFilterElement.getLabel(), 120, GUIFactory.DEFAULT_HEIGHT);
			final JCheckBox lockedCheckBox = createIconCheckBox("locked", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox enabledCheckBox = createIconCheckBox("enabled", "visible", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton editOptionsButton = createTextButton("editConfig", 100, GUIFactory.DEFAULT_HEIGHT);
			final JComboBox extensionComboBox = createExtensionComboBox(model, 140, GUIFactory.DEFAULT_HEIGHT);
			final Box filterPanel = createHorizontalBox(false);
			filterPanel.setBorder(BorderFactory.createEmptyBorder(0, 4, 0, 8));
			filterPanel.add(createSpace());
			filterPanel.add(selectedCheckBox);
			filterPanel.add(label);
			filterPanel.add(createSpace());
			filterPanel.add(lockedCheckBox);
			filterPanel.add(createSpace());
			filterPanel.add(enabledCheckBox);
			filterPanel.add(createSpace());
			filterPanel.add(Box.createHorizontalGlue());
			filterPanel.add(extensionComboBox);
			filterPanel.add(createSpace());
			filterPanel.add(editOptionsButton);
			setLayout(new BorderLayout());
			setBorder(BorderFactory.createEmptyBorder(2, 0, 2, 0));
			add(filterPanel, BorderLayout.CENTER);
			lockedCheckBox.setSelected(frameFilterElement.isLocked());
			enabledCheckBox.setSelected(frameFilterElement.isEnabled());
			editOptionsButton.setEnabled((frameFilterElement != null) && (frameFilterElement.getReference() != null));
			if (frameFilterElement.getReference() != null) {
				model.setSelectedItemByExtensionId(frameFilterElement.getReference().getExtensionId());
			}
			final ValueChangeListener lockedListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					lockedCheckBox.setSelected(frameFilterElement.isLocked());
				}
			};
			final ValueChangeListener enabledListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					enabledCheckBox.setSelected(frameFilterElement.isEnabled());
				}
			};
			final ValueChangeListener labelListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					label.setText(frameFilterElement.getLabel());
					if (configView != null) {
						configView.setName(createFiltersPanelName(frameFilterElement));
					}
				}
			};
			final ActionListener comboListener = new ActionListener() {
				@SuppressWarnings("unchecked")
				public void actionPerformed(ActionEvent e) {
					ConfigurableExtension<FrameFilterExtensionRuntime, FrameFilterExtensionConfig> extension = (ConfigurableExtension<FrameFilterExtensionRuntime, FrameFilterExtensionConfig>) extensionComboBox.getSelectedItem();
					try {
						context.stopRenderers();
						frameFilterElement.getContext().updateTimestamp();
						if (extension instanceof NullConfigurableExtension) {
							frameFilterElement.setReference(null);
						}
						else {
							frameFilterElement.setReference(extension.createConfigurableExtensionReference());
						}
						context.startRenderers();
						context.refresh();
						if (configView != null) {
							viewContext.removeComponent(configView);
							configView = null;
						}
					}
					catch (ExtensionException x) {
						x.printStackTrace();
					}
				}
			};
			final ValueChangeListener filterListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ExtensionConfigElementEvents.EXTENSION_REFERENCE_CHANGED: {
							editOptionsButton.setEnabled(frameFilterElement.getReference() != null);
							extensionComboBox.removeActionListener(comboListener);
							if (frameFilterElement.getReference() != null) {
								model.setSelectedItemByExtensionId(frameFilterElement.getReference().getExtensionId());
							}
							else {
								model.setSelectedItem(0);
							}
							if (configView != null) {
								viewContext.removeComponent(configView);
								configView = null;
							}
							extensionComboBox.addActionListener(comboListener);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			frameFilterElement.getLabelElement().addChangeListener(labelListener);
			frameFilterElement.getLockedElement().addChangeListener(lockedListener);
			frameFilterElement.getEnabledElement().addChangeListener(enabledListener);
			extensionComboBox.addActionListener(comboListener);
			editOptionsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					if (frameFilterElement.getReference() != null) {
						if (configView == null) {
							try {
								final Extension<ViewExtensionRuntime> extension = TwisterSwingRegistry.getInstance().getViewExtension(frameFilterElement.getReference().getExtensionId());
								configView = extension.createExtensionRuntime().createView(frameFilterElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
								configView = new NavigatorViewRuntime().createView(frameFilterElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
						}
						if (configView != null) {
							configView.setName(createFiltersPanelName(frameFilterElement));
						}
						viewContext.setComponent(configView);
					}
				}
			});
			label.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					// frameFilterElement.getLabelElement().removeChangeListener(labelListener);
					frameFilterElement.getContext().updateTimestamp();
					frameFilterElement.setLabel(label.getText());
					if (configView != null) {
						configView.setName(createFiltersPanelName(frameFilterElement));
					}
					// frameFilterElement.getLabelElement().addChangeListener(labelListener);
				}
			});
			selectedCheckBox.addItemListener(new ItemListener() {
				public void itemStateChanged(final ItemEvent e) {
					if (selectedCheckBox.isSelected()) {
						frameFilterElement.setUserData(Boolean.TRUE);
					}
					else {
						frameFilterElement.setUserData(null);
					}
				}
			});
			frameFilterElement.getExtensionElement().addChangeListener(filterListener);
			lockedCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					frameFilterElement.getContext().updateTimestamp();
					frameFilterElement.setLocked(lockedCheckBox.isSelected());
				}
			});
			enabledCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					frameFilterElement.getContext().updateTimestamp();
					frameFilterElement.setEnabled(enabledCheckBox.isSelected());
					context.refresh();
				}
			});
		}

		private String createFiltersPanelName(final FrameFilterConfigElement frameFilterElement) {
			return frameFilterElement.getLabel();
		}

		/**
		 * @param selected
		 */
		public void setSelected(final boolean selected) {
			selectedCheckBox.setSelected(selected);
		}

		/**
		 * @return
		 */
		public boolean isSelected() {
			return selectedCheckBox.isSelected();
		}
	}

	private class ImagePanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private JComponent configView;

		/**
		 * @param layerElement
		 * @param imageElement
		 */
		public ImagePanel(final ImageLayerConfigElement layerElement, final ImageConfigElement imageElement) {
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(TwisterRegistry.getInstance().getImageRegistry(), true);
			final JButton editOptionsButton = createTextButton("editConfig", 100, GUIFactory.DEFAULT_HEIGHT);
			final JComboBox extensionComboBox = createExtensionComboBox(model, 140, GUIFactory.DEFAULT_HEIGHT);
			final Box imagePanel = createHorizontalBox(false);
			imagePanel.add(Box.createHorizontalGlue());
			imagePanel.add(extensionComboBox);
			imagePanel.add(createSpace());
			imagePanel.add(editOptionsButton);
			setLayout(new BorderLayout());
			add(imagePanel, BorderLayout.CENTER);
			editOptionsButton.setEnabled((imageElement != null) && (imageElement.getReference() != null));
			if (imageElement.getReference() != null) {
				model.setSelectedItemByExtensionId(imageElement.getReference().getExtensionId());
			}
			final ActionListener comboListener = new ActionListener() {
				@SuppressWarnings("unchecked")
				public void actionPerformed(ActionEvent e) {
					ConfigurableExtension<ImageExtensionRuntime, ImageExtensionConfig> extension = (ConfigurableExtension<ImageExtensionRuntime, ImageExtensionConfig>) extensionComboBox.getSelectedItem();
					try {
						context.stopRenderers();
						imageElement.getContext().updateTimestamp();
						if (extension instanceof NullConfigurableExtension) {
							imageElement.setReference(null);
						}
						else {
							imageElement.setReference(extension.createConfigurableExtensionReference());
						}
						context.startRenderers();
						context.refresh();
						if (configView != null) {
							viewContext.removeComponent(configView);
							configView = null;
						}
					}
					catch (ExtensionException x) {
						x.printStackTrace();
					}
				}
			};
			final ValueChangeListener imageListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ExtensionConfigElementEvents.EXTENSION_REFERENCE_CHANGED: {
							editOptionsButton.setEnabled(imageElement.getReference() != null);
							extensionComboBox.removeActionListener(comboListener);
							if (imageElement.getReference() != null) {
								model.setSelectedItemByExtensionId(imageElement.getReference().getExtensionId());
							}
							else {
								model.setSelectedItem(0);
							}
							if (configView != null) {
								viewContext.removeComponent(configView);
								configView = null;
							}
							extensionComboBox.addActionListener(comboListener);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			imageElement.getExtensionElement().addChangeListener(imageListener);
			extensionComboBox.addActionListener(comboListener);
			editOptionsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					if (imageElement.getReference() != null) {
						if (configView == null) {
							try {
								final Extension<ViewExtensionRuntime> extension = TwisterSwingRegistry.getInstance().getViewExtension(imageElement.getReference().getExtensionId());
								configView = extension.createExtensionRuntime().createView(imageElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
								configView = new NavigatorViewRuntime().createView(imageElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
						}
						if (configView != null) {
							configView.setName(createImagePanelName(layerElement));
						}
						viewContext.setComponent(configView);
					}
				}
			});
		}

		private String createImagePanelName(final ImageLayerConfigElement layerElement) {
			return layerElement.getLabel();
		}
	}

	private class EffectPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private JComponent configView;

		/**
		 * @param config
		 * @param effectElement
		 */
		public EffectPanel(final TwisterConfig config, final EffectConfigElement effectElement) {
			final ConfigurableExtensionComboBoxModel model = new ConfigurableExtensionComboBoxModel(TwisterRegistry.getInstance().getEffectRegistry(), true);
			final JLabel label = createTextLabel("effect", SwingConstants.LEFT, 80, GUIFactory.DEFAULT_HEIGHT);
			final JCheckBox lockedCheckBox = createIconCheckBox("locked", "locked", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JCheckBox enabledCheckBox = createIconCheckBox("enabled", "visible", GUIFactory.ICON_WIDTH, GUIFactory.ICON_HEIGHT);
			final JButton editOptionsButton = createTextButton("editConfig", 100, GUIFactory.DEFAULT_HEIGHT);
			final JComboBox extensionComboBox = createExtensionComboBox(model, 140, GUIFactory.DEFAULT_HEIGHT);
			label.setPreferredSize(new Dimension(55, GUIFactory.DEFAULT_HEIGHT));
			final Box effectPanel = createHorizontalBox(false);
			effectPanel.add(Box.createHorizontalGlue());
			effectPanel.add(label);
			effectPanel.add(createSpace());
			effectPanel.add(lockedCheckBox);
			effectPanel.add(createSpace());
			effectPanel.add(enabledCheckBox);
			effectPanel.add(createSpace());
			effectPanel.add(extensionComboBox);
			effectPanel.add(createSpace());
			effectPanel.add(editOptionsButton);
			effectPanel.add(Box.createHorizontalGlue());
			setLayout(new BorderLayout());
			add(effectPanel, BorderLayout.CENTER);
			setBorder(BorderFactory.createMatteBorder(1, 0, 0, 0, Color.DARK_GRAY));
			editOptionsButton.setEnabled((effectElement != null) && (effectElement.getReference() != null));
			if (effectElement.getReference() != null) {
				model.setSelectedItemByExtensionId(effectElement.getReference().getExtensionId());
			}
			lockedCheckBox.setSelected(effectElement.isLocked());
			enabledCheckBox.setSelected(effectElement.isEnabled());
			final ValueChangeListener lockedListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					lockedCheckBox.setSelected(effectElement.isLocked());
				}
			};
			final ValueChangeListener enabledListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					enabledCheckBox.setSelected(effectElement.isEnabled());
				}
			};
			final ActionListener comboListener = new ActionListener() {
				@SuppressWarnings("unchecked")
				public void actionPerformed(ActionEvent e) {
					ConfigurableExtension<EffectExtensionRuntime, EffectExtensionConfig> extension = (ConfigurableExtension<EffectExtensionRuntime, EffectExtensionConfig>) extensionComboBox.getSelectedItem();
					try {
						context.stopRenderers();
						effectElement.getContext().updateTimestamp();
						if (extension == NullConfigurableExtension.getInstance()) {
							effectElement.setReference(null);
						}
						else {
							effectElement.setReference(extension.createConfigurableExtensionReference());
						}
						context.startRenderers();
						context.refresh();
						if (configView != null) {
							viewContext.removeComponent(configView);
							configView = null;
						}
					}
					catch (ExtensionException x) {
						x.printStackTrace();
					}
				}
			};
			extensionComboBox.addActionListener(comboListener);
			final ValueChangeListener effectListener = new ValueChangeListener() {
				public void valueChanged(ValueChangeEvent e) {
					switch (e.getEventType()) {
						case ExtensionConfigElementEvents.EXTENSION_REFERENCE_CHANGED: {
							editOptionsButton.setEnabled(effectElement.getReference() != null);
							extensionComboBox.removeActionListener(comboListener);
							if (effectElement.getReference() != null) {
								model.setSelectedItemByExtensionId(effectElement.getReference().getExtensionId());
							}
							else {
								model.setSelectedItem(0);
							}
							if (configView != null) {
								viewContext.removeComponent(configView);
								configView = null;
							}
							extensionComboBox.addActionListener(comboListener);
							break;
						}
						default: {
							break;
						}
					}
				}
			};
			effectElement.getLockedElement().addChangeListener(lockedListener);
			effectElement.getEnabledElement().addChangeListener(enabledListener);
			effectElement.getExtensionElement().addChangeListener(effectListener);
			editOptionsButton.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					if (effectElement.getReference() != null) {
						if (configView == null) {
							try {
								final Extension<ViewExtensionRuntime> extension = TwisterSwingRegistry.getInstance().getViewExtension(effectElement.getReference().getExtensionId());
								configView = extension.createExtensionRuntime().createView(effectElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
							catch (final ExtensionException x) {
								x.printStackTrace();
								configView = new NavigatorViewRuntime().createView(effectElement.getReference().getExtensionConfig(), viewContext, context, session);
							}
						}
						if (configView != null) {
							configView.setName(createEffectPanelName());
						}
						viewContext.setComponent(configView);
					}
				}
			});
			lockedCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					effectElement.getContext().updateTimestamp();
					effectElement.setLocked(lockedCheckBox.isSelected());
				}
			});
			enabledCheckBox.addActionListener(new ActionListener() {
				public void actionPerformed(final ActionEvent e) {
					effectElement.getContext().updateTimestamp();
					effectElement.setEnabled(enabledCheckBox.isSelected());
					context.refresh();
				}
			});
		}

		private String createEffectPanelName() {
			return TwisterSwingResources.getInstance().getString("name.effect");
		}
	}

	private class PreviewPanel extends JComponent {
		private static final long serialVersionUID = 1L;
		private final Surface surface;

		/**
		 * @param surface
		 */
		public PreviewPanel(final Surface surface) {
			this.surface = surface;
		}

		/**
		 * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
		 */
		@Override
		protected void paintComponent(final Graphics g) {
			g.drawImage(surface.getImage(), 0, 0, null);
		}
	}
}
