/*
 * $Id:NavigatorViewRuntime.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister.swing.view;

import java.awt.BorderLayout;
import java.awt.Color;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

import net.sf.jame.core.extension.Extension;
import net.sf.jame.core.extension.ExtensionConfig;
import net.sf.jame.core.extension.ExtensionException;
import net.sf.jame.core.tree.Node;
import net.sf.jame.core.tree.NodeAction;
import net.sf.jame.core.tree.NodeBuilder;
import net.sf.jame.core.tree.NodeEvent;
import net.sf.jame.core.tree.NodeSession;
import net.sf.jame.core.tree.RootNode;
import net.sf.jame.core.tree.Tree;
import net.sf.jame.core.tree.TreeListener;
import net.sf.jame.twister.TwisterRegistry;
import net.sf.jame.twister.renderer.RenderContext;
import net.sf.jame.twister.swing.NavigatorPanel;
import net.sf.jame.twister.swing.NavigatorTree;
import net.sf.jame.twister.swing.ViewContext;
import net.sf.jame.twister.swing.view.extension.ViewExtensionRuntime;
import net.sf.jame.twister.tree.extension.NodeBuilderExtensionRuntime;

/**
 * @author Andrea Medeghini
 */
public class NavigatorViewRuntime extends ViewExtensionRuntime {
	/**
	 * @see net.sf.jame.twister.swing.view.extension.ViewExtensionRuntime#createView(net.sf.jame.core.extension.ExtensionConfig, net.sf.jame.twister.swing.ViewContext, net.sf.jame.twister.renderer.RenderContext, net.sf.jame.core.tree.NodeSession)
	 */
	@Override
	public JComponent createView(final ExtensionConfig config, final ViewContext viewContext, final RenderContext context, final NodeSession session) {
		try {
			final Extension<NodeBuilderExtensionRuntime> extension = TwisterRegistry.getInstance().getNodeBuilderExtension(config.getExtensionId());
			final NodeBuilder nodeBuilder = extension.createExtensionRuntime().createNodeBuilder(config);
			final Tree tree = new Tree(new RootNode("navigator.root", extension.getExtensionName() + " extension"));
			nodeBuilder.createNodes(tree.getRootNode());
			tree.getRootNode().setContext(config.getContext());
			tree.getRootNode().setSession(new NavigatorNodeSession(session));
			tree.getRootNode().addNodeListener(new TreeListener() {
				/**
				 * @see net.sf.jame.core.tree.TreeListener#nodeChanged(net.sf.jame.core.tree.NodeEvent)
				 */
				public void nodeChanged(final NodeEvent e) {
				}

				/**
				 * @see net.sf.jame.core.tree.TreeListener#nodeAdded(net.sf.jame.core.tree.NodeEvent)
				 */
				public void nodeAdded(final NodeEvent e) {
				}

				/**
				 * @see net.sf.jame.core.tree.TreeListener#nodeRemoved(net.sf.jame.core.tree.NodeEvent)
				 */
				public void nodeRemoved(final NodeEvent e) {
				}

				/**
				 * @see net.sf.jame.core.tree.TreeListener#nodeAccepted(net.sf.jame.core.tree.NodeEvent)
				 */
				public void nodeAccepted(final NodeEvent e) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							context.refresh();
						}
					});
				}

				/**
				 * @see net.sf.jame.core.tree.TreeListener#nodeCancelled(net.sf.jame.core.tree.NodeEvent)
				 */
				public void nodeCancelled(final NodeEvent e) {
				}
			});
			return new NavigatorViewPanel(viewContext, context, tree);
		}
		catch (final ExtensionException e) {
			e.printStackTrace();
		}
		return new JPanel();
	}

	private class NavigatorViewPanel extends JPanel {
		private static final long serialVersionUID = 1L;
		private final NavigatorTreeSelectionListener listener;
		private final NavigatorPanel navigatorPanel;
		private final NavigatorTree navigatorTree;

		/**
		 * @param viewContext
		 * @param context
		 * @param tree
		 */
		public NavigatorViewPanel(final ViewContext viewContext, final RenderContext context, final Tree tree) {
			setLayout(new BorderLayout());
			setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20), BorderFactory.createLineBorder(Color.DARK_GRAY)));
			navigatorTree = new NavigatorTree(tree.getRootNode());
			navigatorPanel = new NavigatorPanel(viewContext, tree.getRootNode());
			navigatorTree.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createMatteBorder(0, 0, 0, 1, Color.DARK_GRAY), BorderFactory.createEmptyBorder(2, 2, 2, 2)));
			add(navigatorTree, BorderLayout.WEST);
			add(navigatorPanel, BorderLayout.CENTER);
			listener = new NavigatorTreeSelectionListener();
			navigatorPanel.addChangeListener(new NavigatorPanelSelectionListener());
			navigatorTree.getModel().addTreeModelListener(new NavigatorTreeModelListener());
			navigatorTree.getSelectionModel().addTreeSelectionListener(listener);
		}

		private class NavigatorPanelSelectionListener implements ChangeListener {
			/**
			 * @see javax.swing.event.ChangeListener#stateChanged(javax.swing.event.ChangeEvent)
			 */
			public void stateChanged(final ChangeEvent e) {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						if (navigatorPanel.getEditorNode() != null) {
							final TreePath path = navigatorTree.creareTreePath(navigatorPanel.getEditorNode().getNodePath());
							navigatorTree.expandPath(path.getParentPath());
							navigatorTree.getSelectionModel().removeTreeSelectionListener(listener);
							navigatorTree.setSelectionPath(path);
							navigatorTree.getSelectionModel().addTreeSelectionListener(listener);
						}
					}
				});
			}
		}

		private class NavigatorTreeSelectionListener implements TreeSelectionListener {
			/**
			 * @see javax.swing.event.TreeSelectionListener#valueChanged(javax.swing.event.TreeSelectionEvent)
			 */
			public void valueChanged(final TreeSelectionEvent e) {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						if (navigatorTree.getSelectionPath() != null) {
							final Node node = (Node) ((DefaultMutableTreeNode) navigatorTree.getSelectionPath().getLastPathComponent()).getUserObject();
							navigatorPanel.loadNode(node);
						}
						else {
							navigatorPanel.loadNode(null);
						}
					}
				});
			}
		}

		private class NavigatorTreeModelListener implements TreeModelListener {
			/**
			 * @see javax.swing.event.TreeModelListener#treeNodesChanged(javax.swing.event.TreeModelEvent)
			 */
			public void treeNodesChanged(final TreeModelEvent e) {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						navigatorTree.expandPath(e.getTreePath());
					}
				});
			}

			/**
			 * @see javax.swing.event.TreeModelListener#treeNodesInserted(javax.swing.event.TreeModelEvent)
			 */
			public void treeNodesInserted(final TreeModelEvent e) {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						navigatorTree.expandPath(e.getTreePath());
					}
				});
			}

			/**
			 * @see javax.swing.event.TreeModelListener#treeNodesRemoved(javax.swing.event.TreeModelEvent)
			 */
			public void treeNodesRemoved(final TreeModelEvent e) {
			}

			/**
			 * @see javax.swing.event.TreeModelListener#treeStructureChanged(javax.swing.event.TreeModelEvent)
			 */
			public void treeStructureChanged(final TreeModelEvent e) {
			}
		}
	}

	private class NavigatorNodeSession implements NodeSession {
		private final NodeSession session;

		/**
		 * @param session
		 */
		public NavigatorNodeSession(final NodeSession session) {
			this.session = session;
		}

		/**
		 * @see net.sf.jame.core.tree.NodeSession#appendAction(net.sf.jame.core.tree.NodeAction)
		 */
		public void appendAction(final NodeAction action) {
			session.appendAction(action);
		}

		/**
		 * @see net.sf.jame.core.tree.NodeSession#getActions()
		 */
		public List<NodeAction> getActions() {
			return session.getActions();
		}

		/**
		 * @see net.sf.jame.core.tree.NodeSession#getSessionName()
		 */
		public String getSessionName() {
			return session.getSessionName();
		}

		/**
		 * @see net.sf.jame.core.tree.NodeSession#getTimestamp()
		 */
		public long getTimestamp() {
			return session.getTimestamp();
		}

		/**
		 * @see net.sf.jame.core.tree.NodeSession#isAcceptImmediatly()
		 */
		public boolean isAcceptImmediatly() {
			return true;
		}

		/**
		 * @see net.sf.jame.core.tree.NodeSession#setAcceptImmediatly(boolean)
		 */
		public void setAcceptImmediatly(final boolean isApplyImmediatly) {
		}

		/**
		 * @see net.sf.jame.core.tree.NodeSession#setTimestamp(long)
		 */
		public void setTimestamp(final long timestamp) {
			session.setTimestamp(timestamp);
		}
	}
}
