package sample.form;

import jaxcent.*;
import java.util.*;

/**
  * Jaxcent sample.
  *
  * Does some data data verification, if everything fine, presents a summary of data.
  */

public class Summary extends jaxcent.JaxcentPage {

    // Page items.

    HtmlDiv okDiv = new HtmlDiv( this, "okDiv" );
    HtmlDiv errorDiv = new HtmlDiv( this, "errorDiv" );
    HtmlTable problems = new HtmlTable( this, "problems" );
    HtmlTable summary = new HtmlTable( this, "summary" );

    boolean haveError = false;

    Map dataMap;

    // Constructor checks the session data, and makes either
    // okDiv or errorDiv visible, the other one invisible.

    public Summary()
    {
        try {

            dataMap = getAllSessionData( false );

            checkShoppingCart();  // Check the shopping cart.

            // Check if user has visited all pages, and check page items.
            checkNameAddress();
            checkShippingAddress();
            checkPayment();

            // All done.  If found any errors, make error DIV visible, else make ok DIV visible.

            if ( haveError ) {
                errorDiv.show();
                okDiv.hide();
            } else {
                errorDiv.hide();
                okDiv.show();
            }
        } catch (Jaxception jax) {
            jax.printStackTrace();
        }
    }

    void addError( String error ) throws Jaxception
    {
        haveError = true;
        problems.insertRow( -1, new String[]{ error } );
    }

    void addSummary( String str1, String str2 ) throws Jaxception
    {
        summary.insertRow( -1, new String[]{ str1, str2 } );
    }

    void checkShoppingCart() throws Jaxception
    {
        // Check if shopping cart has any items.
        ArrayList itemsInCart = (ArrayList) dataMap.get( "shopping.cart" );
        if ( itemsInCart == null || itemsInCart.size() == 0 ) {
            addError( "No items in shopping cart" );
        } else {
            addSummary( "<B>Shopping Cart Items</B>", "&nbsp;" );
            // Describe items, prices, and a final total.
            int total = 0;
            for ( int i = 0; i < itemsInCart.size(); i += 2 ) {
                int itemIndex = ((Integer) itemsInCart.get( i )).intValue();
                int itemCount = ((Integer) itemsInCart.get( i+1 )).intValue();
                String item = ShoppingCart.items[ itemIndex ][ 0 ];
                int unitPrice = Integer.parseInt( ShoppingCart.items[ itemIndex ][ 1 ] );
                int cost = unitPrice * itemCount;
                total += cost;
                addSummary( "&nbsp;&nbsp;" + itemCount + " " + item, "$" + cost );
            }
            addSummary( "<B>Total Price:</B>", "$" + total );
            addSummary( "&nbsp;", "&nbsp;" );
        }
    }

    // To see if user has visited a page, we need any single item on page.
    // If not visited, the item will have a null value.  If visited,
    // the item will have a value or an empty string.

    boolean checkPage( String item, String description ) throws Jaxception
    {
        if ( dataMap.get( item ) == null ) {
            addError( "You have not visited the \"" + description + "\" page!" );
            return false;
        }
        return true;
    }

    // Check the value of an item.  If required and item not entered, register as error.
    // Otherwise, add to summary.
 
    boolean checkItem( String item, String description, boolean required ) throws Jaxception
    {
        String value = (String) dataMap.get( item );
        if ( value.equals( "" )) {
            if ( required ) {
                addError( "Item \"" + description + "\" is required, you have not entered it." );
            }
            return false;
        }
        summary.insertRow( -1, new String[]{ description, value } );
        return true;
    }

    void checkNameAddress() throws Jaxception
    {
        if ( checkPage( "firstName", "Name and Address Page" )) {
            checkItem( "firstName", "First Name", true );
            checkItem( "lastName", "Last Name", true );
            checkItem( "billingAddress", "Address", true );
            checkItem( "billingCity", "City", true );
            checkItem( "billingState", "State", true );
            checkItem( "billingZip", "ZIP/Postal Code", true );
            checkItem( "billingCountry", "Country", false );
        }
    }

    void checkShippingAddress() throws Jaxception
    {
        if ( checkPage( "shippingAddress", "Shipping Address Page" )) {
            checkItem( "shippingAddress", "Shipping Address", true );
            checkItem( "shippingCity", "Shipping City", true );
            checkItem( "shippingState", "Shipping State", true );
            checkItem( "shippingZip", "Shipping ZIP/Postal Code", true );
            checkItem( "shippingCountry", "Shipping Country", false );
        }
    }

    void checkPayment() throws Jaxception
    {
        if ( checkPage( "cardType", "Payment Page" )) {
            if ( checkItem( "cardNumber", "Card Number", true )) {
                String cn = (String) dataMap.get( "cardNumber" );
                boolean seenDigit = false;
                for ( int i = 0; i < cn.length(); i++ ) {
                    char ch = cn.charAt( i );
                    if ( Character.isDigit( ch )) {
                        seenDigit = true;
                    } else if ((! Character.isSpaceChar( ch )) &&
                        ( ch != '-' ) &&
                        ( ch != '/' )) {
                        addError( "Invalid character '" + ch + "' in card number " + cn );
                        seenDigit = true; // Don't add another error message
                        break;
                    }
                }
                if ( ! seenDigit ) {
                    addError( "There are no numbers in the credit card number!" );
                }
            }
            if ( checkItem( "cardExpirationYear", "Card Expiration Year", true )) {
                String year = (String) dataMap.get( "cardExpirationYear" );
                try {
                    int nYear = Integer.parseInt( year );
                    Calendar cal = Calendar.getInstance();
                    if ( nYear < 100 )
                        nYear += 2000;
                    if ( nYear < cal.get( cal.YEAR )) {
                        addError( "Year " + year + " is already gone!" );
                    }
                    if ( nYear > cal.get( cal.YEAR ) + 20 ) {
                        addError( "Year " + year + " is kind of late for card expiry!" );
                    }
                } catch (NumberFormatException ex) {
                    addError( "Invalid character(s) in year: " + year );
                }
            }
        }
    }

}
