/*
 * This file is part of jDiffChaser.
 *
 *  jDiffChaser is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  jDiffChaser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with jDiffChaser; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package org.jdiffchaser.utils;

import java.text.SimpleDateFormat;
import java.util.GregorianCalendar;
import java.util.Calendar;
import java.util.Locale;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;

/** 
 * Simple formatting of logs
 */
public class LogFormatter extends Formatter {
     
    private static final SimpleDateFormat SDF = new SimpleDateFormat(  "dd MMM yyyy HH:mm:ss.SSS", 
                                                                       Locale.ENGLISH);
    protected static final String LINE_SEP = System.getProperty("line.separator"); 
   
    private static final String PAD_STACK_TRACE = "               ";
     
    /**
     * gregorian calendar instance used to retrieve a date object
     */
    protected static final Calendar gregoCal = GregorianCalendar.getInstance();
    
    /**
     * @see buildHeader
     */
    private int counter=0;
    
    /**
     * e.g :
     *
     * counter date        time          level Thread name  method name
     * 123     06 Jan 2004 15:54:40.772  INFO  main         JMSHelper.initJNDI()
     */
    protected StringBuffer buildHeader(LogRecord record){
        
        StringBuffer sb = new StringBuffer();
        
        sb.append(Integer.toString(counter++));
        sb.append(' ');
        sb.append('|');
        sb.append(' ');
        
        // timeHeader
        gregoCal.setTimeInMillis(record.getMillis());        
        sb.append(SDF.format(gregoCal.getTime()));
        sb.append(' ');
        
        // level
        sb.append(record.getLevel());
        sb.append(' ');
        
        // Thread id
        sb.append(Thread.currentThread().getName());
        sb.append(' ');
        
        String loggerName = record.getLoggerName();
        if( loggerName!=null && !loggerName.startsWith("org.jdiffchaser") ){
            sb.append('[').append(record.getLoggerName()).append(']');
            sb.append(' ');
        }

        // class.method()
        String className = record.getSourceClassName();
        int indexDot = className.lastIndexOf('.');
        if(indexDot>0){
            sb.append(className.substring(indexDot+1, className.length()));    
        }else {
            sb.append(className);    
        }   
        if(record.getSourceMethodName()!=null){
            sb.append('.');        
            sb.append(record.getSourceMethodName());        
            sb.append("() ");   
        }else {
            sb.append(' ');  
        }
        return sb;
    }
       
    
    
    public String format(LogRecord record) {
        
        StringBuffer sb = buildHeader(record);
    
        // message
        sb.append(record.getMessage());
        sb.append(LINE_SEP);
        
        // When logger.log(level, msg, throwable) is used
        Throwable throwable = record.getThrown();
        if(throwable!=null){
            sb.append(PAD_STACK_TRACE);
            sb.append(throwable.toString());
            sb.append(LINE_SEP);
            
            StackTraceElement [] traces = throwable.getStackTrace();            
            
            for(int i=0; i<traces.length; i++){
                sb.append(PAD_STACK_TRACE);
                sb.append(traces[i].toString());
                sb.append(LINE_SEP);
            }
        }
        
        return sb.toString();
    }
    
    
    public String formatMessage(LogRecord record) {
        return format(record);
    }
    

    /**
     * Not a singleton !
     */
    public static Formatter getInstance(){
        return new LogFormatter();
    }
}

