package jp.co.sra.jun.collections.linkedlist;

import java.io.*;
import jp.co.sra.jun.collections.sequences.JunSequence;
import jp.co.sra.smalltalk.*;

/**
 * JunSingleLinkedList class
 * 
 *  @author    nisinaka
 *  @created   1998/11/09 (by nisinaka)
 *  @updated   2002/07/18 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSingleLinkedList.java,v 8.10 2008/02/20 06:30:52 nisinaka Exp $
 */
public class JunSingleLinkedList extends JunSequence {
	/** The first link of the list. */
	protected JunSingleLink firstLink = null;

	/** The last link of the list. */
	protected JunSingleLink lastLink = null;

	/**
	 * Add the object as an entity of the receiver.
	 * 
	 * @param anObject java.lang.Object
	 */
	public void add_(Object anObject) {
		this.addLast_(anObject);
	}

	/**
	 * Add the object as the first entity of the receiver.
	 * 
	 * @param anObject java.lang.Object
	 */
	public void addFirst_(Object anObject) {
		JunSingleLink aLink = JunSingleLink.Entity_(anObject);

		if (this.isEmpty()) {
			lastLink = aLink;
		} else {
			aLink.nextLink = firstLink;
		}

		firstLink = aLink;
	}

	/**
	 * Add the object as the last entity of the receiver.
	 * 
	 * @param anObject java.lang.Object
	 */
	public void addLast_(Object anObject) {
		JunSingleLink aLink = JunSingleLink.Entity_(anObject);

		if (this.isEmpty()) {
			firstLink = aLink;
		} else {
			lastLink.nextLink_(aLink);
		}

		lastLink = aLink;
	}

	/**
	 * Enumerate the objects and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object do_(final StBlockClosure aBlock) {
		return this.linksDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunSingleLink aLink = (JunSingleLink) anObject;

				return aBlock.value_(aLink.entityObject());
			}
		});
	}

	/**
	 * Answer the object of the first entity.
	 * 
	 * @return java.lang.Object
	 */
	public Object first() {
		this.emptyCheck();

		return firstLink.entityObject;
	}

	/**
	 * Answer true if the receiver includes the Object, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * 
	 * @return boolean
	 */
	public boolean includes_(Object anObject) {
		JunSingleLink aLink = firstLink;

		while (aLink != null) {
			if (aLink.entityObject == anObject) {
				return true;
			}

			aLink = aLink.nextLink;
		}

		return false;
	}

	/**
	 * Answer true if the receiver is empty, otherwise false.
	 * 
	 * @return boolean
	 */
	public boolean isEmpty() {
		return (firstLink == null);
	}

	/**
	 * Answer the object of the last entity.
	 * 
	 * @return java.lang.Object
	 */
	public Object last() {
		this.emptyCheck();

		return lastLink.entityObject;
	}

	/**
	 * Print my string representation on the Writer.
	 * 
	 * @param aWriter java.io.Writer
	 * 
	 * @throws IOException DOCUMENT ME!
	 */
	public void printOn_(final Writer aWriter) throws IOException {
		aWriter.write('(');

		if (this.isEmpty() == false) {
			this.do_(new StBlockClosure() {
				public Object value_(Object each) {
					try {
						aWriter.write(each.toString());
						aWriter.write(' ');
					} catch (IOException e) {
					}

					return null;
				}
			});
		}

		aWriter.write(')');
	}

	/**
	 * Remove the first entity from the receiver.
	 * 
	 * @return java.lang.Object
	 */
	public Object removeFirst() {
		this.emptyCheck();

		JunSingleLink oldLink = firstLink;

		if (firstLink == lastLink) {
			firstLink = null;
			lastLink = null;
		} else {
			firstLink = oldLink.nextLink;
		}

		oldLink.nextLink = null;

		return oldLink.entityObject;
	}

	/**
	 * Answer the size of the list.
	 * 
	 * @return int
	 */
	public int size() {
		int size = 0;
		JunSingleLink aLink = firstLink;

		while (aLink != null) {
			size++;
			aLink = aLink.nextLink();
		}

		return size;
	}

	/**
	 * If the receiver is empty, throw an exception.
	 */
	protected void emptyCheck() {
		if (this.isEmpty()) {
			throw SmalltalkException.Error("this collection is empty");
		}
	}

	/**
	 * Enumerate all links and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	protected Object linksDo_(StBlockClosure aBlock) {
		JunSingleLink aLink = firstLink;

		while (aLink != null) {
			Object result = aBlock.value_(aLink);

			if (result != null) {
				return result;
			}

			aLink = aLink.nextLink();
		}

		return null;
	}
}
