package jp.co.sra.jun.dxf.support;

import java.awt.Color;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StReadStream;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsCurve;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolylineLoop;

/**
 * JunDXFParser class
 * 
 *  @author    nisinaka
 *  @created   2002/07/11 (by nisinaka)
 *  @updated   2007/08/30 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun678 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDXFParser.java,v 8.15 2008/02/20 06:30:54 nisinaka Exp $
 */
public class JunDXFParser extends JunDXFScanner {
	protected static Color[] ColorTable = null;
	protected Hashtable layers;
	protected String layer;
	protected Hashtable dxfNames;
	protected Hashtable itemCount;
	protected boolean happy;
	protected Object[] item;
	protected Vector definedObjects;
	protected boolean doNurbs = false;
	protected Hashtable settings = null;
	protected Vector outside = null;
	protected Hashtable layerColors = null;

	/**
	 * Create a new instance of <code>JunDXFParser</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunDXFParser() {
		super();
	}

	/**
	 * Insert the method's description here. Creation date: (2002/07/15 14:37:33)
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param r double
	 * @param a1 double
	 * @param a2 double
	 * @return java.util.Vector
	 * @category Utilities
	 */
	public static Vector CalculateARC_radius_from_to_(Jun3dPoint aPoint, double r, double a1, double a2) {
		double x = aPoint.x();
		double y = aPoint.y();
		double pi2 = Math.PI * 2;
		double angle1 = (a1 * pi2) / 360.0;
		double angle2 = (a2 * pi2) / 360.0;
		if (angle2 == 0) {
			angle2 = pi2;
		}
		Vector points = new Vector(8);
		double xStart = angle1;
		double xEnd = angle2;
		double diff = angle2 - angle1;
		double sweep = diff - (Math.floor(diff / pi2) * pi2);
		double delta = sweep / 8.0;
		if (xStart > xEnd) {
			xEnd = xEnd + pi2;
		}

		double nx;
		double ny;
		while (xStart < xEnd) {
			nx = x + (Math.cos(xStart) * r);
			ny = y + (Math.sin(xStart) * r);
			points.add(new Jun3dPoint(nx, ny, 0));
			xStart += delta;
		}
		nx = x + (Math.cos(xEnd) * r);
		ny = y + (Math.sin(xEnd) * r);
		points.add(new Jun3dPoint(nx, ny, 0));

		return points;
	}

	/**
	 * Create a new instance of JunDXFParser and parse the specified file with
	 * it. Determines from the file itself whether to use CR or LF.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.dxf.JunDXFParser
	 * @category Instance creation
	 */
	public static JunDXFParser OnSmart_(File aFile) {
		JunDXFParser parser = new JunDXFParser();
		try {
			Reader reader = new BufferedReader(new FileReader(aFile));
			StringWriter writer = new StringWriter();
			int ch;
			while ((ch = reader.read()) > 0) {
				writer.write(ch);
			}
			parser.onSmart_(new StReadStream(writer.toString()));
		} catch (IOException e) {
			return null;
		}
		parser.parse();
		return parser;
	}

	/**
	 * Answer the color specified with the index number.
	 * 
	 * @param anInteger int
	 * @return java.awt.Color
	 * @category Accessing
	 */
	protected static Color ColorAt_(int anInteger) {
		int index = anInteger;
		if (ColorTable == null) {
			ColorTable = new Color[] { Color.red, Color.yellow, Color.green, Color.cyan, Color.blue, new Color(0.624954f, 0.125015f, 0.874985f), Color.white, Color.black };
		}
		if ((index < 1) || (7 < index)) {
			index = 7;
		}
		return ColorTable[index - 1];
	}

	/**
	 * Parse a 3DFACE.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parse3DFACE() {
		String markName = Integer.toString(mark);
		String layer = "";
		double x1 = 0.0, y1 = 0.0, z1 = 0.0;
		double x2 = 0.0, y2 = 0.0, z2 = 0.0;
		double x3 = 0.0, y3 = 0.0, z3 = 0.0;
		double x4 = 0.0, y4 = 0.0, z4 = 0.0;
		boolean isQuad = false;
		ArrayList vertexes = new ArrayList(4);

		this.nextPair();
		if (dataString.equals("EOF")) {
			return false;
		}
		while (groupCode != 0) {
			switch (groupCode) {
				case 10:
					x1 = dataNumber.doubleValue();
					break;
				case 20:
					y1 = dataNumber.doubleValue();
					break;
				case 30:
					z1 = dataNumber.doubleValue();
					break;
				case 11:
					x2 = dataNumber.doubleValue();
					break;
				case 21:
					y2 = dataNumber.doubleValue();
					break;
				case 31:
					z2 = dataNumber.doubleValue();
					break;
				case 12:
					x3 = dataNumber.doubleValue();
					break;
				case 22:
					y3 = dataNumber.doubleValue();
					break;
				case 32:
					z3 = dataNumber.doubleValue();
					break;
				case 13:
					isQuad = true;
					x4 = dataNumber.doubleValue();
					break;
				case 23:
					isQuad = true;
					y4 = dataNumber.doubleValue();
					break;
				case 33:
					isQuad = true;
					z4 = dataNumber.doubleValue();
					break;
			}
			this.nextPair();
			if (dataString.equals("EOF")) {
				return false;
			}
		}

		vertexes.add(new Jun3dPoint(x1, y1, z1));
		vertexes.add(new Jun3dPoint(x2, y2, z2));
		vertexes.add(new Jun3dPoint(x3, y3, z3));
		if (isQuad) {
			vertexes.add(new Jun3dPoint(x4, y4, z4));
		}

		JunOpenGL3dPolygon newItem = new JunOpenGL3dPolygon(vertexes);
		for (int i = 0; i < vertexes.size(); i++) {
			this.count_($("vertex"));
		}
		this.unNextPair();
		newItem.name_(markName);
		item = new Object[] { layer, newItem };
		this.count_($("polygon"));

		return true;
	}

	/**
	 * Parse an ARC.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseARC() {
		double x = 0;
		double y = 0;
		double z = 0;
		double r = 0;
		double a1 = 0;
		double a2 = 0;
		String markName = String.valueOf(mark);
		this.nextPair();
		if (dataString.equals("EOF")) {
			return false;
		}
		while (groupCode != 0) {
			switch (groupCode) {
				case 8:
					layer = dataString;
					break;
				case 10:
					x = dataNumber.doubleValue();
					break;
				case 20:
					y = dataNumber.doubleValue();
					break;
				case 30:
					z = dataNumber.doubleValue();
					break;
				case 40:
					r = dataNumber.doubleValue();
					break;
				case 50:
					a1 = dataNumber.doubleValue();
					break;
				case 51:
					a2 = dataNumber.doubleValue();
					break;
			}
			this.nextPair();
			if (dataString.equals("EOF")) {
				return false;
			}
		}

		JunOpenGL3dPolyline newItem = new JunOpenGL3dPolyline(CalculateARC_radius_from_to_(new Jun3dPoint(x, y, z), r, a1, a2));
		this.unNextPair();
		newItem.name_(markName);
		item = new Object[] { layer, newItem };
		this.count_($("arc"));

		return true;
	}

	/**
	 * Parse a block.
	 * 
	 * @return boolean
	 * @category parsing
	 */
	public boolean parseBlock() {
		while ((groupCode != 0) || (dataString.equals("ENDBLK") == false)) {
			this.nextPair();
			if ((groupCode == 3) && (dataString.charAt(0) != '$')) {
				this.newDxfDefinition_(dataString);
			}
		}
		return true;
	}

	/**
	 * Parse a block section.
	 * 
	 * @return boolean
	 * @category parsing
	 */
	public boolean parseBlocks() {
		while ((groupCode != 0) || (dataString.equals("ENDSEC") == false)) {
			this.nextPair();
			if ((groupCode == 0) && dataString.equals("BLOCK")) {
				this.parseBlock();
			}
		}
		return true;
	}

	/**
	 * Parse a CIRCLE.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseCIRCLE() {
		double x = 0;
		double y = 0;
		// double z = 0;
		double r = 0;
		double[] weights = new double[] { 1.0, 0.707107, 1.0, 0.707107, 1.0, 0.707107, 1.0, 0.707107, 1.0 };
		double[] knots = new double[] { 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 4 };
		Jun3dPoint[] controls = new Jun3dPoint[9];
		String markName = String.valueOf(mark);
		this.nextPair();
		if (dataString.equals("EOF")) {
			return false;
		}
		while (groupCode != 0) {
			switch (groupCode) {
				case 8:
					layer = dataString;
					break;
				case 10:
					x = dataNumber.doubleValue();
					break;
				case 20:
					y = dataNumber.doubleValue();
					break;
				case 30:
					// z = dataNumber.doubleValue();
					break;
				case 40:
					r = dataNumber.doubleValue();
					break;
			}
			this.nextPair();
			if (dataString.equals("EOF")) {
				return false;
			}
		}
		// this.debug_("parseCIRCLE got " + x + " " + y + " " + z);

		JunOpenGL3dObject newItem = null;
		if (this.doNurbs()) {
			controls[0] = new Jun3dPoint(x, (y - r), 0);
			controls[1] = new Jun3dPoint(x + r, (y - r), 0);
			controls[2] = new Jun3dPoint(x + r, y, 0);
			controls[3] = new Jun3dPoint(x + r, (y + r), 0);
			controls[4] = new Jun3dPoint(x, (y + r), 0);
			controls[5] = new Jun3dPoint(x - r, (y + r), 0);
			controls[6] = new Jun3dPoint(x - r, y, 0);
			controls[7] = new Jun3dPoint(x - r, (y - r), 0);
			controls[8] = new Jun3dPoint(x, (y - r), 0);
			newItem = new JunOpenGL3dNurbsCurve(controls, weights, knots);
		} else {
			double pi = Math.PI;
			double pi2 = Math.PI * 2;
			double delta = pi / 16.0;
			double angle1 = 0;
			double angle2 = pi2;
			Vector points = new Vector(16);
			double xx = angle1;
			while (xx < (angle2 + delta)) {
				double nx = x + (Math.cos(xx) * r);
				double ny = y + (Math.sin(xx) * r);
				points.add(new Jun3dPoint(nx, ny, 0));
				xx += delta;
			}
			newItem = new JunOpenGL3dPolyline(points);
		}

		this.unNextPair();
		newItem.name_(markName);
		item = new Object[] { layer, newItem };
		this.count_($("circle"));

		return true;
	}

	/**
	 * Parse classes section.
	 * 
	 * @return boolean
	 * @category parsing
	 */
	public boolean parseClasses() {
		while ((groupCode != 0) || (dataString.equals("ENDSEC") == false)) {
			this.nextPair();
		}
		return true;
	}

	/**
	 * Answer true if the parsing was succeeded, otherwise false.
	 * 
	 * @return boolean
	 * @category parsing
	 */
	public boolean parsedOk() {
		return happy;
	}

	/**
	 * Parse entities section.
	 * 
	 * @return boolean
	 * @category parsing
	 */
	public boolean parseEntities() {
		Hashtable blocks = new Hashtable(5);
		blocks.put("POLYLINE", "parsePOLYLINE");
		blocks.put("LINE", "parseLINE");
		blocks.put("ARC", "parseARC");
		blocks.put("CIRCLE", "parseCIRCLE");
		blocks.put("INSERT", "parseINSERT");
		blocks.put("3DFACE", "parse3DFACE");

		definedObjects = new Vector(10);
		this.setBoundingBox();

		while (dataString.equals("ENDSEC") == false) {
			boolean rc = false;
			this.nextPair();
			if (groupCode == 0) {
				String selector = (String) blocks.get(dataString);
				if (selector != null) {
					try {
						Boolean result = (Boolean) this.perform_(selector);
						rc = result.booleanValue();
					} catch (Exception e) {
					}
					if (rc == true) {
						this.addLayeredItem_(item);
						this.count_($("entities"));
					}
				}
			}
		}
		this.nextPair();

		return true;
	}

	/**
	 * Parse an ENTITYCOLOR.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseENTITYCOLOR() {
		Number nn = null;
		this.nextPair();
		if (dataString.equals("EOF")) {
			return false;
		}
		while ((groupCode != 0) && (groupCode != 9)) {
			if (groupCode == 62) {
				nn = dataNumber;
			}
			this.nextPair();
			if (dataString.equals("EOF")) {
				return false;
			}
		}
		item = new Object[] { nn };
		this.unNextPair();

		return true;
	}

	/**
	 * Parse a EXTMAX.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseEXTMAX() {
		this.parseEXTMINMAX();
		this.settings().put("EXTMAX", item[0]);
		this.count_($("settings"));

		return true;
	}

	/**
	 * Parse a EXTMIN.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseEXTMIN() {
		this.parseEXTMINMAX();
		this.settings().put("EXTMIN", item[0]);
		this.count_($("settings"));

		return true;
	}

	/**
	 * Parse a header section.
	 * 
	 * @return boolean
	 * @category parsing
	 */
	public boolean parseHeader() {
		Hashtable blocks = new Hashtable(3);
		blocks.put("$EXTMIN", "parseEXTMIN");
		blocks.put("$EXTMAX", "parseEXTMAX");
		blocks.put("$CECOLOR", "parseENTITYCOLOR");
		while ((groupCode != 0) || (dataString.equals("ENDSEC") == false)) {
			if (groupCode == 9) {
				String selector = (String) blocks.get(dataString);
				if (selector != null) {
					try {
						this.perform_(selector);
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			}
			this.nextPair();
		}

		return true;
	}

	/**
	 * Parse an INSERT.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseINSERT() {
		double x = 0;
		double y = 0;
		double z = 0;
		double extrudeX = 0;
		double extrudeY = 0;
		double extrudeZ = 0;
		double xScale = 0;
		double yScale = 0;
		double zScale = 0;
		double angle = 0;
		String defName = "";
		String markName = String.valueOf(mark);
		this.nextPair();
		if (dataString.equals("EOF")) {
			return false;
		}
		while (groupCode != 0) {
			switch (groupCode) {
				case 2:
					defName = dataString;
					break;
				case 8:
					layer = dataString;
					break;
				case 10:
					x = dataNumber.doubleValue();
					break;
				case 20:
					y = dataNumber.doubleValue();
					break;
				case 30:
					z = dataNumber.doubleValue();
					break;
				case 41:
					xScale = dataNumber.doubleValue();
					break;
				case 42:
					yScale = dataNumber.doubleValue();
					break;
				case 43:
					zScale = dataNumber.doubleValue();
					break;
				case 50:
					angle = dataNumber.doubleValue();
					break;
			}
			this.nextPair();
			if (dataString.equals("EOF")) {
				return false;
			}
		}

		JunOpenGL3dObject newItem = (JunOpenGL3dObject) dxfNames.get(defName);
		newItem = newItem.scaledBy_(new Jun3dPoint(xScale, yScale, zScale));
		if (angle != 0) {
			newItem = newItem.rotatedBy_(JunAngle.FromDeg_(angle));
		}
		newItem = newItem.translatedBy_(new Jun3dPoint(x, y, z));
		if ((extrudeX != 0) || (extrudeY != 0) || (extrudeZ != 0)) {
			newItem = this.applyExtrusion_to_(new Jun3dPoint(extrudeX, extrudeY, extrudeZ), newItem);
		}
		this.unNextPair();
		newItem.name_(markName);
		item = new Object[] { layer, newItem };
		this.count_($("insert"));

		return true;
	}

	/**
	 * Parse a LAYER. Answer a color for this layer.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseLAYER() {
		String name = "";
		Number color = new Integer(0);
		this.nextPair();
		if (dataString.equals("EOF")) {
			return false;
		}
		while (groupCode != 0) {
			switch (groupCode) {
				case 62:
					color = dataNumber;
					break;
				case 2:
					name = dataString;
					break;
			}
			this.nextPair();
			if (dataString.equals("EOF")) {
				return false;
			}
		}

		this.setLayerColor_to_(name, color.intValue());
		this.count_($("layers"));
		item = new Object[] { name, color };
		this.unNextPair();

		return true;
	}

	/**
	 * Parse a LINE.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseLINE() {
		double x = 0;
		double y = 0;
		double z = 0;
		double x2 = 0;
		double y2 = 0;
		double z2 = 0;
		int color = 0;
		String markName = String.valueOf(mark);
		this.nextPair();
		if (dataString.equals("EOF")) {
			return false;
		}
		while (groupCode != 0) {
			switch (groupCode) {
				case 62:
					color = dataNumber.intValue();
					break;
				case 8:
					layer = dataString;
					break;
				case 10:
					x = dataNumber.doubleValue();
					break;
				case 20:
					y = dataNumber.doubleValue();
					break;
				case 30:
					z = dataNumber.doubleValue();
					break;
				case 11:
					x2 = dataNumber.doubleValue();
					break;
				case 21:
					y2 = dataNumber.doubleValue();
					break;
				case 31:
					z2 = dataNumber.doubleValue();
					break;
			}
			this.nextPair();
			if (dataString.equals("EOF")) {
				return false;
			}
		}

		JunOpenGL3dPolyline newItem = new JunOpenGL3dPolyline(new Jun3dPoint[] { new Jun3dPoint(x, y, z), new Jun3dPoint(x2, y2, z2) });
		if (color > 0) {
			newItem.paint_(this.paintColor_(color));
		}
		this.unNextPair();
		newItem.name_(markName);
		item = new Object[] { layer, newItem };
		this.count_($("line"));

		return true;
	}

	/**
	 * Parse objects section.
	 * 
	 * @return boolean
	 * @category parsing
	 */
	public boolean parseObjects() {
		while ((groupCode != 0) || (dataString.equals("ENDSEC") == false)) {
			this.nextPair();
		}
		return true;
	}

	/**
	 * Parse a POLYLINE.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parsePOLYLINE() {
		String markName = String.valueOf(mark);
		boolean isClosed = false;
		boolean rc = false;
		String layer = "";
		Vector vertexes = new Vector(10);
		double extrudeX = 0;
		double extrudeY = 0;
		double extrudeZ = 0;
		this.nextPair();
		while ((dataString.equals("SEQEND") == false) && (dataString.equals("EOF") == false)) {
			rc = false;
			switch (groupCode) {
				case 8:
					layer = dataString;
					break;
				case 70:
					isClosed = ((dataNumber.longValue() & 1) == 1);
					break;
				case 210:
					extrudeX = dataNumber.doubleValue();
					break;
				case 220:
					extrudeY = dataNumber.doubleValue();
					break;
				case 230:
					extrudeZ = dataNumber.doubleValue();
					break;
			}
			if (dataString.equals("VERTEX")) {
				rc = this.parseVERTEX();
				if (rc) {
					vertexes.add(this.itemValue());
				}
			} else {
				this.nextPair();
			}
		}

		JunOpenGL3dObject newItem = isClosed ? new JunOpenGL3dPolylineLoop(vertexes) : new JunOpenGL3dPolyline(vertexes);
		if ((extrudeX != 0) || (extrudeY != 0) || (extrudeZ != 0)) {
			newItem = this.applyExtrusion_to_(new Jun3dPoint(extrudeX, extrudeY, extrudeZ), newItem);
		}
		newItem.name_(markName);
		item = new Object[] { layer, newItem };
		this.count_($("polyline"));

		return true;
	}

	/**
	 * Parse a table section.
	 * 
	 * @return boolean
	 * @category parsing
	 */
	public boolean parseTables() {
		Hashtable blocks = new Hashtable(1);
		blocks.put("LAYER", "parseLAYER");
		while (dataString.equals("ENDSEC") == false) {
			this.nextPair();
			if (groupCode == 0) {
				String selector = (String) blocks.get(dataString);
				if (selector != null) {
					try {
						this.perform_(selector);
					} catch (Exception e) {
					}
				}
			}
		}

		return true;
	}

	/**
	 * Parse a VERTEX.
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	public boolean parseVERTEX() {
		double x = 0;
		double y = 0;
		double z = 0;

		this.nextPair();

		if (dataString.equals("EOF")) {
			return false;
		}

		while (groupCode != 0) {
			switch (groupCode) {
				case 8:
					layer = dataString;
					break;
				case 10:
					x = dataNumber.doubleValue();
					break;
				case 20:
					y = dataNumber.doubleValue();
					break;
				case 30:
					z = dataNumber.doubleValue();
					break;
			}

			this.nextPair();

			if (dataString.equals("EOF")) {
				return false;
			}
		}

		item = new Object[] { layer, new Jun3dPoint(x, y, z) };
		this.count_($("vertex"));

		return true;
	}

	/**
	 * Show the layers on the new JunCADModel.
	 * 
	 * @category graphics
	 */
	public void showCAD() {
		JunCADModel cad = new JunCADModel();
		cad.open();
		cad.openParser_(this);
	}

	/**
	 * Add the specified item to an appropriate layer.
	 * 
	 * @param anItem java.lang.Object[]
	 * @category graphics
	 */
	protected void addLayeredItem_(Object[] anItem) {
		String layerName = (String) anItem[0];
		JunOpenGL3dObject layerItem = (JunOpenGL3dObject) anItem[1];
		if (layerItem.isCompound() && (((JunOpenGL3dCompoundObject) layerItem).components().length == 0)) {
			throw SmalltalkException.Error("internal error");
		}
		if (layers.containsKey(layerName) == false) {
			layers.put(layerName, new JunOpenGL3dCompoundObject());
		}
		if (this.contained_(layerItem)) {
			((JunOpenGL3dCompoundObject) layers.get(layerName)).add_(layerItem);
		} else {
			this.outside().add(layerItem);
			this.count_($("outside"));
		}
	}

	/**
	 * Answer the given object, transformed.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category parsing entities
	 */
	protected JunOpenGL3dObject applyExtrusion_to_(Jun3dPoint aPoint, JunOpenGL3dObject anObject) {
		Jun3dPoint v1 = new Jun3dPoint(0, 0, 1);
		Jun3dPoint ev = aPoint.unitVector();
		Jun3dPoint av = v1.product_(ev);
		Jun3dLine rl = new Jun3dLine(Jun3dPoint.Zero(), av);
		double th = Math.acos(v1.dotProduct_(ev));
		Jun3dTransformation tf = Jun3dTransformation.Rotate_around_(JunAngle.FromRad_(th), rl);
		return anObject.transform_(tf);
	}

	/**
	 * Answer true if the specified item is contained in the current bounding box.
	 * 
	 * @param anItem jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @return boolean
	 * @category accessing
	 */
	protected boolean contained_(JunOpenGL3dObject anItem) {
		Jun3dBoundingBox bb = (Jun3dBoundingBox) this.settings().get("box");
		if (bb == null) {
			return true;
		}
		return bb.contains_(anItem.boundingBox());
	}

	/**
	 * Count for the symbol.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category counting
	 */
	protected void count_(StSymbol aSymbol) {
		Number number = (Number) itemCount.get(aSymbol);
		if (number == null) {
			itemCount.put(aSymbol, new Integer(1));
		} else {
			itemCount.put(aSymbol, new Integer(number.intValue() + 1));
		}
	}

	/**
	 * Answer true if use NURBS, otherwise false.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	protected boolean doNurbs() {
		return doNurbs;
	}

	/**
	 * Initialize the count information.
	 * 
	 * @category initialize-release
	 */
	protected void initCounts() {
		itemCount = new Hashtable(10);
	}

	/**
	 * Initialize the JunDXFParser.
	 * 
	 * @see jp.co.sra.jun.dxf.support.JunDXFScanner#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.initParser();
		this.initCounts();
	}

	/**
	 * Initialize the parser information.
	 * 
	 * @category initialize-release
	 */
	protected void initParser() {
		layers = new Hashtable(20);
		dxfNames = new Hashtable(20);
	}

	/**
	 * Answer the key of the current item.
	 * 
	 * @return java.lang.Object
	 * @category accessing
	 */
	protected Object itemKey() {
		return item[0];
	}

	/**
	 * Answer the value of the current item.
	 * 
	 * @return java.lang.Object
	 * @category accessing
	 */
	protected Object itemValue() {
		return item[1];
	}

	/**
	 * Answer a Hashtable which contains colors for each layers.
	 * 
	 * @return java.util.Hashtable
	 * @category accessing
	 */
	protected Hashtable layerColors() {
		if (layerColors == null) {
			layerColors = new Hashtable(50);
		}
		return layerColors;
	}

	/**
	 * Answer a color for the specified layer.
	 * 
	 * @param aString java.lang.String
	 * @return java.awt.Color
	 * @category accessing
	 */
	protected Color layerColorsAt_(String aString) {
		Color color = (Color) this.layerColors().get(aString);
		if (color == null) {
			color = Color.blue;
		}
		return color;
	}

	/**
	 * Answer the layers table.
	 * 
	 * @return java.util.Hashtable
	 * @category accessing
	 */
	protected Hashtable layers() {
		return layers;
	}

	/**
	 * Parse a new DXF definition.
	 * 
	 * @param aString java.lang.String
	 * @return boolean
	 * @category parsing
	 */
	protected boolean newDxfDefinition_(String aString) {
		Hashtable blocks = new Hashtable(4);
		blocks.put("POLYLINE", "parsePOLYLINE");
		blocks.put("LINE", "parseLINE");
		blocks.put("ARC", "parseARC");
		blocks.put("CIRCLE", "parseCIRCLE");

		definedObjects = new Vector(10);

		while (dataString.equals("ENDBLK") == false) {
			boolean rc = false;
			this.nextPair();
			if (groupCode == 0) {
				String selector = (String) blocks.get(dataString);
				if (selector != null) {
					try {
						Boolean result = (Boolean) this.perform_(selector);
						rc = result.booleanValue();
					} catch (Exception e) {
					}
					if (rc == true) {
						definedObjects.add(item[1]);
					}
				}
			}
		}

		this.count_($("names"));
		dxfNames.put(aString, new JunOpenGL3dCompoundObject(definedObjects));

		return true;
	}

	/**
	 * Answer the collection which contains the out-of-the-layers objects.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	protected Vector outside() {
		if (outside == null) {
			outside = new Vector(50);
		}

		return outside;
	}

	/**
	 * Answer a color specified with the index number.
	 * 
	 * @param anInteger int
	 * @return java.awt.Color
	 * @category accessing
	 */
	protected Color paintColor_(int anInteger) {
		int index = anInteger;
		if ((index < 1) || (7 < index)) {
			index = 1;
		}
		return ColorAt_(index);
	}

	/**
	 * Parse the source stream.
	 * 
	 * @category parsing
	 */
	protected void parse() {
		Hashtable blocks = new Hashtable(5);
		blocks.put("HEADER", "parseHeader");
		blocks.put("CLASSES", "parseClasses");
		blocks.put("TABLES", "parseTables");
		blocks.put("BLOCKS", "parseBlocks");
		blocks.put("ENTITIES", "parseEntities");
		blocks.put("OBJECTS", "parseObjects");

		happy = false;
		this.nextPair();

		while (dataString.equals("EOF") == false) {
			if (groupCode != 0) {
				failBlock.value_("invalid groupcode (not 0)");
				return;
			}
			if (dataString.equals("SECTION") == false) {
				failBlock.value_("invalid section string (not SECTION)");
				return;
			}
			this.nextPair();
			if (groupCode != 2) {
				failBlock.value_("invalid section groupcode (not 2)");
				return;
			}
			boolean rc = true;
			String selector = (String) blocks.get(dataString);
			try {
				Boolean result = (Boolean) this.perform_(selector);
				rc = result.booleanValue();
			} catch (Exception e) {
				rc = false;
			}
			if (rc == true) {
				this.count_($("sections"));
			}
			this.nextPair();
		}

		//this.showCounts();
		this.setLayerNames();
		this.updateLayerColors();
		happy = true;
	}

	/**
	 * Parse a EXTMIN or a EXTMAX
	 * 
	 * @return boolean
	 * @category parsing entities
	 */
	protected boolean parseEXTMINMAX() {
		double x = 0;
		double y = 0;
		double z = 0;
		this.nextPair();
		if (dataString.equals("EOF")) {
			return false;
		}
		while ((groupCode != 0) && (groupCode != 9)) {
			switch (groupCode) {
				case 10:
					x = dataNumber.doubleValue();
					break;
				case 20:
					y = dataNumber.doubleValue();
					break;
				case 30:
					z = dataNumber.doubleValue();
					break;
			}
			this.nextPair();
			if (dataString.equals("EOF")) {
				return false;
			}
		}

		Jun3dPoint newItem = new Jun3dPoint(x, y, z);
		item = new Object[] { newItem };
		this.unNextPair();

		return true;
	}

	/**
	 * Set the current bounding box.
	 * 
	 * @category graphics
	 */
	protected void setBoundingBox() {
		Jun3dPoint b1 = (Jun3dPoint) this.settings().get("EXTMIN");
		Jun3dPoint b2 = (Jun3dPoint) this.settings().get("EXTMAX");
		if ((b1 != null) && (b2 != null)) {
			this.settings().put("box", Jun3dBoundingBox.Origin_corner_(b1, b2));
		}
	}

	/**
	 * Set the color of the named layer.
	 * 
	 * @param aName java.lang.String
	 * @param anInteger int
	 * @category accessing
	 */
	protected void setLayerColor_to_(String aName, int anInteger) {
		this.layerColors().put(aName, ColorAt_(anInteger));
	}

	/**
	 * Set layer names to the objects which belong to each layer.
	 * 
	 * @category accessing
	 */
	protected void setLayerNames() {
		Enumeration keys = this.layers().keys();

		while (keys.hasMoreElements()) {
			String key = (String) keys.nextElement();
			JunOpenGL3dObject object = (JunOpenGL3dObject) this.layers().get(key);
			object.name_(key);
		}
	}

	/**
	 * Answer the settings.
	 * 
	 * @return java.util.Hashtable
	 * @category accessing
	 */
	protected Hashtable settings() {
		if (settings == null) {
			settings = new Hashtable(10);
		}

		return settings;
	}

	/**
	 * Set layer colors to the objects which belong to each layer.
	 * 
	 * @category accessing
	 */
	protected void updateLayerColors() {
		Enumeration keys = this.layers().keys();

		while (keys.hasMoreElements()) {
			String key = (String) keys.nextElement();
			JunOpenGL3dObject object = (JunOpenGL3dObject) this.layers().get(key);
			object.paint_(this.layerColorsAt_(key));
		}
	}
}
