package jp.co.sra.jun.geometry.basic;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * JunAngle class
 * 
 *  @author    nisinaka
 *  @created   1998/10/05 (by nisinaka)
 *  @updated   2004/10/19 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun678 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunAngle.java,v 8.16 2008/02/20 06:30:55 nisinaka Exp $
 */
public class JunAngle extends JunGeometry implements Comparable {

	/** Some useful values. */
	protected static final double _RadiansPerDegree = Math.PI / 180.0d;

	protected double rad;
	protected Double deg = null;
	protected Double sin = null;
	protected Double cos = null;
	protected Double tan = null;

	/**
	 * Convert the value from degrees to radians.
	 * 
	 * @param degrees double
	 * @return double
	 * @category Utilities
	 */
	public static double _DegreesToRadians(double degrees) {
		return degrees * _RadiansPerDegree;
	}

	/**
	 * Convert the value from radians to degrees.
	 * 
	 * @param radians double
	 * @return double
	 * @category Utilities
	 */
	public static double _RadiansToDegrees(double radians) {
		return radians / _RadiansPerDegree;
	}

	/**
	 * Create a new JunAngle with the degrees.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param degrees double
	 * @category Instance creation
	 */
	public static final JunAngle FromDeg_(double degrees) {
		JunAngle anAngle = new JunAngle();
		anAngle.deg_(degrees);
		return anAngle;
	}

	/**
	 * Create a new JunAngle with the radians.
	 * 
	 * @param radians double
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Instance creation
	 */
	public static final JunAngle FromRad_(double radians) {
		JunAngle anAngle = new JunAngle();
		anAngle.rad_(radians);
		return anAngle;
	}

	/**
	 * Answer a constant, double PI, angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Constants access
	 */
	public static JunAngle DoublePiAngle() {
		return JunAngle.FromDeg_(JunGeometry.DoublePi());
	}

	/**
	 * Answer a constant, half PI, angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Constants access
	 */
	public static JunAngle HalfPiAngle() {
		return JunAngle.FromDeg_(JunGeometry.HalfPi());
	}

	/**
	 * Answer a constant, double PI, angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Constants access
	 */
	public static JunAngle PiAngle() {
		return JunAngle.FromDeg_(JunGeometry.Pi());
	}

	/**
	 * Answer a constant, double PI, angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Constants access
	 */
	public static JunAngle QuarterPiAngle() {
		return JunAngle.FromDeg_(JunGeometry.QuarterPi());
	}

	/**
	 * Answer a constant, unity, angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category Constants access
	 */
	public static JunAngle Unity() {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Answer a constant, zero, angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Constants access
	 */
	public static JunAngle Zero() {
		return JunAngle.FromRad_(0);
	}

	/**
	 * Convert to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		JunOpenGL3dObject aBall = JunOpenGL3dObject.PieFrom_to_by_radius_(0, this.radiansToDegrees(), 5, 1);
		compoundObject.add_(aBall.reversed());
		compoundObject.add_(aBall);
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunOpenGL3dObject) each).paint_alpha_(JunAngle.this.defaultColor(), JunAngle.this.defaultAlpha());
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Compares this object with the specified object for order.  Returns a
	 * negative integer, zero, or a positive integer as this object is less
	 * than, equal to, or greater than the specified object.
	 * 
	 * @param o java.lang.Object
	 * @return int
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 * @category comparing
	 */
	public int compareTo(Object o) {
		double value = Double.NaN;

		if (o instanceof JunAngle) {
			value = ((JunAngle) o).rad();
		} else if (o instanceof Number) {
			value = ((Number) o).doubleValue();
		}

		if (!Double.isNaN(value)) {
			if (this.rad() < value) {
				return -1;
			} else if (value < this.rad()) {
				return 1;
			} else {
				return 0;
			}
		}

		return ((Comparable) o).compareTo(new Double(this.rad()));
	}

	/**
	 * Answer as cos.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double cos() {
		if (cos == null) {
			cos = new Double(Math.cos(rad));
		}

		return cos.doubleValue();
	}

	/**
	 * Answer as degrees.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double deg() {
		if (deg == null) {
			deg = new Double(_RadiansToDegrees(rad));
		}

		return deg.doubleValue();
	}

	/**
	 * Set this angle from degrees.
	 * 
	 * @param degrees double
	 * @category accessing
	 */
	public void deg_(double degrees) {
		this.setRad_(_DegreesToRadians(degrees));
	}

	/**
	 * Answer my value as degrees.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double degrees() {
		return this.deg();
	}

	/**
	 * Set my value as degrees.
	 * 
	 * @param degrees double
	 * @category accessing
	 */
	public void degrees_(double degrees) {
		this.deg_(degrees);
	}

	/**
	 * Divide the receiver by aNumber.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aNumber double
	 * @category arithmetic
	 */
	public JunAngle div_(double aNumber) {
		return JunAngle.FromRad_(rad / aNumber);
	}

	/**
	 * Answer the result of dividing the receiver by delta.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param delta double
	 * @category arithmetic
	 */
	public JunAngle dividedBy_(double delta) {
		return JunAngle.FromRad_(rad / delta);
	}

	/**
	 * Answer the result of dividing the receiver by the JunAngle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category arithmetic
	 */
	public JunAngle dividedBy_(JunAngle anAngle) {
		return this.dividedBy_(anAngle.rad());
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		JunAngle anAngle = (JunAngle) anObject;
		return this.isEqualNumber_to_(rad, anAngle.rad);
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		JunAngle anAngle = (JunAngle) anObject;
		return rad == anAngle.rad;
	}

	/**
	 * Answer the number representing the ordering of the receiver in the
	 * generality hierarchy.
	 * 
	 * @return int
	 * @category coercing
	 */
	public int generality() {
		return 120;
	}

	/**
	 * Answer true if the receiver is an angle geometry element, otherwise
	 * false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isAngle()
	 * @category testing
	 */
	public boolean isAngle() {
		return true;
	}

	/**
	 * Answer true if this angle is ZERO, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isZero() {
		return this.rad() == 0.0d;
	}

	/**
	 * Answer the difference between the receiver and delta.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param delta double
	 * @category arithmetic
	 */
	public JunAngle minus_(double delta) {
		return JunAngle.FromRad_(rad - delta);
	}

	/**
	 * Answer the difference between the receiver and the JunAngle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category arithmetic
	 */
	public JunAngle minus_(JunAngle anAngle) {
		return this.minus_(anAngle.rad());
	}

	/**
	 * Answer an instance of JunAngle which angle is multiplied by the
	 * specified number.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aNumber double
	 * @category arithmetic
	 */
	public JunAngle mul_(double aNumber) {
		return JunAngle.FromRad_(rad * aNumber);
	}

	/**
	 * Answer the result of multiplying the receiver by delta.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param delta double
	 * @category arithmetic
	 */
	public JunAngle multipliedBy_(double delta) {
		return JunAngle.FromRad_(rad * delta);
	}

	/**
	 * Answer the result of multiplying the receiver by the JunAngle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category arithmetic
	 */
	public JunAngle multipliedBy_(JunAngle anAngle) {
		return this.multipliedBy_(anAngle.rad());
	}

	/**
	 * Answer the negation of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category arithmetic
	 */
	public JunAngle negated() {
		return JunAngle.FromRad_(-rad);
	}

	/**
	 * Answer the sum of the receiver and delta.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param delta double
	 * @category arithmetic
	 */
	public JunAngle plus_(double delta) {
		return JunAngle.FromRad_(rad + delta);
	}

	/**
	 * Answer the sum of the receiver and the JunAngle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category arithmetic
	 */
	public JunAngle plus_(JunAngle anAngle) {
		return this.plus_(anAngle.rad());
	}

	/**
	 * Answer the reciprocal of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category arithmetic
	 */
	public JunAngle reciprocal() {
		return JunAngle.FromDeg_(1.0d / this.rad());
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		aWriter.write(" fromDeg: ");
		aWriter.write(String.valueOf(this.deg()));
		aWriter.write(')');
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		aWriter.write(" fromRad: ");
		aWriter.write(String.valueOf(this.rad()));
		aWriter.write(')');
	}

	/**
	 * Answer as radians.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double rad() {
		return rad;
	}

	/**
	 * Convert from radians to degrees.
	 * 
	 * @return double
	 * @category converting
	 */
	public double radiansToDegrees() {
		return this.deg();
	}

	/**
	 * Set this angle from radians.
	 * 
	 * @param radians double
	 * @category accessing
	 */
	public void rad_(double radians) {
		this.setRad_(radians);
	}

	/**
	 * Answer my value as radians.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double radians() {
		return this.rad();
	}

	/**
	 * Set my value as radians.
	 * 
	 * @param radians double
	 * @category accessing
	 */
	public void radians_(double radians) {
		this.rad_(radians);
	}

	/**
	 * Answer as sin.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double sin() {
		if (sin == null) {
			sin = new Double(Math.sin(rad));
		}

		return sin.doubleValue();
	}

	/**
	 * Answer as tan.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double tan() {
		if (tan == null) {
			tan = new Double(Math.tan(rad));
		}

		return tan.doubleValue();
	}

	/**
	 * Answer the transformation to rotate around this.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming
	 */
	public Jun2dTransformation transformationToRotate() {
		return Jun2dTransformation.Rotate_(this);
	}

	/**
	 * Answer the transformation to rotate around the line.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTransformation transformationToRotate_(Jun3dLine aLine) {
		return aLine.transformationToRotate_(this);
	}

	/**
	 * Answer the transformation to rotate around the receiver's x.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTransformation transformationToRotateX() {
		return Jun3dTransformation.RotateX_(this);
	}

	/**
	 * Answer the transformation to rotate around the receiver's x.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTransformation transformationToRotateY() {
		return Jun3dTransformation.RotateY_(this);
	}

	/**
	 * Answer the transformation to rotate around the receiver's x.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTransformation transformationToRotateZ() {
		return Jun3dTransformation.RotateZ_(this);
	}

	/**
	 * Set the new value of 'rad' and reset other cached values.
	 * 
	 * @param radians double
	 * @category private
	 */
	private void setRad_(double radians) {
		rad = radians;
		deg = null;
		sin = null;
		cos = null;
		tan = null;
	}

}
