package jp.co.sra.jun.geometry.curves;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.TreeMap;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.geometry.surfaces.Jun3dPolygon;
import jp.co.sra.jun.geometry.surfaces.Jun3dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;

/**
 * Jun3dPolyline class
 * 
 *  @author    nisinaka
 *  @created   2006/08/18 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun692 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dPolyline.java,v 8.18 2008/02/20 06:30:57 nisinaka Exp $
 */
public class Jun3dPolyline extends JunPolyline {

	protected Jun3dPoint[] points;
	protected Jun3dBoundingBox boundingBox;
	protected double[] segmentLengths;

	/**
	 * Create a new instance of <code>Jun3dPolyline</code> and initialize it.
	 *
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public Jun3dPolyline(Jun3dPoint aPoint1, Jun3dPoint aPoint2) {
		this(new Jun3dPoint[] { aPoint1, aPoint2 });
	}

	/**
	 * Create a new instance of <code>Jun3dPolyline</code> and initialize it.
	 *
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public Jun3dPolyline(Jun3dPoint aPoint1, Jun3dPoint aPoint2, Jun3dPoint aPoint3) {
		this(new Jun3dPoint[] { aPoint1, aPoint2, aPoint3 });
	}

	/**
	 * Create a new instance of <code>Jun3dPolyline</code> and initialize it.
	 *
	 * @param points jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public Jun3dPolyline(Jun3dPoint[] points) {
		this.setPoints_(points);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		points = null;
		boundingBox = null;
		segmentLengths = null;
	}

	/**
	 * Answer my points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] points() {
		if (points == null) {
			points = new Jun3dPoint[0];
		}
		return points;
	}

	/**
	 * Answer my points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @see jp.co.sra.jun.geometry.curves.JunPolyline#_points()
	 * @category accessing
	 */
	protected JunPoint[] _points() {
		return this.points();
	}

	/**
	 * Answer the point at the specified index.
	 * 
	 * @param index int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint pointAt_(int index) {
		return this.points()[index];
	}

	/**
	 * Answer a regularized point the polyline at the specified number.
	 * 
	 * @param t double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atT_(double t) {
		if (t <= 0) {
			return this.pointAt_(0);
		} else if (1 <= t) {
			return this.pointAt_(this.numberOfPoints() - 1);
		}

		double targetLength = this.length() * t;
		for (int i = 1; i < this.numberOfPoints(); i++) {
			double minLength = segmentLengths[i - 1];
			double maxLength = segmentLengths[i];
			if (minLength <= targetLength && targetLength <= maxLength) {
				Jun3dLine aLine = this.pointAt_(i - 1).to_(this.pointAt_(i));
				return aLine.atT_((targetLength - minLength) / (maxLength - minLength));
			}
		}

		throw SmalltalkException.Error("unexpected error");
	}

	/**
	 * Answer a point on the receiver at the specified x-coordinate.
	 * 
	 * @param xValue double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atX_(double xValue) {
		Jun3dLine[] lines = this.asArrayOfLines();
		for (int i = 0; i < lines.length; i++) {
			if (lines[i].from().x() <= xValue && xValue <= lines[i].to().x()) {
				return lines[i].atX_(xValue);
			}
		}

		throw SmalltalkException.Error("unexpected error");
	}

	/**
	 * Answer a point on the receiver at the specified y-coordinate.
	 * 
	 * @param yValue double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atY_(double yValue) {
		Jun3dLine[] lines = this.asArrayOfLines();
		for (int i = 0; i < lines.length; i++) {
			if (lines[i].from().y() <= yValue && yValue <= lines[i].to().y()) {
				return lines[i].atY_(yValue);
			}
		}

		throw SmalltalkException.Error("unexpected error");
	}

	/**
	 * Answer a point on the receiver at the specified z-coordinate.
	 * 
	 * @param zValue double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atZ_(double zValue) {
		Jun3dLine[] lines = this.asArrayOfLines();
		for (int i = 0; i < lines.length; i++) {
			if (lines[i].from().z() <= zValue && zValue <= lines[i].to().z()) {
				return lines[i].atZ_(zValue);
			}
		}

		throw SmalltalkException.Error("unexpected error");
	}

	/**
	 * Answer the point at the beginning.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint from() {
		return this.atT_(0.0d);
	}

	/**
	 * Answer the point at the end.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint to() {
		return this.atT_(1.0d);
	}

	/**
	 * Answer the number of the points.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfPoints() {
		return this.size();
	}

	/**
	 * Answer the length of the polyline.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double length() {
		if (segmentLengths == null) {
			double[] lengths = new double[this.numberOfPoints()];
			double length = 0;
			lengths[0] = length;
			for (int i = 1; i < this.numberOfPoints(); i++) {
				length += this.pointAt_(i - 1).distance_(this.pointAt_(i));
				lengths[i] = length;
			}
			segmentLengths = lengths;
		}
		return segmentLengths[segmentLengths.length - 1];
	}

	/**
	 * Answer my normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint normalVector() {
		Jun3dPoint vector = Jun3dPoint.Zero();
		int size = this.numberOfPoints();
		for (int i = 0; i < size; i++) {
			Jun3dPoint point1 = this.pointAt_(i);
			Jun3dPoint point2 = this.pointAt_((i + 1) % size);
			Jun3dPoint point3 = this.pointAt_((i + 2) % size);
			vector = vector.plus_(point2.minus_(point1).vectorProduct_(point3.minus_(point2)));
		}
		return vector;
	}

	/**
	 * Answer my normal unit vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint normalUnitVector() {
		return this.normalVector().unitVector();
	}

	/**
	 * Answer the receiver's size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return this.points().length;
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		if (boundingBox == null) {
			boundingBox = this.preferredBoundingBox();
		}
		return boundingBox;
	}

	/**
	 * Answer my preferred bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category bounds accessing
	 */
	protected Jun3dBoundingBox preferredBoundingBox() {
		double minX = Double.MAX_VALUE;
		double minY = Double.MAX_VALUE;
		double minZ = Double.MAX_VALUE;
		double maxX = Double.MIN_VALUE;
		double maxY = Double.MIN_VALUE;
		double maxZ = Double.MIN_VALUE;

		Jun3dPoint[] points = this.points();
		for (int i = 0; i < points.length; i++) {
			minX = Math.min(minX, points[i].x());
			minY = Math.min(minY, points[i].y());
			minZ = Math.min(minZ, points[i].z());
			maxX = Math.max(maxX, points[i].x());
			maxY = Math.max(maxY, points[i].y());
			maxZ = Math.max(maxZ, points[i].z());
		}

		if (minX == Double.MAX_VALUE) {
			minX = 0;
		}
		if (minY == Double.MAX_VALUE) {
			minY = 0;
		}
		if (minZ == Double.MAX_VALUE) {
			minZ = 0;
		}
		if (maxX == Double.MIN_VALUE) {
			maxX = 0;
		}
		if (maxY == Double.MIN_VALUE) {
			maxY = 0;
		}
		if (maxZ == Double.MIN_VALUE) {
			maxZ = 0;
		}

		return Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(minX, minY, minZ), new Jun3dPoint(maxX, maxY, maxZ));
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dPolyline aPolyline = (Jun3dPolyline) anObject;
		if (this.numberOfPoints() != aPolyline.numberOfPoints()) {
			return false;
		}
		for (int i = 0; i < this.numberOfPoints(); i++) {
			if (this.pointAt_(i).equal_(aPolyline.pointAt_(i)) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dPolyline aPolyline = (Jun3dPolyline) anObject;
		if (this.numberOfPoints() != aPolyline.numberOfPoints()) {
			return false;
		}
		for (int i = 0; i < this.numberOfPoints(); i++) {
			if (this.pointAt_(i).equals(aPolyline.pointAt_(i)) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Convert to an array of Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @see jp.co.sra.jun.geometry.curves.JunPolyline#as2dPoints()
	 * @category converting
	 */
	public Jun2dPoint[] as2dPoints() {
		Jun3dPoint[] _3dPoints = this.points();
		Jun2dPoint[] _2dPoints = new Jun2dPoint[_3dPoints.length];
		for (int i = 0; i < _2dPoints.length; i++) {
			_2dPoints[i] = _3dPoints[i].as2dPoint();
		}
		return _2dPoints;
	}

	/**
	 * Convert to an array of Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @see jp.co.sra.jun.geometry.curves.JunPolyline#as3dPoints()
	 * @category converting
	 */
	public Jun3dPoint[] as3dPoints() {
		return this.points();
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @category converting
	 */
	public Jun3dLine[] asArrayOfLines() {
		return this.asArrayOf3dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dLines()
	 * @category converting
	 */
	public Jun3dLine[] asArrayOf3dLines() {
		Jun3dPoint[] points = this.points();
		Jun3dLine[] lines = new Jun3dLine[points.length - 1];
		for (int i = 0; i < points.length - 1; i++) {
			lines[i] = points[i].to_(points[i + 1]);
		}
		return lines;
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category converting
	 */
	public Jun3dPoint[] asArrayOfPoints() {
		return this.points();
	}

	/**
	 * Convert the receiver as an array of the <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category converting
	 */
	public Jun3dTriangle[] asArrayOfTriangles() {
		return this.asArrayOf3dTriangles();
	}

	/**
	 * Convert the receiver as a <code>Jun3dPolygon</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @category converting
	 */
	public Jun3dPolygon asPolygon() {
		return Jun3dPolygon.Vertexes_(this.points());
	}

	/**
	 * Convert the receiver as a <code>Jun3dPolyline</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category converting
	 */
	public Jun3dPolyline asPolyline() {
		return this;
	}

	/**
	 * Convert the receiver as a <code>Jun3dPolygon</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon
	 * @category converting
	 */
	public Jun3dPolyline reversed() {
		Jun3dPoint[] points = this.points();
		Jun3dPoint[] reversePoints = new Jun3dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			reversePoints[reversePoints.length - 1 - i] = points[i];
		}
		return new Jun3dPolyline(reversePoints);
	}

	/**
	 * Convert to an array of Jun3dTriangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category converting
	 */
	public Jun3dTriangle[] asArrayOf3dTriangles() {
		if (this.numberOfPoints() < 3) {
			return new Jun3dTriangle[0];
		}

		return this.asArrayOf3dTrianglesOn_(JunPlane.On_normalVector_(this.from(), this.normalVector()));
	}

	/**
	 * Convert to an array of Jun3dTriangle on the plane.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category converting
	 */
	public Jun3dTriangle[] asArrayOf3dTrianglesOn_(JunPlane aPlane) {
		if (this.numberOfPoints() < 3) {
			return new Jun3dTriangle[0];
		}

		Object[] anArray = this.asArrayOf2dPointsAndTableOn_(aPlane);
		Jun2dPoint[] points = (Jun2dPoint[]) anArray[0];
		Map aTable = (Map) anArray[1];

		ArrayList aList = new ArrayList();
		Jun2dTriangle[] triangles = new Jun2dPolyline(points).asArrayOfTriangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun3dPoint p1 = (Jun3dPoint) aTable.get(triangles[i].p1());
			Jun3dPoint p2 = (Jun3dPoint) aTable.get(triangles[i].p2());
			Jun3dPoint p3 = (Jun3dPoint) aTable.get(triangles[i].p3());
			aList.add(Jun3dTriangle.On_on_on_(p1, p2, p3));
		}

		return (Jun3dTriangle[]) aList.toArray(new Jun3dTriangle[aList.size()]);
	}

	/**
	 * Convert to an array of 2D points on the plane and a mapping table.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return java.lang.Object[]
	 * @category converting
	 */
	protected Object[] asArrayOf2dPointsAndTableOn_(JunPlane aPlane) {
		ArrayList aList = new ArrayList();
		HashMap aMap = new HashMap();

		Jun3dPoint upVector = aPlane.normalUnitVector();
		Jun3dLine xLine = aPlane.p1().to_(aPlane.p2());
		Jun3dLine yLine = xLine.transform_(Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(90), Jun3dPoint.Zero().to_(upVector)));
		JunPlane yzPlane = JunPlane.On_vertical_(aPlane.p1(), xLine);
		JunPlane xzPlane = JunPlane.On_vertical_(aPlane.p1(), yLine);
		Jun3dPoint[] points = this.points();
		for (int i = 0; i < points.length; i++) {
			Jun3dLine aLine = points[i].to_(points[i].plus_(upVector));
			Jun3dPoint aPoint = aPlane.intersectingPointWithLine_(aLine);
			double xValue = yzPlane.distanceFromPoint_(aPoint) * aPoint.whichSideOf_(yzPlane);
			double yValue = xzPlane.distanceFromPoint_(aPoint) * aPoint.whichSideOf_(xzPlane);
			Jun2dPoint a2dPoint = new Jun2dPoint(xValue, yValue);
			aList.add(a2dPoint);
			aMap.put(a2dPoint, points[i]);
		}

		return new Object[] { aList.toArray(new Jun2dPoint[aList.size()]), aMap };
	}

	/**
	 * Answer the receiver's subdivide polyline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category subdividing
	 */
	public Jun3dPolyline subdivide() {
		Jun3dPoint[] points = this.points();
		Collection vertexes = new ArrayList(this.size() * 2 - 1);
		for (int i = 0; i < points.length - 1; i++) {
			vertexes.add(points[i]);
			vertexes.add(points[i].to_(points[i + 1]).center());
		}
		vertexes.add(points[points.length - 1]);
		Jun3dPolyline polyline = new Jun3dPolyline((Jun3dPoint[]) vertexes.toArray(new Jun3dPoint[vertexes.size()]));
		return polyline;
	}

	/**
	 * Answer the receiver's subdivide polyline with specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category subdividing
	 */
	public Jun3dPolyline subdivideLevel_(int levelNumber) {
		Jun3dPolyline polyline = (Jun3dPolyline) this.copy();
		for (int i = 0; i < levelNumber; i++) {
			polyline = polyline.subdivide();
		}
		return polyline;
	}

	/**
	 * Subdivide myself as a spline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category subdividing
	 */
	public Jun3dPolyline subdivideAsSpline() {
		return new Jun3dPolyline(this.computeSplinePoints_(this.points()));
	}

	/**
	 * Subdivide myself as a spline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @category subdividing
	 */
	public Jun3dPolyline subdivideAsSplineLevel_(int levelNumber) {
		Jun3dPolyline polyline = this;
		for (int i = 0; i < levelNumber; i++) {
			polyline = polyline.subdivideAsSpline();
		}
		return polyline;
	}

	/**
	 * Answer true if the receiver is a 3d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Sew with the triangles between the polylines.
	 * 
	 * @param aPolyline jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category sewing
	 */
	public Jun3dTriangle[] sew_(Jun3dPolyline aPolyline) {
		return this.sew_interim_(aPolyline, null);
	}

	/**
	 * Sew with the triangles between the polylines.
	 * 
	 * @param aPolyline jp.co.sra.jun.geometry.curves.Jun3dPolyline
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category sewing
	 */
	public Jun3dTriangle[] sew_interim_(Jun3dPolyline aPolyline, StBlockClosure aBlock) {
		ArrayList aList = new ArrayList();

		Jun3dPoint[] points = aPolyline.points();
		for (int i = 1; i < points.length; i++) {
			Jun3dPoint startPoint = points[i - 1];
			Jun3dPoint endPoint = points[i];

			int index = this.findIndexOfPoints_from_to_interim_(this.points(), startPoint, endPoint, aBlock);
			if (index >= 0) {
				Jun3dPoint aPoint = this.pointAt_(index);
				aList.add(new Jun3dTriangle(startPoint, endPoint, aPoint));
			}
		}

		points = this.points();
		for (int i = 1; i < points.length; i++) {
			Jun3dPoint startPoint = points[i - 1];
			Jun3dPoint endPoint = points[i];

			int index = this.findIndexOfPoints_from_to_interim_(aPolyline.points(), startPoint, endPoint, aBlock);
			if (index >= 0) {
				Jun3dPoint aPoint = aPolyline.pointAt_(index);
				aList.add(new Jun3dTriangle(endPoint, startPoint, aPoint));
			}
		}

		return (Jun3dTriangle[]) aList.toArray(new Jun3dTriangle[aList.size()]);
	}

	/**
	 * Find an index of points.
	 * 
	 * @param candidatePoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param startPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param endPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return int
	 * @category sewing
	 */
	protected int findIndexOfPoints_from_to_interim_(Jun3dPoint[] candidatePoints, Jun3dPoint startPoint, Jun3dPoint endPoint, StBlockClosure aBlock) {
		if (candidatePoints == null || candidatePoints.length == 0) {
			return -1;
		}

		double distance = startPoint.distance_(endPoint);

		TreeMap aTreeMap = new TreeMap(new Comparator() {
			public int compare(Object o1, Object o2) {
				double delta = ((JunAngle) o1).rad() - ((JunAngle) o2).rad();
				return (delta < 0) ? 1 : (delta > 0) ? -1 : 0;
			}
		});

		for (int i = 0; i < candidatePoints.length; i++) {
			Jun3dPoint aPoint = candidatePoints[i];
			double distance1 = aPoint.distance_(startPoint);
			double distance2 = aPoint.distance_(endPoint);
			JunAngle key = startPoint.to_(aPoint).angleWithLine_(endPoint.to_(aPoint));
			double[] value = new double[] { distance1, distance2, i };
			aTreeMap.put(key, value);

			if (aBlock != null) {
				aBlock.valueWithArguments_(new Object[] { new Jun3dPoint[] { startPoint, endPoint, aPoint }, $("pending") });
			}
		}

		Iterator i = aTreeMap.keySet().iterator();
		while (i.hasNext()) {
			JunAngle anAngle = (JunAngle) i.next();
			double[] value = (double[]) aTreeMap.get(anAngle);
			double distance1 = value[0];
			double distance2 = value[1];
			int index = (int) value[2];
			if (anAngle.rad() > Accuracy() && Math.abs(distance + distance1 - distance2) > Accuracy() && Math.abs(distance + distance2 - distance1) > Accuracy()) {
				if (aBlock != null) {
					aBlock.valueWithArguments_(new Object[] { new Jun3dPoint[] { startPoint, endPoint, candidatePoints[index] }, $("decided") });
				}
				return index;
			}
		}

		return -1;
	}

	/**
	 * Set my points.
	 * 
	 * @param newPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category private
	 */
	protected void setPoints_(Jun3dPoint[] newPoints) {
		points = newPoints;
		boundingBox = null;
		segmentLengths = null;
	}

	/**
	 * Compute the spline points from the specified control points.
	 * 
	 * @param controlPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category private
	 */
	protected Jun3dPoint[] computeSplinePoints_(Jun3dPoint[] controlPoints) {
		int size = controlPoints.length;
		if (size < 3) {
			Jun3dPoint[] points = new Jun3dPoint[size];
			System.arraycopy(controlPoints, 0, points, 0, size);
			return points;
		}

		int extras = 0;
		Jun3dPoint[] values = controlPoints;
		if (this.size() > 3 && this.from().equals(this.to())) {
			extras = 2;
			values = new Jun3dPoint[2 * extras + size];
			for (int i = 0; i < extras; i++) {
				values[i] = this.points()[size - extras + i - 1];
				values[size + extras + i] = this.points()[i + 1];
			}
			System.arraycopy(controlPoints, 0, values, extras, size);
		}

		Jun3dPoint[][] derivatives = new Jun3dPoint[3][];
		for (int i = 0; i < derivatives.length; i++) {
			derivatives[i] = new Jun3dPoint[values.length];
		}

		// Beginning of the derivs123 block.
		int n = values.length;
		if (n > 2) {
			double[] v = new double[n];
			v[0] = 4.0;
			Jun3dPoint[] h = new Jun3dPoint[n];
			h[0] = values[0].minus_(values[1].multipliedBy_(2.0)).plus_(values[2]).multipliedBy_(6.0);
			for (int i = 1; i <= n - 3; i++) {
				v[i] = 4.0 - 1.0 / v[i - 1];
				h[i] = values[i].minus_(values[i + 1].multipliedBy_(2.0)).plus_(values[i + 2]).multipliedBy_(6.0).minus_(h[i - 1].dividedBy_(v[i - 1]));
			}
			derivatives[1][n - 2] = h[n - 3].dividedBy_(v[n - 3]);
			for (int i = n - 3; i >= 1; i--) {
				derivatives[1][i] = h[i - 1].minus_(derivatives[1][i + 1]).dividedBy_(v[i - 1]);
			}
		}
		derivatives[1][0] = Jun3dPoint.Zero();
		derivatives[1][n - 1] = Jun3dPoint.Zero();
		for (int i = 0; i <= n - 2; i++) {
			derivatives[0][i] = values[i + 1].minus_(values[i]).minus_(derivatives[1][i].multipliedBy_(2.0).plus_(derivatives[1][i + 1]).dividedBy_(6.0));
			derivatives[2][i] = derivatives[1][i + 1].minus_(derivatives[1][i]);
		}
		// End of the derivs123 block.

		if (extras > 0) {
			for (int i = 0; i < 3; i++) {
				Jun3dPoint[] points = new Jun3dPoint[size];
				System.arraycopy(derivatives[i], extras, points, 0, size);
				derivatives[i] = points;
			}
		}

		LinkedList spline = new LinkedList();
		spline.add(this.from());
		for (int k = 0; k < this.size() - 1; k++) {
			Jun3dPoint d = controlPoints[k];
			Jun3dPoint c = derivatives[0][k];
			Jun3dPoint b = derivatives[1][k].dividedBy_(2.0);
			Jun3dPoint a = derivatives[2][k].dividedBy_(6.0);
			Jun3dPoint p = derivatives[1][k].abs().plus_(derivatives[1][k + 1].abs());
			int steps = (int) Math.max(4, Math.floor((p.x() + p.y()) / 100));
			for (int j = 1; j <= steps; j++) {
				double t = (double) j / steps;
				p = a.multipliedBy_(t).plus_(b).multipliedBy_(t).plus_(c).multipliedBy_(t).plus_(d);
				if (p.equal_(spline.getLast()) == false) {
					spline.add(p);
				}
			}
			p = controlPoints[k + 1];
			if (p.equal_(spline.getLast()) == false) {
				spline.add(p);
			}
		}
		return (Jun3dPoint[]) spline.toArray(new Jun3dPoint[spline.size()]);
	}

}
