package jp.co.sra.jun.geometry.forms;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dProcessor;

/**
 * JunFormTriangulation1 class
 * 
 *  @author    nisinaka
 *  @created   2007/06/19 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun632 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFormTriangulation1.java,v 8.5 2008/02/20 06:30:57 nisinaka Exp $
 */
public class JunFormTriangulation1 extends JunAbstractFormTriangulation {

	protected Jun2dTriangle[] normalDelaunayTriangles;
	protected Jun2dTriangle[] insideTriangles;
	protected Jun2dTriangle[] outsideTriangles;

	/**
	 * Create a new instance of JunFormTriangulation1 and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunFormTriangulation1() {
		super();
	}

	/**
	 * Create a new instance of JunFormTriangulation1 and initialize it.
	 *
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunFormTriangulation1(Jun2dPoint[] points) {
		super(points);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.forms.JunForm2dRegion#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		normalDelaunayTriangles = null;
		insideTriangles = null;
		outsideTriangles = null;
	}

	/**
	 * Set my new points.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @see jp.co.sra.jun.geometry.forms.JunForm2dRegion#points_(jp.co.sra.jun.geometry.basic.Jun2dPoint[])
	 * @category accessing
	 */
	public void points_(Jun2dPoint[] points) {
		super.points_(points);
		normalDelaunayTriangles = null;
		insideTriangles = null;
		outsideTriangles = null;
	}

	/**
	 * Answer my current triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @see jp.co.sra.jun.geometry.forms.JunAbstractFormTriangulation#triangles()
	 * @category accessing
	 */
	public Jun2dTriangle[] triangles() {
		return this.trianglesInterim_(null);
	}

	/**
	 * Answer my current triangles.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category accessing
	 */
	public Jun2dTriangle[] trianglesInterim_(StBlockClosure aBlock) {
		if (this.pointsWithoutLast().length < 3) {
			return new Jun2dTriangle[0];
		}

		return this.normalDelaunayTrianglesInterim_(aBlock);
	}

	/**
	 * Compute the normal delaunay triangles.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category computing
	 */
	protected Jun2dTriangle[] normalDelaunayTrianglesInterim_(StBlockClosure aBlock) {
		if (normalDelaunayTriangles == null) {
			Jun2dTriangle[] triangles = null;
			if (aBlock == null) {
				JunVoronoi2dProcessor aProcessor = new JunVoronoi2dProcessor(this.pointsWithoutLast());
				aProcessor.compute();
				triangles = this._triangles(aProcessor.triangles());
			} else {
				aBlock.value_value_value_(new Jun2dTriangle[0], null, null);

				ArrayList aList = new ArrayList();
				Jun2dPoint[] points = this.pointsWithoutLast();
				for (int i = 0; i < points.length; i++) {
					aList.add(points[i]);
					if (i >= 2) {
						JunVoronoi2dProcessor aProcessor = new JunVoronoi2dProcessor((Jun2dPoint[]) aList.toArray(new Jun2dPoint[aList.size()]));
						aProcessor.compute();
						triangles = this._triangles(aProcessor.triangles());
						aBlock.value_value_value_(triangles, null, null);
					}
				}
			}

			normalDelaunayTriangles = triangles;
			this.computeInsideAndOutsideTriangles_interim_(normalDelaunayTriangles, aBlock);
		}
		return insideTriangles;
	}

	/**
	 * Compute the inside and outside triangles.
	 * 
	 * @param triangles jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category computing
	 */
	private void computeInsideAndOutsideTriangles_interim_(Jun2dTriangle[] triangles, StBlockClosure aBlock) {
		ArrayList insideTriangleList = new ArrayList();
		ArrayList outsideTriangleList = new ArrayList();

		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint p1 = triangles[i].p1();
			Jun2dPoint p2 = triangles[i].p2();
			Jun2dPoint p3 = triangles[i].p3();
			Jun2dLine line1 = new Jun2dLine(p1, p2);
			Jun2dLine line2 = new Jun2dLine(p2, p3);
			Jun2dLine line3 = new Jun2dLine(p3, p1);

			Jun2dLine[] segments = this.segments();
			for (int j = 0; j < segments.length; j++) {
				if (line1 != null) {
					if (segments[j].equals(line1) || segments[j].equals(line1.reversed())) {
						line1 = null;
					}
				}
				if (line2 != null) {
					if (segments[j].equals(line2) || segments[j].equals(line2.reversed())) {
						line2 = null;
					}
				}
				if (line3 != null) {
					if (segments[j].equals(line3) || segments[j].equals(line3.reversed())) {
						line3 = null;
					}
				}
			}

			if ((line1 == null || this.containsLineSegment_(line1)) && (line2 == null || this.containsLineSegment_(line2)) && (line3 == null || this.containsLineSegment_(line3))) {
				insideTriangleList.add(triangles[i]);
			} else {
				outsideTriangleList.add(triangles[i]);
			}
			if (aBlock != null) {
				aBlock.value_value_value_(triangles, (Jun2dTriangle[]) insideTriangleList.toArray(new Jun2dTriangle[insideTriangleList.size()]), (Jun2dTriangle[]) outsideTriangleList.toArray(new Jun2dTriangle[outsideTriangleList.size()]));
			}
		}

		insideTriangles = (Jun2dTriangle[]) insideTriangleList.toArray(new Jun2dTriangle[insideTriangleList.size()]);
		outsideTriangles = (Jun2dTriangle[]) outsideTriangleList.toArray(new Jun2dTriangle[outsideTriangleList.size()]);
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayOn_at_(aGraphics, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		this.displayOn_at_triangles_color_insides_color_outsides_color_(aGraphics, aPoint, this.triangles(), Color.red, null, Color.green, null, Color.blue);
	}

	/**
	 * Display the triangles on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param triangles jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category displaying
	 */
	public void displayOn_triangles_(Graphics aGraphics, Jun2dTriangle[] triangles) {
		this.displayOn_at_triangles_color_insides_color_outsides_color_(aGraphics, new Point(0, 0), triangles, Color.red, null, null, null, null);
	}

	/**
	 * Display the triangles on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param triangles jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param insides jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param outsides jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category displaying
	 */
	public void displayOn_triangles_insides_outsides_(Graphics2D aGraphics, Jun2dTriangle[] triangles, Jun2dTriangle[] insides, Jun2dTriangle[] outsides) {
		this.displayOn_at_triangles_color_insides_color_outsides_color_(aGraphics, new Point(0, 0), triangles, Color.red, insides, Color.green, outsides, Color.blue);
	}

	/**
	 * Display the triangles on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param triangles jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param triangleColor java.awt.Color
	 * @param insides jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param insideColor java.awt.Color
	 * @param outsides jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param outsideColor java.awt.Color
	 * @category displaying
	 */
	protected void displayOn_at_triangles_color_insides_color_outsides_color_(Graphics aGraphics, Point aPoint, Jun2dTriangle[] triangles, Color triangleColor, Jun2dTriangle[] insides, Color insideColor, Jun2dTriangle[] outsides, Color outsideColor) {
		Rectangle boundingBox = this.boundingBox().asRectangle();
		int dx = aPoint.x - boundingBox.x;
		int dy = aPoint.y - boundingBox.y;

		Graphics2D g = (Graphics2D) aGraphics;
		try {
			g.translate(dx, dy);

			g.setColor(Color.white);
			g.fillRect(boundingBox.x, boundingBox.y, boundingBox.width, boundingBox.height);

			g.setColor(Color.gray);
			g.drawRect(boundingBox.x, boundingBox.y, boundingBox.width - 1, boundingBox.height - 1);

			for (int i = 0; i < triangles.length; i++) {
				JunForm2dRegion aPolyline = new JunForm2dRegion(triangles[i].asPointArray());
				GeneralPath aGeneralPath = aPolyline.toGeneralPath();
				g.setColor(StColorValue.Blend(triangleColor, Color.white));
				g.fill(aGeneralPath);
				g.setColor(triangleColor);
				g.draw(aGeneralPath);
			}
			if (insides != null) {
				for (int i = 0; i < insides.length; i++) {
					JunForm2dRegion aPolyline = new JunForm2dRegion(insides[i].asPointArray());
					GeneralPath aGeneralPath = aPolyline.toGeneralPath();
					g.setColor(StColorValue.Blend(insideColor, Color.white));
					g.fill(aGeneralPath);
					g.setColor(insideColor);
					g.draw(aGeneralPath);
				}
			}
			if (outsides != null) {
				for (int i = 0; i < outsides.length; i++) {
					JunForm2dRegion aPolyline = new JunForm2dRegion(outsides[i].asPointArray());
					GeneralPath aGeneralPath = aPolyline.toGeneralPath();
					g.setColor(StColorValue.Blend(outsideColor, Color.white));
					g.fill(aGeneralPath);
					g.setColor(outsideColor);
					g.draw(aGeneralPath);
				}
			}

			g.setColor(Color.black);
			g.draw(this.toGeneralPath());

		} finally {
			g.translate(-dx, -dy);
		}
	}

	/**
	 * Convert the array of points to the array of triangles.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category private
	 */
	private Jun2dTriangle[] _triangles(Jun2dPoint[][] anArrayOfPoints) {
		if (anArrayOfPoints == null) {
			return null;
		}

		Jun2dTriangle[] triangles = new Jun2dTriangle[anArrayOfPoints.length];
		for (int i = 0; i < triangles.length; i++) {
			triangles[i] = new Jun2dTriangle(anArrayOfPoints[i][0], anArrayOfPoints[i][1], anArrayOfPoints[i][2]);
		}
		return triangles;
	}

}
