package jp.co.sra.jun.geometry.surfaces;

import java.awt.Graphics;
import java.awt.Polygon;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.TreeSet;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.curves.Jun2dPolyline;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dProcessor;

/**
 * Jun2dPolygon class
 * 
 *  @author    nisinaka
 *  @created   1999/07/28 (by nisinaka)
 *  @updated   2004/10/19 (by Mitsuhiro Asada)
 *  @updated   2007/06/29 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun691 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dPolygon.java,v 8.17 2008/02/20 06:30:58 nisinaka Exp $
 */
public class Jun2dPolygon extends JunGeometry {

	/** A collection of Jun2dPoint. */
	protected Jun2dPoint[] points;

	/** A number to represent my signed area. */
	protected double signedArea = Double.NaN;

	/** My bounding box. */
	protected Jun2dBoundingBox bounds;

	/**
	 * Create a new instance of Jun2dPolygon and initialize it with the array
	 * of points.
	 * 
	 * @param anArrayOfPoint jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public Jun2dPolygon(Jun2dPoint[] anArrayOfPoint) {
		this.setPoints_(anArrayOfPoint);
	}

	/**
	 * Create a new instance of Jun2dPolygon and initialize it with the
	 * collection of points.
	 * 
	 * @param aCollectionOfPoint java.util.Collection
	 * @category Instance creation
	 */
	public Jun2dPolygon(Collection aCollectionOfPoint) {
		Jun2dPoint[] points = new Jun2dPoint[aCollectionOfPoint.size()];
		aCollectionOfPoint.toArray(points);
		this.setPoints_(points);
	}

	/**
	 * Answer a number which represents the area for the array of points.
	 * 
	 * @param anArrayOfPoint jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return double
	 * @category Geometric functions
	 */
	public static double AreaForPoints_(Jun2dPoint[] anArrayOfPoint) {
		return Math.abs(SignedAreaForPoints_(anArrayOfPoint));
	}

	/**
	 * Answer the area of the triangle.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category Geometric functions
	 */
	public static double AreaForTriangle_and_and_(Jun2dPoint aPoint1, Jun2dPoint aPoint2, Jun2dPoint aPoint3) {
		return Math.abs(SignedAreaForTriangle_and_and_(aPoint1, aPoint2, aPoint3));
	}

	/**
	 * Answer a number which represents the signed area for the array of
	 * points.
	 * 
	 * @param anArrayOfPoint jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return double
	 * @category Geometric functions
	 */
	public static double SignedAreaForPoints_(Jun2dPoint[] anArrayOfPoint) {
		int size = anArrayOfPoint.length;
		double area = 0.0d;
		for (int i = 0; i < size; i++) {
			Jun2dPoint p1 = anArrayOfPoint[i];
			Jun2dPoint p2 = anArrayOfPoint[(i + 1) % size];
			area += ((p1.y() + p2.y()) * (p1.x() - p2.x()));
		}
		return area / 2;
	}

	/**
	 * Answer a number which represents the signed area for the array of
	 * points.
	 * 
	 * @param anArrayOfPoint jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return double
	 * @category Geometric functions
	 */
	public static double SignedAreaForPoints_(Jun3dPoint[] anArrayOfPoint) {
		int size = anArrayOfPoint.length;
		double area = 0.0d;

		for (int i = 0; i < size; i++) {
			Jun3dPoint p1 = anArrayOfPoint[i];
			Jun3dPoint p2 = anArrayOfPoint[(i + 1) % size];
			area += ((p1.y() + p2.y()) * (p1.x() - p2.x()));
		}

		return area / 2;
	}

	/**
	 * Answer the signed area of the triangle.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category Geometric functions
	 */
	public static double SignedAreaForTriangle_and_and_(Jun2dPoint aPoint1, Jun2dPoint aPoint2, Jun2dPoint aPoint3) {
		return (((aPoint1.y() + aPoint2.y()) * (aPoint1.x() - aPoint2.x())) + ((aPoint2.y() + aPoint3.y()) * (aPoint2.x() - aPoint3.x())) + (aPoint3.y() + aPoint1.y()) * (aPoint3.x() - aPoint1.x())) / 2;
	}

	/**
	 * Answer true if the array of points make a consistent polygon.
	 * 
	 * @param anArrayOfJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return boolean
	 * @category Private
	 */
	protected static boolean IsConsistentPolygonFromPoints_(Jun2dPoint[] anArrayOfJun2dPoint) {
		final Jun2dPoint[] thePoints = (new Jun2dPolygon(anArrayOfJun2dPoint)).asPositivePolygon().points();
		final int size = thePoints.length;
		StBlockClosure isConcaveBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				int index = ((Number) anObject).intValue();
				Jun2dPoint p1 = thePoints[(index - 1) % size];
				Jun2dPoint p2 = thePoints[index];
				Jun2dPoint p3 = thePoints[(index + 1) % size];
				return new Boolean((SignedAreaForTriangle_and_and_(p1, p2, p3) + ACCURACY) <= 0);
			}
		};

		StBlockClosure isConvexBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				int index = ((Number) anObject).intValue();
				Jun2dPoint p1 = thePoints[(index - 1) % size];
				Jun2dPoint p2 = thePoints[index];
				Jun2dPoint p3 = thePoints[(index + 1) % size];
				return new Boolean((SignedAreaForTriangle_and_and_(p1, p2, p3) + ACCURACY) >= 0);
			}
		};

		StBlockClosure containsBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				int index = ((Number) anObject).intValue();
				Jun2dPoint point = thePoints[index];
				ArrayList v = new ArrayList();
				for (int i = 0; i < thePoints.length; i++) {
					if (point.equals(thePoints[i]) == false) {
						v.add(thePoints[i]);
					}
				}
				Jun2dPoint[] ps = new Jun2dPoint[v.size()];
				v.toArray(ps);
				return new Boolean(PolygonFromPoints_containsPoint_(ps, point));
			}
		};

		//
		for (int i = 0; i < size; i++) {
			Integer index = new Integer(i);
			boolean contains = ((Boolean) containsBlock.value_(index)).booleanValue();
			if (((Boolean) isConcaveBlock.value_(index)).booleanValue() && !contains) {
				return false;
			}
			if (((Boolean) isConvexBlock.value_(index)).booleanValue() && contains) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Answer true if the polygon contains the point.
	 * 
	 * @param anArrayOfPoint jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * 
	 * @return boolean
	 * @category Private
	 */
	protected static boolean PolygonFromPoints_containsPoint_(Jun2dPoint[] anArrayOfPoint, Jun2dPoint aPoint) {
		int size = anArrayOfPoint.length;
		double around = 0.0d;
		for (int i = 0; i < size; i++) {
			Jun2dPoint p1 = anArrayOfPoint[i];
			Jun2dPoint p2 = anArrayOfPoint[(i + 1) % size];
			around += aPoint.angleBetween_and_(p1, p2).rad();
		}
		return (Math.abs(around) > Math.PI / 4.0d);
	}

	/**
	 * Answer a number which represents my area.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return Math.abs(this.signedArea());
	}

	/**
	 * Answer the point at the specified index.
	 * 
	 * @param index int
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint pointAt_(int index) {
		return points[index];
	}

	/**
	 * Answer the array of my points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		return points;
	}

	/**
	 * Answer the number of my points.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int pointsSize() {
		return points.length;
	}

	/**
	 * Answer a number which represents my signed area.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double signedArea() {
		if (Double.isNaN(signedArea)) {
			signedArea = SignedAreaForPoints_(points);
		}

		return signedArea;
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category bounds accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		if (bounds == null) {
			Jun2dPoint min = points[0];
			Jun2dPoint max = points[0];
			for (int i = 1; i < points.length; i++) {
				min = (Jun2dPoint) min.min_(points[i]);
				max = (Jun2dPoint) max.max_(points[i]);
			}
			bounds = Jun2dBoundingBox.Origin_corner_(min, max);
		}
		return bounds;
	}

	/**
	 * Answer the extent of my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category bounds accessing
	 */
	public Jun2dPoint extent() {
		return (Jun2dPoint) this.boundingBox().extent();
	}

	/**
	 * Answer the height of my bounding box.
	 * 
	 * @return double
	 * @category bounds accessing
	 */
	public double height() {
		return this.extent().y();
	}

	/**
	 * Answer the width of my bounding box.
	 * 
	 * @return double
	 * @category bounds accessing
	 */
	public double width() {
		return this.extent().x();
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dPolygon aPolygon = (Jun2dPolygon) anObject;
		int size = this.pointsSize();
		if (size != aPolygon.pointsSize()) {
			return false;
		}

		for (int i = 0; i < size; i++) {
			if (this.pointAt_(i).equal_(aPolygon.pointAt_(i)) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dPolygon aPolygon = (Jun2dPolygon) anObject;
		int size = this.pointsSize();
		if (size != aPolygon.pointsSize()) {
			return false;
		}

		for (int i = 0; i < size; i++) {
			if (this.pointAt_(i).equals(aPolygon.pointAt_(i)) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Convert the receiver as an array of the convex polygons.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon[]
	 * @category converting
	 */
	public Jun2dPolygon[] asArrayOfConvexPolygons() {
		final ArrayList polygons = new ArrayList();
		this.asConvexPolygonsDo_(new StBlockClosure() {
			public Object value_(Object polygon) {
				polygons.add(polygon);
				return null;
			}
		});
		return (Jun2dPolygon[]) polygons.toArray(new Jun2dPolygon[polygons.size()]);
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category converting
	 */
	public Jun2dLine[] asArrayOfLines() {
		return this.asArrayOf2dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf2dLines()
	 * @category converting
	 */
	public Jun2dLine[] asArrayOf2dLines() {
		Jun2dPoint[] points = this.points();
		boolean isLoop = points[0].equal_(points[points.length - 1]);
		Jun2dPoint[] collection = new Jun2dPoint[isLoop ? points.length : points.length + 1];
		for (int i = 0; i < points.length; i++) {
			collection[i] = new Jun2dPoint(points[i]);
		}
		if (isLoop == false) {
			collection[collection.length - 1] = new Jun2dPoint(points[0]);
		}
		Jun2dPolyline polyline = new Jun2dPolyline(collection);
		return polyline.asArrayOfLines();
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category converting
	 */
	public Jun2dPoint[] asArrayOfPoints() {
		return this.points();
	}

	/**
	 * Convert the receiver to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		final ArrayList polygons = new ArrayList();
		this.asJunOpenGL3dObjectsDo_(new StBlockClosure() {
			public Object value_(Object polygon) {
				polygons.add(polygon);
				return null;
			}
		});
		JunOpenGL3dObject aBody = (polygons.size() == 1) ? (JunOpenGL3dObject) polygons.get(0) : new JunOpenGL3dCompoundObject(polygons);
		aBody.objectsDo_(new StBlockClosure() {
			public Object value_(Object object) {
				((JunOpenGL3dObject) object).paint_alpha_(Jun2dPolygon.this.defaultColor(), Float.NaN);
				return null;
			}
		});
		return aBody;
	}

	/**
	 * Convert the receiver as a negative polygon.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category converting
	 */
	public Jun2dPolygon asNegativePolygon() {
		if (this.signedArea() <= 0) {
			return this;
		} else {
			return this.reversed();
		}
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category converting
	 */
	public Jun2dPoint[] asPointArray() {
		return this.points();
	}

	/**
	 * Convert the receiver as a positive polygon.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category converting
	 */
	public Jun2dPolygon asPositivePolygon() {
		if (this.signedArea() >= 0) {
			return this;
		} else {
			return this.reversed();
		}
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.jp.co.sra.jun.geometry.surfaces[]
	 * @category converting
	 */
	public Jun2dTriangle[] asTriangles() {
		final ArrayList triangleStream = new ArrayList();
		this.asTrianglesDo_(new StBlockClosure() {
			public Object value_(Object polygon) {
				triangleStream.add(polygon);
				return null;
			}
		});
		return (Jun2dTriangle[]) triangleStream.toArray(new Jun2dTriangle[triangleStream.size()]);
	}

	/**
	 * Create a reversed polygon.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category converting
	 */
	public Jun2dPolygon reversed() {
		Jun2dPoint[] reversedPoints = new Jun2dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			reversedPoints[i] = points[points.length - i - 1];
		}
		return new Jun2dPolygon(reversedPoints);
	}

	/**
	 * Answer the transposed polygon of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category converting
	 */
	public Jun2dPolygon transposed() {
		Jun2dPoint[] transposedPoints = new Jun2dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			transposedPoints[i] = new Jun2dPoint(points[i].y(), points[i].x());
		}
		return new Jun2dPolygon(transposedPoints);
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		int npoints = this.pointsSize();
		int[] xpoints = new int[npoints];
		int[] ypoints = new int[npoints];
		for (int i = 0; i < npoints; i++) {
			xpoints[i] = (int) points[i].x();
			ypoints[i] = (int) points[i].y();
		}

		aGraphics.drawPolygon(new Polygon(xpoints, ypoints, npoints));
	}

	/**
	 * Enumerate every convex polygons and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object asConvexPolygonsDo_(final StBlockClosure aBlock) {
		if (this.isNegative()) {
			return this.asPositivePolygon().asConvexPolygonsDo_(new StBlockClosure() {
				public Object value_(Object anObject) {
					Jun2dPolygon polygon = (Jun2dPolygon) anObject;
					return aBlock.value_(polygon.asNegativePolygon());
				}
			});
		}

		if (this.isConvex()) {
			return aBlock.value_(this);
		}

		int size = this.pointsSize();
		int concaveIndex = this.indexOfFirstConcaveVertex();
		Jun2dPoint concavePoint = this.pointAt_(concaveIndex);
		Jun2dPoint divisionVector = (Jun2dPoint) concavePoint.minus_(this.pointAt_((concaveIndex - 1 + size) % size)).unitVector();
		divisionVector = (Jun2dPoint) divisionVector.plus_(concavePoint.minus_(this.pointAt_((concaveIndex + 1) % size)).unitVector());
		int divisionIndex;
		Jun2dPoint divisionPoint;
		if (Math.abs(divisionVector.x()) > Math.abs(divisionVector.y())) {
			if (divisionVector.x() > 0) {
				Jun2dPoint minPoint = null;
				int minIndex = -1;
				for (int i = concaveIndex; i <= ((concaveIndex + size) - 3); i++) {
					Jun2dPoint p1 = this.pointAt_((i + 1) % size);
					Jun2dPoint p2 = this.pointAt_((i + 2) % size);
					if ((p1.y() < p2.y()) && (p1.y() < concavePoint.y()) && (p2.y() >= concavePoint.y())) {
						double x = (((concavePoint.y() - p1.y()) * (p2.x() - p1.x())) / (p2.y() - p1.y())) + p1.x();
						if ((concavePoint.x() < x) && (minPoint == null || x <= minPoint.x())) {
							minPoint = new Jun2dPoint(x, concavePoint.y());
							minIndex = (i + 1) % size;
						}
					}
				}

				if (minIndex < 0) {
					throw SmalltalkException.Error("internal error");
				}
				divisionIndex = minIndex;
				divisionPoint = minPoint;
			} else {
				Jun2dPoint maxPoint = null;
				int maxIndex = -1;
				for (int i = concaveIndex; i <= ((concaveIndex + size) - 3); i++) {
					Jun2dPoint p1 = this.pointAt_((i + 1) % size);
					Jun2dPoint p2 = this.pointAt_((i + 2) % size);
					if ((p1.y() > p2.y()) && (p1.y() > concavePoint.y()) && (p2.y() <= concavePoint.y())) {
						double x = (((concavePoint.y() - p1.y()) * (p2.x() - p1.x())) / (p2.y() - p1.y())) + p1.x();
						if ((concavePoint.x() > x) && (maxPoint == null || x >= maxPoint.x())) {
							maxPoint = new Jun2dPoint(x, concavePoint.y());
							maxIndex = (i + 1) % size;
						}
					}
				}

				if (maxIndex < 0) {
					throw SmalltalkException.Error("internal error");
				}
				divisionIndex = maxIndex;
				divisionPoint = maxPoint;
			}
		} else {
			if (divisionVector.y() > 0) {
				Jun2dPoint minPoint = null;
				int minIndex = -1;
				for (int i = concaveIndex; i <= ((concaveIndex + size) - 3); i++) {
					Jun2dPoint p1 = this.pointAt_((i + 1) % size);
					Jun2dPoint p2 = this.pointAt_((i + 2) % size);
					if ((p1.x() > p2.x()) && (p1.x() > concavePoint.x()) && (p2.x() <= concavePoint.x())) {
						double y = (((concavePoint.x() - p1.x()) * (p2.y() - p1.y())) / (p2.x() - p1.x())) + p1.y();
						if ((concavePoint.y() < y) && (minPoint == null || y <= minPoint.y())) {
							minPoint = new Jun2dPoint(concavePoint.x(), y);
							minIndex = (i + 1) % size;
						}
					}
				}

				if (minIndex < 0) {
					throw SmalltalkException.Error("internal error");
				}
				divisionIndex = minIndex;
				divisionPoint = minPoint;
			} else {
				Jun2dPoint maxPoint = null;
				int maxIndex = -1;
				for (int i = concaveIndex; i <= ((concaveIndex + size) - 3); i++) {
					Jun2dPoint p1 = this.pointAt_((i + 1) % size);
					Jun2dPoint p2 = this.pointAt_((i + 2) % size);
					if ((p1.x() < p2.x()) && (p1.x() < concavePoint.x()) && (p2.x() >= concavePoint.x())) {
						double y = (((concavePoint.x() - p1.x()) * (p2.y() - p1.y())) / (p2.x() - p1.x())) + p1.y();
						if ((concavePoint.y() > y) && (maxPoint == null || y >= maxPoint.y())) {
							maxPoint = new Jun2dPoint(concavePoint.x(), y);
							maxIndex = (i + 1) % size;
						}
					}
				}

				if (maxIndex < 0) {
					throw SmalltalkException.Error("internal error");
				}

				divisionIndex = maxIndex;
				divisionPoint = maxPoint;
			}
		}

		ArrayList newPoints1 = new ArrayList();
		ArrayList newPoints2 = new ArrayList();

		if (divisionIndex < concaveIndex) {
			for (int i = 0; i <= divisionIndex; i++) {
				newPoints1.add(this.pointAt_(i));
			}
			if (divisionPoint.equals(this.pointAt_(divisionIndex)) == false) {
				newPoints1.add(divisionPoint);
			}
			for (int i = concaveIndex; i < size; i++) {
				newPoints1.add(this.pointAt_(i));
			}
			if (divisionPoint.equals(this.pointAt_((divisionIndex + 1) % size)) == false) {
				newPoints2.add(divisionPoint);
			}
			for (int i = divisionIndex + 1; i <= concaveIndex; i++) {
				newPoints2.add(this.pointAt_(i));
			}
		} else {
			for (int i = 0; i <= concaveIndex; i++) {
				newPoints1.add(this.pointAt_(i));
			}
			if (divisionPoint.equals(this.pointAt_((divisionIndex + 1) % size)) == false) {
				newPoints1.add(divisionPoint);
			}
			for (int i = divisionIndex + 1; i < size; i++) {
				newPoints1.add(this.pointAt_(i));
			}
			for (int i = concaveIndex; i <= divisionIndex; i++) {
				newPoints2.add(this.pointAt_(i));
			}
			if (divisionPoint.equals(this.pointAt_(divisionIndex)) == false) {
				newPoints2.add(divisionPoint);
			}
		}

		Object result = (new Jun2dPolygon(newPoints1)).asConvexPolygonsDo_(aBlock);
		if (result != null) {
			return result;
		}
		result = (new Jun2dPolygon(newPoints2)).asConvexPolygonsDo_(aBlock);
		if (result != null) {
			return result;
		}
		return null;
	}

	/**
	 * Enumerate the convex polygons as JunOpenGL3dObjects and evaluate the
	 * block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object asJunOpenGL3dObjectsDo_(final StBlockClosure aBlock) {
		return this.asConvexPolygonsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun2dPolygon polygon = (Jun2dPolygon) anObject;
				Jun2dPoint[] points = polygon.points();
				Jun3dPoint[] vertexes = new Jun3dPoint[points.length];
				for (int i = 0; i < points.length; i++) {
					vertexes[i] = new Jun3dPoint(points[i], 0);
				}
				return aBlock.value_(new JunOpenGL3dPolygon(vertexes));
			}
		});
	}

	/**
	 * Enumerate the triangles and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object asTrianglesDo_(final StBlockClosure aBlock) {
		boolean isPositive = this.isPositive();
		JunVoronoi2dProcessor voronoiProcessor = new JunVoronoi2dProcessor(this.points());
		voronoiProcessor.compute();
		Jun2dPoint[][] triangles = voronoiProcessor.triangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint p1 = triangles[i][0];
			Jun2dPoint p2 = triangles[i][1];
			Jun2dPoint p3 = triangles[i][2];
			if (AreaForTriangle_and_and_(p1, p2, p3) > Jun2dPolygon.Accuracy() && this.absContainsPoint_(p1.plus_(p2).plus_(p3).dividedBy_(3))) {
				Jun2dPolygon polygon = new Jun2dPolygon(triangles[i]);
				aBlock.value_(polygon.asPositivePolygon_(isPositive));
			}
		}
		return null;
	}

	/**
	 * Enumerate the edges and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object edgesDo_(StBlockClosure aBlock) {
		for (int i = 0; i < points.length; i++) {
			Object result = aBlock.value_value_(points[i], points[(i + 1) % points.length]);
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Answer the nearest point on the receiver from the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint nearestPointFromPoint_(Jun2dPoint aPoint) {
		if (this.containsPoint_(aPoint)) {
			return aPoint;
		} else {
			return this.nearestPointOnEdgeFromPoint_(aPoint);
		}
	}

	/**
	 * Answer the nearest point on the receiver's edge from the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint nearestPointOnEdgeFromPoint_(Jun2dPoint aPoint) {
		final Jun2dPoint thePoint = new Jun2dPoint(aPoint.x(), aPoint.y());
		final StValueHolder nearestPoint = new StValueHolder(null);
		final StValueHolder minDistance = new StValueHolder(Double.POSITIVE_INFINITY);
		this.edgesDo_(new StBlockClosure() {
			public Object value_value_(Object anObject1, Object anObject2) {
				Jun2dPoint point1 = (Jun2dPoint) anObject1;
				Jun2dPoint point2 = (Jun2dPoint) anObject2;
				if (point1.equals(point2) == false) {
					Jun2dLine edge = new Jun2dLine(point1, point2);
					Jun2dPoint point = edge.lineSegmentNearestPointFromPoint_(thePoint);
					double distance = thePoint.distance_(point);
					if (minDistance._doubleValue() > distance) {
						minDistance.value_(distance);
						nearestPoint.value_(point);
					}
				}
				return null;
			}
		});

		return (Jun2dPoint) nearestPoint.value();
	}

	/**
	 * Answer true if the receiver contains the line segment.
	 * 
	 * @param from jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean absContainsLineSegmentFrom_to_(final Jun2dPoint from, final Jun2dPoint to) {
		final double dx = to.x() - from.x();
		final double dy = to.y() - from.y();
		final StBlockClosure tBlock = new StBlockClosure() {
			public Object value_value_(Object anObject1, Object anObject2) {
				Jun2dPoint point1 = (Jun2dPoint) anObject1;
				Jun2dPoint point2 = (Jun2dPoint) anObject2;
				double pdx = point2.x() - point1.x();
				double pdy = point2.y() - point1.y();
				double d = (pdx * dy) - (dx * pdy);
				if (d < ACCURACY) {
					return null;
				} else {
					return new Double((pdx * (point1.y() - from.y()) - pdy * (point1.x() - from.x())) / d);
				}
			}
		};

		final TreeSet intersectingTs = new TreeSet(new Comparator() {
			public int compare(Object arg0, Object arg1) {
				return (((Number) arg0).doubleValue() <= ((Number) arg1).doubleValue()) ? -1 : 1;
			}
		});
		intersectingTs.add(new Double(0));
		intersectingTs.add(new Double(1));
		this.edgesDo_(new StBlockClosure() {
			public Object value_value_(Object p1, Object p2) {
				Double t = (Double) tBlock.value_value_(p1, p2);
				if ((t != null) && (0.0d < t.doubleValue()) && (t.doubleValue() < 1.0d)) {
					intersectingTs.add(t);
				}
				return null;
			}
		});

		if (intersectingTs.size() == 2) {
			return this.absContainsPoint_((Jun2dPoint) from.plus_(to).dividedBy_(2));
		}

		Double[] ts = (Double[]) intersectingTs.toArray(new Double[intersectingTs.size()]);
		for (int i = 0; i < (ts.length - 1); i++) {
			double t1 = ts[i].doubleValue();
			double t2 = ts[i + 1].doubleValue();
			double t = (t1 + t2) / 2.0d;
			Jun2dPoint intersectingPoint = (Jun2dPoint) to.minus_(from).multipliedBy_(t).plus_(from);

			if (this.absContainsPoint_(intersectingPoint) == false) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Answer true if the receiver contains the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean absContainsPoint_(Jun2dPoint aPoint) {
		if (this.boundingBox().containsOrTouchesPoint_(aPoint) == false) {
			return false;
		}

		final double x = aPoint.x();
		final double y = aPoint.y();
		final StValueHolder count = new StValueHolder(0);
		Object result = this.edgesDo_(new StBlockClosure() {
			public Object value_value_(Object anObject1, Object anObject2) {
				Jun2dPoint p1 = (Jun2dPoint) anObject1;
				Jun2dPoint p2 = (Jun2dPoint) anObject2;
				if ((Math.abs(p1.x() - x) < ACCURACY) && (Math.abs(p1.y() - y) < ACCURACY)) {
					return Boolean.TRUE;
				}
				Jun2dPoint point1;
				Jun2dPoint point2;
				if (p1.x() <= p2.x()) {
					point1 = p1;
					point2 = p2;
				} else {
					point1 = p2;
					point2 = p1;
				}
				if ((point1.x() <= x) && (x < point2.x())) {
					double y0 = ((x - point1.x()) / (point2.x() - point1.x()) * (point2.y() - point1.y())) + point1.y();
					if (Math.abs(y0 - y) < ACCURACY) {
						return Boolean.TRUE;
					}
					if (y0 >= y) {
						count.value_(count._intValue() + 1);
					}
				}

				return null;
			}
		});

		if (result != null) {
			return true;
		}
		return (count._intValue() % 2 == 1);
	}

	/**
	 * Answer true if the receiver is a convex polygon.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean absIsConvex() {
		return this.asPositivePolygon().isConvex();
	}

	/**
	 * Answer true if the receiver contains the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun2dPoint aPoint) {
		return (this.absContainsPoint_(aPoint) != this.isNegative());
	}

	/**
	 * Answer true if the receiver is a 2d geometry element.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Answer true if the receiver is consistent.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConsistent() {
		return IsConsistentPolygonFromPoints_(this.points());
	}

	/**
	 * Answer true if the receiver is a convex polygon.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConvex() {
		if (this.isPositive()) {
			return (this.indexOfFirstConcaveVertex() < 0);
		} else {
			return (this.asPositivePolygon().indexOfFirstConcaveVertex() < 0);
		}
	}

	/**
	 * Answer true if the receiver is a negative polygon.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNegative() {
		return (this.isPositive() == false);
	}

	/**
	 * Answer true if the receiver is a positive polygon.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPositive() {
		return (this.signedArea() + this.ACCURACY >= 0);
	}

	/**
	 * Answer the new <code>Jun2dPolygon</code> which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category transforming
	 */
	public Jun2dPolygon rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun2dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer a copy of the receiver scaled by the specified extent.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category transforming
	 */
	public Jun2dPolygon scaledBy_(double aNumber) {
		Jun2dPoint[] scaledPoints = new Jun2dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			scaledPoints[i] = points[i].scaledBy_(aNumber);
		}
		return new Jun2dPolygon(scaledPoints);
	}

	/**
	 * Answer a copy of the receiver scaled by the specified extent.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category transforming
	 */
	public Jun2dPolygon scaledBy_(Jun2dPoint aPoint) {
		Jun2dPoint[] scaledPoints = new Jun2dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			scaledPoints[i] = points[i].scaledBy_(aPoint);
		}
		return new Jun2dPolygon(scaledPoints);
	}

	/**
	 * Answer the copy of the receiver which is applied the specified
	 * transformation.
	 * 
	 * @param aJun2dTransformation jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category transforming
	 */
	public Jun2dPolygon transform_(Jun2dTransformation aJun2dTransformation) {
		Jun2dPoint[] points = this.points();
		Jun2dPoint[] newPoints = new Jun2dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			newPoints[i] = aJun2dTransformation.applyToPoint_(points[i]);
		}
		return new Jun2dPolygon(newPoints);
	}

	/**
	 * Answer a copy of the receiver translated by the specified extent.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category transforming
	 */
	public Jun2dPolygon translatedBy_(double aNumber) {
		Jun2dPoint[] scaledPoints = new Jun2dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			scaledPoints[i] = points[i].translatedBy_(aNumber);
		}
		return new Jun2dPolygon(scaledPoints);
	}

	/**
	 * Answer a copy of the receiver translated by the specified extent.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category transforming
	 */
	public Jun2dPolygon translatedBy_(Jun2dPoint aPoint) {
		Jun2dPoint[] scaledPoints = new Jun2dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			scaledPoints[i] = points[i].translatedBy_(aPoint);
		}
		return new Jun2dPolygon(scaledPoints);
	}

	/**
	 * Answer the receiver's positive/negative polygon with the specified flag.
	 * 
	 * @param aBoolean boolean
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category private
	 */
	protected Jun2dPolygon asPositivePolygon_(boolean aBoolean) {
		return aBoolean ? this.asPositivePolygon() : this.asNegativePolygon();
	}

	/**
	 * Answer the index of the first concave vertex. If not found, return -1.
	 * 
	 * @return int
	 * @category private
	 */
	protected int indexOfFirstConcaveVertex() {
		int pointsSize = this.pointsSize();
		for (int i = 0; i < pointsSize; i++) {
			Jun2dPoint p1 = this.pointAt_((i - 1 + pointsSize) % pointsSize);
			Jun2dPoint p2 = this.pointAt_(i);
			Jun2dPoint p3 = this.pointAt_((i + 1) % pointsSize);
			if ((SignedAreaForTriangle_and_and_(p1, p2, p3) + ACCURACY) < 0) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Set the points.
	 * 
	 * @param anArrayOfPoint jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category private
	 */
	protected void setPoints_(Jun2dPoint[] anArrayOfPoint) {
		int size = anArrayOfPoint.length;
		points = new Jun2dPoint[size];
		System.arraycopy(anArrayOfPoint, 0, points, 0, size);
		signedArea = Double.NaN;
	}
}
