package jp.co.sra.jun.goodies.calendar;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Polygon;
import java.util.Iterator;
import java.util.Vector;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.system.framework.JunAbstractViewCanvas;
import jp.co.sra.jun.system.support.JunSystem;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StModel;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

/**
 * JunCalendarTimeViewAwt class
 * 
 *  @author    Nobuto Matsubara
 *  @created   2003/10/06 (by Nobuto Matsubara)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCalendarTimeViewAwt.java,v 8.14 2008/02/20 06:31:13 nisinaka Exp $
 */
public class JunCalendarTimeViewAwt extends JunAbstractViewCanvas implements JunCalendarTimeView {

	protected Vector frames;

	/**
	 * Create a new instance of <code>JunCalendarTimeViewAwt</code> and initialize it.
	 * 
	 * @param aCalendarTimeModel JunCalendarTimeModel
	 * @category Instance creation
	 */
	public JunCalendarTimeViewAwt(JunCalendarTimeModel aCalendarTimeModel) {
		super(aCalendarTimeModel);
	}

	/**
	 * Return a frames.
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector frames() {
		if (frames == null) {
			this.makeFrames();
		}
		return frames;
	}

	/**
	 * Make frames.
	 * @category private
	 */
	private void makeFrames() {
		Vector collection = new Vector();
		int width = this.getSize().width / 10;
		int height = this.getSize().height / 8;
		StRectangle button = new StRectangle(0, 0, 12, 16);
		int x = width * 0;
		int y = height * 0;
		StRectangle box = new StRectangle(x, y, width * 4, height);
		box = box.insetBy_(StRectangle.Origin_corner_(new Point(8, 0), new Point(0, 0)));
		collection.add(new Object[] { $("label"), JunSystem.$String("hours:"), box });
		x = width * 5;
		y = height * 0;
		box = StRectangle.Origin_extent_(new Point(x, y), new Point(width * 5, height));
		box = box.insetBy_(StRectangle.Origin_corner_(new Point(button.width(), 0), new Point(button.width() + 8, 0)));
		collection.add(new Object[] { $("hours"), box });
		button = button.align_with_(button.rightCenter(), box.leftCenter());
		collection.add(new Object[] { $("button"), $("hourDown"), button });
		button = button.align_with_(button.leftCenter(), box.rightCenter());
		collection.add(new Object[] { $("button"), $("hourUp"), button });
		x = width * 0;
		y = height * 1;
		box = StRectangle.Origin_extent_(new Point(x, y), new Point(width * 4, height));
		box = box.insetBy_(StRectangle.Origin_corner_(new Point(8, 0), new Point(0, 0)));
		collection.add(new Object[] { $("label"), JunSystem.$String("minutes:"), box });
		x = width * 5;
		y = height * 1;
		box = StRectangle.Origin_extent_(new Point(x, y), new Point(width * 5, height));
		box = box.insetBy_(StRectangle.Origin_corner_(new Point(button.width(), 0), new Point(button.width() + 8, 0)));
		collection.add(new Object[] { $("minutes"), box });
		button = button.align_with_(button.rightCenter(), box.leftCenter());
		collection.add(new Object[] { $("button"), $("minuteDown"), button });
		button = button.align_with_(button.leftCenter(), box.rightCenter());
		collection.add(new Object[] { $("button"), $("minuteUp"), button });
		x = width * 0;
		y = height * 2;
		box = StRectangle.Origin_extent_(new Point(x, y), new Point(width * 4, height));
		box = box.insetBy_(StRectangle.Origin_corner_(new Point(8, 0), new Point(0, 0)));
		collection.add(new Object[] { $("label"), JunSystem.$String("seconds:"), box });
		x = width * 5;
		y = height * 2;
		box = StRectangle.Origin_extent_(new Point(x, y), new Point(width * 5, height));
		box = box.insetBy_(StRectangle.Origin_corner_(new Point(button.width(), 0), new Point(button.width() + 8, 0)));
		collection.add(new Object[] { $("seconds"), box });
		button = button.align_with_(button.rightCenter(), box.leftCenter());
		collection.add(new Object[] { $("button"), $("secondDown"), button });
		button = button.align_with_(button.leftCenter(), box.rightCenter());
		collection.add(new Object[] { $("button"), $("secondUp"), button });
		x = width * 0;
		y = height * 3;
		box = StRectangle.Origin_extent_(new Point(x, y), new Point(width * 10, height * 5));
		collection.add(new Object[] { $("clock"), box });
		frames = collection;
	}

	/**
	 * Return rotateBlock.
	 *
	 * @return StBlockClosure
	 * @category private
	 */
	private StBlockClosure rotateBlock() {
		StBlockClosure block = new StBlockClosure() {
			public Object value_value_(Object p, Object d) {
				Jun2dPoint point = (Jun2dPoint) p;
				double radians = Math.toRadians(((java.lang.Double) d).doubleValue() * -1.0);
				double sin = Math.sin(radians);
				double cos = Math.cos(radians);
				double x = point.x() * cos + point.y() * sin;
				double y = -1.0 * point.x() * sin + point.y() * cos;
				return (Object) (new Jun2dPoint(x, y));
			}
		};
		return block;
	}

	/**
	 * paint.
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayBordersOn_(aGraphics);
		this.displayObjectsOn_(aGraphics);
	}

	/**
	 * Paint borders.
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayBordersOn_(Graphics aGraphics) {
		StSymbol key;
		StRectangle box;
		StRectangle area;
		StRectangle clip;
		for (int i = 0; i < this.frames().size(); ++i) {
			Object[] array = (Object[]) this.frames().get(i);
			key = (StSymbol) array[0];
			box = (StRectangle) array[array.length - 1];
			clip = new StRectangle(aGraphics.getClipBounds());
			if (clip.intersects_(box)) {
				if (key == $("hours") || key == $("minutes") || key == $("seconds") || key == $("clock")) {
					aGraphics.setColor(veryLightGray);
					aGraphics.fillRect(box.x(), box.y(), box.width(), box.height());
					area = box.insetBy_(new StRectangle(1, 1, 2, 2));
					aGraphics.setColor(veryDarkGray);
					aGraphics.drawRect(area.x(), area.y(), area.width(), area.height());
					if (key != $("clock")) {
						area = box.insetBy_(new StRectangle(2, 2, 3, 3));
						aGraphics.setColor(Color.black);
						aGraphics.drawRect(area.x(), area.y(), area.width(), area.height());
					}
				}
			}
		}
	}

	/**
	 * Paint objects.
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayObjectsOn_(Graphics aGraphics) {
		StSymbol key;
		StRectangle box;
		for (int i = 0; i < this.frames().size(); ++i) {
			Object[] array = (Object[]) this.frames().get(i);
			key = (StSymbol) array[0];
			box = (StRectangle) array[array.length - 1];
			if (key == $("hours")) {
				this.displayOn_hours_clip_(aGraphics, key, box);
			}
			if (key == $("minutes")) {
				this.displayOn_minutes_clip_(aGraphics, key, box);
			}
			if (key == $("seconds")) {
				this.displayOn_seconds_clip_(aGraphics, key, box);
			}
			if (key == $("label")) {
				this.displayOn_label_clip_(aGraphics, (String) array[1], box);
			}
			if (key == $("button")) {
				this.displayOn_button_clip_(aGraphics, (StSymbol) array[1], box);
			}
			if (key == $("clock")) {
				this.displayOn_clock_clip_(aGraphics, key, box);
			}
		}
	}

	/**
	 * Paint hours.
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_hours_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		int hours = this.getModel().hours();
		String string = String.valueOf(hours);
		if (string.length() < 2) {
			string = "0".concat(string);
		}
		StComposedText para = new StComposedText(string);
		StRectangle bounds = new StRectangle(para.bounds());
		StRectangle rect = bounds.align_with_(bounds.center(), aRectangle.center());
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(rect)) {
			aGraphics.setColor(Color.black);
			para.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint minutes.
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_minutes_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		int minutes = this.getModel().minutes();
		String string = String.valueOf(minutes);
		if (string.length() < 2) {
			string = "0".concat(string);
		}
		StComposedText para = new StComposedText(string);
		StRectangle bounds = new StRectangle(para.bounds());
		StRectangle rect = bounds.align_with_(bounds.center(), aRectangle.center());
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(rect)) {
			aGraphics.setColor(Color.black);
			para.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint seconds.
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_seconds_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		int seconds = this.getModel().seconds();
		String string = String.valueOf(seconds);
		if (string.length() < 2) {
			string = "0".concat(string);
		}
		StComposedText para = new StComposedText(string);
		StRectangle bounds = new StRectangle(para.bounds());
		StRectangle rect = bounds.align_with_(bounds.center(), aRectangle.center());
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(rect)) {
			aGraphics.setColor(Color.black);
			para.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint label.
	 * @param aGraphics Graphics
	 * @param aString String
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_label_clip_(Graphics aGraphics, String aString, StRectangle aRectangle) {
		StComposedText para = new StComposedText(aString);
		StRectangle bounds = new StRectangle(para.bounds());
		StRectangle rect = bounds.align_with_(bounds.leftCenter(), aRectangle.leftCenter());
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(rect)) {
			aGraphics.setColor(Color.black);
			para.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint button.
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_button_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		Polygon polygon;
		StRectangle box = aRectangle;
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(box)) {
			if (aSymbol == $("hourDown") || aSymbol == $("minuteDown") || aSymbol == $("secondDown")) {
				polygon = new Polygon();
				polygon.addPoint(box.topRight().x, box.topRight().y);
				polygon.addPoint(box.bottomRight().x, box.bottomRight().y);
				polygon.addPoint(box.leftCenter().x, box.leftCenter().y);
				polygon.addPoint(box.topRight().x, box.topRight().y);
			} else {
				polygon = new Polygon();
				polygon.addPoint(box.topLeft().x, box.topLeft().y);
				polygon.addPoint(box.bottomLeft().x, box.bottomLeft().y);
				polygon.addPoint(box.rightCenter().x, box.rightCenter().y);
				polygon.addPoint(box.topLeft().x, box.topLeft().y);
			}
			aGraphics.setColor(veryLightGray);
			aGraphics.fillPolygon(polygon);
			box = box.insetBy_(new StRectangle(1, 1, 2, 2));
			if (aSymbol == $("hourDown") || aSymbol == $("minuteDown") || aSymbol == $("secondDown")) {
				polygon = new Polygon();
				polygon.addPoint(box.topRight().x, box.topRight().y);
				polygon.addPoint(box.bottomRight().x, box.bottomRight().y);
				polygon.addPoint(box.leftCenter().x, box.leftCenter().y);
				polygon.addPoint(box.topRight().x, box.topRight().y);
			} else {
				polygon = new Polygon();
				polygon.addPoint(box.topLeft().x, box.topLeft().y);
				polygon.addPoint(box.bottomLeft().x, box.bottomLeft().y);
				polygon.addPoint(box.rightCenter().x, box.rightCenter().y);
				polygon.addPoint(box.topLeft().x, box.topLeft().y);
			}
			aGraphics.setColor(veryDarkGray);
			aGraphics.drawPolygon(polygon);
		}
	}

	/**
	 * Paint clock.
	 *
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_clock_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		this.displayOn_clockBackground_clip_(aGraphics, aSymbol, aRectangle);
		this.displayOn_clockShortHand_clip_(aGraphics, aSymbol, aRectangle);
		this.displayOn_clockLongHand_clip_(aGraphics, aSymbol, aRectangle);
		this.displayOn_clockTickHand_clip_(aGraphics, aSymbol, aRectangle);
	}

	/**
	 * Paint clock background.
	 *
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_clockBackground_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		StComposedText paragraph;
		if (clip.intersects_(aRectangle)) {
			aGraphics.setColor(Color.black);
			Jun2dPoint center = new Jun2dPoint(aRectangle.center());
			double radius = Math.min(aRectangle.width(), aRectangle.height()) / 2 - (new StComposedText(" ")).height();
			StRectangle rect = StRectangle.Origin_extent_(new Point(0, 0), new Point(2, 2));
			for (int number = 1; number <= 12; number++) {
				paragraph = new StComposedText(String.valueOf(number));
				double degree = (number - 3) * 30;
				double x = Math.cos(Math.toRadians(degree));
				double y = Math.sin(Math.toRadians(degree));
				paragraph.displayOn_at_(aGraphics, center.plus_(1).plus_((new Jun2dPoint(x, y)).multipliedBy_(radius)).minus_((new Jun2dPoint(paragraph.extent().width, paragraph.extent().height)).dividedBy_(2))._toPoint());
				Point displayPoint = center.plus_((new Jun2dPoint(x, y)).multipliedBy_(radius - 10))._toPoint();
				aGraphics.fillRect(displayPoint.x, displayPoint.y, rect.width(), rect.height());
			}
			paragraph = new StComposedText(JunSystem.$String("a.m.", "am"));
			rect = new StRectangle(paragraph.bounds());
			rect = rect.align_with_(rect.bottomLeft(), aRectangle.insetBy_(8).bottomLeft());
			if (this.getModel().hours() >= 12) {
				aGraphics.setColor(Color.lightGray);
			} else {
				aGraphics.setColor(Color.black);
			}
			paragraph.displayOn_at_(aGraphics, rect.origin());
			paragraph = new StComposedText(JunSystem.$String("p.m.", "pm"));
			rect = new StRectangle(paragraph.bounds());
			rect = rect.align_with_(rect.bottomRight(), aRectangle.insetBy_(8).bottomRight());
			if (this.getModel().hours() >= 12) {
				aGraphics.setColor(Color.black);
			} else {
				aGraphics.setColor(Color.lightGray);
			}
			paragraph.displayOn_at_(aGraphics, rect.origin());
		}
	}

	/**
	 * Paint clock short hand.
	 *
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_clockShortHand_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(aRectangle)) {
			Jun2dPoint center = new Jun2dPoint(aRectangle.center());
			double radius = Math.min(aRectangle.width(), aRectangle.height()) / 2 - (new StComposedText(" ")).height() - 8;
			double minute = this.getModel().minutes();
			double hour = this.getModel().hours() * 5 + minute / 12;
			double x = Math.cos(Math.toRadians(hour * 6 - 90));
			double y = Math.sin(Math.toRadians(hour * 6 - 90));
			Vector points = new Vector();
			points.add(center.plus_((Jun2dPoint) this.rotateBlock().value_value_(new Jun2dPoint(x * 3, y * 3), new java.lang.Double(-180))));
			points.add(center.plus_((Jun2dPoint) this.rotateBlock().value_value_(new Jun2dPoint(x * 3, y * 3), new java.lang.Double(-90))));
			points.add(center.plus_((new Jun2dPoint(x, y)).multipliedBy_(radius * 6 / 10)));
			points.add(center.plus_((Jun2dPoint) this.rotateBlock().value_value_(new Jun2dPoint(x * 3, y * 3), new java.lang.Double(90))));
			points.add(((Jun2dPoint) points.firstElement()).copy());
			Polygon polygon = new Polygon();
			Iterator pointsIterator = points.iterator();
			while (pointsIterator.hasNext()) {
				Jun2dPoint aPoint = ((Jun2dPoint) pointsIterator.next()).rounded();
				int px = (int) aPoint.x();
				int py = (int) aPoint.y();
				polygon.addPoint(px, py);
			}
			((Graphics2D) aGraphics).setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			aGraphics.setColor(Color.lightGray);
			aGraphics.fillPolygon(polygon);
			aGraphics.setColor(Color.black);
			aGraphics.drawPolygon(polygon);
		}
	}

	/**
	 * Paint clock long hand.
	 *
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_clockLongHand_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(aRectangle)) {
			Jun2dPoint center = new Jun2dPoint(aRectangle.center());
			double radius = Math.min(aRectangle.width(), aRectangle.height()) / 2 - (new StComposedText(" ")).height() - 8;
			double minute = this.getModel().minutes();
			double x = Math.cos(Math.toRadians(minute * 6 - 90));
			double y = Math.sin(Math.toRadians(minute * 6 - 90));
			Vector points = new Vector();
			points.add(center.plus_((Jun2dPoint) this.rotateBlock().value_value_(new Jun2dPoint(x * 3, y * 3), new java.lang.Double(-180))));
			points.add(center.plus_((Jun2dPoint) this.rotateBlock().value_value_(new Jun2dPoint(x * 3, y * 3), new java.lang.Double(-90))));
			points.add(center.plus_((new Jun2dPoint(x, y)).multipliedBy_(radius * 9 / 10)));
			points.add(center.plus_((Jun2dPoint) this.rotateBlock().value_value_(new Jun2dPoint(x * 3, y * 3), new java.lang.Double(90))));
			points.add(((Jun2dPoint) points.firstElement()).copy());
			Polygon polygon = new Polygon();
			Iterator pointsIterator = points.iterator();
			while (pointsIterator.hasNext()) {
				Jun2dPoint aPoint = ((Jun2dPoint) pointsIterator.next()).rounded();
				int px = (int) aPoint.x();
				int py = (int) aPoint.y();
				polygon.addPoint(px, py);
			}
			((Graphics2D) aGraphics).setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			aGraphics.setColor(Color.lightGray);
			aGraphics.fillPolygon(polygon);
			aGraphics.setColor(Color.black);
			aGraphics.drawPolygon(polygon);
		}
	}

	/**
	 * Paint clock tick hand.
	 *
	 * @param aGraphics Graphics
	 * @param aSymbol StSymbol
	 * @param aRectangle StRectangle
	 * @category displaying
	 */
	public void displayOn_clockTickHand_clip_(Graphics aGraphics, StSymbol aSymbol, StRectangle aRectangle) {
		StRectangle clip = new StRectangle(aGraphics.getClipBounds());
		if (clip.intersects_(aRectangle)) {
			Jun2dPoint center = new Jun2dPoint(aRectangle.center());
			double radius = Math.min(aRectangle.width(), aRectangle.height()) / 2 - (new StComposedText(" ")).height() - 4;
			double length = radius * 9.5 / 10;
			double seconds = this.getModel().seconds();
			double x = Math.cos(Math.toRadians(seconds * 6 - 90));
			double y = Math.sin(Math.toRadians(seconds * 6 - 90));
			((Graphics2D) aGraphics).setStroke(new BasicStroke(1, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			aGraphics.setColor(Color.black);
			Point fromPoint = center._toPoint();
			Point toPoint = center.plus_((new Jun2dPoint(x, y)).multipliedBy_(length))._toPoint();
			aGraphics.drawLine(fromPoint.x, fromPoint.y, toPoint.x, toPoint.y);
		}
	}

	/**
	 * Update.
	 *
	 * @param evt DependentEvent
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		this.makeFrames();

		if (this.isShowing() == false) {
			return;
		}

		if (evt.getAspect() == $("hours")) {
			repaint();
		}
		if (evt.getAspect() == $("minutes")) {
			repaint();
		}
		if (evt.getAspect() == $("seconds")) {
			repaint();
		}
		super.update_(evt);
	}

	/**
	 * Set bounds.
	 * 
	 * @param x The new x-coordinate of this component.
	 * @param y The new y-coordinate of this component.
	 * @param width The new width-coordinate of this component.
	 * @param height The new height-coordinate of this component.
	 * @category bounds accessing
	 */
	public void setBounds(int x, int y, int width, int height) {
		super.setBounds(x, y, width, height);
		this.makeFrames();
	}

	/**
	 * Build component.
	 * @category interface opening
	 */
	protected void buildComponent() {
		this.setSize(150, 200);
	}

	/**
	 * Get model.
	 *
	 * @return JunCalendarTimeModel
	 * @category accessing
	 */
	public JunCalendarTimeModel getModel() {
		return (JunCalendarTimeModel) this.model();
	}

	/**
	 * Get default controller.
	 *
	 * @return JunCalendarTimeController
	 * @category accessing
	 */
	protected StController defaultController() {
		return new JunCalendarTimeController();
	}

	/**
	 * Get default model.
	 *
	 * @return jp.co.sra.smalltalk.StModel
	 * @category accessing
	 */
	protected StModel defaultModel() {
		return new JunCalendarTimeModel();
	}

}
