package jp.co.sra.jun.goodies.catalog;

import java.io.File;
import java.util.Arrays;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.qt4jun.JunQTMovie;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunMovieCatalog class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2003/05/21 (by Mitsuhiro Asada)
 *  @updated   2006/04/11 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun640 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunMovieCatalog.java,v 8.12 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunMovieCatalog extends JunVisualCatalog {
	protected double[] visualClip;

	/**
	 * Create a new instance of <code>JunMovieCatalog</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunMovieCatalog() {
		super();
	}

	/**
	 * Create a new instance of <code>JunMovieCatalog</code> and set a directory.
	 * 
	 * @param directoryFilename java.io.File
	 * @category Instance creation
	 */
	public JunMovieCatalog(File directoryFilename) {
		super(directoryFilename);
	}

	/**
	 * Create a new instance of <code>JunMovieCatalog</code> and set a directory.
	 * 
	 * @param fromDirectoryFilename java.io.File
	 * @param toDirectoryFilename java.io.File
	 * @category Instance creation
	 */
	public JunMovieCatalog(File fromDirectoryFilename, File toDirectoryFilename) {
		super(fromDirectoryFilename, toDirectoryFilename);
	}

	/**
	 * Create a new instance of <code>JunMovieCatalog</code> and dive to a directory.
	 * 
	 * @return jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @param directoryFilename java.io.File
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#Dive_(java.io.File)
	 * @category Utilities
	 */
	public static JunVisualCatalog Dive_(File directoryFilename) {
		return _Dive_catalog_(directoryFilename, new JunMovieCatalog());
	}

	/**
	 * Create a new instance of <code>JunMovieCatalog</code> and dive to a directory.
	 * 
	 * @return jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @param directoryFilename java.io.File
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#Dive_block_(java.io.File, jp.co.sra.smalltalk.StBlockClosure)
	 * @category Utilities
	 */
	public static JunVisualCatalog Dive_block_(File directoryFilename, StBlockClosure aBlock) {
		return _Dive_block_catalog_(directoryFilename, aBlock, new JunMovieCatalog());
	}

	/**
	 * Initialize this when created.
	 * 
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		visualClip = null;
	}

	/**
	 * Answer the visual clip.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] visualClip() {
		return visualClip;
	}

	/**
	 * Set the visual clip.
	 * 
	 * @param anArray double[]
	 * @category accessing
	 */
	public void visualClip_(double[] anArray) {
		visualClip = anArray;
	}

	/**
	 * Answer the default title string.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#defaultTitleString()
	 * @category defaults
	 */
	public String defaultTitleString() {
		return "Movie Catalog";
	}

	/**
	 * Answer the default visual catalog directory name.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#defaultVisualCatalogDirectoryName()
	 * @category defaults
	 */
	public String defaultVisualCatalogDirectoryName() {
		return "MovieCatalog";
	}

	/**
	 * Answer the default visual clip directory name.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#defaultVisualClipDirectoryName()
	 * @category defaults
	 */
	public String defaultVisualClipDirectoryName() {
		return "movies";
	}

	/**
	 * Answer the visual clip files.
	 * 
	 * @return java.io.File[]
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#visualClipFiles()
	 * @category file accessing
	 */
	public File[] visualClipFiles() {
		File[] originalVisualFiles = this.originalVisualFiles();
		File[] visualClipFiles = new File[originalVisualFiles.length];
		for (int i = 0; i < originalVisualFiles.length; i++) {
			String aString = originalVisualFiles[i].getName();
			File aFilename = new File(this.visualClipDirectory(), aString);
			visualClipFiles[i] = aFilename;
		}
		return visualClipFiles;
	}

	/**
	 * Make visual clip files.
	 * 
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#makeVisualClipFiles()
	 * @category making
	 */
	protected void makeVisualClipFiles() {
		progressValue.value_(0.0f);
		int anIndex = 1;
		File[] originalVisualFiles = this.originalVisualFiles();
		File[] visualClipFiles = this.visualClipFiles();

		for (int i = 0; i < originalVisualFiles.length; i++) {
			String aString = $String("Movie:") + " " + visualClipFiles[i].getName();
			progressMessage.value_(JunStringUtility._ContractString_to_(aString, this.defaultContractLength()));

			if (originalVisualFiles[i].equals(visualClipFiles[i]) == false) {
				JunMoviePlayer aMoviePlayer = null;
				try {
					aMoviePlayer = new JunMoviePlayer(originalVisualFiles[i]);
					aMoviePlayer.trackerModel().enableMarkers();
					if (this.visualClip() == null) {
						aMoviePlayer.trackerModel().lastMarker_(1.0d);
						aMoviePlayer.trackerModel().firstMarker_(0.0d);
					} else {
						aMoviePlayer.trackerModel().lastMarker_(this.visualClip()[1]);
						aMoviePlayer.trackerModel().firstMarker_(this.visualClip()[0]);
					}
					this.saveMovie_into_(aMoviePlayer, visualClipFiles[i]);
				} finally {
					if (aMoviePlayer != null) {
						aMoviePlayer.release();
						aMoviePlayer = null;
					}
				}
			}

			progressValue.value_(((float) anIndex) / this.numberOfVisuals());
			anIndex = anIndex + 1;
		}
	}

	/**
	 * Answer true if the specified extension string is for a movie, otherwise false.
	 * 
	 * @param extensionString java.lang.String
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.catalog.JunVisualCatalog#incluesExtensions_(java.lang.String)
	 * @category testing
	 */
	public boolean incluesExtensions_(String extensionString) {
		String extension = extensionString.toLowerCase();
		return Arrays.asList(JunSystem.DefaultMovieExtensions()).contains(extension);
	}

	/**
	 * Save movie into aFilename.
	 * 
	 * @param aMoviePlayer jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer
	 * @param aFilename java.io.File
	 * @category private
	 */
	protected void saveMovie_into_(JunMoviePlayer aMoviePlayer, File aFilename) {
		int start = 0;
		int duration = 0;
		if (aMoviePlayer.isIntervalNotEmpty()) {
			start = Math.round((float) (aMoviePlayer.trackerModel().firstMarker() * aMoviePlayer.movieHandle().duration()));
			duration = Math.round((float) ((aMoviePlayer.trackerModel().lastMarker() - aMoviePlayer.trackerModel().firstMarker()) * aMoviePlayer.movieHandle().duration()));
		} else {
			start = 0;
			duration = aMoviePlayer.movieHandle().duration();
		}

		JunQTMovie newMovie = null;
		JunCursors cursor = new JunCursors(JunCursors.ExecuteCursor());
		try {
			cursor._show();
			aMoviePlayer.movieHandle().selection_duration_(start, duration);
			newMovie = aMoviePlayer.movieHandle().copyMovie();
		} finally {
			cursor._restore();
		}

		if (newMovie == null) {
			return;
		}

		cursor = new JunCursors(JunCursors.WriteCursor());
		try {
			cursor._show();
			newMovie.flattenToFile(aFilename);
		} finally {
			newMovie.release();
			cursor._restore();
		}
	}
}
