package jp.co.sra.jun.goodies.catalog;

import java.awt.Dimension;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.TreeSet;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.image.streams.JunJpegImageStream;
import jp.co.sra.jun.goodies.image.support.JunImageAdjuster;
import jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer;
import jp.co.sra.jun.goodies.progress.JunProgress;
import jp.co.sra.jun.goodies.tips.JunURL;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunVisualCatalog class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2003/05/19 (by Mitsuhiro Asada)
 *  @updated   2006/04/11 (by Mitsuhiro Asada)
 *  @updated   2006/12/06 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun640 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVisualCatalog.java,v 8.13 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunVisualCatalog extends StObject {
	public static final String DefaultIndexHtmlFileName = "index.html";
	public static final int DIVE_MAX_LEVEL = 8;
	public static final int THUMBNAIL_IMAGE_MIN_SIZE = 32;
	public static final int THUMBNAIL_IMAGE_MAX_SIZE = 256;

	protected File originalVisualDirectory;
	protected File[] originalVisualFiles;
	protected File visualCatalogDirectory;
	protected double keyFramePosition;
	protected Dimension thumbnailImageSize;
	protected int numberOfColumns;
	protected boolean withFileName;
	protected StValueHolder progressValue;
	protected StValueHolder progressMessage;

	/**
	 * Create a new instance of <code>JunVisualCatalog</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunVisualCatalog() {
		super();
		this.initialize();
	}

	/**
	 * Create a new instance of <code>JunVisualCatalog</code> and initialize it.
	 * 
	 * @param directoryFilename java.io.File
	 * @category Instance creation
	 */
	public JunVisualCatalog(File directoryFilename) {
		this();
		this.originalVisualDirectory_(directoryFilename);
	}

	/**
	 * Create a new instance of <code>JunVisualCatalog</code> and initialize it.
	 * 
	 * @param fromDirectoryFilename java.io.File
	 * @param toDirectoryFilename java.io.File
	 * @category Instance creation
	 */
	public JunVisualCatalog(File fromDirectoryFilename, File toDirectoryFilename) {
		this();
		this.originalVisualDirectory_(fromDirectoryFilename);
		this.visualCatalogDirectory_(toDirectoryFilename);
	}

	/**
	 * Create a new instance of <code>JunVisualCatalog</code> and dive to a directory.
	 * 
	 * @return jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @param directoryFilename java.io.File
	 * @category Utilities
	 */
	public static JunVisualCatalog Dive_(File directoryFilename) {
		return _Dive_catalog_(directoryFilename, new JunVisualCatalog());
	}

	/**
	 * Create a new instance of <code>JunVisualCatalog</code> and dive to a directory.
	 * 
	 * @return jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @param directoryFilename java.io.File
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category Utilities
	 */
	public static JunVisualCatalog Dive_block_(File directoryFilename, StBlockClosure aBlock) {
		return _Dive_block_catalog_(directoryFilename, aBlock, new JunVisualCatalog());
	}

	/**
	 * Answer separated strings.
	 *
	 * @return java.lang.String[]
	 * @param separateCollection java.lang.String
	 * @param dividerCollection char[]
	 * @category Utilities
	 */
	public static String[] Separate_dividers_(String separateCollection, char[] dividerCollection) {
		return JunStringUtility.Separate_dividers_(separateCollection, dividerCollection);
	}

	/**
	 * Create a new instance of <code>JunVisualCatalog</code> and dive to a directory.
	 * 
	 * @return jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @param directoryFilename java.io.File
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param visualCatalog jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @category Private
	 */
	protected static JunVisualCatalog _Dive_block_catalog_(File directoryFilename, StBlockClosure aBlock, JunVisualCatalog visualCatalog) {
		Collection originalDirectoryNames = new Vector();
		Dive_into_(directoryFilename, originalDirectoryNames);
		originalDirectoryNames = new TreeSet(originalDirectoryNames);
		Vector targetDirectoryNames = new Vector(originalDirectoryNames.size());
		String directoryName = directoryFilename.getPath();
		if (directoryName.charAt(directoryName.length() - 1) == File.separatorChar) {
			directoryName = directoryName.substring(0, directoryName.length() - 1);
		}
		String baseString = new File(visualCatalog.defaultDirectory(), visualCatalog.defaultVisualCatalogDirectoryName()).getPath();
		String[] strings = new String[originalDirectoryNames.size()];
		originalDirectoryNames.toArray(strings);
		for (int i = 0; i < strings.length; i++) {
			String aString = directoryFilename.getParent();
			aString = strings[i].substring(aString.length() + 1, strings[i].length());
			File aFilename = new File(baseString, aString);
			aString = aFilename.getPath();
			targetDirectoryNames.add(aString);
		}

		Vector removeCollection = new Vector();
		for (int i = 0; i < targetDirectoryNames.size(); i++) {
			String targetDirectoryName = (String) targetDirectoryNames.get(i);
			JunVisualCatalog aVisualCatalog = (JunVisualCatalog) StObject._New(visualCatalog.getClass(), new File(strings[i]), new File(targetDirectoryName));
			aBlock.value_(aVisualCatalog);
			if (aVisualCatalog.isEmpty()) {
				removeCollection.add(targetDirectoryName);
			}
		}
		targetDirectoryNames.removeAll(removeCollection);

		BufferedWriter writer = null;
		JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
		try {
			cursor._show();

			writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(visualCatalog.indexHtmlFile()), "MS932"));
			writer.write("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">");
			writer.newLine();
			writer.write("<HTML>");
			writer.newLine();
			writer.write("<HEAD>");
			writer.newLine();
			writer.write("<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=Shift_JIS\">");
			writer.newLine();
			writer.write("<LINK REV=\"MADE\" HREF=\"mailto:jun@ktlsva.sra.co.jp\">");
			writer.newLine();
			writer.write("<LINK REL=\"INDEX\" HREF=\"./index.html\">");
			writer.newLine();
			writer.write("<TITLE>" + visualCatalog.defaultTitleString() + "s" + "</TITLE>");
			writer.newLine();
			writer.write("</HEAD>");
			writer.newLine();
			writer.write("<BODY BGCOLOR=\"#FFFFFF\" TEXT=\"#000000\" LINK=\"#0000FF\" VLINK=\"#663399\" ALINK=\"#FF0000\">");
			writer.newLine();
			writer.write("<H2>" + visualCatalog.defaultTitleString() + "s" + "</H2>");
			writer.newLine();
			writer.write("<OL>");
			writer.newLine();
			for (Enumeration enumeration = targetDirectoryNames.elements(); enumeration.hasMoreElements();) {
				String each = ((String) enumeration.nextElement());
				String aString = each.substring(baseString.length(), each.length());
				aString = aString.replace(File.separatorChar, '/');
				aString = "." + aString;
				writer.write("<LI><A HREF=\"");
				if (aString.charAt(aString.length() - 1) == '/') {
					writer.write(aString + visualCatalog.DefaultIndexHtmlFileName);
				} else {
					writer.write(aString + "/" + visualCatalog.DefaultIndexHtmlFileName);
				}
				writer.write("\">" + aString + "</A>");
				writer.newLine();
			}
			writer.write("</OL>");
			writer.newLine();
			writer.write("<HR>");
			writer.newLine();
			writer.write("This page was created by " + visualCatalog._className() + " on " + JunSystem.FullName());
			writer.newLine();
			writer.write("</BODY>");
			writer.newLine();
			writer.write("</HTML>");
			writer.newLine();
		} catch (IOException e) {
			System.out.println(e.getMessage());
			e.printStackTrace();
		} finally {
			if (writer != null) {
				try {
					writer.flush();
					writer.close();
				} catch (IOException e) {
				} finally {
					writer = null;
				}
			}
			cursor._restore();
		}

		return visualCatalog;
	}

	/**
	 * Create a new instance of <code>JunVisualCatalog</code> and dive to a directory.
	 * 
	 * @return jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @param directoryFilename java.io.File
	 * @param visualCatalog jp.co.sra.jun.goodies.catalog.JunVisualCatalog
	 * @category Private
	 */
	protected static JunVisualCatalog _Dive_catalog_(File directoryFilename, JunVisualCatalog visualCatalog) {
		return _Dive_block_catalog_(directoryFilename, new StBlockClosure() {
			public Object value_(Object anObject) {
				final JunVisualCatalog visualCatalog = (JunVisualCatalog) anObject;
				final JunProgress aProgress = new JunProgress();
				visualCatalog.computeWhenProgressMessageChanged_(new StBlockClosure() {
					public Object value_(Object m) {
						aProgress.message_((String) m);
						return null;
					}
				});
				visualCatalog.computeWhenProgressValueChanged_(new StBlockClosure() {
					public Object value_(Object v) {
						aProgress.value_(((Number) v).floatValue());
						return null;
					}
				});
				aProgress.do_(new StBlockClosure() {
					public Object value() {
						visualCatalog.make();
						return null;
					}
				});

				return null;
			}
		}, visualCatalog);
	}

	/**
	 * Answer the collection of files contains the specifiied directory.
	 * 
	 * @return java.util.Collection
	 * @param directoryName java.io.File
	 * @param aCollection java.util.Collection
	 * @category Private
	 */
	protected static Collection Dive_into_(File directoryName, Collection aCollection) {
		return Dive_into_(directoryName.getPath(), aCollection);
	}

	/**
	 * Answer the collection of files contains the specifiied directory.
	 * 
	 * @return java.util.Collection
	 * @param directoryName java.lang.String
	 * @param aCollection java.util.Collection
	 * @category Private
	 */
	protected static Collection Dive_into_(String directoryName, Collection aCollection) {
		File aFilename = new File(directoryName);
		int startLevel = Separate_dividers_(aFilename.getPath(), new char[] { File.separatorChar }).length;
		return Dive_into_start_(aFilename.getPath(), aCollection, startLevel);
	}

	/**
	 * Answer the collection of files contains the specifiied directory with start level.
	 * 
	 * @return java.util.Collection
	 * @param directoryName java.lang.String
	 * @param resultCollection java.util.Set
	 * @param startLevel int
	 * @category Private
	 */
	protected static Collection Dive_into_start_(String directoryName, Collection resultCollection, int startLevel) {
		File aFilename = new File(directoryName);
		int diveLevel = Separate_dividers_(aFilename.getPath(), new char[] { File.separatorChar }).length;
		if (diveLevel - startLevel > JunVisualCatalog.DIVE_MAX_LEVEL) {
			return resultCollection;
		}
		if (resultCollection.contains(aFilename.getPath())) {
			return resultCollection;
		}
		if (aFilename.exists() && aFilename.isDirectory() && aFilename.canRead()) {
			resultCollection.add(aFilename.getPath());
			String[] aCollection = aFilename.list();
			for (int i = 0; i < aCollection.length; i++) {
				String aString = null;
				if (directoryName.charAt(directoryName.length() - 1) == File.separatorChar) {
					aString = directoryName + aCollection[i];
				} else {
					aString = directoryName + File.separator + aCollection[i];
				}
				Dive_into_start_(aString, resultCollection, startLevel);
			}
			return resultCollection;
		}
		return resultCollection;
	}

	/**
	 * Initialize this when created.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		originalVisualDirectory = null;
		originalVisualFiles = null;
		visualCatalogDirectory = null;
		this.keyFramePosition_(this.defaultKeyFramePosition());
		this.thumbnailImageSize_(this.defaultThumbnailImageSize());
		this.numberOfColumns_(this.defaultNumberOfColumns());
		this.withFileName_(false);
		progressValue = new StValueHolder(0.0d);
		progressMessage = new StValueHolder("");
	}

	/**
	 * Answer the key frame position.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double keyFramePosition() {
		return keyFramePosition;
	}

	/**
	 * Set the key frame position.
	 * 
	 * @param normalizedValue double
	 * @category accessing
	 */
	public void keyFramePosition_(double normalizedValue) {
		keyFramePosition = normalizedValue;
	}

	/**
	 * Answer the number of columns.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfColumns() {
		return numberOfColumns;
	}

	/**
	 * Set the number of columns.
	 * 
	 * @param anInteger int
	 * @category accessing
	 */
	public void numberOfColumns_(int anInteger) {
		numberOfColumns = anInteger;
	}

	/**
	 * Answer the number of visuals.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfVisuals() {
		return this.originalVisualFiles().length;
	}

	/**
	 * Answer the original visual directory.
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File originalVisualDirectory() {
		return originalVisualDirectory;
	}

	/**
	 * Set the original visual directory.
	 * 
	 * @param aFilename java.io.File
	 * @category accessing
	 */
	public void originalVisualDirectory_(File aFilename) {
		originalVisualDirectory = aFilename;
		originalVisualFiles = null;
	}

	/**
	 * Answer the thumbnail image size.
	 * 
	 * @return java.awt.Dimension
	 * @category accessing
	 */
	public Dimension thumbnailImageSize() {
		return thumbnailImageSize;
	}

	/**
	 * Set the thumbnail image size.
	 * 
	 * @param sizeObject int
	 * @category accessing
	 */
	public void thumbnailImageSize_(int sizeObject) {
		this.thumbnailImageSize_(new Dimension(sizeObject, sizeObject));
	}

	/**
	 * Set the thumbnail image size.
	 * 
	 * @param sizeObject java.awt.Dimension
	 * @category accessing
	 */
	public void thumbnailImageSize_(Dimension sizeObject) {
		int width = Math.max(THUMBNAIL_IMAGE_MIN_SIZE, Math.min(sizeObject.width, THUMBNAIL_IMAGE_MAX_SIZE));
		int height = Math.max(THUMBNAIL_IMAGE_MIN_SIZE, Math.min(sizeObject.height, THUMBNAIL_IMAGE_MAX_SIZE));
		thumbnailImageSize = new Dimension(width, height);
	}

	/**
	 * Answer the visual catalog directory.
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File visualCatalogDirectory() {
		if (visualCatalogDirectory == null) {
			File aFilename = new File(this.defaultDirectory(), this.defaultVisualCatalogDirectoryName());
			this.visualCatalogDirectory_(aFilename);
		}
		return visualCatalogDirectory;
	}

	/**
	 * Set the visual catalog directory.
	 * 
	 * @param aFilename java.io.File
	 * @category accessing
	 */
	public void visualCatalogDirectory_(File aFilename) {
		visualCatalogDirectory = aFilename;
		if (visualCatalogDirectory.exists()) {
			if (visualCatalogDirectory.isDirectory() == false) {
				new SmalltalkException("Can't access directory: " + visualCatalogDirectory.toString());
				visualCatalogDirectory = null;
				return;
			}
		} else {
			visualCatalogDirectory.mkdirs();
		}
		if (this.isEmpty() == false) {
			File visualClipDirectory = this.visualClipDirectory();
			if (visualClipDirectory.exists() == false) {
				visualClipDirectory.mkdir();
			}
			File thumbnailImageDirectory = this.thumbnailImageDirectory();
			if (thumbnailImageDirectory.exists() == false) {
				thumbnailImageDirectory.mkdir();
			}
		}
	}

	/**
	 * Answer true if this object with file name, otherwise false.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean withFileName() {
		return withFileName == true;
	}

	/**
	 * Set the with file name.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void withFileName_(boolean aBoolean) {
		withFileName = aBoolean;
	}

	/**
	 * Answer the default contract length.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultContractLength() {
		return 32;
	}

	/**
	 * Answer the default directory name.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultDirectoryName() {
		return (String) System.getProperty("user.dir");
	}

	/**
	 * Answer the default key frame position.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultKeyFramePosition() {
		return 0.5d;
	}

	/**
	 * Answer the default number of columns.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultNumberOfColumns() {
		return 5;
	}

	/**
	 * Answer the default thumbnail image directory name.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultThumbnailImageDirectoryName() {
		return "thumbnails";
	}

	/**
	 * Answer the default thumbnail image size.
	 * 
	 * @return java.awt.Dimension
	 * @category defaults
	 */
	public Dimension defaultThumbnailImageSize() {
		return new Dimension(72, 72);
	}

	/**
	 * Answer the default title string.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultTitleString() {
		return "Visual Catalog";
	}

	/**
	 * Answer the default visual catalog directory name.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultVisualCatalogDirectoryName() {
		return "VisualCatalog";
	}

	/**
	 * Answer the default visual clip directory name.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultVisualClipDirectoryName() {
		return "visuals";
	}

	/**
	 * Answer the default directory.
	 * 
	 * @return java.io.File
	 * @category directory accessing
	 */
	public File defaultDirectory() {
		return new File(this.defaultDirectoryName());
	}

	/**
	 * Answer the thumbnail image directory.
	 * 
	 * @return java.io.File
	 * @category directory accessing
	 */
	public File thumbnailImageDirectory() {
		return new File(this.visualCatalogDirectory(), this.defaultThumbnailImageDirectoryName());
	}

	/**
	 * Answer the visual clip directory.
	 * 
	 * @return java.io.File
	 * @category directory accessing
	 */
	public File visualClipDirectory() {
		return new File(this.visualCatalogDirectory(), this.defaultVisualClipDirectoryName());
	}

	/**
	 * Compute when progress message changed.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category progress
	 */
	public void computeWhenProgressMessageChanged_(StBlockClosure aBlock) {
		progressMessage.compute_(aBlock);
	}

	/**
	 * Compute when progress value changed.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category progress
	 */
	public void computeWhenProgressValueChanged_(StBlockClosure aBlock) {
		progressValue.compute_(aBlock);
	}

	/**
	 * Answer the index.html file.
	 * 
	 * @return java.io.File
	 * @category file accessing
	 */
	public File indexHtmlFile() {
		return new File(this.visualCatalogDirectory(), JunVisualCatalog.DefaultIndexHtmlFileName);
	}

	/**
	 * Answer the original visual files.
	 * 
	 * @return java.io.File[]
	 * @category file accessing
	 */
	public File[] originalVisualFiles() {
		if (this.originalVisualDirectory() == null) {
			return new File[0];
		}

		ArrayList aList = new ArrayList();
		String[] lists = this.originalVisualDirectory().list();
		for (int i = 0; i < lists.length; i++) {
			String each = lists[i];
			int index = each.lastIndexOf('.');
			if (index >= 0) {
				String extensionString = each.substring(index + 1).toLowerCase();
				if (this.incluesExtensions_(extensionString)) {
					File aFile = new File(this.originalVisualDirectory(), each);
					aList.add(aFile);
				}
			}
		}

		originalVisualFiles = (File[]) aList.toArray(new File[aList.size()]);
		return originalVisualFiles;
	}

	/**
	 * Answer the thumbnail image files.
	 * 
	 * @return java.io.File[]
	 * @category file accessing
	 */
	public File[] thumbnailImageFiles() {
		File[] originalVisualFiles = this.originalVisualFiles();
		File[] thumbnailImageFiles = new File[originalVisualFiles.length];
		for (int i = 0; i < originalVisualFiles.length; i++) {
			File each = originalVisualFiles[i];
			String aString = each.getName();
			int index = aString.lastIndexOf('.');
			String baseString = (index < 0) ? aString : aString.substring(0, index);
			String extensionString = (index < 0) ? "" : aString.substring(index + 1);
			aString = baseString + '_' + extensionString + '.' + JunSystem.DefaultImageExtension();
			File aFilename = new File(this.thumbnailImageDirectory(), aString);
			thumbnailImageFiles[i] = aFilename;
		}

		return thumbnailImageFiles;
	}

	/**
	 * Answer the visual clip files.
	 * 
	 * @return java.io.File[]
	 * @category file accessing
	 */
	public File[] visualClipFiles() {
		File[] originalVisualFiles = this.originalVisualFiles();
		File[] visualClipFiles = new File[originalVisualFiles.length];
		for (int i = 0; i < originalVisualFiles.length; i++) {
			File each = originalVisualFiles[i];
			String aString = each.getName();
			File aFilename = new File(this.visualClipDirectory(), aString);
			visualClipFiles[i] = aFilename;
		}

		return visualClipFiles;
	}

	/**
	 * Make visual clip files, thumenail image files and index.html file.
	 * 
	 * @category making
	 */
	public void make() {
		this.makeVisualClipFiles();
		this.makeThumbnailImageFiles();
		this.makeIndexHtmlFile();
	}

	/**
	 * Make index.html file.
	 * 
	 * @category making
	 */
	protected void makeIndexHtmlFile() {
		final File[] visualClipFiles = this.visualClipFiles();
		final File[] thumbnailImageFiles = this.thumbnailImageFiles();
		BufferedWriter bufferedWriter = null;
		JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
		try {
			cursor._show();

			bufferedWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(this.indexHtmlFile()), "MS932"));
			final BufferedWriter aWriter = bufferedWriter;
			final JunVisualCatalog self = this;

			aWriter.write("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">");
			aWriter.newLine();
			this.on_tag_do_(aWriter, "HTML", new StBlockClosure() {
				public Object value() {
					try {
						aWriter.newLine();

						JunVisualCatalog.this.on_tag_do_(aWriter, "HEAD", new StBlockClosure() {
							public Object value() {
								try {
									aWriter.newLine();
									aWriter.write("<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=Shift_JIS\">");
									aWriter.newLine();
									aWriter.write("<LINK REV=\"MADE\" HREF=\"mailto:jun@ktlsva.sra.co.jp\">");
									aWriter.newLine();
									aWriter.write("<LINK REL=\"INDEX\" HREF=\"./index.html\">");
									aWriter.newLine();
									aWriter.write("<TITLE>" + self.defaultTitleString() + "</TITLE>");
									aWriter.newLine();
									return null;
								} catch (IOException e) {
									throw new RuntimeException(e);
								}
							}
						});
						aWriter.newLine();

						JunVisualCatalog.this.on_tag_attributes_do_(aWriter, "BODY", "BGCOLOR=\"#FFFFFF\" TEXT=\"#000000\" LINK=\"#0000FF\" VLINK=\"#663399\" ALINK=\"#FF0000\"", new StBlockClosure() {
							public Object value() {
								try {
									aWriter.newLine();

									JunVisualCatalog.this.on_tag_do_(aWriter, "H2", new StBlockClosure() {
										public Object value() {
											try {
												aWriter.write(self.defaultTitleString());
											} catch (IOException e) {
												throw new RuntimeException(e);
											}
											return null;
										}
									});
									aWriter.newLine();

									if (self.isEmpty() == false) {
										JunVisualCatalog.this.on_tag_attributes_do_(aWriter, "TABLE", "BORDER=\"0\" CELLPADDING=\"5\"", new StBlockClosure() {
											public Object value() {
												try {
													aWriter.newLine();

													JunVisualCatalog.this.on_tag_do_(aWriter, "TR", new StBlockClosure() {
														public Object value() {
															try {
																aWriter.newLine();

																self.urlsDo_(new StBlockClosure() {
																	public Object valueWithArguments_(Object[] anArray) {
																		final String movieClipUrl = (String) anArray[0];
																		final String thumbnailImageUrl = (String) anArray[1];
																		final Dimension thumbnailImageExtent = (Dimension) anArray[2];
																		final int anIndex = ((Number) anArray[3]).intValue();

																		try {
																			JunVisualCatalog.this.on_tag_attributes_do_(aWriter, "TD", "ALIGN=\"CENTER\" VALIGN=\"MIDDLE\"", new StBlockClosure() {
																				public Object value() {
																					try {
																						aWriter.newLine();

																						JunVisualCatalog.this.on_tag_attributes_do_(aWriter, "A", "HREF=\"" + movieClipUrl + "\"", new StBlockClosure() {
																							public Object value() {
																								try {
																									aWriter.write("<IMG SRC=\"");
																									aWriter.write(thumbnailImageUrl);
																									aWriter.write("\" WIDTH=\"");
																									aWriter.write(Integer.toString(thumbnailImageExtent.width));
																									aWriter.write("\" HEIGHT=\"");
																									aWriter.write(Integer.toString(thumbnailImageExtent.height));
																									aWriter.write("\" BORDER=\"1\" ALT=\"");
																									aWriter.write(thumbnailImageFiles[anIndex].getName());
																									aWriter.write("\">");
																									if (self.withFileName()) {
																										aWriter.write("<BR>");
																										aWriter.write(visualClipFiles[anIndex].getName());
																									}
																								} catch (IOException e) {
																									throw new RuntimeException(e);
																								}
																								return null;
																							}
																						});
																						aWriter.newLine();
																					} catch (IOException e) {
																						throw new RuntimeException(e);
																					}
																					return null;
																				}
																			});

																			aWriter.newLine();

																			if ((anIndex + 1) % self.numberOfColumns() == 0 && anIndex < self.numberOfVisuals()) {
																				aWriter.write("</TR>");
																				aWriter.newLine();
																				aWriter.write("<TR>");
																				aWriter.newLine();
																			}
																		} catch (IOException e) {
																			throw new RuntimeException(e);
																		}
																		return null;
																	}
																});
																return null;
															} catch (IOException e) {
																throw new RuntimeException(e);
															}
														}
													});
													aWriter.newLine();
													return null;
												} catch (IOException e) {
													throw new RuntimeException(e);
												}
											}
										});
										aWriter.newLine();

										aWriter.write("<HR>");
										aWriter.newLine();
										aWriter.write("This page was created by ");
										aWriter.write(self._className().toString());
										aWriter.write(" on ");
										aWriter.write(JunSystem.FullName());
										aWriter.newLine();
									}
								} catch (IOException e) {
									throw new RuntimeException(e);
								}
								return null;

							}
						});
						aWriter.newLine();
					} catch (IOException e) {
						throw new RuntimeException(e);
					}
					return null;
				}
			});
			aWriter.newLine();
		} catch (Exception e) {
			System.out.println(e.getMessage());
			e.printStackTrace();
		} finally {
			if (bufferedWriter != null) {
				try {
					bufferedWriter.flush();
					bufferedWriter.close();
				} catch (IOException e) {
				} finally {
					bufferedWriter = null;
				}
			}
			cursor._restore();
		}
	}

	/**
	 * Make thumbnail image files.
	 * 
	 * @category making
	 */
	protected void makeThumbnailImageFiles() {
		progressValue.value_(0.0f);
		int anIndex = 1;
		File[] originalVisualFiles = this.originalVisualFiles();
		File[] thumbnailImageFiles = this.thumbnailImageFiles();
		for (int i = 0; i < originalVisualFiles.length; i++) {
			String aString = $String("Thumbnail:") + " " + thumbnailImageFiles[i].getName();
			progressMessage.value_(JunStringUtility._ContractString_to_(aString, this.defaultContractLength()));

			StImage anImage = null;
			JunMoviePlayer aMoviePlayer = null;
			try {
				aMoviePlayer = new JunMoviePlayer(originalVisualFiles[i]);
				aMoviePlayer.goto_(this.keyFramePosition());
				anImage = aMoviePlayer.asImage();
			} finally {
				if (aMoviePlayer != null) {
					aMoviePlayer.release();
					aMoviePlayer = null;
				}
			}

			JunImageAdjuster anAdjuster = new JunImageAdjuster();
			Dimension imageExtent = this.proportionalExtentOfImage_within_(anImage, this.thumbnailImageSize());
			anImage = anAdjuster.adjust_extent_(anImage, imageExtent);

			JunImageStream writeStream = null;
			try {
				writeStream = JunJpegImageStream.On_(new BufferedOutputStream(new FileOutputStream(thumbnailImageFiles[i])));
				writeStream.nextPutImage_(anImage);
			} catch (IOException e) {
				System.out.println(e.getMessage());
				e.printStackTrace();
			} finally {
				if (writeStream != null) {
					try {
						writeStream.flush();
						writeStream.close();
					} catch (IOException e) {
					} finally {
						writeStream = null;
					}
				}
			}

			progressValue.value_(((float) anIndex) / this.numberOfVisuals());
			anIndex = anIndex + 1;
		}
	}

	/**
	 * Make visual clip files.
	 * 
	 * @category making
	 */
	protected void makeVisualClipFiles() {
		progressValue.value_(0.0f);
		int anIndex = 1;
		File[] originalVisualFiles = this.originalVisualFiles();
		File[] visualClipFiles = this.visualClipFiles();
		for (int i = 0; i < originalVisualFiles.length; i++) {
			String aString = $String("Visual:") + " " + visualClipFiles[i].getName();
			progressMessage.value_(JunStringUtility._ContractString_to_(aString, this.defaultContractLength()));

			if (originalVisualFiles[i].equals(visualClipFiles[i]) == false) {
				InputStream readStream = null;
				OutputStream writeStream = null;
				try {
					readStream = new BufferedInputStream(new FileInputStream(originalVisualFiles[i]));
					writeStream = new BufferedOutputStream(new FileOutputStream(visualClipFiles[i]));
					byte[] buffer = new byte[1024];
					int size;
					while ((size = readStream.read(buffer, 0, buffer.length)) != -1) {
						writeStream.write(buffer, 0, size);
					}
				} catch (IOException e) {
					System.out.println(e.getMessage());
					e.printStackTrace();
				} finally {
					if (writeStream != null) {
						try {
							writeStream.flush();
							writeStream.close();
						} catch (IOException e) {
						} finally {
							writeStream = null;
						}
					}
					if (readStream != null) {
						try {
							readStream.close();
						} catch (IOException e) {
						} finally {
							readStream = null;
						}
					}
				}
			}

			progressValue.value_(((float) anIndex) / this.numberOfVisuals());
			anIndex = anIndex + 1;
		}
	}

	/**
	 * Answer true if this visuals is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.numberOfVisuals() == 0;
	}

	/**
	 * Answer true if the specified extension string is either an image or a movie, otherwise false.
	 * 
	 * @param extensionString java.lang.String
	 * @return boolean
	 * @category testing
	 */
	public boolean incluesExtensions_(String extensionString) {
		String extension = extensionString.toLowerCase();
		return Arrays.asList(JunSystem.DefaultImageExtensions()).contains(extension) || Arrays.asList(JunSystem.DefaultMovieExtensions()).contains(extension);
	}

	/**
	 * Answer the thumbnail image extents.
	 * 
	 * @return java.awt.Dimension[]
	 * @category url accessing
	 */
	protected Dimension[] thumbnailImageExtents() {
		File[] thumbnailImageFiles = this.thumbnailImageFiles();
		Dimension[] thumbnailImageExtents = new Dimension[thumbnailImageFiles.length];

		StImage image = null;
		JunImageStream stream = null;
		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		for (int i = 0; i < thumbnailImageFiles.length; i++) {
			try {
				cursor._show();
				stream = JunJpegImageStream.On_(new FileInputStream(thumbnailImageFiles[i]));
				image = stream.nextImage();
			} catch (IOException e) {
				System.out.println(e.getMessage());
				e.printStackTrace();
			} finally {
				if (stream != null) {
					try {
						stream.close();
					} catch (IOException e) {
					} finally {
						stream = null;
					}
				}
			}
			thumbnailImageExtents[i] = image.bounds().getSize();
		}
		return thumbnailImageExtents;
	}

	/**
	 * Answer the strings of thumbnail image URL.
	 * 
	 * @return java.lang.String[]
	 * @category url accessing
	 */
	protected String[] thumbnailImageUrls() {
		File[] thumbnailImageFiles = this.thumbnailImageFiles();
		String[] thumbnailImageUrls = new String[thumbnailImageFiles.length];
		for (int i = 0; i < thumbnailImageFiles.length; i++) {
			thumbnailImageUrls[i] = this.convertFilenameToUrlString_(thumbnailImageFiles[i]);
		}
		return thumbnailImageUrls;
	}

	/**
	 * URLs do a block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @exception java.io.IOException
	 * @category url accessing
	 */
	public void urlsDo_(StBlockClosure aBlock) throws IOException {
		String[] visualClipUrls = this.visualClipUrls();
		String[] thumbnailImageUrls = this.thumbnailImageUrls();
		Dimension[] thumbnailImageExtents = this.thumbnailImageExtents();
		for (int i = 0; i < this.numberOfVisuals(); i++) {
			Object result = aBlock.valueWithArguments_(new Object[] { visualClipUrls[i], thumbnailImageUrls[i], thumbnailImageExtents[i], new Integer(i) });
			if (result != null && result instanceof IOException) {
				throw (IOException) result;
			}
		}
	}

	/**
	 * Answer the strings of visual clip URL.
	 * 
	 * @return java.lang.String[]
	 * @category url accessing
	 */
	protected String[] visualClipUrls() {
		File[] visualClipFiles = this.visualClipFiles();
		String[] visualClipUrls = new String[visualClipFiles.length];
		for (int i = 0; i < visualClipFiles.length; i++) {
			visualClipUrls[i] = this.convertFilenameToUrlString_(visualClipFiles[i]);
		}
		return visualClipUrls;
	}

	/**
	 * Open this visual catlog with browser.
	 * 
	 * @category viewing
	 */
	public void show() {
		File aFilename = this.indexHtmlFile();
		if (aFilename.exists() == false) {
			return;
		}
		try {
			JunURL.Browse_(aFilename.toURL());
		} catch (MalformedURLException e) {
		}
	}

	/**
	 * Convert a filename to url string and answer it.
	 * 
	 * @return java.lang.String
	 * @param aFilename java.io.File
	 * @category private
	 */
	protected String convertFilenameToUrlString_(File aFilename) {
		String baseString = this.visualCatalogDirectory().toString();
		String aString = aFilename.toString();
		aString = aString.substring(baseString.length() + 1);
		if (aString.charAt(0) == File.separatorChar) {
			aString = aString.substring(1);
		}
		aString = aString.replace(File.separatorChar, '/');
		return aString;
	}

	/**
	 * Write html tag and attribute string on a writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @param aString java.lang.String
	 * @param attributeString java.lang.String
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @throws java.io.IOException
	 * @category private
	 */
	protected void on_tag_attributes_do_(Writer aWriter, String aString, String attributeString, StBlockClosure aBlock) throws IOException {
		JunStringUtility.On_tag_attributes_do_(aWriter, aString, attributeString, aBlock);
	}

	/**
	 * Write html tag and attribute string on a writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @param aString java.lang.String
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @throws java.io.IOException
	 * @category private
	 */
	protected void on_tag_do_(Writer aWriter, String aString, StBlockClosure aBlock) throws IOException {
		JunStringUtility.On_tag_do_(aWriter, aString, aBlock);
	}

	/**
	 * Answer the propotional extent of image with in a aDimension.
	 * 
	 * @return java.awt.Dimension
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aSize java.awt.Dimension
	 * @category private
	 */
	protected Dimension proportionalExtentOfImage_within_(StImage anImage, Dimension aSize) {
		Dimension imageExtent = anImage.bounds().getSize();
		if (imageExtent.width > aSize.width) {
			int imageClipHeight = Math.round(imageExtent.height * aSize.width / (float) imageExtent.width);
			imageExtent = new Dimension(aSize.width, imageClipHeight);
		}
		if (imageExtent.height > aSize.height) {
			int imageClipWidth = Math.round(imageExtent.width * aSize.height / (float) imageExtent.height);
			imageExtent = new Dimension(imageClipWidth, aSize.height);
		}

		return imageExtent;
	}
}
