package jp.co.sra.jun.goodies.colors;

import java.awt.Cursor;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunAbstractController;

/**
 * JunColorBarController class
 * 
 *  @author    nisinaka
 *  @created   2004/06/11 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun473 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunColorBarController.java,v 8.10 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunColorBarController extends JunAbstractController implements MouseListener, MouseMotionListener {

	protected StSymbol _tracking;
	protected Point _offsetPoint;
	protected Point _previousPoint;
	protected boolean _redisplayBoolean;

	/**
	 * Remove references to objects that may refer to the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		super.release();
		this.view().toComponent().removeMouseListener(this);
		this.view().toComponent().removeMouseMotionListener(this);
	}

	/**
	 * Setup the listeners.
	 * 
	 * @param aView
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category initialize-release
	 */
	protected void buildListener(StView aView) {
		this.view().toComponent().addMouseListener(this);
		this.view().toComponent().addMouseMotionListener(this);
	}

	/**
	 * Answer my model as JunColorBarModel.
	 * 
	 * @return JunColorBarModel
	 * @category view accessing
	 */
	public JunColorBarModel getColorBarModel() {
		return (JunColorBarModel) this.model();
	}

	/**
	 * Answer my view as JunColorBarView.
	 * 
	 * @return jp.co.sra.jun.goodies.colors.JunColorBarView
	 * @category view accessing
	 */
	public JunColorBarView getColorBarView() {
		return (JunColorBarView) this.view();
	}

	/**
	 * Invoked when a mouse button has been pressed on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mousePressed(java.awt.event.MouseEvent)
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events 
	 */
	public void mousePressed(MouseEvent e) {
		if (e.isPopupTrigger()) {
			super.mousePressed(e);
			return;
		}

		_previousPoint = null;
		_redisplayBoolean = false;

		Point startPoint = e.getPoint();
		_tracking = this._state(startPoint);
		if (_tracking == $("slider")) {
			if (this.getColorBarView().sliderRectangle().containsPoint_(startPoint)) {
				_offsetPoint = this.getColorBarView().sliderRectangle().center();
				_offsetPoint.translate(-startPoint.x, -startPoint.y);
			} else {
				_offsetPoint = new Point(0, 0);
			}
			this.trackSlider_(startPoint);
		} else if (_tracking == $("firstMarker")) {
			_offsetPoint = this.getColorBarView().firstMarkerRectangle().rightCenter();
			_offsetPoint.translate(-startPoint.x, -startPoint.y);
			this.trackFirstMarker_(startPoint);
		} else if (_tracking == $("lastMarker")) {
			_offsetPoint = this.getColorBarView().lastMarkerRectangle().leftCenter();
			_offsetPoint.translate(-startPoint.x, -startPoint.y);
			this.trackLastMarker_(startPoint);
		} else if (_tracking == $("interval")) {
			_offsetPoint = this.getColorBarView().anotherIntervalRectangle().center();
			_offsetPoint.translate(-startPoint.x, -startPoint.y);
			this.trackInterval_(startPoint);
		}

	}

	/**
	 * Invoked when a mouse button has been released on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseReleased(java.awt.event.MouseEvent)
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events 
	 */
	public void mouseReleased(MouseEvent e) {
		if (e.isPopupTrigger()) {
			super.mouseReleased(e);
			return;
		}

		if (_tracking == $("slider")) {
			if (_redisplayBoolean) {
				this.view().toComponent().repaint();
			}
		} else if (_tracking == $("firstMarker")) {
			// nothing to do
		} else if (_tracking == $("lastMarker")) {
			// nothing to do
		} else if (_tracking == $("interval")) {
			// nothing to do
		}

		_tracking = null;
		_offsetPoint = null;
		_redisplayBoolean = false;
	}

	/**
	 * Invoked when a mouse is dragged on the view.
	 *
	 * @param e java.awt.event.MouseEvent 
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent e) {
		Point cursorPoint = e.getPoint();
		if ((new StRectangle(this.view().toComponent().getSize())).expandedBy_(16, 16).containsPoint_(cursorPoint)) {
			if (cursorPoint.equals(_previousPoint) == false) {
				if (_tracking == $("slider")) {
					this.trackSlider_(e.getPoint());
				} else if (_tracking == $("firstMarker")) {
					this.trackFirstMarker_(e.getPoint());
				} else if (_tracking == $("lastMarker")) {
					this.trackLastMarker_(e.getPoint());
				} else if (_tracking == $("interval")) {
					this.trackInterval_(e.getPoint());
				}
				_previousPoint = cursorPoint;
			}
			_redisplayBoolean = true;
		}

		Thread.yield();
	}

	/**
	 * Invoked when a mouse is moved on the view.
	 *
	 * @param e java.awt.event.MouseEvent 
	 * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseMoved(MouseEvent e) {
		StSymbol state = this._state(e.getPoint());
		if (state == $("slider")) {
			this.view().toComponent().setCursor(JunCursors.CrossCursor());
		} else if (state == $("firstMarker")) {
			if (this.getColorBarView().isVertical()) {
				this.view().toComponent().setCursor(JunCursors.Corner1Cursor());
			} else {
				this.view().toComponent().setCursor(JunCursors.Corner2Cursor());
			}
		} else if (state == $("lastMarker")) {
			this.view().toComponent().setCursor(JunCursors.Corner4Cursor());
		} else if (state == $("interval")) {
			if (this.getColorBarView().isVertical()) {
				this.view().toComponent().setCursor(JunCursors.UpDownCursor());
			} else {
				this.view().toComponent().setCursor(JunCursors.BetweenCursor());
			}
		} else {
			this.view().toComponent().setCursor(Cursor.getDefaultCursor());
		}
	}

	/**
	 * Do the slider tracking.
	 * 
	 * @param aPoint java.awt.Point
	 * @category tracking
	 */
	protected void trackSlider_(Point aPoint) {
		if (aPoint.equals(_previousPoint) == false) {
			this.getColorBarView().setSliderAt(new Point(aPoint.x + _offsetPoint.x, aPoint.y + _offsetPoint.y));
			_previousPoint = aPoint;
		}
		_redisplayBoolean = true;
	}

	/**
	 * Do the first marker tracking.
	 * 
	 * @param aPoint java.awt.Point
	 * @category tracking
	 */
	protected void trackFirstMarker_(Point aPoint) {
		if (aPoint.equals(_previousPoint) == false) {
			this.getColorBarView().setFirstMarkerAt(new Point(aPoint.x + _offsetPoint.x, aPoint.y + _offsetPoint.y));
			_previousPoint = aPoint;
		}
		_redisplayBoolean = true;
	}

	/**
	 * Do the last marker tracking.
	 * 
	 * @param aPoint java.awt.Point
	 * @category tracking
	 */
	protected void trackLastMarker_(Point aPoint) {
		if (aPoint.equals(_previousPoint) == false) {
			this.getColorBarView().setLastMarkerAt(new Point(aPoint.x + _offsetPoint.x, aPoint.y + _offsetPoint.y));
			_previousPoint = aPoint;
		}
		_redisplayBoolean = true;
	}

	/**
	 * Do the interval tracking.
	 * 
	 * @param aPoint java.awt.Point
	 * @category tracking
	 */
	protected void trackInterval_(Point aPoint) {
		if (aPoint.equals(_previousPoint) == false) {
			this.getColorBarView().setIntervalAt(new Point(aPoint.x + _offsetPoint.x, aPoint.y + _offsetPoint.y));
			_previousPoint = aPoint;
		}
		_redisplayBoolean = true;
	}

	/**
	 * Answer the state at the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category private
	 */
	protected StSymbol _state(Point aPoint) {
		StSymbol symbol = null;
		if (this.getColorBarView().sliderRectangle().containsPoint_(aPoint)) {
			symbol = $("slider");
		} else if (this.getColorBarView().firstMarkerRectangle().containsPoint_(aPoint)) {
			symbol = $("firstMarker");
		} else if (this.getColorBarView().lastMarkerRectangle().containsPoint_(aPoint)) {
			symbol = $("lastMarker");
		} else if (this.getColorBarView().anotherIntervalRectangle().containsPoint_(aPoint)) {
			symbol = $("interval");
		} else if (this.getColorBarView().rangeArea().containsPoint_(aPoint)) {
			symbol = $("slider");
		}
		return symbol;
	}

}
