package jp.co.sra.jun.goodies.colors;

import java.awt.Color;
import java.awt.Frame;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StBlockValued;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunColorBarModel class
 * 
 *  @author    nisinaka
 *  @created   2004/06/10 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun462 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunColorBarModel.java,v 8.11 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunColorBarModel extends JunApplicationModel implements StBlockValued {

	protected StValueHolder valueHolder;
	protected StValueHolder intervalHolder;
	protected StValueHolder colorHolder;
	protected Color ultraFirstColor;
	protected Color ultraLastColor;
	protected double[] intervalBounds;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		valueHolder = null;
		intervalHolder = null;
		colorHolder = null;
		ultraFirstColor = null;
		ultraLastColor = null;
		intervalBounds = null;
	}

	/**
	 * Answer my current value.
	 * 
	 * @return java.lang.Object
	 * @see jp.co.sra.smalltalk.StValued#value()
	 * @category accessing
	 */
	public Object value() {
		return this.valueHolder().value();
	}

	/**
	 * Answer my current value as double.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double _doubleValue() {
		return this.valueHolder()._doubleValue();
	}

	/**
	 * Set my new value.
	 * 
	 * @param newValue double
	 * @category accessing
	 */
	public void value_(double newValue) {
		this.valueHolder().value_(Math.max(0, Math.min(newValue, 1)));
		this.changed_($("value"));
		this.colorHolder().value_(this.color());
	}

	/**
	 * Answer my current value holder.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	protected StValueHolder valueHolder() {
		if (valueHolder == null) {
			valueHolder = new StValueHolder(0);
		}
		return valueHolder;
	}

	/**
	 * Answer my current interval.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] interval() {
		return (double[]) this.intervalHolder().value();
	}

	/**
	 * Set my current interval.
	 * 
	 * @param newValue double[]
	 * @category accessing
	 */
	public void interval_(double[] newValue) {
		this.intervalHolder().value_(this.adjustInterval_(newValue));
		this.changed_($("interval"));
		this.colorHolder().value_(this.color());
	}

	/**
	 * Answer my current interval holder.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	protected StValueHolder intervalHolder() {
		if (intervalHolder == null) {
			intervalHolder = new StValueHolder(new double[] { 0, 0 });
		}
		return intervalHolder;
	}

	/**
	 * Answer the current position of my first marker.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double firstMarker() {
		return this.interval()[0];
	}

	/**
	 * Set the new position of my first marker.
	 * 
	 * @param normalizedValue double
	 * @category accessing
	 */
	public void firstMarker_(double normalizedValue) {
		double first = Math.max(normalizedValue, this.intervalBounds()[0]);
		double[] newInterval = new double[] { first, Math.max(first, this.lastMarker())};
		this.interval_(newInterval);
	}

	/**
	 * Answer the current position of my last marker.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double lastMarker() {
		return this.interval()[1];
	}

	/**
	 * Set the new position of my last marker.
	 * 
	 * @param normalizedValue double
	 * @category accessing
	 */
	public void lastMarker_(double normalizedValue) {
		double last = Math.min(normalizedValue, this.intervalBounds()[1]);
		double[] newInterval = new double[] { Math.min(this.firstMarker(), last), last };
		this.interval_(newInterval);
	}

	/**
	 * Answer my current color.
	 * 
	 * @return java.awt.Color
	 * @category color accessing
	 */
	public Color color() {
		if (this._doubleValue() < this.firstMarker()) {
			return this.ultraFirstColor();
		} else if (this.lastMarker() < this._doubleValue()) {
			return this.ultraLastColor();
		}

		double normalizedValue = 0;
		double denominator = this.lastMarker() - this.firstMarker();
		if (denominator > 0) {
			normalizedValue = (this._doubleValue() - this.firstMarker()) / denominator;
		}
		return this.getColor_(normalizedValue);
	}

	/**
	 * Answer my current color holder.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category color accessing
	 */
	protected StValueHolder colorHolder() {
		if (colorHolder == null) {
			colorHolder = new StValueHolder(null);
		}
		return colorHolder;
	}

	/**
	 * Answer my current ultra first color.
	 * 
	 * @return java.awt.Color
	 * @category color accessing
	 */
	public Color ultraFirstColor() {
		if (ultraFirstColor == null) {
			ultraFirstColor = this.defaultUltraFirstColor();
		}
		return ultraFirstColor;
	}

	/**
	 * Set my new ultra first color.
	 * 
	 * @param newColor java.awt.Color
	 * @category color accessing
	 */
	public void ultraFirstColor_(Color newColor) {
		if (ultraFirstColor != null && ultraFirstColor.equals(newColor)) {
			return;
		}

		ultraFirstColor = newColor;
		this.changed_($("color"));
	}

	/**
	 * Answer my current ultra last color.
	 * 
	 * @return java.awt.Color
	 * @category color accessing
	 */
	public Color ultraLastColor() {
		if (ultraLastColor == null) {
			ultraLastColor = this.defaultUltraLastColor();
		}
		return ultraLastColor;
	}

	/**
	 * Set my new ultra last color.
	 * 
	 * @param newColor java.awt.Color
	 * @category color accessing
	 */
	public void ultraLastColor_(Color newColor) {
		if (ultraLastColor != null && ultraLastColor.equals(newColor)) {
			return;
		}

		ultraLastColor = newColor;
		this.changed_($("color"));
	}

	/**
	 * Answer a StBlockValue that computes aBlock with the receiver's value as the argument.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return this.colorCompute_(aBlock);
	}

	/**
	 * Answer a StBlockValue that computes aBlock with the receiver's value as the argument.
	 * This one is for the value.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue valueCompute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this.valueHolder());
	}

	/**
	 * Answer a StBlockValue that computes aBlock with the receiver's value as the argument.
	 * This one is for the interval value.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue intervalCompute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this.intervalHolder());
	}

	/**
	 * Answer a StBlockValue that computes aBlock with the receiver's value as the argument.
	 * This one is for the color value.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue colorCompute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this.colorHolder());
	}

	/**
	 * Answer my default ultra first color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultUltraFirstColor() {
		return StColorValue.Brightness_(0.8);
	}

	/**
	 * Answer my default ultra last color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultUltraLastColor() {
		return StColorValue.Brightness_(0.8);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunColorBarViewAwt.OnBorderedPanel_(this);
		} else {
			return JunColorBarViewSwing.OnBorderedPanel_(this);
		}
	}

	/**
	 * Answer a default vertical view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultVerticalView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunColorBarVerticalViewAwt.OnBorderedPanel_(this);
		} else {
			return JunColorBarVerticalViewSwing.OnBorderedPanel_(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Color Bar");
	}

	/**
	 * Answer the color which correspond to the normalized value.
	 * My subclasses may override this method.
	 * 
	 * @param normalizedValue double
	 * @return java.awt.Color
	 * @category private
	 */
	protected Color getColor_(double normalizedValue) {
		return Color.getHSBColor((float) normalizedValue, 1, 1);
	}

	/**
	 * Adjust the double array appropriate as an interval.
	 * 
	 * @param interval double[]
	 * @return double[]
	 * @category private
	 */
	protected double[] adjustInterval_(double[] interval) {
		double first = Math.max(0, Math.min(Math.min(interval[0], interval[1]), 1));
		double last = Math.max(first, Math.min(Math.max(interval[0], interval[1]), 1));
		return new double[] { first, last };
	}

	/**
	 * Answer my current interval bounds.
	 * 
	 * @return double[]
	 * @category private
	 */
	protected double[] intervalBounds() {
		if (intervalBounds == null) {
			intervalBounds = new double[] { 0, 1 };
		}
		return intervalBounds;
	}

	/**
	 * Open as a vertical bar.
	 * 
	 * @return java.awt.Frame
	 * @see jp.co.sra.smalltalk.StApplicationModel#open()
	 * @category interface opening
	 */
	public Frame openVertical() {
		return this.openView_(this.defaultVerticalView());
	}

}
