package jp.co.sra.jun.goodies.drawing.element;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.datatransfer.DataFlavor;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.SystemResourceSupport;
import jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel;
import jp.co.sra.jun.goodies.drawing.properties.JunTextPropertiesModel;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunTextElement class
 * 
 *  @author    m-asada
 *  @created   2005/04/07 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTextElement.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public class JunTextElement extends JunRectangularElement {
	protected String text;
	protected String fontName;
	protected int fontSize;
	protected int fontStyle;
	protected Color foregroundColor;

	protected transient Font font;
	protected transient StComposedText composedText;
	protected transient JunTextPropertiesModel propertiesModel;

	protected static transient Dimension DefaultMinimumExtent = null;

	public static DataFlavor DataFlavor = new DataFlavor(JunTextElement.class, "JunTextElement");

	/**
	 * Create a new instance of JunTextElement and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunTextElement() {
		super();
	}

	/**
	 * Create a new instance of JunTextElement and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunTextElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		boolean isMacOS = System.getProperty("os.name").toLowerCase().matches(".*mac.*os.*x.*");
		text = null;
		fontName = isMacOS ? "Osaka-Mono" : "Monospaced";
		fontSize = isMacOS ? 12 : SystemResourceSupport.getFont().getSize();
		fontStyle = Font.PLAIN;
		foregroundColor = Color.black;
		font = null;
		composedText = null;
		propertiesModel = null;
	}

	/**
	 * Release the resources of the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		super.release();

		if (propertiesModel != null) {
			propertiesModel.closeRequest();
			propertiesModel = null;
		}
	}

	/**
	 * Answer the receiver's text.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String text() {
		return text;
	}

	/**
	 * Set the receiver's text.
	 * 
	 * @param newText java.lang.String
	 * @category accessing
	 */
	public void text_(String newText) {
		text = newText;
		this.flushBounds();
	}

	/**
	 * Answer the receiver's composed text object.
	 * 
	 * @return jp.co.sra.smalltalk.StComposedText
	 * @category accessing
	 */
	protected StComposedText composedText() {
		if (composedText == null) {
			composedText = new StComposedText(this.text(), this.font());
			if (composedText.width() > this.width()) {
				composedText = new StComposedText(this.text(), this.width());
			}
		}
		return composedText;
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);

		if (text != null) {
			this.text_(new String(text));
		}
		if (fontName != null) {
			this.fontName_(new String(fontName));
		}
		this.foregroundColor_(new Color(foregroundColor.getRGB(), true));
		font = null;
		composedText = null;
		propertiesModel = null;

		return this;
	}

	/**
	 * Get the default extent.
	 *
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#defaultExtent()
	 * @category defaults
	 */
	public Dimension defaultExtent() {
		return new Dimension(96, this.defaultMinimumExtent().height + 4);
	}

	/**
	 * Get the default minimum extent.
	 *
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#defaultMinimumExtent()
	 * @category defaults
	 */
	public Dimension defaultMinimumExtent() {
		if (DefaultMinimumExtent == null) {
			FontMetrics metrics = SystemResourceSupport.getFontMetrics(SystemResourceSupport.getFont());
			DefaultMinimumExtent = new Dimension(metrics.getWidths()[0] * 2, metrics.getHeight());
		}
		return DefaultMinimumExtent;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		Rectangle clipBounds = aGraphics.getClipBounds();
		if (clipBounds != null && clipBounds.intersects(this.bounds()) == false) {
			return;
		}

		Graphics graphicsContext = aGraphics.create();
		try {
			StComposedText text = this.composedText();
			int offsetX = (text.width() == this.width()) ? 0 : (this.width() - text.width()) / 2;
			int offsetY = (this.height() - text.height()) / 2;
			clipBounds = (clipBounds == null) ? this.bounds() : this.bounds().intersection(clipBounds);
			graphicsContext.setClip(clipBounds);
			graphicsContext.setColor(this.foregroundColor());
			text.displayOn_at_(graphicsContext, new Point(this.x() + offsetX, this.y() + offsetY));
		} finally {
			if (graphicsContext != null) {
				graphicsContext.dispose();
			}
		}
	}

	/**
	 * Answer the receiver's font.
	 * 
	 * @return java.awt.Font
	 * @category fonts
	 */
	public Font font() {
		if (font == null) {
			font = new Font(this.fontName(), this.fontStyle(), this.fontSize());
		}
		return font;
	}

	/**
	 * Flush the receiver's preferred bounds.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunRectangularElement#flushBounds()
	 * @category flushing
	 */
	protected void flushBounds() {
		super.flushBounds();
		if (composedText != null) {
			if ((this.text() == null && composedText.string() != null) || (this.text() != null && this.text().equals(composedText.string())) == false) {
				composedText = null;
			} else if (composedText.numberOfLines() == 1 && this.width() < composedText.width()) {
				composedText = null;
			} else if (composedText.numberOfLines() > 1 && this.width() != composedText.width()) {
				composedText = null;
			}
		}
	}

	/**
	 * Flush the receiver's font.
	 * 
	 * @category flushing
	 */
	protected void flushFont() {
		font = null;
		composedText = null;
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be provided in. The array should be
	 * ordered according to preference for providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return java.awt.datatransfer.DataFlavor[]
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#getTransferDataFlavors()
	 * @category transfering
	 */
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { JunTextElement.DataFlavor };
	}

	/**
	 * Answer the receiver's font name.
	 * 
	 * @return java.lang.String
	 * @category visual properties
	 */
	public String fontName() {
		return fontName;
	}

	/**
	 * Set the receiver's font name.
	 * 
	 * @param newFontName java.lang.String
	 * @category visual properties
	 */
	public void fontName_(String newFontName) {
		fontName = newFontName;
		this.flushFont();
	}

	/**
	 * Answer the receiver's font size.
	 * 
	 * @return fontSize
	 * @category visual properties
	 */
	public int fontSize() {
		return fontSize;
	}

	/**
	 * Set the receiver's font size.
	 * 
	 * @param newFontSize int
	 * @category visual properties
	 */
	public void fontSize_(int newFontSize) {
		fontSize = newFontSize;
		this.flushFont();
	}

	/**
	 * Answer the receiver's font style.
	 * 
	 * @return fontSize
	 * @category visual properties
	 */
	public int fontStyle() {
		return fontStyle;
	}

	/**
	 * Set the receiver's font style.
	 * 
	 * @param newFontStyle int
	 * @category visual properties
	 */
	public void fontStyle_(int newFontStyle) {
		fontStyle = newFontStyle;
		this.flushFont();
	}

	/**
	 * Answer the receiver's foreground color.
	 * 
	 * @return java.awt.Color
	 * @category visual properties
	 */
	public Color foregroundColor() {
		return foregroundColor;
	}

	/**
	 * Set the receiver's foreground color.
	 * 
	 * @param aColor java.awt.Color
	 * @category visual properties
	 */
	public void foregroundColor_(Color aColor) {
		foregroundColor = aColor;
	}

	/**
	 * Answer the receiver's properties model.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.properties.JunAbstractElementPropertiesModel
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#propertiesModel()
	 * @category visual properties
	 */
	public JunDrawingElementPropertiesModel propertiesModel() {
		if (propertiesModel == null) {
			propertiesModel = new JunTextPropertiesModel(this);
		}
		return propertiesModel;
	}

	/**
	 * Convert the receiver to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		JunLispCons textList = this.textToLispList();
		if (textList != null) {
			list.add_(textList);
		}
		JunLispCons fontNameList = this.fontNameToLispList();
		if (fontNameList != null) {
			list.add_(fontNameList);
		}
		JunLispCons fontSizeList = this.fontSizeToLispList();
		if (fontSizeList != null) {
			list.add_(fontSizeList);
		}
		JunLispCons fontStyleList = this.fontStyleToLispList();
		if (fontStyleList != null) {
			list.add_(fontStyleList);
		}
		JunLispCons foregroundColorList = this.foregroundColorToLispList();
		if (foregroundColorList != null) {
			list.add_(foregroundColorList);
		}
		return list;
	}

	/**
	 * Convert the text to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons textToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("text"));
		list.tail_(this.text());
		return list;
	}

	/**
	 * Convert the font name to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons fontNameToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("fontName"));
		list.tail_(this.fontName());
		return list;
	}

	/**
	 * Convert the font size to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons fontSizeToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("fontSize"));
		list.tail_(new Integer(this.fontSize()));
		return list;
	}

	/**
	 * Convert the font style to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons fontStyleToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("fontStyle"));
		list.tail_(new Integer(this.fontStyle()));
		return list;
	}

	/**
	 * Convert the foreground color to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons foregroundColorToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("foregroundColor"));
		list.tail_(this.foregroundColor());
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.textFromLispList_(aList);
		this.fontNameFromLispList_(aList);
		this.fontSizeFromLispList_(aList);
		this.fontStyleFromLispList_(aList);
		this.foregroundColorFromLispList_(aList);
	}

	/**
	 * Get the text from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void textFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("text"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.text_((String) list.tail());
	}

	/**
	 * Get the font name from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void fontNameFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("fontName"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.fontName_((String) list.tail());
	}

	/**
	 * Get the font size from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void fontSizeFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("fontSize"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.fontSize_(((Number) list.tail()).intValue());
	}

	/**
	 * Get the font style from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void fontStyleFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("fontStyle"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.fontStyle_(((Number) list.tail()).intValue());
	}

	/**
	 * Get the foreground color from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void foregroundColorFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("foregroundColor"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.foregroundColor_((Color) list.tail());
	}
}
