package jp.co.sra.jun.goodies.drawing.element;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.datatransfer.DataFlavor;
import java.util.HashMap;
import java.util.Map;
import jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel;
import jp.co.sra.jun.goodies.drawing.properties.JunTextboxPropertiesModel;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunTextboxElement class
 * 
 *  @author    m-asada
 *  @created   2005/04/07 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTextboxElement.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public class JunTextboxElement extends JunCompositeElement {
	protected transient JunTextboxPropertiesModel propertiesModel;

	public static int DefaultMargin = 5;

	public static DataFlavor DataFlavor = new DataFlavor(JunTextboxElement.class, "JunTextboxElement");

	/**
	 * Create a new instance of JunTextboxElement and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunTextboxElement() {
		super();
	}

	/**
	 * Create a new instance of JunTextboxElement and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunTextboxElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		propertiesModel = null;
	}

	/**
	 * Initialize the receiver's component elements.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunCompositeElement#initializeComponentElements()
	 * @category initialize-release
	 */
	protected void initializeComponentElements() {
		super.initializeComponentElements();

		this.addElement_(new JunRectangleElement());
		this.addElement_(new JunTextElement());
	}

	/**
	 * Release the resources of the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		super.release();

		if (propertiesModel != null) {
			propertiesModel.closeRequest();
			propertiesModel = null;
		}
	}

	/**
	 * Answer the receiver's border element at component elements.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunRectangularShapeElement
	 * @category accessing
	 */
	public JunRectangularShapeElement borderElement() {
		if (this.componentElementSize() == 0) {
			this.initializeComponentElements();
		}
		return (JunRectangularShapeElement) this.componentElementAt_(0);
	}

	/**
	 * Answer the receiver's text element at component elements.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunTextElement
	 * @category accessing
	 */
	public JunTextElement textElement() {
		if (this.componentElementSize() == 0) {
			this.initializeComponentElements();
		}
		return (JunTextElement) this.componentElementAt_(1);
	}

	/**
	 * Answer the receiver's text.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodeis.drawing.element.JunTextElement#text()
	 * @category accessing
	 */
	public String text() {
		return this.textElement().text();
	}

	/**
	 * Set the receiver's text.
	 * 
	 * @param newText java.lang.String
	 * @see jp.co.sra.jun.goodeis.drawing.element.JunTextElement#text_(String)
	 * @category accessing
	 */
	public void text_(String newText) {
		this.textElement().text_(newText);
	}

	/**
	 * Set the receiver's location point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location_(java.awt.Point)
	 * @category bounds accessing
	 */
	public void location_(Point aPoint) {
		Point newLocation = new Point(Math.max(aPoint.x, 0), Math.max(aPoint.y, 0));
		this.borderElement().location_(newLocation);
		int textX = 0;
		int textY = 0;
		int textWidth = 0;
		int textHeight = 0;
		if (this.textElement().defaultMinimumExtent().width + DefaultMargin * 2 + this.lineWidth() * 2 <= this.borderElement().width()) {
			textX = newLocation.x + DefaultMargin + this.borderElement().lineWidth();
			textWidth = this.textElement().width();
		} else {
			textX = newLocation.x + (this.borderElement().width() - this.textElement().defaultMinimumExtent().width) / 2;
			textWidth = this.textElement().defaultMinimumExtent().width;
		}
		if (this.textElement().defaultMinimumExtent().height + DefaultMargin * 2 + this.lineWidth() * 2 <= this.borderElement().height()) {
			textY = newLocation.y + DefaultMargin + this.borderElement().lineWidth();
			textHeight = this.textElement().height();
		} else {
			textY = newLocation.y + (this.borderElement().height() - this.textElement().defaultMinimumExtent().height) / 2;
			textHeight = this.textElement().defaultMinimumExtent().height;
		}
		this.textElement().location_(textX, textY);
		this.textElement().extent_(textWidth, textHeight);
	}

	/**
	 * Set the receiver's extent size.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#extent_(java.awt.Dimension)
	 * @category bounds accessing
	 */
	public void extent_(Dimension aDimension) {
		Dimension minExtent = this.defaultMinimumExtent();
		Dimension newExtent = new Dimension(Math.max(aDimension.width, minExtent.width), Math.max(aDimension.height, minExtent.height));
		this.borderElement().extent_(newExtent);
		int textWidth = 0;
		int textHeight = 0;
		if (this.textElement().defaultMinimumExtent().width + DefaultMargin * 2 + this.lineWidth() * 2 <= newExtent.width) {
			textWidth = newExtent.width - this.borderElement().lineWidth() * 2 - DefaultMargin * 2;
		} else {
			textWidth = newExtent.width - (newExtent.width - this.textElement().defaultMinimumExtent().width) / 2;
		}
		if (this.textElement().defaultMinimumExtent().height + DefaultMargin * 2 + this.lineWidth() * 2 <= newExtent.height) {
			textHeight = newExtent.height - this.borderElement().lineWidth() * 2 - DefaultMargin * 2;
		} else {
			textHeight = newExtent.height - (newExtent.height - this.textElement().defaultMinimumExtent().height) / 2;
		}
		this.textElement().extent_(textWidth, textHeight);
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#preferredBounds()
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		return this.borderElement().preferredBounds();
	}

	/**
	 * Answer the collection of receiver's controll point.
	 * 
	 * @return java.util.HashMap
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#controllPoints()
	 * @category bounds accessing
	 */
	public HashMap controllPoints() {
		return this.borderElement().controllPoints();
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);
		propertiesModel = null;

		return this;
	}

	/**
	 * Get the default extent.
	 *
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#defaultExtent()
	 * @category defaults
	 */
	public Dimension defaultExtent() {
		return new Dimension(96, this.defaultMinimumExtent().height + 4);
	}

	/**
	 * Get the default minimum extent.
	 *
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#defaultMinimumExtent()
	 * @category defaults
	 */
	public Dimension defaultMinimumExtent() {
		Dimension textExtent = this.textElement().defaultMinimumExtent();
		Dimension borderExtent = this.borderElement().defaultMinimumExtent();
		return new Dimension(Math.max(textExtent.width + DefaultMargin * 2, borderExtent.width), Math.max(textExtent.height + DefaultMargin * 2, borderExtent.height));
	}

	/**
	 * Answer the receiver's font.
	 * 
	 * @return java.awt.Font
	 * @category fonts
	 */
	public Font font() {
		return this.textElement().font();
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be provided in. The array should be
	 * ordered according to preference for providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return java.awt.datatransfer.DataFlavor[]
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#getTransferDataFlavors()
	 * @category transfering
	 */
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { JunTextboxElement.DataFlavor };
	}

	/**
	 * Answer the receiver's font name.
	 * 
	 * @return java.lang.String
	 * @category visual properties
	 */
	public String fontName() {
		return this.textElement().fontName();
	}

	/**
	 * Set the receiver's font name.
	 * 
	 * @param newFontName java.lang.String
	 * @category visual properties
	 */
	public void fontName_(String newFontName) {
		this.textElement().fontName_(newFontName);
	}

	/**
	 * Answer the receiver's font size.
	 * 
	 * @return fontSize
	 * @category visual properties
	 */
	public int fontSize() {
		return this.textElement().fontSize();
	}

	/**
	 * Set the receiver's font size.
	 * 
	 * @param newFontSize int
	 * @category visual properties
	 */
	public void fontSize_(int newFontSize) {
		this.textElement().fontSize_(newFontSize);
	}

	/**
	 * Answer the receiver's font style.
	 * 
	 * @return fontSize
	 * @category visual properties
	 */
	public int fontStyle() {
		return this.textElement().fontStyle();
	}

	/**
	 * Set the receiver's font style.
	 * 
	 * @param newFontStyle int
	 * @category visual properties
	 */
	public void fontStyle_(int newFontStyle) {
		this.textElement().fontStyle_(newFontStyle);
	}

	/**
	 * Answer the receiver's foreground color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.goodies.drawing.element.JunTextElement#foregroundColor()
	 * @category visual properties
	 */
	public Color foregroundColor() {
		return this.textElement().foregroundColor();
	}

	/**
	 * Set the receiver's foreground color.
	 * 
	 * @param aColor java.awt.Color
	 * @see jp.co.sra.jun.goodies.drawing.element.JunTextElement#foregroundColor_(Color)
	 * @category visual properties
	 */
	public void foregroundColor_(Color aColor) {
		this.textElement().foregroundColor_(aColor);
	}

	/**
	 * Answer the receiver's background color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.goodies.drawing.element.JunRectangularShapeElement#backgroundColor()
	 * @category visual properties
	 */
	public Color backgroundColor() {
		return this.borderElement().backgroundColor();
	}

	/**
	 * Set the receiver's background color.
	 * 
	 * @param aColor java.awt.Color
	 * @see jp.co.sra.jun.goodies.drawing.element.JunRectangularShapeElement#backgroundColor_(Color)
	 * @category visual properties
	 */
	public void backgroundColor_(Color aColor) {
		this.borderElement().backgroundColor_(aColor);
	}

	/**
	 * Answer the receiver's border color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.goodies.drawing.element.JunRectangularShapeElement#foregroundColor()
	 * @category visual properties
	 */
	public Color borderColor() {
		return this.borderElement().foregroundColor();
	}

	/**
	 * Set the receiver's border color.
	 * 
	 * @param aColor java.awt.Color
	 * @see jp.co.sra.jun.goodies.drawing.element.JunRectangularShapeElement#foregroundColor_(Color)
	 * @category visual properties
	 */
	public void borderColor_(Color aColor) {
		this.borderElement().foregroundColor_(aColor);
	}

	/**
	 * Answer the receiver's line width.
	 * 
	 * @return int
	 * @category visual properties
	 */
	public int lineWidth() {
		return this.borderElement().lineWidth();
	}

	/**
	 * Set the receiver's line width.
	 * 
	 * @param aNumber int
	 * @category visual properties
	 */
	public void lineWidth_(int aNumber) {
		this.borderElement().lineWidth_(aNumber);
		this.location_(this.location());
		this.extent_(this.extent());
	}

	/**
	 * Answer the receiver's properties model.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.properties.JunAbstractElementPropertiesModel
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#propertiesModel()
	 * @category visual properties
	 */
	public JunDrawingElementPropertiesModel propertiesModel() {
		if (propertiesModel == null) {
			propertiesModel = new JunTextboxPropertiesModel(this);
		}
		return propertiesModel;
	}
}
