package jp.co.sra.jun.goodies.drawing.map;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.util.HashMap;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StViewJPanel;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenuViewSwing;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.drawing.element.JunDrawingElement;
import jp.co.sra.jun.goodies.drawing.element.JunLinkElement;
import jp.co.sra.jun.goodies.drawing.element.JunPathElement;
import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;

/**
 * JunDrawingMapViewSwing class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDrawingMapViewSwing.java,v 8.11 2008/02/20 06:31:24 nisinaka Exp $
 */
public class JunDrawingMapViewSwing extends JunAbstractViewJPanel implements JunDrawingMapView {
	protected int editMode;
	protected StViewJPanel canvas;
	protected JScrollPane scrollPane;

	protected transient HashMap popupMenuTable;
	protected transient Image offScreenImage;
	protected transient Dimension offScreenSize;

	/**
	 * Create a new instance of JunDrawingMapViewSwing and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunDrawingMapViewSwing() {
		super();
	}

	/**
	 * Create a new instance of JunDrawingMapViewSwing and initialize it.
	 *
	 * @param aModel jp.co.sra.jun.goodies.drawing.map.JunDrawingMapModel
	 * @category Instance creation
	 */
	public JunDrawingMapViewSwing(JunDrawingMapModel aModel) {
		super(aModel);
	}

	/**
	 * Initialize the receiver. 
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#initialize()
	 * @category initialize-release 
	 */
	protected void initialize() {
		super.initialize();

		editMode = JunDrawingMapController.EDITMODE_SELECT;
		canvas = null;
		scrollPane = null;

		popupMenuTable = null;
		offScreenImage = null;
		offScreenSize = null;
	}

	/**
	 * Initialize the receiver's off screen image.
	 * 
	 * @category initialize-release
	 */
	protected void initializeOffScreen() {
		if (offScreenImage == null) {
			return;
		}

		Graphics2D aGraphics = (Graphics2D) offScreenImage.getGraphics().create();
		try {
			aGraphics.setColor(this.canvas().getBackground());
			aGraphics.setClip(0, 0, offScreenSize.width, offScreenSize.height);
			aGraphics.fillRect(0, 0, offScreenSize.width, offScreenSize.height);
			aGraphics.translate(-this.scrollPane().getViewport().getViewRect().x, -this.scrollPane().getViewport().getViewRect().y);
			if (this.getDrawingMapModel().isZooming()) {
				Jun2dPoint scalePoint = this.getDrawingMapModel().scalePoint();
				aGraphics.scale(scalePoint.x(), scalePoint.y());
			}
			this.displayCanvasOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
	}

	/**
	 * Answer the receiver's edit mode.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#editMode()
	 * @category accessing
	 */
	public int editMode() {
		return editMode;
	}

	/**
	 * Set the receiver's edit mode.
	 * 
	 * @param newEditMode int
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#editMode_(int)
	 * @category accessing
	 */
	public void editMode_(int newEditMode) {
		if (editMode != newEditMode) {
			int oldEditMode = editMode;
			editMode = newEditMode;
			this.firePropertyChange("editMode", oldEditMode, newEditMode);

			if (this.getDrawingMapController().startPoint() != null) {
				this.canvas().repaint(0);
				this.getDrawingMapController().initializeTemporaryVariables();
			}
		}
	}

	/**
	 * Answer the rectangle contains the elements.
	 * 
	 * @return java.awt.Rectangle or null
	 * @param anArray jp.co.sra.jun.goodies.drawing.element.JunDrawingElement[]
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#areaOfElements_(jp.co.sra.jun.goodies.drawing.element.JunDrawingElement[])
	 * @category accessing
	 */
	public Rectangle areaOfElements_(JunDrawingElement[] anArray) {
		Rectangle area = null;
		for (int i = 0; i < anArray.length; i++) {
			if (area == null) {
				area = new Rectangle(anArray[i].bounds());
			} else {
				area.add(anArray[i].bounds());
			}
		}
		return area;
	}

	/**
	 * Answer the receiver's model as JunDrawingMapModel.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMapModel
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#getDrawingMapModel()
	 * @category model accessing
	 */
	public JunDrawingMapModel getDrawingMapModel() {
		return (JunDrawingMapModel) this.model();
	}

	/**
	 * Answer the receiver's controller as JunDrawingMapController.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#getDrawingMapController()
	 * @category accessing
	 */
	public JunDrawingMapController getDrawingMapController() {
		return (JunDrawingMapController) this.canvas().controller();
	}

	/**
	 * Build the component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#buildComponent()
	 * @category interface opening
	 */
	protected void buildComponent() {
		this.setLayout(new BorderLayout());
		this.add(this.scrollPane(), BorderLayout.CENTER);
		this.setPreferredSize(new Dimension(300, 200));
	}

	/**
	 * Answer the draw view.
	 * 
	 * @return jp.co.sra.smalltalk.StViewJPanel
	 * @category interface opening
	 */
	public StViewJPanel canvas() {
		if (canvas == null) {
			canvas = new StViewJPanel(this.getDrawingMapModel()) {
				/**
				 * Answer the receiver's default controller
				 * 
				 * @return jp.co.sra.smalltalk.StController
				 * @category controller accessing
				 */
				protected StController defaultController() {
					return new JunDrawingMapController();
				}

				/**
				 * Display the receiver on the graphics.
				 * 
				 * @param aGraphics java.awt.Graphics
				 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
				 * @category displaying
				 */
				public void displayOn_(Graphics aGraphics) {
					JunDrawingMapViewSwing self = JunDrawingMapViewSwing.this;
					Image anImage = self.offScreen();
					if (anImage != null) {
						Rectangle clipBounds = aGraphics.getClipBounds();
						Graphics2D offScreenGraphics = self.graphicsWithZoomFrom_(anImage);
						try {
							if (clipBounds != null) {
								if (self.getDrawingMapModel().isZooming()) {
									Jun2dPoint scalePoint = self.getDrawingMapModel().scalePoint();
									clipBounds = new Rectangle((int) Math.floor(clipBounds.x / scalePoint.x()), (int) Math.floor(clipBounds.y / scalePoint.y()), (int) Math.ceil(clipBounds.width / scalePoint.x()), (int) Math.ceil(clipBounds.height
											/ scalePoint.y()));
								}
								offScreenGraphics.setColor(self.canvas().getBackground());
								offScreenGraphics.setClip(clipBounds);
								offScreenGraphics.fillRect(clipBounds.x, clipBounds.y, clipBounds.width, clipBounds.height);
							}
							self.displayCanvasOn_(offScreenGraphics);
						} finally {
							if (offScreenGraphics != null) {
								offScreenGraphics.dispose();
								offScreenGraphics = null;
							}
						}

						self.displayCanvasOn_scaledImage_unscaledClip_(aGraphics, anImage, null);
					} else {
						Graphics2D graphicsContext = (Graphics2D) anImage.getGraphics().create();
						try {
							if (self.getDrawingMapModel().isZooming()) {
								Jun2dPoint scalePoint = self.getDrawingMapModel().scalePoint();
								graphicsContext.scale(scalePoint.x(), scalePoint.y());
							}
							self.displayCanvasOn_(graphicsContext);
						} finally {
							if (graphicsContext != null) {
								graphicsContext.dispose();
								graphicsContext = null;
							}
						}
					}
				}

				/**
				 * Show the popup menu at the specified point on the view.
				 *
				 * @param x int
				 * @param y int
				 * @see jp.co.sra.smalltalk.StView#_showPopupMenu(int, int)
				 * @category popup menu accessing
				 */
				public void _showPopupMenu(int x, int y) {
					JPopupMenu popupMenu = JunDrawingMapViewSwing.this.popupMenuView(x, y);
					if (popupMenu != null) {
						popupMenu.show(this, x, y);
					}
				}

				/**
				 * Updates this canvas.
				 *
				 * @param aGraphics java.awt.Graphics
				 * @category updating
				 */
				public void update(Graphics aGraphics) {
					aGraphics.setClip(0, 0, this.getWidth(), this.getHeight());
					this.displayOn_(aGraphics);
				}

				/**
				 * Receive a change notice from an object of whom the receiver is a
				 * dependent.  The argument anAspectSymbol is typically a Symbol that
				 * indicates what change has occurred.
				 * 
				 * @param evt jp.co.sra.smalltalk.DependentEvent
				 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
				 * @category updating
				 */
				public void update_(DependentEvent evt) {
				}
			};
			canvas.setBackground(Color.white);
			canvas.setLayout(null);
		}
		return canvas;
	}

	/**
	 * Answer the scroll pane.
	 * 
	 * @return javax.swing.JScrollPane
	 * @category interface opening
	 */
	protected JScrollPane scrollPane() {
		if (scrollPane == null) {
			final JunDrawingMapViewSwing self = this;
			scrollPane = new JScrollPane(this.canvas(), JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
			scrollPane.getHorizontalScrollBar().setFocusable(false);
			scrollPane.getVerticalScrollBar().setFocusable(false);
			scrollPane.getViewport().setBackground(Color.white);
			scrollPane.addComponentListener(new ComponentAdapter() {
				public void componentResized(ComponentEvent e) {
					self.updateCanvasSize_(self.canvas().getSize());
				}
			});
			scrollPane.getViewport().addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					self.repaint(0);
				}
			});
			this.updateCanvasSize_(this.canvas().getSize());
		}
		return scrollPane;
	}

	/**
	 * Answer the center point on scroll pane.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#centerPointOnScrollPane()
	 * @category bounds accessing
	 */
	public Point centerPointOnScrollPane() {
		Point scrollPosition = this.scrollPane().getViewport().getViewPosition();
		Dimension size = this.scrollPane().getSize();
		return new Point(scrollPosition.x + size.width / 2, scrollPosition.y + size.height / 2);
	}

	/**
	 * Display the canvas part of the receiver on the graphics.
	 * The subclasses may override this.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#displayCanvasOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayCanvasOn_(Graphics aGraphics) {
		if (this.canvas().isShowing() == false) {
			return;
		}

		StDisplayable displayObject = this.getDrawingMapModel().displayObject();
		if (displayObject != null) {
			displayObject.displayOn_(aGraphics);
		}
	}

	/**
	 * Display the scaled image on a graphics with the unscaled clipping bounds.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param anImage java.awt.Image
	 * @param aRectangle java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#displayCanvasOn_scaledImage_unscaledClip_(java.awt.Graphics, java.awt.Image, java.awt.Rectangle)
	 * @category displaying
	 */
	public void displayCanvasOn_scaledImage_unscaledClip_(Graphics aGraphics, Image anImage, Rectangle aRectangle) {
		if (aGraphics == null) {
			return;
		}

		try {
			if (aRectangle != null) {
				Rectangle clipBounds = new Rectangle(aRectangle);
				if (this.getDrawingMapModel().isZooming()) {
					Jun2dPoint scalePoint = this.getDrawingMapModel().scalePoint();
					clipBounds.setBounds((int) Math.floor(aRectangle.x * scalePoint.x()), (int) Math.floor(aRectangle.y * scalePoint.y()), (int) Math.ceil(aRectangle.width * scalePoint.x()), (int) Math.ceil(aRectangle.height * scalePoint.y()));
				}
				aGraphics.setClip(clipBounds.x, clipBounds.y, clipBounds.width + 1, clipBounds.height + 1);
			}
			Rectangle viewRect = this.scrollPane().getViewport().getViewRect();
			aGraphics.drawImage(anImage, viewRect.x, viewRect.y, viewRect.width, viewRect.height, this);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
	}

	/**
	 * Invalidate the Rectangle aRectangle. If aBoolean is false, repair later.
	 * Propagate a damage rectangle up the containment hierarchy.
	 * This will result in a displayCanvasOn_(Graphics) being sent to the receiver.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @param aBoolean boolean
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#_invalidateRectangle_repairNow_(java.awt.Rectangle, boolean)
	 * @category displaying
	 */
	public void _invalidateRectangle_repairNow_(Rectangle aRectangle, boolean aBoolean) {
		if (aRectangle == null) {
			return;
		}

		Rectangle zoomBounds = null;
		if (this.getDrawingMapModel().isZooming()) {
			Jun2dPoint scalePoint = this.getDrawingMapModel().scalePoint();
			zoomBounds = new Rectangle((int) Math.floor(aRectangle.x * scalePoint.x()), (int) Math.floor(aRectangle.y * scalePoint.y()), (int) Math.ceil(aRectangle.width * scalePoint.x()), (int) Math.ceil(aRectangle.height * scalePoint.y()));
			zoomBounds.grow(5, 5);
		} else {
			zoomBounds = aRectangle;
		}

		Graphics aGraphics = this.canvas().getGraphics();
		if (aGraphics == null) {
			return;
		}
		try {
			aGraphics.setClip(zoomBounds.x, zoomBounds.y, zoomBounds.width, zoomBounds.height);
			this.canvas().displayOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
	}

	/**
	 * Create the off screen image.
	 * 
	 * @return java.awt.Image
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#createOffScreen()
	 * @category displaying
	 */
	public Image createOffScreen() {
		this.initializeOffScreen();
		return this.offScreen();
	}

	/**
	 * Answer the off screen image.
	 * 
	 * @return java.awt.Image
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#offScreen()
	 * @category displaying
	 */
	public Image offScreen() {
		Dimension viewSize = this.scrollPane().getViewport().getViewRect().getSize();
		if (viewSize.equals(offScreenSize) == false || offScreenImage == null) {
			offScreenImage = this.createImage(viewSize.width, viewSize.height);
			offScreenSize = viewSize;

			this.initializeOffScreen();
		}
		return offScreenImage;
	}

	/**
	 * Answer the graphics context with zooming factor from an image.
	 * 
	 * @param anImage java.awt.Image
	 * @return java.awt.Graphics2D
	 * @see jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView#graphicsWithZoomFrom_(java.awt.Image)
	 * @category displaying
	 */
	public Graphics2D graphicsWithZoomFrom_(Image anImage) {
		Graphics2D aGraphics = (Graphics2D) anImage.getGraphics().create();
		aGraphics.translate(-this.scrollPane().getViewport().getViewRect().x, -this.scrollPane().getViewport().getViewRect().y);
		if (this.getDrawingMapModel().isZooming()) {
			Jun2dPoint scalePoint = this.getDrawingMapModel().scalePoint();
			aGraphics.scale(scalePoint.x(), scalePoint.y());
		}
		return aGraphics;
	}

	/**
	 * Update the vertex menu indication.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @category menu accessing
	 */
	public void updateVertexMenuIndication(JunDrawingElement anElement) {
		StMenu aMenu = (StMenu) this.getDrawingMapModel().popupMenuFor_(anElement).atNameKey_($("vertexMenu"));
		if (aMenu == null) {
			return;
		}
		JunPathElement pathElement = null;
		if (anElement.isPath()) {
			pathElement = (JunPathElement) anElement;
		} else if (anElement.isLink()) {
			pathElement = ((JunLinkElement) anElement).pathElement();
		} else {
			return;
		}

		Point aPoint = this.getDrawingMapController().popupMenuPoint();
		StMenuItem menuElement = aMenu.atNameKey_($("addVertex"));
		if (menuElement != null) {
			if (aPoint != null && pathElement.containsPoint_(aPoint)) {
				menuElement.beEnabled(true);
			} else {
				menuElement.beEnabled(false);
			}
		}

		menuElement = aMenu.atNameKey_($("removeVertex"));
		if (menuElement != null) {
			if (aPoint != null && pathElement.containsPointInControllArea_(aPoint) && pathElement.points().size() > 2) {
				menuElement.beEnabled(true);
			} else {
				menuElement.beEnabled(false);
			}
		}
	}

	/**
	 * Answer the receiver's popup menu view specified position.
	 * 
	 * @param x int
	 * @param y int
	 * @return javax.swing.JPopupMenu
	 * @see jp.co.sra.smalltalk.StViewJPanel#popupMenuView()
	 * @category popup menu
	 */
	public JPopupMenu popupMenuView(int x, int y) {
		JunDrawingElement anElement = this.getDrawingMapModel().mapObject().currentElement();
		StSymbol key = (anElement != null) ? anElement._className() : null;

		if (this.popupMenuTable().containsKey(key) == false) {
			JPopupMenu newPopupMenu = new StPopupMenuViewSwing(this.getDrawingMapModel().popupMenuFor_(anElement)).toPopupMenu();
			this.popupMenuTable().put(key, newPopupMenu);
			this.canvas().add(newPopupMenu);
		}

		this.updateVertexMenuIndication(anElement);
		return (JPopupMenu) this.popupMenuTable().get(key);
	}

	/**
	 * Answer the receiver's popup menu table.
	 * 
	 * @return java.util.HashMap
	 * @category popup menu
	 */
	public HashMap popupMenuTable() {
		if (popupMenuTable == null) {
			popupMenuTable = new HashMap();
		}
		return popupMenuTable;
	}

	/**
	 * Update the receiver according to the change notification from the model.
	 * 
	 * @param evt jp.co.sra.smalltalk.DevendentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (this.isShowing() == false) {
			return;
		}

		this.getDrawingMapModel().updateMenuIndication();
		StSymbol aSymbol = evt.getAspect();
		if (aSymbol == $("redisplay")) {
			if (evt.getParameter() == null) {
				this.updateCanvasSize_(this.canvas().getSize());
			} else {
				this._invalidateRectangle_repairNow_((Rectangle) evt.getParameter(), true);
			}
			return;
		}
		if (aSymbol == $("bounds")) {
			this.updateCanvasSize_(this.scrollPane().getViewport().getExtentSize());
			return;
		}
		if (aSymbol == $("selection")) {
			if (evt.getParameter() == null) {
				Rectangle repaintArea = this.areaOfElements_(this.getDrawingMapModel().mapObject()._componentElements());
				if (repaintArea != null) {
					repaintArea.grow(JunDrawingElement.CONTROLL_AREA_SIZE / 2, JunDrawingElement.CONTROLL_AREA_SIZE / 2);
				}
				this._invalidateRectangle_repairNow_(repaintArea, true);
			} else {
				this._invalidateRectangle_repairNow_((Rectangle) evt.getParameter(), true);
			}
			return;
		}
		if (aSymbol == $("location")) {
			this.updateCanvasSize_(this.canvas().getSize());
			return;
		}
		if (aSymbol == $("zoom")) {
			offScreenImage = null;
			offScreenSize = null;
			this.updateCanvasSize_(this.scrollPane().getViewport().getExtentSize());
			return;
		}

		super.update_(evt);
	}

	/**
	 * Update the canvas size.
	 * 
	 * @param minimumSize java.awt.Dimension
	 * @category updating
	 */
	protected void updateCanvasSize_(Dimension minimumSize) {
		Dimension scaledSize = this.getDrawingMapModel().scaledSize();
		Dimension newSize = new Dimension(Math.max(scaledSize.width, minimumSize.width), Math.max(scaledSize.height, minimumSize.height));
		if (this.canvas().getSize().equals(newSize) == false) {
			this.canvas().setPreferredSize(newSize);
			this.scrollPane().getViewport().doLayout();

			offScreenImage = null;
			offScreenSize = null;
		}

		Rectangle viewportArea = this.scrollPane().getViewport().getViewRect();
		if (this.getDrawingMapModel().isZooming()) {
			Jun2dPoint scalePoint = this.getDrawingMapModel().scalePoint();
			viewportArea = new Rectangle((int) Math.floor(viewportArea.x / scalePoint.x()), (int) Math.floor(viewportArea.y / scalePoint.y()), (int) Math.ceil(viewportArea.width / scalePoint.x()), (int) Math.ceil(viewportArea.height / scalePoint.y()));
			viewportArea.grow(5, 5);
		}
		this._invalidateRectangle_repairNow_(viewportArea, true);
	}
}
