package jp.co.sra.jun.goodies.dump;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;

import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunDumpModel class
 * 
 *  @author    nisinaka
 *  @created   2003/03/18 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun436 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDumpModel.java,v 8.11 2008/02/20 06:31:33 nisinaka Exp $
 */
public class JunDumpModel extends JunApplicationModel {
	protected byte[] byteArray;
	protected int startAddress;
	protected StValueHolder textModel;

	/**
	 * Create a new instance of <code>JunDumpModel</code> and initialize it.
	 * 
	 * @param bytes byte[]
	 * @category Instance creation
	 */
	public JunDumpModel(byte[] bytes) {
		super();
		this.bytes_start_(bytes, 0);
	}

	/**
	 * Create a new instance of <code>JunDumpModel</code> from the file.
	 *
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunDumpModel(File aFile) {
		this(aFile, 0, 0);
	}

	/**
	 * Create a new instance of <code>JunDumpModel</code> from the file.
	 *
	 * @param aFile java.io.File
	 * @param startAddress int
	 * @category Instance creation
	 */
	public JunDumpModel(File aFile, int startAddress) {
		this(aFile, startAddress, 0);
	}

	/**
	 * Create a new instance of <code>JunDumpModel</code> from the file.
	 *
	 * @param aFile java.io.File
	 * @param startAddress int
	 * @param stopAddress int
	 * @category Instance creation
	 */
	public JunDumpModel(File aFile, int startAddress, int stopAddress) {
		byte[] byteArray = null;

		long fileSize = aFile.length();
		long stopLocation = (stopAddress <= 0) ? fileSize : Math.min(stopAddress, fileSize);
		int startLocation = (int) Math.min(startAddress, stopLocation);

		InputStream input = null;
		ByteArrayOutputStream output = null;
		try {
			input = new BufferedInputStream(new FileInputStream(aFile));
			output = new ByteArrayOutputStream(1024);

			input.skip(startLocation);
			for (int i = 0; i < stopLocation - startLocation; i++) {
				output.write(input.read());
			}
			byteArray = output.toByteArray();
		} catch (FileNotFoundException e) {
			byteArray = new byte[0];
		} catch (IOException e) {
			byteArray = new byte[0];
		} finally {
			try {
				if (input != null) {
					input.close();
				}
				if (output != null) {
					output.flush();
					output.close();
				}
			} catch (IOException e) {
			}
		}

		this.bytes_start_(byteArray, startLocation);
	}

	/**
	 * Answer my bytes.
	 *
	 * @return byte[]
	 * @category accessing
	 */
	public byte[] bytes() {
		if (byteArray == null) {
			byteArray = new byte[0];
		}
		return byteArray;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category Interface opening
	 */
	public jp.co.sra.smalltalk.StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunDumpViewAwt(this);
		} else {
			return new JunDumpViewSwing(this);
		}
	}

	/**
	 * Answer true if the specified object is a JunDumpModel and its value equals to the receiver.
	 *
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == this) {
			return true;
		}

		if (anObject instanceof JunDumpModel == false) {
			return false;
		}

		if (this.start() != ((JunDumpModel) anObject).start()) {
			return false;
		}

		byte[] bytes1 = this.bytes();
		byte[] bytes2 = ((JunDumpModel) anObject).bytes();
		if (bytes1.length != bytes2.length) {
			return false;
		}

		for (int i = 0; i < bytes1.length; i++) {
			if (bytes1[i] != bytes2[i]) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		String aString = (String) this.textModel().value();
		if (aString == null) {
			aString = "null";
		}
		aWriter.write(aString);
	}

	/**
	 * Answer the current start position.
	 *
	 * @return int
	 * @category accessing
	 */
	public int start() {
		return startAddress;
	}

	/**
	 * Answer the current stop position.
	 *
	 * @return int
	 * @category accessing
	 */
	public int stop() {
		return this.bytes().length;
	}

	/**
	 * Answer my text model.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category model accessing
	 */
	public StValueHolder textModel() {
		if (textModel == null) {
			textModel = new StValueHolder(new String());
		}
		return textModel;
	}

	/**
	 * Convert the address number to the string representation.
	 *
	 * @return java.lang.String
	 * @param address int
	 * @category private
	 */
	protected String addressString_(int address) {
		StringWriter sw = new StringWriter(9);
		String string = Integer.toHexString(address).toUpperCase();
		if (string.length() > 8) {
			string = string.substring(string.length() - 8);
		} else {
			for (int i = 8 - string.length(); i > 0; i--) {
				sw.write('0');
			}
		}
		sw.write(string);
		sw.write(':');
		return sw.toString();
	}

	/**
	 * Initialize with the byte array and the start index.
	 *
	 * @param bytes byte[]
	 * @param start int
	 * @category accessing
	 */
	protected void bytes_start_(byte[] bytes, int start) {
		byteArray = bytes;
		startAddress = start;
		this.textModel().value_(this.dump());
	}

	/**
	 * Dump the bytes.
	 *
	 * @return java.lang.String
	 * @category private
	 */
	protected String dump() {
		StringWriter sw = new StringWriter(1024);
		PrintWriter pw = null;
		try {
			pw = new PrintWriter(sw);

			int count = this.start();
			StringWriter chars = new StringWriter(16);
			if (count % 16 != 0) {
				pw.write(this.addressString_(count - count % 16));
				for (int n = 0; n < count % 16; n++) {
					if (n % 8 == 0) {
						pw.write(' ');
					}
					if (n % 2 == 0) {
						pw.write(' ');
					}
					pw.write("  ");
					chars.write(' ');
				}
			}

			byte[] bytes = this.bytes();
			for (int i = 0; i < bytes.length; i++) {
				if (count % 16 == 0) {
					pw.write(this.addressString_(count));
				}
				if (count % 8 == 0) {
					pw.write(' ');
				}
				if (count % 2 == 0) {
					pw.write(' ');
				}
				try {
					String string = Integer.toHexString(bytes[i] & 0xFF).toUpperCase();
					for (int j = 2 - string.length(); j > 0; j--) {
						pw.write('0');
					}
					pw.write(string);
					chars.write(this.printable_(bytes[i]));
				} finally {
					count = count + 1;
					if (count % 16 == 0) {
						pw.write("  ");
						pw.println(chars.toString());
						chars = new StringWriter(16);
					}
				}
			}

			while (count % 16 != 0) {
				if (count % 8 == 0) {
					pw.write(' ');
				}
				if (count % 2 == 0) {
					pw.write(' ');
				}
				try {
					pw.write("  ");
				} finally {
					count = count + 1;
					if (count % 16 == 0) {
						pw.write("  ");
						pw.println(chars.toString());
						chars = new StringWriter(16);
					}
				}
			}
		} finally {
			if (pw != null) {
				pw.flush();
				pw.close();
			}
		}
		return sw.toString();
	}

	/**
	 * Initialize the ApplicationModel when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		byteArray = null;
		startAddress = 0;
		textModel = null;
	}

	/**
	 * Answer the printable char for the byte number.
	 *
	 * @return char
	 * @param byteNumber byte
	 * @category private
	 */
	protected char printable_(byte byteNumber) {
		if (33 <= byteNumber && byteNumber <= 126) {
			return (char) byteNumber;
		}
		return ' ';
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Dump");
	}

	/**
	 * Answre one of my views.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category private
	 */
	public StView getView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunDumpView && ((JunDumpView) dependents[i]).model() == this) {
				return (JunDumpView) dependents[i];
			}
		}

		return null;
	}
}
