package jp.co.sra.jun.goodies.files;

import java.awt.Cursor;
import java.awt.Image;
import java.io.File;
import java.util.Collections;
import java.util.LinkedList;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StBlockValued;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.SystemResourceSupport;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunFileModel class
 * 
 *  @author    nisinaka
 *  @created   2003/03/05 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun433 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileModel.java,v 8.11 2008/02/20 06:31:33 nisinaka Exp $
 */
public class JunFileModel extends JunApplicationModel implements StBlockValued {

	/**
	 * The FileType class defines a file type name with its extensions.
	 */
	public static class FileType {
		protected String name;
		protected String[] extensions;

		public FileType(String nameString, String[] extensionStrings) {
			name = nameString;
			extensions = extensionStrings;
		}

		public static FileType All(String nameString) {
			return new FileType(nameString, new String[] { "*" });
		}

		public String name() {
			return name;
		}

		public String[] extensions() {
			return extensions;
		}
	}

	protected static final String DefaultDirectoryItem = JunSystem.$String("Default directory");

	private static Image ClosedFolderIcon = null;
	private static Image DeviceIcon = null;
	private static Image OpenedFolderIcon = null;

	protected StValueHolder fileName;
	protected StValueHolder fileList;
	protected StValueHolder fileType;
	protected JunFileModel.FileType[] fileTypes;
	protected File currentDirectory;
	protected StValueHolder currentDirectoryLabel;
	protected int currentDirectoryLevel;
	protected StValueHolder directoryList;
	protected StValueHolder currentDrive;
	protected File[] drives;

	/**
	 * Answer the default directory.
	 *
	 * @return java.io.File
	 * @category Accessing
	 */
	public static File DefaultDirectory() {
		return new File(System.getProperty("user.dir"));
	}

	/**
	 * Create a closedFolderIcon image.
	 * 
	 * @return java.awt.Image
	 * @category Resources
	 */
	public static Image ClosedFolderIcon() {
		if (ClosedFolderIcon == null) {
			ClosedFolderIcon = SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/files/closedFolderIcon.png");
		}
		return ClosedFolderIcon;
	}

	/**
	 * Create a DeviceIcon image.
	 * 
	 * @return java.awt.Image
	 * @category Resources
	 */
	public static Image DeviceIcon() {
		if (DeviceIcon == null) {
			DeviceIcon = SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/files/deviceIcon.png");
		}
		return DeviceIcon;
	}

	/**
	 * Create a openedFolderIcon image.
	 * 
	 * @return java.awt.Image
	 * @category Resources
	 */
	public static Image OpenedFolderIcon() {
		if (OpenedFolderIcon == null) {
			OpenedFolderIcon = SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/files/openedFolderIcon.png");
		}
		return OpenedFolderIcon;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		fileName = null;
		fileList = null;
		fileType = null;
		fileTypes = null;
		currentDirectory = null;
		currentDirectoryLabel = null;
		currentDirectoryLevel = -1;
		directoryList = null;
		fileList = null;
		currentDrive = null;
		drives = null;
	}

	/**
	 * Answer the current directory.
	 *
	 * @return java.io.File
	 * @category accessing
	 */
	public File currentDirectory() {
		if (currentDirectory == null) {
			this.currentDirectory_(DefaultDirectory());
		}
		return currentDirectory;
	}

	/**
	 * Set the new directory as a current directory.
	 *
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void currentDirectory_(File aFile) {
		File newDirectory = null;

		if (aFile == null) {
			newDirectory = DefaultDirectory();
		} else {
			newDirectory = aFile;
			if (newDirectory.exists() && newDirectory.isDirectory()) {
				if (newDirectory.isAbsolute() == false) {
					newDirectory = newDirectory.getAbsoluteFile();
				}
			} else {
				newDirectory = DefaultDirectory();
			}
		}

		currentDirectory = newDirectory;
		currentDirectoryLevel = -1;
		this.updateCurrentDrive();
		this.updateCurrentDirectoryLabel();
		this.updateDirectoryAndFileList();
	}

	/**
	 * Answer the current directory level.
	 *
	 * @return int
	 * @category accessing
	 */
	public int currentDirectoryLevel() {
		if (currentDirectoryLevel < 0) {
			currentDirectoryLevel = 0;
			File aFile = this.currentDirectory();
			while (aFile != null) {
				currentDirectoryLevel++;
				aFile = aFile.getParentFile();
			}
		}
		return currentDirectoryLevel;
	}

	/**
	 * Answer the current file.
	 *
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object value() {
		return new File(this.currentDirectory(), (String) this.fileName().value());
	}

	/**
	 * Set the file as an initial value.
	 *
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void value_(File aFile) {
		if (aFile == null) {
			return;
		}

		if (aFile.exists()) {
			if (aFile.isDirectory()) {
				this.currentDirectory_(aFile);
			} else {
				this.currentDirectory_(aFile.getParentFile());
				this.fileName().value_(aFile.getName());
			}
		} else {
			this.fileName().value_(aFile.getName());
			File directory = aFile.getParentFile();
			while (directory != null && directory.exists() == false) {
				directory = directory.getParentFile();
			}
			this.currentDirectory_(directory);
		}
	}

	/**
	 * Answer the value holder for the current directory label.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder currentDirectoryLabel() {
		if (currentDirectoryLabel == null) {
			currentDirectoryLabel = new StValueHolder(new String());
		}
		return currentDirectoryLabel;
	}

	/**
	 * Answer the value holder for the current drive.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder currentDrive() {
		if (currentDrive == null) {
			currentDrive = new StValueHolder(null);
			this.updateCurrentDrive();
		}
		return currentDrive;
	}

	/**
	 * Answer the value holder for the directory list.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder directoryList() {
		if (directoryList == null) {
			directoryList = new StValueHolder(new String[0]);
		}
		return directoryList;
	}

	/**
	 * Answer the array of all drives.
	 *
	 * @return java.io.File[]
	 * @category aspects
	 */
	public File[] drives() {
		return File.listRoots();
	}

	/**
	 * Answer the value holder for the file list.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder fileList() {
		if (fileList == null) {
			fileList = new StValueHolder(new String[0]);
		}
		return fileList;
	}

	/**
	 * Answer the value holder for the file name.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder fileName() {
		if (fileName == null) {
			fileName = new StValueHolder(new String());
		}
		return fileName;
	}

	/**
	 * Answer the value holder for the file type.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder fileType() {
		if (fileType == null) {
			fileType = new StValueHolder(this.fileTypes()[0]);
		}
		return fileType;
	}

	/**
	 * Answer the current file types.
	 *
	 * @return jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @category aspects
	 */
	public FileType[] fileTypes() {
		if (fileTypes == null) {
			fileTypes = this.defaultFileTypes();
		}
		return fileTypes;
	}

	/**
	 * Set the new file types.
	 *
	 * @param newFileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @category aspects
	 */
	public void fileTypes_(FileType[] newFileTypes) {
		fileTypes = newFileTypes;
	}

	/**
	 * Answer a StBlockValue that computes aBlock with the receiver's value as the argument.
	 *
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this);
	}

	/**
	 * Called when the current drive is changed.
	 *
	 * @param directoryName java.lang.String
	 * @category actions
	 */
	protected void currentDriveChanged(String directoryName) {
		if (DefaultDirectoryItem.equals(directoryName)) {
			this.currentDirectory_(DefaultDirectory());
		} else {
			this.currentDirectory_(new File(directoryName));
		}
		this.changed_($("value"));
	}

	/**
	 * Called when the item on the directory list is double clicked.
	 *
	 * @param index int
	 * @param aString java.lang.String
	 * @category actions
	 */
	protected void directoryListDoubleClicked(int index, String aString) {
		String name = aString.trim();
		File directory = this.currentDirectory();
		if (index >= this.currentDirectoryLevel()) {
			this.currentDirectory_(new File(directory, name));
		} else {
			for (int i = this.currentDirectoryLevel() - 1; i > index; i--) {
				directory = directory.getParentFile();
			}
			this.currentDirectory_(directory);
		}
		this.changed_($("value"));
	}

	/**
	 * Called when a file is selected on a file list.
	 *
	 * @param fileName java.lang.String
	 * @category actions
	 */
	protected void fileListChanged(String fileName) {
		this.fileName().value_(fileName);
		this.changed_($("value"));
	}

	/**
	 * Called when the file name is changed.
	 *
	 * @param fileName java.lang.String
	 * @category actions
	 */
	protected void fileNameChanged(String fileName) {
		this.fileName().value_(fileName);
		this.changed_($("value"));
	}

	/**
	 * Called when a file type is changed.
	 *
	 * @param index int
	 * @category actions
	 */
	protected void fileTypeChanged(int index) {
		this.fileType().value_(this.fileTypes()[index]);
		this.updateDirectoryAndFileList();
		this.changed_($("value"));
	}

	/**
	 * Answer the current file types.
	 *
	 * @return jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @category defaults
	 */
	protected JunFileModel.FileType[] defaultFileTypes() {
		return new JunFileModel.FileType[] { JunFileModel.FileType.All(JunSystem.$String("All files") + " (*)") };
	}

	/**
	 * Update the current directory label.
	 * 
	 * @category updating
	 */
	protected void updateCurrentDirectoryLabel() {
		int maxSize = 32;
		String directoryString = this.currentDirectory().getPath();
		if (directoryString.length() > maxSize) {
			directoryString = directoryString.substring(0, 3) + "..." + directoryString.substring(directoryString.length() - maxSize + 5, directoryString.length());
		}
		this.currentDirectoryLabel().value_(directoryString);
	}

	/**
	 * Update the current drive.
	 * 
	 * @category updating
	 */
	protected void updateCurrentDrive() {
		File directory = this.currentDirectory();
		while (directory.getParent() != null) {
			directory = directory.getParentFile();
		}
		this.currentDrive().value_(directory);
	}

	/**
	 * Update the directory list and the file list.
	 * 
	 * @category updating
	 */
	protected void updateDirectoryAndFileList() {
		JunCursors cursor = new JunCursors(new Cursor(Cursor.WAIT_CURSOR));
		try {
			cursor._show();

			String listingName;
			LinkedList dirs1 = new LinkedList();
			File directory = this.currentDirectory();
			while (directory != null) {
				listingName = directory.getName();
				if (listingName.length() == 0) {
					listingName = directory.getPath();
				}
				dirs1.addFirst(listingName);
				directory = directory.getParentFile();
			}

			LinkedList dirs2 = new LinkedList();
			LinkedList files = new LinkedList();
			File[] directoryContents = this.currentDirectory().listFiles();
			for (int i = 0; i < directoryContents.length; i++) {
				File aFile = directoryContents[i];
				listingName = aFile.getName();
				if (aFile.canRead()) {
					if (aFile.isDirectory()) {
						dirs2.add(listingName);
					} else {
						if (this.checkFileTypeMatch_(listingName)) {
							files.add(listingName);
						}
					}
				}
			}

			Collections.sort(dirs2);
			dirs1.addAll(dirs2);
			this.directoryList().value_(dirs1.toArray());

			Collections.sort(files);
			this.fileList().value_(files.toArray());

		} finally {
			cursor._restore();
		}
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunFileViewAwt(this);
		} else {
			return new JunFileViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("File Model");
	}

	/**
	 * Answer true if the filename string matches the current file type, otherwise false.
	 *
	 * @return boolean
	 * @param filenameString java.lang.String
	 * @category private
	 */
	protected boolean checkFileTypeMatch_(String filenameString) {
		FileType aFileType = (FileType) this.fileType().value();
		if (aFileType == null) {
			return true;
		}

		String[] extensions = aFileType.extensions();
		if (extensions == null) {
			return true;
		}

		for (int i = 0; i < extensions.length; i++) {
			if (JunStringUtility.StringMatch_and_(filenameString, extensions[i])) {
				return true;
			}
		}

		return false;
	}

}
