package jp.co.sra.jun.goodies.gauge;

import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.IOException;
import java.io.Writer;
import java.text.DecimalFormat;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StBlockValued;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunGaugeModel class
 * 
 *  @author    nisinaka
 *  @created   2004/04/13 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun491 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGaugeModel.java,v 8.12 2008/02/20 06:31:34 nisinaka Exp $
 */
public class JunGaugeModel extends JunApplicationModel implements StBlockValued {

	protected double gaugeValue;
	protected double[] gaugeRange;
	protected StBlockClosure valueStringBlock;

	/**
	 * Create a new instance of JunGaugeModel with default values.
	 * 
	 * @category Instance creation
	 */
	public JunGaugeModel() {
		super();
	}

	/**
	 * Create a new instance of JunGaugeModel with an integer value and the default range.
	 * 
	 * @param value double
	 * @category Instance creation
	 */
	public JunGaugeModel(double value) {
		this(value, null);
	}

	/**
	 * Create a new instance of JunGaugeModel with a value and a range.
	 * 
	 * @param value double
	 * @param range double[]
	 * @category Instance creation
	 */
	public JunGaugeModel(double value, double[] range) {
		super();
		this.range_(range);
		this.value_(value);
	}

	/**
	 * Create a new instance of JunGaugeModel and initialize it.
	 * 
	 * @param value double
	 * @param fromValue double
	 * @param toValue double
	 * @param roundValue double
	 * @category Instance creation
	 */
	public JunGaugeModel(double value, double fromValue, double toValue, double roundValue) {
		this(value, new double[] { fromValue, toValue, 10, roundValue });
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		gaugeValue = Double.NaN;
		gaugeRange = null;
		valueStringBlock = null;
	}

	/**
	 * Answer my current value.
	 * 
	 * @return java.lang.Object
	 * @see jp.co.sra.smalltalk.StValued#value()
	 * @category accessing
	 */
	public Object value() {
		return new Double(this._doubleValue());
	}

	/**
	 * Answer my current value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double _doubleValue() {
		if (Double.isNaN(gaugeValue)) {
			gaugeValue = this.defaultGaugeValue();
		}

		double roundValue = this.roundValue();
		if (roundValue >= 1) {
			return Math.round((this.fromValue() + gaugeValue * (this.toValue() - this.fromValue())) / roundValue) * roundValue;
		} else {
			roundValue = 1 / roundValue;
			return Math.round((this.fromValue() + gaugeValue * (this.toValue() - this.fromValue())) * roundValue) / roundValue;
		}
	}

	/**
	 * Set my new value.
	 * 
	 * @param newValue double
	 * @category accessing
	 */
	public void value_(double newValue) {
		newValue = (newValue - this.fromValue()) / (this.toValue() - this.fromValue());
		newValue = Math.max(0, Math.min(newValue, 1));
		if (this.normalizedValue() != newValue) {
			gaugeValue = newValue;
			this.changed_($("value"));
		}
	}

	/**
	 * Answer my current range.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] range() {
		if (gaugeRange == null) {
			gaugeRange = this.defaultGaugeRange();
		}
		return gaugeRange;
	}

	/**
	 * Set my new range.
	 * 
	 * @param newRange double[]
	 * @category accessing
	 */
	public void range_(double[] newRange) {
		gaugeRange = newRange;
		this.changed_($("range"));
	}

	/**
	 * Answer my current FROM value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double fromValue() {
		return this.range()[0];
	}

	/**
	 * Set my new FROM value.
	 * 
	 * @param newFromValue double
	 * @category accessing
	 */
	public void fromValue_(double newFromValue) {
		if (this.fromValue() != newFromValue) {
			double[] newRange = this.range();
			newRange[0] = newFromValue;
			this.range_(newRange);
		}
	}

	/**
	 * Answer my current TO value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double toValue() {
		return this.range()[1];
	}

	/**
	 * Set my new TO value.
	 * 
	 * @param newToValue double
	 * @category accessing
	 */
	public void toValue_(double newToValue) {
		if (this.toValue() != newToValue) {
			double[] newRange = this.range();
			newRange[1] = newToValue;
			this.range_(newRange);
		}
	}

	/**
	 * Answer my current DIVIDE value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double divideValue() {
		return this.range()[2];
	}

	/**
	 * Set my new DIVIDE value.
	 * 
	 * @param newDivideValue double
	 * @category accessing
	 */
	public void divideValue_(double newDivideValue) {
		if (this.divideValue() != newDivideValue) {
			double[] newRange = this.range();
			newRange[2] = newDivideValue;
			this.range_(newRange);
		}
	}

	/**
	 * Answer my current ROUND value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double roundValue() {
		return this.range()[3];
	}

	/**
	 * Set my new ROUND value.
	 * 
	 * @param newRoundValue double
	 * @category accessing
	 */
	public void roundValue_(double newRoundValue) {
		if (this.roundValue() != newRoundValue) {
			double[] newRange = this.range();
			newRange[3] = newRoundValue;
			this.range_(newRange);
		}
	}

	/**
	 * Answer my current normalized value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double normalizedValue() {
		return (this._doubleValue() - this.fromValue()) / (this.toValue() - this.fromValue());
	}

	/**
	 * Set my new normalized value.
	 * 
	 * @param newNormalizedValue double
	 * @category accessing
	 */
	public void normalizedValue_(double newNormalizedValue) {
		newNormalizedValue = Math.max(0, Math.min(newNormalizedValue, 1));
		double newValue = Math.round((this.fromValue() + newNormalizedValue * (this.toValue() - this.fromValue())) / this.roundValue()) * this.roundValue();
		this.value_(newValue);
	}

	/**
	 * Answer the step value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double stepValue() {
		double stepValue = this.roundValue();
		if (this.fromValue() > this.toValue()) {
			stepValue *= -1;
		}
		return stepValue;
	}

	/**
	 * Answer my current value string block.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public StBlockClosure valueStringBlock() {
		if (valueStringBlock == null) {
			valueStringBlock = this.defaultValueStringBlock();
		}
		return valueStringBlock;
	}

	/**
	 * Set my new value string block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public void valueStringBlock_(StBlockClosure aBlock) {
		valueStringBlock = aBlock;
	}

	/**
	 * Answer an StBlockValue that computes aBlock with the receiver's value as the argument.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this);
	}

	/**
	 * Answer my KeyListener which handles my keyboard events.
	 *
	 * @return java.awt.event.KeyListener
	 * @category keyboard
	 */
	public KeyListener _keyListener() {
		return new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				if (_keyboardEvent(e)) {
					e.consume();
				}
			}
		};
	}

	/**
	 * Process the key event.
	 *
	 * @param ev java.awt.event.KeyEvent
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean _keyboardEvent(KeyEvent ev) {
		if (ev.isConsumed()) {
			return false;
		}

		int code = ev.getKeyCode();
		switch (code) {
			case KeyEvent.VK_DOWN :
			case KeyEvent.VK_LEFT :
				this.value_(this._doubleValue() - this.stepValue());
				return true;
			case KeyEvent.VK_UP :
			case KeyEvent.VK_RIGHT :
				this.value_(this._doubleValue() + this.stepValue());
				return true;
		}

		return false;
	}

	/**
	 * Print my string representation on a writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		super.printOn_(aWriter);
		aWriter.write(" on: " + this.value());
	}

	/**
	 * Answer my default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunGaugeViewAwt._OnBorderedPanel(this);
		} else {
			return JunGaugeViewSwing._OnBorderedPanel(this);
		}
	}

	/**
	 * Answer the defualt value of gauge value.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultGaugeValue() {
		return 0;
	}

	/**
	 * Answer the defualt value of gauge range.
	 * 
	 * @return double[]
	 * @category defaults
	 */
	protected double[] defaultGaugeRange() {
		return new double[] { this.defaultFromValue(), this.defaultToValue(), this.defaultDivideValue(), this.defaultRoundValue()};
	}

	/**
	 * Answer the defualt value of FROM value.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultFromValue() {
		return 0;
	}

	/**
	 * Answer the defualt value of TO value.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultToValue() {
		return 100;
	}

	/**
	 * Answer the defualt value of DIVIDE value.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultDivideValue() {
		return 10;
	}

	/**
	 * Answer the defualt value of ROUND value.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultRoundValue() {
		return 1;
	}

	/**
	 * Answer the default value string block.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category defaults
	 */
	protected StBlockClosure defaultValueStringBlock() {
		return new StBlockClosure() {
			public Object value_(Object anObject) {
				return JunGaugeModel.this.formatValueString_(((Number) anObject).doubleValue());
			}
		};
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Gauge");
	}

	/**
	 * Answer the formatted string for the value.
	 * 
	 * @param value double
	 * @return java.lang.String
	 * @category private
	 */
	protected String formatValueString_(double value) {
		String pattern = "0";
		if (this.roundValue() < 1) {
			pattern += '.';
			int scaleValue = (int) Math.round(Math.log(this.roundValue()) / Math.log(10) * -1);
			for (int i = 0; i < scaleValue; i++) {
				pattern += '0';
			}
		}

		DecimalFormat formatter = new DecimalFormat(pattern);
		return formatter.format(value);
	}

}
