package jp.co.sra.jun.goodies.gauge;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.StViewPanel;
import jp.co.sra.jun.system.framework.JunAbstractViewCanvas;

/**
 * JunGaugeViewAwt class
 * 
 *  @author    nisinaka
 *  @created   2004/04/13 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGaugeViewAwt.java,v 8.12 2008/02/20 06:31:34 nisinaka Exp $
 */
public class JunGaugeViewAwt extends JunAbstractViewCanvas implements JunGaugeView {

	protected Object[] scalesLayout;

	/**
	 * Answer an StView which contains the JunGaugeViewAwt on a bordered panel.
	 * 
	 * @param aGaugeModel jp.co.sra.jun.goodies.gauge.JunGaugeModel
	 * @return jp.co.sra.smalltalk.StView
	 * @category Instance creation
	 */
	public static StView _OnBorderedPanel(JunGaugeModel aGaugeModel) {
		return new StViewPanel(aGaugeModel) {

			protected JunGaugeViewAwt gaugeView;

			protected void buildComponent() {
				gaugeView = new JunGaugeViewAwt((JunGaugeModel) this.model());
				this.setLayout(new BorderLayout());
				this.add(gaugeView._onBorderedPanel(), BorderLayout.CENTER);
			}

			public void _setupKeyboard(Window aWindow) {
				gaugeView._setupKeyboard(aWindow);
			}
		};
	}

	/**
	 * Create a new instance of JunGaugeViewAwt and initialize it.
	 * 
	 * @param aGaugeModel jp.co.sra.jun.goodies.gauge.JunGaugeModel
	 * @category Instance creation
	 */
	public JunGaugeViewAwt(JunGaugeModel aGaugeModel) {
		super(aGaugeModel);
	}

	/**
	 * Answer my model as JunGaugeModel.
	 * 
	 * @return jp.co.sra.jun.goodies.gauge.JunGaugeModel
	 * @see jp.co.sra.jun.goodies.gauge.JunGaugeView#getGaugeModel()
	 * @category model accessing
	 */
	public JunGaugeModel getGaugeModel() {
		return (JunGaugeModel) this.model();
	}

	/**
	 * Answer my default controller
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewCanvas#defaultController()
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunGaugeController();
	}

	/**
	 * Build the component
	 * 
	 * @see jp.co.sra.smalltalk.StViewCanvas#buildComponent()
	 * @category user interface
	 */
	protected void buildComponent() {
		this.setSize(120, 180);
	}

	/**
	 * Answer the border rectangle of the gauge.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.gauge.JunGaugeView#gaugeBorder()
	 * @category accessing
	 */
	public StRectangle gaugeBorder() {
		StRectangle box = (new StRectangle(this.getBounds())).insetBy_(this.defaultMargin());
		int x = box.left() + this.gaugeScalesExtent().width + 13;
		return new StRectangle(x, box.top(), box.right(), box.bottom());
	}

	/**
	 * Answer the gague scales.
	 * 
	 * @return double[][]
	 * @category accessing
	 */
	protected double[][] gaugeScales() {
		return (double[][]) this.scalesLayout()[0];
	}

	/**
	 * Answer the extent of the gauge scales.
	 * 
	 * @return java.awt.Dimension
	 * @category accessing
	 */
	protected Dimension gaugeScalesExtent() {
		return (Dimension) this.scalesLayout()[1];
	}

	/**
	 * Answer the string representation of the value.
	 * 
	 * @param value double
	 * @return java.lang.String
	 * @category adaptor
	 */
	protected String valueString_(double value) {
		return (String) this.getGaugeModel().valueStringBlock().value_(new Double(value));
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		if (this.isShowing() == false) {
			return;
		}

		this.displayBorderOn_(aGraphics);
		this.displayScalesOn_(aGraphics);
		this.displayValueOn_(aGraphics);
	}

	/**
	 * Display the border on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayBorderOn_(Graphics aGraphics) {
		StRectangle border = this.gaugeBorder();
		aGraphics.setColor(StColorValue.Brightness_(0.75));
		aGraphics.drawLine(border.left(), border.bottom(), border.right(), border.bottom());
		aGraphics.setColor(StColorValue.Brightness_(0.75));
		aGraphics.drawLine(border.right(), border.top(), border.right(), border.bottom());
		aGraphics.setColor(StColorValue.Brightness_(0.25));
		aGraphics.drawLine(border.left(), border.top(), border.right(), border.top());
		aGraphics.setColor(StColorValue.Brightness_(0.25));
		aGraphics.drawLine(border.left(), border.top(), border.left(), border.bottom());
	}

	/**
	 * Display the scales on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayScalesOn_(Graphics aGraphics) {
		FontMetrics fontMetrics = aGraphics.getFontMetrics();
		aGraphics.setColor(this.getForeground());
		StRectangle box = (new StRectangle(this.getBounds())).insetBy_(this.defaultMargin());
		Dimension extent = this.gaugeScalesExtent();
		double[][] scales = this.gaugeScales();
		for (int i = 0; i < scales.length; i++) {
			String label = this.valueString_(scales[i][0]);
			double location = scales[i][1];

			int x = box.left() + extent.width - fontMetrics.stringWidth(label);
			int y = box.top() + (int) (box.height() * (1 - location)) + fontMetrics.getDescent();
			aGraphics.drawString(label, x, y);

			x = box.left() + extent.width;
			y = box.top() + (int) (box.height() * (1 - location));
			aGraphics.drawLine(x + 3, y, x + 7, y);
		}
	}

	/**
	 * Display the value on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayValueOn_(Graphics aGraphics) {
		double value = this.getGaugeModel().normalizedValue();
		StRectangle border = this.gaugeBorder().insetBy_(new StRectangle(1, 1, 0, 0));
		int deltaY = (int) (border.height() * (1 - value));
		aGraphics.setColor(SelectionForegroundColor);
		aGraphics.fillRect(border.originX(), border.originY(), border.width(), deltaY);
		aGraphics.setColor(SelectionBackgroundColor);
		aGraphics.fillRect(border.originX(), border.originY() + deltaY, border.width(), border.height() - deltaY);
	}

	/**
	 * Answer the default margin.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category defaults
	 */
	protected StRectangle defaultMargin() {
		return new StRectangle(10, 10, 10, 10);
	}

	/**
	 * Calculate the layout information of the scales.
	 * 
	 * @return java.lang.Object[]
	 * @category private
	 */
	protected Object[] scalesLayout() {
		if (scalesLayout == null) {
			Graphics aGraphics = null;
			try {
				aGraphics = this.getGraphics();
				FontMetrics aFontMetrics = aGraphics.getFontMetrics();

				JunGaugeModel gaugeModel = this.getGaugeModel();
				double increment = (gaugeModel.toValue() - gaugeModel.fromValue()) / gaugeModel.divideValue();
				ArrayList labels = new ArrayList();
				labels.add(new double[] { gaugeModel.fromValue(), 0 });
				int maxWidth = aFontMetrics.stringWidth(this.valueString_(gaugeModel.fromValue()));
				double value = gaugeModel.fromValue();
				double location = 0;
				for (int i = 1; i < gaugeModel.divideValue(); i++) {
					value += increment;
					location += (1 / gaugeModel.divideValue());
					labels.add(new double[] { value, location });
					maxWidth = Math.max(maxWidth, aFontMetrics.stringWidth(this.valueString_(gaugeModel._doubleValue())));
				}
				labels.add(new double[] { gaugeModel.toValue(), 1 });
				maxWidth = Math.max(maxWidth, aFontMetrics.stringWidth(this.valueString_(gaugeModel.toValue())));

				scalesLayout = new Object[] { labels.toArray(new double[labels.size()][]), new Dimension(maxWidth, aFontMetrics.getHeight())};

			} finally {
				if (aGraphics != null) {
					aGraphics.dispose();
					aGraphics = null;
				}
			}

		}
		return scalesLayout;
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.  The argument anAspectSymbol is typically a Symbol that
	 * indicates what change has occurred.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (this.isShowing() == false) {
			return;
		}

		if (evt.getAspect() == $("value")) {
			Graphics aGraphics = null;
			try {
				aGraphics = this.getGraphics();
				this.displayValueOn_(aGraphics);
			} finally {
				if (aGraphics != null) {
					aGraphics.dispose();
					aGraphics = null;
				}
			}
		} else if (evt.getAspect() == $("range")) {
			this.flushScalesLayout();
			this.repaint();
		} else {
			super.update_(evt);
		}
	}

	/**
	 * Flush the scales layout. 
	 * 
	 * @category flushing
	 */
	protected void flushScalesLayout() {
		scalesLayout = null;
	}

	/**
	 * Set up the keyboard for the view on the window.
	 * 
	 * @param aWindow java.awt.Window
	 * @see jp.co.sra.smalltalk.StView#_setupKeyboard(java.awt.Window)
	 * @category keyboard
	 */
	public void _setupKeyboard(final Window aWindow) {
		this.addKeyListener(this.getGaugeModel()._keyListener());

		this.addMouseListener(new MouseAdapter() {
			public void mouseEntered(MouseEvent e) {
				Component focusOwner = aWindow.getFocusOwner();
				if (focusOwner != null && focusOwner != JunGaugeViewAwt.this) {
					JunGaugeViewAwt.this.requestFocus();
				}
			}
		});
	}

}
