package jp.co.sra.jun.goodies.image.streams;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.SystemResourceSupport;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunImageStream class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1999/01/20 (by Hirotsugu Kondo)
 *  @updated   2000/11/12 (by Mitsuhiro Asada)
 *  @updated   2001/04/12 (by nisinaka)
 *  @updated   2005/08/15 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageStream.java,v 8.14 2008/02/20 06:31:35 nisinaka Exp $
 */
public abstract class JunImageStream extends JunAbstractObject {

	protected InputStream inStream;
	protected OutputStream outStream;
	protected StImage imageObject;
	protected StValueHolder progressValue = new StValueHolder(0.0f);

	protected static HashMap ImageKindTable;

	static {
		FlushImageKindTable();
	}

	/**
	 * Flush the image kind table.
	 * 
	 * @category Class initialization
	 */
	protected static void FlushImageKindTable() {
		ImageKindTable = null;
	}

	/**
	 * Answer the image kind table.
	 * 
	 * @return java.util.Map
	 * @category Accessing
	 */
	protected static Map ImageKindTable() {
		if (ImageKindTable == null) {
			HashMap aMap = new HashMap();
			aMap.put("bmp", "jp.co.sra.jun.goodies.image.streams.JunBmpImageStream");
			aMap.put("jpg", "jp.co.sra.jun.goodies.image.streams.JunJpegImageStream");
			aMap.put("jpeg", "jp.co.sra.jun.goodies.image.streams.JunJpegImageStream");
			aMap.put("png", "jp.co.sra.jun.goodies.image.streams.JunPngImageStream");
			aMap.put("gif", "jp.co.sra.jun.goodies.image.streams.JunGifImageStream");
			aMap.put("giff", "jp.co.sra.jun.goodies.image.streams.JunGifImageStream");
			ImageKindTable = aMap;
		}
		return ImageKindTable;
	}

	/**
	 * Answer an image stream class for the file.
	 * 
	 * @param aFile java.io.File
	 * @return java.lang.Class
	 * @category Accessing
	 */
	public static Class ImageStreamClassForFileName_(File aFile) {
		return ImageStreamClassForFileName_(aFile.toString());
	}

	/**
	 * Answer an image stream class for the filename.
	 * 
	 * @param aFilename java.lang.String
	 * @return java.lang.Class
	 * @category Accessing
	 */
	public static Class ImageStreamClassForFileName_(String aFilename) {
		String aString = aFilename.substring(aFilename.lastIndexOf('.') + 1).toLowerCase();
		String aSymbol = (String) ImageKindTable().get(aString);
		if (aSymbol == null) {
			return null;
		}

		Class aClass = null;
		try {
			aClass = Class.forName(aSymbol);
		} catch (ClassNotFoundException e) {
			return null;
		}
		return aClass;
	}

	/**
	 * Answer an image of SmalltalkBalloon 128x128.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Accessing
	 */
	public static StImage ImageSmalltalkBalloon128x128() {
		return new StImage(SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/image/streams/imageSmalltalkBalloon128x128.jpg"));
	}

	/**
	 * Answer an image of SmalltalkBalloon 256x256.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Accessing
	 */
	public static StImage ImageSmalltalkBalloon256x256() {
		return new StImage(SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/image/streams/imageSmalltalkBalloon256x256.jpg"));
	}

	/**
	 * Constructor with an input stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.InputStream
	 * @exception java.io.IOException
	 * @category Instance creation
	 */
	public static JunImageStream On_(InputStream stream) throws IOException {
		throw new IOException("subclassResponsibilitty");
	}

	/**
	 * Constructor with an output stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.OutputStream
	 * @exception java.io.IOException
	 * @category Instance creation
	 */
	public static JunImageStream On_(OutputStream stream) throws IOException {
		throw new IOException("subclassResponsibilitty");
	}

	/**
	 * Constructor with an image stream and an input stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param imageStream jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.InputStream
	 * @exception java.io.IOException
	 * @category Instance creation
	 */
	protected static JunImageStream On_(JunImageStream imageStream, InputStream stream) throws IOException {
		if (imageStream == null) {
			throw new IOException("could not load image");
		}
		imageStream.on_(stream);
		return imageStream;
	}

	/**
	 * Constructor with an image stream and an output stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param imageStream jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.OutputStream
	 * @exception java.io.IOException
	 * @category Instance creation
	 */
	protected static JunImageStream On_(JunImageStream imageStream, OutputStream stream) throws IOException {
		if (imageStream == null) {
			throw new IOException("could not save image");
		}
		imageStream.on_(stream);
		return imageStream;
	}

	/**
	 * This image stream close.
	 * 
	 * @exception java.io.IOException
	 * @category stream access
	 */
	public void close() throws IOException {
		if (inStream != null) {
			inStream.close();
			inStream = null;
		}

		if (outStream != null) {
			outStream.close();
			outStream = null;
		}
	}

	/**
	 * This image stream flush.
	 * 
	 * @exception java.io.IOException
	 * @category stream access
	 */
	public void flush() throws IOException {
		if (outStream != null) {
			outStream.flush();
		}
	}

	/**
	 * Write the byte on the output stream.
	 * 
	 * @param b byte
	 * @throws java.io.IOException
	 * @category stream access
	 */
	protected void nextPut_(int b) throws IOException {
		outStream.write(b);
	}

	/**
	 * Write the byte array on the output stream.
	 * 
	 * @param bytes byte[]
	 * @throws java.io.IOException
	 * @category stream access
	 */
	protected void nextPutAll_(byte[] bytes) throws IOException {
		outStream.write(bytes);
	}

	/**
	 * Read the image from the input stream.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @category accessing
	 */
	public abstract StImage nextImage() throws IOException;

	/**
	 * Write the image on output stream.
	 * 
	 * @param newImage jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @category accessing
	 */
	public abstract void nextPutImage_(StImage newImage) throws IOException;

	/**
	 * Answer the receiver's image file extension.
	 *
	 * @return java.lang.String
	 * @category defaults
	 */
	public abstract String defaultImageExtension();

	/**
	 * Set the input stream.
	 * 
	 * @param stream java.io.InputStream
	 * @exception java.io.IOException
	 * @category initialize-release
	 */
	protected void on_(InputStream stream) throws IOException {
		inStream = stream;
	}

	/**
	 * Set the output stream.
	 * 
	 * @param stream java.io.OutputStream
	 * @exception java.io.IOException
	 * @category initialize-release
	 */
	protected void on_(OutputStream stream) throws IOException {
		outStream = stream;
	}

	/**
	 * Do something with the progress value.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category progress
	 */
	public void compute_(StBlockClosure aBlock) {
		this.progressValue().compute_(aBlock);
	}

	/**
	 * Set the progress value.
	 * 
	 * @param normalizedNumber float
	 * @category progress
	 */
	public void progress_(float normalizedNumber) {
		if ((0 <= normalizedNumber) && (normalizedNumber <= 1)) {
			float truncatedValue = Math.round(normalizedNumber / 0.005) * 0.005f;

			if (this.progressValue()._floatValue() != truncatedValue) {
				this.progressValue().value_(truncatedValue);
			}
		}
	}

	/**
	 * Answer the progress value.
	 * 
	 * @return float
	 * @category progress
	 */
	public void progress() {
		this.progressValue()._floatValue();
	}

	/**
	 * Set the StValueHolder for the progress value.
	 * 
	 * @param aValueHolder jp.co.sra.smalltalk.StValueHolder
	 * @category progress
	 */
	protected void progressValue_(StValueHolder aValueHolder) {
		progressValue = aValueHolder;
	}

	/**
	 * Answer the StValueHolder for the progress value.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category progress
	 */
	protected StValueHolder progressValue() {
		return progressValue;
	}

}
