package jp.co.sra.jun.goodies.image.streams;

import java.awt.image.IndexColorModel;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.HashSet;

import jp.co.sra.smalltalk.StImage;

import jp.co.sra.jun.goodies.image.support.JunImageProcessor;

/**
 * JunSraGifImageStream class
 * 
 *  @author    nisinaka
 *  @created   2007/10/04 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSraGifImageStream.java,v 8.5 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunSraGifImageStream extends JunGifImageStream {

	protected static final byte ImageSeparator = ','; // 0x2c
	protected static final byte Extension = '!'; // 0x21
	protected static final byte Terminator = ';'; // 0x3b
	protected static final byte GraphicControlLabel = (byte) 0xf9;
	protected static final byte ApplicationExtensionLabel = (byte) 0xff;
	protected static final byte CommentExtensionLabel = (byte) 0xfe;
	protected static final byte PlainTextExtensionLabel = (byte) 0x01;

	protected int width;
	protected int height;
	protected int bitsPerPixel;
	protected IndexColorModel colorModel;
	protected int rowByteSize;
	protected int xpos;
	protected int ypos;
	protected int pass;
	protected boolean interlace;
	protected int codeSize;
	protected int clearCode;
	protected int eoiCode;
	protected int freeCode;
	protected int maxCode;
	protected int remainBitCount;
	protected int bufByte;
	protected ByteBuffer bufStream;
	protected int transparentColorIndex = -1;
	protected StImage image8;

	/**
	 * Create a new instance of JunGifAnimationStream with the specified input stream.
	 * 
	 * @param stream java.io.InputStream
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @throws java.io.IOException
	 * @category Instance creation
	 */
	public static JunImageStream On_(InputStream stream) throws IOException {
		return On_(new JunSraGifImageStream(), stream);
	}

	/**
	 * Create a new instance of JunGifAnimationStream with the specified output stream.
	 * 
	 * @param stream java.io.InputStream
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @throws java.io.IOException
	 * @category Instance creation
	 */
	public static JunImageStream On_(OutputStream stream) throws IOException {
		return On_(new JunSraGifImageStream(), stream);
	}

	/**
	 * A utility method for the bit shift.
	 * 
	 * @param integer int
	 * @param shift int
	 * @return int
	 * @category Utilities
	 */
	protected static int BitShift(int integer, int shift) {
		if (shift > 0) {
			return integer << shift;
		} else if (shift < 0) {
			return integer >> -shift;
		} else {
			return integer;
		}
	}

	/**
	 * Write the image on the output stream.
	 *
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunGifImageStream#nextPutImage_(jp.co.sra.smalltalk.StImage)
	 * @category accessing
	 */
	public void nextPutImage_(StImage anImage) throws IOException {
		try {
			byte[] bits = this.bitsFor_(anImage);

			this.writeHeader();
			this.writeBlocks(bits);

		} finally {
			this.close();
		}
	}

	/**
	 * Write the header of the image.
	 * 
	 * @throws java.io.IOException 
	 * @category encoding
	 */
	protected void writeHeader() throws IOException {
		this.nextPutAll_("GIF89a".getBytes());
		this.writeWord_(width);
		this.writeWord_(height);

		int b = 128;
		b |= (bitsPerPixel - 1) << 4;
		b |= (bitsPerPixel - 1);
		this.nextPut_(b);

		// Background Color Index
		if (transparentColorIndex >= 0) {
			this.nextPut_(transparentColorIndex);
		} else {
			this.nextPut_(0);
		}

		// Pixel Aspect Ratio
		this.nextPut_(0);

		// Global Color Table
		int size = 1 << bitsPerPixel;
		int[] array = new int[colorModel.getMapSize()];
		colorModel.getRGBs(array);
		for (int i = 0; i < size; i++) {
			if (i < array.length) {
				int rgb = array[i];
				this.nextPut_((rgb >> 16) & 0xFF);
				this.nextPut_((rgb >> 8) & 0xFF);
				this.nextPut_(rgb & 0xFF);
			} else {
				this.nextPut_(0);
				this.nextPut_(0);
				this.nextPut_(0);
			}
		}
	}

	/**
	 * Write the image block of the image.
	 * 
	 * @param bits byte[]
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void writeBlocks(byte[] bits) throws IOException {
		if (transparentColorIndex >= 0) {
			this.writeGraphicControlExtensionBlock();
		}

		// Image Block
		this.nextPut_(ImageSeparator);
		this.writeWord_(0); // Image Left Position
		this.writeWord_(0); // Image Top Position
		this.writeWord_(width); // Image Width
		this.writeWord_(height); // Image Height
		this.nextPut_(interlace ? 64 : 0);

		this.writeBitData_(bits);

		this.nextPut_(Terminator);
	}

	/**
	 * Write the graphic control extension block of the image.
	 * 
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void writeGraphicControlExtensionBlock() throws IOException {
		this.nextPut_(Extension);
		this.nextPut_(GraphicControlLabel);
		this.nextPut_(4);
		if (transparentColorIndex >= 0) {
			this.nextPut_(1);
		} else {
			this.nextPut_(0);
		}
		this.nextPut_(0); // Delay Time
		this.nextPut_(0); // Delay Time
		if (transparentColorIndex >= 0) {
			this.nextPut_(transparentColorIndex);
		} else {
			this.nextPut_(0);
		}
		this.nextPut_(0); // Block Terminator
	}

	/**
	 * Write the bit data.
	 * 
	 * @param bits byte[]
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void writeBitData_(byte[] bits) throws IOException {
		pass = 0;
		xpos = 0;
		ypos = 0;
		rowByteSize = (width * 8 + 31) / 32 * 4;
		remainBitCount = 0;
		bufByte = 0;
		bufStream = ByteBuffer.allocate(256);
		int maxBits = 12;
		int maxMaxCode = 1 << maxBits;
		int tSize = 5003;
		int[] prefixTable = new int[tSize];
		int[] suffixTable = new int[tSize];
		int initCodeSize = (bitsPerPixel <= 1) ? 2 : bitsPerPixel;

		this.nextPut_(initCodeSize);

		this.setParameters_(initCodeSize);
		int tShift = 0;
		int fCode = tSize;
		while (fCode < 65536) {
			tShift++;
			fCode *= 2;
		}
		tShift = 8 - tShift;
		for (int i = 0; i < tSize; i++) {
			suffixTable[i] = -1;
		}

		HashSet set = new HashSet(height);
		this.progress_((float) set.size() / height);
		set.add(new Integer(ypos));
		this.progress_((float) set.size() / height);

		this.writeCodeAndCheckCodeSize_(clearCode);
		int ent = this.readPixelFrom_(bits);
		int pixel;
		while ((pixel = this.readPixelFrom_(bits)) >= 0) {
			set.add(new Integer(ypos));
			this.progress_((float) set.size() / height);

			fCode = (pixel << maxBits) + ent;
			int index = (pixel << tShift) ^ ent;
			if (suffixTable[index] == fCode) {
				ent = prefixTable[index];
			} else {
				boolean nomatch = true;
				if (suffixTable[index] > 0) {
					int disp = tSize - index;
					if (index == 0) {
						disp = 1;
					}

					do {
						index -= disp;
						if (index < 0) {
							index += tSize;
						}
						if (suffixTable[index] == fCode) {
							ent = prefixTable[index];
							nomatch = false;
						}
					} while (nomatch && suffixTable[index] > 0);
				}
				if (nomatch) {
					this.writeCodeAndCheckCodeSize_(ent);
					ent = pixel;
					if (freeCode < maxMaxCode) {
						prefixTable[index] = freeCode;
						suffixTable[index] = fCode;
						freeCode += 1;
					} else {
						this.writeCodeAndCheckCodeSize_(clearCode);
						for (int i = 0; i < tSize; i++) {
							suffixTable[i] = -1;
						}
						this.setParameters_(initCodeSize);
					}
				}
			}

			set.add(new Integer(ypos));
			this.progress_((float) set.size() / height);
		}

		prefixTable = null;
		suffixTable = null;
		this.writeCodeAndCheckCodeSize_(ent);
		this.writeCodeAndCheckCodeSize_(eoiCode);
		this.flushCode();
		this.nextPut_(0);
	}

	/**
	 * Write the word.
	 * 
	 * @param word int
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void writeWord_(int word) throws IOException {
		this.nextPut_(word & 0xFF);
		this.nextPut_((word >> 8) & 0xFF);
	}

	/**
	 * Write the code.
	 * 
	 * @param code int
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void writeCode_(int code) throws IOException {
		this.nextBitsPut_(code);
	}

	/**
	 * Write the code and check the code size.
	 * 
	 * @param code int
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void writeCodeAndCheckCodeSize_(int code) throws IOException {
		this.writeCode_(code);
		this.checkCodeSize();
	}

	/**
	 * Read the pixel from the bits.
	 * 
	 * @param bits byte[]
	 * @return int
	 * @category encoding
	 */
	protected int readPixelFrom_(byte[] bits) {
		if (ypos >= height) {
			return -1;
		}

		int pixel = (int) (bits[ypos * rowByteSize + xpos] & 0xFF);
		this.updatePixelPosition();
		return pixel;
	}

	/**
	 * Flush the code.
	 * 
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void flushCode() throws IOException {
		this.flushBits();
	}

	/**
	 * Write the integer as bits on the output stream. 
	 * 
	 * @param anInteger int
	 * @throws java.io.IOException
	 * @category bits access
	 */
	protected void nextBitsPut_(int anInteger) throws IOException {
		int shiftCount = 0;
		int writeBitCount;
		int integer;
		if (remainBitCount == 0) {
			writeBitCount = 8;
			integer = anInteger;
		} else {
			writeBitCount = remainBitCount;
			integer = bufByte + (anInteger << (8 - remainBitCount));
		}
		while (writeBitCount < codeSize) {
			this.nextBytePut_((byte) (BitShift(integer, shiftCount) & 0xFF));
			shiftCount -= 8;
			writeBitCount += 8;
		}
		remainBitCount = writeBitCount - codeSize;
		if (remainBitCount == 0) {
			this.nextBytePut_((byte) (BitShift(integer, shiftCount) & 0xFF));
		} else {
			bufByte = (byte) (BitShift(integer, shiftCount) & 0xFF);
		}
	}

	/**
	 * Flush the bits.
	 * 
	 * @throws java.io.IOException
	 * @category bits access
	 */
	protected void flushBits() throws IOException {
		if (remainBitCount != 0) {
			this.nextBytePut_((byte) (bufByte & 0xFF));
			remainBitCount = 0;
		}
		this.flushBuffer();
	}

	/**
	 * Put the byte on the buffer stream.
	 * 
	 * @param b byte
	 * @throws java.io.IOException
	 * @category packing
	 */
	protected void nextBytePut_(byte b) throws IOException {
		bufStream.put(b);
		if (bufStream.position() >= 254) {
			this.flushBuffer();
		}
	}

	/**
	 * Flush the buffer stream.
	 * 
	 * @throws java.io.IOException
	 * @category packing
	 */
	protected void flushBuffer() throws IOException {
		if (bufStream.position() == 0) {
			return;
		}

		int size = bufStream.position();
		byte[] bytes = new byte[size];
		bufStream.position(0);
		bufStream.get(bytes);

		this.nextPut_(size);
		this.nextPutAll_(bytes);

		bufStream = ByteBuffer.allocate(256);
	}

	/**
	 * Calculate an byte array for the specified image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return byte[]
	 * @category private
	 */
	protected byte[] bitsFor_(StImage anImage) {
		if (anImage.bitsPerPixel() > 8 || anImage.colorModel() instanceof IndexColorModel == false) {
			image8 = anImage._convertToPalette_RenderedByErrorDiffusion(JunImageProcessor.ColorPalette256());
		} else {
			image8 = anImage;
		}

		width = image8.width();
		height = image8.height();
		colorModel = (IndexColorModel) image8.colorModel();
		if (transparentColorIndex < 0) {
			transparentColorIndex = colorModel.getTransparentPixel();
		}

		bitsPerPixel = image8.bitsPerPixel();
		/*
		trueBitsPerPixel := image8 bitsPerPixel.
		achievableBitsPerPixel := 2 
				raisedTo: ((colorPalette size - 1) highBit - 1) highBit.
		bitsPerPixel := achievableBitsPerPixel.
		colorPalette := image8 palette.
		colorPalette paintBasis == CoverageValue 
			ifTrue: 
				[transparentPixel := 0.
				colorPalette := self convertToMappedPalette: colorPalette].
		 */

		byte[] bits = image8.getBits();

		/*
		trueBitsPerPixel < 8 
			ifTrue: 
				[bits := self 
							unpackBits: bits
							depthTo8From: bitsPerPixel
							width: image8 width
							height: image8 height
							pad: 32].
		 */

		interlace = false;
		return bits;
	}

	/**
	 * Set the parameters.
	 * 
	 * @param initCodeSize int
	 * @category private
	 */
	protected void setParameters_(int initCodeSize) {
		clearCode = 1 << initCodeSize;
		eoiCode = clearCode + 1;
		freeCode = clearCode + 2;
		codeSize = initCodeSize + 1;
		maxCode = (1 << codeSize) - 1;
	}

	/**
	 * Check the code size.
	 * 
	 * @category private
	 */
	protected void checkCodeSize() {
		if (freeCode > maxCode && codeSize < 12) {
			codeSize += 1;
			maxCode = (1 << codeSize) - 1;
		}
	}

	/**
	 * Update the pixel position of the bits.
	 * 
	 * @category private
	 */
	protected void updatePixelPosition() {
		xpos += 1;
		if (xpos < width) {
			return;
		}

		xpos = 0;
		if (!interlace) {
			ypos += 1;
			return;
		}

		switch (pass) {
			case 0:
				ypos += 8;
				if (ypos >= height) {
					pass += 1;
					ypos = 4;
				}
				break;
			case 1:
				ypos += 8;
				if (ypos >= height) {
					pass += 1;
					ypos = 2;
				}
				break;
			case 2:
				ypos += 4;
				if (ypos >= height) {
					pass += 1;
					ypos = 1;
				}
				break;
			case 3:
				ypos += 2;
				break;
			default:
				this.error_("can't happen");
		}
	}

}
