package jp.co.sra.jun.goodies.image.support;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StInterval;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.goodies.image.framework.JunImageDisplayModel;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunImageAdjuster class
 * 
 *  @author    nisinaka
 *  @created   2001/04/05 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageAdjuster.java,v 8.12 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunImageAdjuster extends JunAbstractObject {

	/**
	 * Answer an image adjusted to the specified dimension.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aDimension java.awt.Dimension
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Adjusting
	 */
	public static StImage Adjust_extent_(StImage anImage, Dimension aDimension) {
		return (new JunImageAdjuster()).adjust_extent_(anImage, aDimension);
	}

	/**
	 * Create an image adjusted to the specified dimension.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aDimension java.awt.Dimension
	 * @param keepAspect boolean
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Adjusting
	 * @deprecated since Jun637, use AdjustImage_extent_keepAspect_()
	 */
	public static StImage Adjust_extent_keepAspect_(StImage anImage, Dimension aDimension, boolean keepAspect) {
		return AdjustImage_extent_keepAspect_(anImage, aDimension, keepAspect);
	}

	/**
	 * Create an image adjusted to the specified dimension.
	 * 
	 * @param originalImage jp.co.sra.smalltalk.StImage
	 * @param aDimension java.awt.Dimension
	 * @param keepAspect boolean
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Adjusting
	 */
	public static StImage AdjustImage_extent_keepAspect_(StImage originalImage, Dimension aDimension, boolean keepAspect) {
		Dimension newImageExtent = aDimension;
		if (keepAspect) {
			int width = originalImage.width();
			int height = originalImage.height();
			if (width != aDimension.width) {
				height = height * aDimension.width / width;
				width = aDimension.width;
			}
			if (height > aDimension.height) {
				width = width * aDimension.height / height;
				height = aDimension.height;
			}
			newImageExtent = new Dimension(width, height);
		}
		return Adjust_extent_(originalImage, newImageExtent);
	}

	/**
	 * Create an image adjusted to the specified dimension.
	 * 
	 * @param originalImage jp.co.sra.smalltalk.StImage
	 * @param aDimension java.awt.Dimension
	 * @param keepAspect boolean
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Adjusting
	 */
	public static StImage AdjustImage_extent_keepAspect_alignmentSymbol_(StImage originalImage, Dimension aDimension, boolean keepAspect, StSymbol aSymbol) {
		StImage sourceImage = AdjustImage_extent_keepAspect_(originalImage, aDimension, keepAspect);
		if (aSymbol == null) {
			return sourceImage;
		}
		if (sourceImage.width() == aDimension.width && sourceImage.height() == aDimension.height) {
			return sourceImage;
		}

		StImage destinationImage = new StImage(aDimension);
		JunImageProcessor.Fill_color_(destinationImage, Color.white);
		StRectangle aRectangle = new StRectangle(sourceImage.bounds());
		aRectangle = aRectangle.align_with_(aRectangle._pointAt(aSymbol), new StRectangle(destinationImage.bounds())._pointAt(aSymbol));
		destinationImage.copy_from_in_rule_(aRectangle.toRectangle(), new Point(0, 0), sourceImage, StImage.Over);
		return destinationImage;
	}

	/**
	 * Show the image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @category Viewing
	 */
	public static void Show_(StImage anImage) {
		JunImageDisplayModel.Show_(anImage);
	}

	////////////////////////////////////////////////////////////////////////////////

	protected StValueHolder progressValue;

	/**
	 * Create a new instance of JunImageAdjuster and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunImageAdjuster() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.progressValue_(new StValueHolder(0.0f));
	}

	/**
	 * Answer an image adjusted to the specified dimension.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aDimension java.awt.Dimension
	 * @return jp.co.sra.smalltalk.StImage
	 * @category adjusting
	 */
	public StImage adjust_extent_(StImage anImage, Dimension aDimension) {
		if (anImage.width() == aDimension.width && anImage.height() == aDimension.height) {
			return anImage;
		}

		StImage theImage = null;
		try {
			this.progress_(0.0f);

			Image image = anImage.image();
			image = anImage.image().getScaledInstance(aDimension.width, aDimension.height, Image.SCALE_SMOOTH);
			theImage = new StImage(image);

			if (theImage == null) {
				theImage = this.computeImage_magnifyExtent_(anImage, aDimension);
			}

		} finally {
			this.progress_(1.0f);
		}

		return theImage;
	}

	/**
	 * Do something with the progress value.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category progress
	 */
	public void compute_(StBlockClosure aBlock) {
		this.progressValue().compute_(aBlock);
	}

	/**
	 * Answer my current progress value.
	 * 
	 * @return float
	 * @category progress
	 */
	public float progress() {
		return this.progressValue()._floatValue();
	}

	/**
	 * Set my new progress value.
	 * 
	 * @param normalizedNumber float
	 * @category progress
	 */
	public void progress_(float normalizedNumber) {
		if ((0 <= normalizedNumber) && (normalizedNumber <= 1)) {
			float truncatedValue = Math.round(normalizedNumber / 0.005) * 0.005f;

			if (this.progressValue()._floatValue() != truncatedValue) {
				this.progressValue().value_(truncatedValue);
			}
		}
	}

	/**
	 * Answer my current StValueHolder for the progress value.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category progress
	 */
	protected StValueHolder progressValue() {
		return progressValue;
	}

	/**
	 * Set my new StValueHolder for the progress value.
	 * 
	 * @param aValueHolder jp.co.sra.smalltalk.StValueHolder
	 * @category progress
	 */
	protected void progressValue_(StValueHolder aValueHolder) {
		progressValue = aValueHolder;
	}

	/**
	 * Compute a double size image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return jp.co.sra.smalltalk.StImage
	 * @category private-magnifying
	 */
	protected StImage computeDoubleSizeImage_(StImage anImage) {
		int width = anImage.width();
		int height = anImage.height();
		StImage magnifiedImage = new StImage(width * 2, height * 2);
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				int colorValue = anImage.getPixel(x, y);
				magnifiedImage.setPixel(x * 2, y * 2, colorValue);
				magnifiedImage.setPixel((x * 2) + 1, y * 2, colorValue);
				magnifiedImage.setPixel(x * 2, (y * 2) + 1, colorValue);
				magnifiedImage.setPixel((x * 2) + 1, (y * 2) + 1, colorValue);
			}
		}
		return magnifiedImage;
	}

	/**
	 * Compute an image maginfied to the specified dimension.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aDimension java.awt.Dimension
	 * @return jp.co.sra.smalltalk.StImage
	 * @category private-magnifying
	 */
	protected StImage computeImage_magnifyExtent_(StImage anImage, Dimension aDimension) {
		StImage theImage = anImage;
		while ((theImage.width() < aDimension.width) || (theImage.height() < aDimension.height)) {
			theImage = this.computeDoubleSizeImage_(theImage);
		}
		StImage magnifiedImage = this.computeImage_shrinkExtent_(theImage, aDimension);
		return magnifiedImage;
	}

	/**
	 * Compute an image shrunken to the specified dimension.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aDimension java.awt.Dimension
	 * @return jp.co.sra.smalltalk.StImage
	 * @category private-shrinking
	 */
	protected StImage computeImage_shrinkExtent_(StImage anImage, Dimension aDimension) {
		int width = anImage.width();
		int height = anImage.height();
		StInterval[] xIntervals = this.computeIntervals_divisionSize_(width, aDimension.width);
		StInterval[] yIntervals = this.computeIntervals_divisionSize_(height, aDimension.height);
		StImage shrunkenImage = new StImage(aDimension.width, aDimension.height);
		int yValue = 0;
		for (int yi = 0; yi < yIntervals.length; yi++) {
			int xValue = 0;
			for (int xi = 0; xi < xIntervals.length; xi++) {
				int redValue = 0;
				int greenValue = 0;
				int blueValue = 0;
				int count = 0;
				for (int y = (int) yIntervals[yi].start(); y <= (int) yIntervals[yi].stop(); y++) {
					for (int x = (int) xIntervals[xi].start(); x <= (int) xIntervals[xi].stop(); x++) {
						Color color = anImage.valueAtPoint_(new Point(x, y));
						redValue += color.getRed();
						greenValue += color.getGreen();
						blueValue += color.getBlue();
						count++;
					}
				}

				Color color = null;
				if (count > 0) {
					int r = Math.max(0, Math.min(255, redValue / count));
					int g = Math.max(0, Math.min(255, greenValue / count));
					int b = Math.max(0, Math.min(255, blueValue / count));
					color = new Color(r, g, b);
				} else {
					color = new Color(0, 0, 0);
				}

				shrunkenImage.valueAtPoint_put_(new Point(xValue, yValue), color);
				xValue++;
			}

			yValue++;
			this.progress_((float) yValue / yIntervals.length);
		}

		return shrunkenImage;
	}

	/**
	 * Compute appropriate intervals for the division size.
	 * 
	 * @param anInteger int
	 * @param divisionSize int
	 * @return jp.co.sra.smalltalk.StInterval[]
	 * @category private-shrinking
	 */
	public StInterval[] computeIntervals_divisionSize_(int anInteger, int divisionSize) {
		if (anInteger < divisionSize) {
			throw new SmalltalkException(anInteger + " is smaller than " + divisionSize);
		}

		float value = 0;
		float stepValue = anInteger / (float) divisionSize;
		int[] values = new int[divisionSize + 1];
		for (int i = 0; i <= divisionSize; i++, value += stepValue) {
			values[i] = Math.round(value);
		}
		StInterval[] intervals = new StInterval[divisionSize];
		for (int i = 0; i < divisionSize; i++) {
			int start = values[i];
			int stop = values[i + 1] - 1;
			intervals[i] = new StInterval(start, stop);
		}
		return intervals;
	}

}
