package jp.co.sra.jun.goodies.image.support;

import java.awt.BorderLayout;
import java.awt.Canvas;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.SystemResourceSupport;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.collections.support.JunCorrelation;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.image.streams.JunJpegImageStream;
import jp.co.sra.jun.goodies.progress.JunProgress;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;

/**
 * JunImageToAscii class
 * 
 *  @author    nisinaka
 *  @created   2000/12/13 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2006/12/37 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun668 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunImageToAscii.java,v 8.13 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunImageToAscii extends JunApplicationModel {

	protected Dimension divisionPoint;
	protected Font font;
	protected int asciiLength;
	protected StValueHolder progressValue;
	protected String baseName;
	protected StValueHolder textModel;
	protected JunProgress _progress;
	protected StMenuBar _menuBar;

	/**
	 * Show the about dialog.
	 * 
	 * @category About
	 */
	protected static void About() {
		final int offsetX = 24;
		final int offsetY = 24;
		int maxX = 0;
		int maxY = 0;
		final StImage image = new StImage(LogoImage());
		int width = image.width();
		int height = image.height();
		maxX = Math.max(maxX, width);
		maxY += height;

		final Font font = new Font("dialog", Font.PLAIN, 16);
		final FontMetrics fontMetrics = SystemResourceSupport.getFontMetrics(font);
		final String[] text = LogoText();

		for (int i = 0; i < text.length; i++) {
			maxX = Math.max(maxX, fontMetrics.stringWidth(text[i]));
			maxY += fontMetrics.getAscent();
		}

		maxX += (offsetX * 2);
		maxY += (offsetY * 2);

		final int centerX = maxX / 2;

		Frame aFrame = new Frame();
		aFrame.setTitle($String("Image to Ascii Picture"));
		aFrame.setLayout(new BorderLayout());
		aFrame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				e.getWindow().dispose();
			}
		});

		Canvas newCanvas = new Canvas() {
			public void paint(Graphics g) {
				int x = centerX - (image.width() / 2);
				image.displayOn_at_(g, new Point(x, offsetY));
				g.setFont(font);

				int y = offsetY + image.height();

				for (int i = 0; i < text.length; i++) {
					x = centerX - (fontMetrics.stringWidth(text[i]) / 2);
					y += fontMetrics.getAscent();
					g.drawString(text[i], x, y);
				}

				return;
			}
		};

		newCanvas.setBounds(0, 0, maxX, maxY);
		newCanvas.setBackground(Color.white);
		aFrame.add(newCanvas);
		aFrame.pack();

		Rectangle bounds = aFrame.getBounds();
		Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		int x = Math.max(0, (screenSize.width - bounds.width) / 2);
		int y = Math.max(0, (screenSize.height - bounds.height) / 2);
		aFrame.setLocation(x, y);
		aFrame.setVisible(true);
	}

	/**
	 * Answer the image part of the logo.
	 * 
	 * @return java.awt.Image
	 * @category About
	 */
	protected static Image LogoImage() {
		return SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/image/support/JunImageToAsciiLogoImage.jpg");
	}

	/**
	 * Answer the text part of the logo.
	 * 
	 * @return java.lang.String[]
	 * @category About
	 */
	protected static String[] LogoText() {
		String[] text = new String[] { "(1999.12.07)", "", "", $String("AOKI Atsushi"), "", "aoki@sra.co.jp", "", "http://www.sra.co.jp/people/aoki/" };
		return text;
	}

	/**
	 * Answer the default ascii length.
	 * 
	 * @return int
	 * @category Constants
	 */
	protected static int AsciiLength() {
		return 78;
	}

	/**
	 * Answer the default division point.
	 * 
	 * @return java.awt.Dimension
	 * @category Constants
	 */
	protected static Dimension DivisionPoint() {
		Point point = SystemResourceSupport._getStringExtentFor_("A", Font());
		int width = Math.max(point.x / 2, 1);
		int height = Math.max(point.y / 2, 1);
		return new Dimension(width, height);
	}

	/**
	 * Answer the default font name.
	 * 
	 * @return java.lang.String
	 * @category Constants
	 */
	protected static String FontName() {
		return "Courier New";
	}

	/**
	 * Answer the default font size.
	 * 
	 * @return int
	 * @category Constants
	 */
	protected static int FontSize() {
		return 12;
	}

	/**
	 * Answer the default font.
	 * This method corresponds to "textAttribues" of Jun for Smalltalk.
	 * 
	 * @return java.awt.Font
	 * @category Constants
	 */
	protected static Font Font() {
		return new Font(FontName(), Font.PLAIN, FontSize());
	}

	/**
	 * Convert the image to an ascii string.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return java.lang.String
	 * @category Converting
	 */
	public static String Convert_(StImage anImage) {
		return Convert_length_(anImage, AsciiLength());
	}

	/**
	 * Convert the image to an ascii string.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param anInteger int
	 * @return java.lang.String
	 * @category Converting
	 */
	public static String Convert_length_(StImage anImage, int anInteger) {
		JunImageToAscii imageToAscii = new JunImageToAscii();
		imageToAscii.asciiLength_(anInteger);
		return imageToAscii.convert_(anImage);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.divisionPoint();
		this.font();
		this.asciiLength();
		this.progressValue_(new StValueHolder(0.0f));
	}

	/**
	 * Answer the current division point.
	 * 
	 * @return java.awt.Dimension
	 * @category accessing
	 */
	public Dimension divisionPoint() {
		if (divisionPoint == null) {
			divisionPoint = DivisionPoint();
		}
		return divisionPoint;
	}

	/**
	 * Set my new division point.
	 * 
	 * @param width int
	 * @param height int
	 * @category accessing
	 */
	public void divisionPoint_(int width, int height) {
		divisionPoint = new Dimension(Math.max(width, 3), Math.max(height, 6));
	}

	/**
	 * Set the new division point.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @category accessing
	 */
	public void divisionPoint_(Dimension aDimension) {
		this.divisionPoint_(aDimension.width, aDimension.height);
	}

	/**
	 * Set the new division point.
	 * 
	 * @param aPoint java.awt.Dimension
	 * @category accessing
	 */
	public void divisionPoint_(Point aPoint) {
		this.divisionPoint_(aPoint.x, aPoint.y);
	}

	/**
	 * Answer my current font.
	 * 
	 * @return java.awt.Font
	 * @category accessing
	 */
	public Font font() {
		if (font == null) {
			font = Font();
		}
		return font;
	}

	/**
	 * Set my new font.
	 * 
	 * @param aFont java.awt.Font
	 * @category accessing
	 */
	public void font_(Font aFont) {
		font = aFont;
	}

	/**
	 * Answer the current ascii length.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int asciiLength() {
		if (asciiLength <= 0) {
			asciiLength = AsciiLength();
		}

		return asciiLength;
	}

	/**
	 * Set the new ascii length.
	 * 
	 * @param newAsciiLength int
	 * @category accessing
	 */
	public void asciiLength_(int newAsciiLength) {
		asciiLength = Math.max(1, newAsciiLength);
	}

	/**
	 * Answer my current text string.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String _getText() {
		return this.textModel().value().toString();
	}

	/**
	 * Answer my base name.
	 * 
	 * @return java.lang.String
	 * @category aspects
	 */
	protected String baseName() {
		return baseName;
	}

	/**
	 * Answer my text model.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	protected StValueHolder textModel() {
		if (textModel == null) {
			textModel = new StValueHolder(new String());
		}
		return textModel;
	}

	/**
	 * Create features of all printable characters.
	 * 
	 * @return float[][]
	 * @category computing
	 */
	protected float[][] characterFeatures() {
		StImage[] characterImages = this.characterImages();
		float[][] characterFeatures = new float[characterImages.length][];
		for (int i = 0; i < characterImages.length; i++) {
			StImage anImage = characterImages[i];
			StImage shrinkImage = JunImageAdjuster.Adjust_extent_(anImage, this.divisionPoint());
			float[] imageFeature = this.imageFeature_(shrinkImage);
			characterFeatures[i] = imageFeature;
		}

		return characterFeatures;
	}

	/**
	 * Create all printable character images.
	 * 
	 * @return jp.co.sra.smalltalk.StImage[]
	 * @category computing
	 */
	protected StImage[] characterImages() {
		char[] printableCharacters = this.printableCharacters();
		int size = printableCharacters.length;
		StImage[] images = new StImage[size];
		for (int i = 0; i < size; i++) {
			String str = String.valueOf(printableCharacters[i]);
			Point extent = SystemResourceSupport._getStringExtentFor_(str, Font());
			StImage image = new StImage(extent.x, extent.y);
			Graphics graphics = null;
			try {
				graphics = image.image().getGraphics();
				graphics.setColor(Color.white);
				graphics.fillRect(0, 0, extent.x, extent.y);
				graphics.setColor(Color.black);
				int y = SystemResourceSupport.getFontMetrics(Font()).getAscent();
				graphics.drawString(str, 0, y);
				image.image().flush();
			} finally {
				if (graphics != null) {
					graphics.dispose();
				}
			}

			// Convert the image to a bitmap.
			images[i] = this.convertToWhiteBlackPalette_(image);
		}

		return images;
	}

	/**
	 * Create luminances of all printable characters.
	 * 
	 * @return float[]
	 * @category computing
	 */
	protected float[] characterLuminances() {
		float maxLuminance = Float.NaN;
		float minLuminance = Float.NaN;
		StImage[] characterImages = this.characterImages();
		int size = characterImages.length;
		float[] characterLuminances = new float[size];
		for (int i = 0; i < size; i++) {
			StImage anImage = characterImages[i];
			float imageLuminance = this.imageLuminance_(anImage);
			if (Float.isNaN(maxLuminance) == true) {
				maxLuminance = imageLuminance;
			} else {
				maxLuminance = Math.max(maxLuminance, imageLuminance);
			}
			if (Float.isNaN(minLuminance) == true) {
				minLuminance = imageLuminance;
			} else {
				minLuminance = Math.min(minLuminance, imageLuminance);
			}
			characterLuminances[i] = imageLuminance;
		}

		float[] normalizedCharacterLuminances = new float[size];
		for (int i = 0; i < size; i++) {
			normalizedCharacterLuminances[i] = (characterLuminances[i] - minLuminance) / (maxLuminance - minLuminance);
		}

		return normalizedCharacterLuminances;
	}

	/**
	 * Create a character table which contains all information about characters.
	 * 
	 * @return java.util.Map.Entry[]
	 * @category computing
	 */
	protected Map.Entry[] characterTable() {
		char[] printableCharacters = this.printableCharacters();
		float[] characterLuminances = this.characterLuminances();
		float[][] characterFeatures = this.characterFeatures();
		HashMap aMap = new HashMap();
		for (int i = 0; i < printableCharacters.length; i++) {
			Character aCharacter = new Character(printableCharacters[i]);
			Float aLuminance = new Float(characterLuminances[i]);
			float[] aFeature = characterFeatures[i];
			if (aMap.containsKey(aLuminance) == false) {
				aMap.put(aLuminance, new ArrayList());
			}
			List aList = (List) aMap.get(aLuminance);
			aList.add(new Object[] { aCharacter, aFeature });
		}

		TreeMap characterTable = new TreeMap(new Comparator() {
			public int compare(Object o1, Object o2) {
				float l1 = ((Number) o1).floatValue();
				float l2 = ((Number) o2).floatValue();
				return (l1 < l2) ? -1 : (l2 < l1) ? 1 : 0;
			}
		});

		Iterator i = aMap.entrySet().iterator();
		while (i.hasNext()) {
			Map.Entry entry = (Map.Entry) i.next();
			Number aNumber = (Number) entry.getKey();
			List aList = (List) entry.getValue();
			characterTable.put(aNumber, aList.toArray());
		}

		return (Map.Entry[]) characterTable.entrySet().toArray(new Map.Entry[characterTable.size()]);
	}

	/**
	 * Answer the array of luminances of all pixels.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return float[]
	 * @category computing
	 */
	protected float[] imageFeature_(StImage anImage) {
		int width = anImage.width();
		int height = anImage.height();
		float[] imageFeature = new float[width * height];
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				Color color = anImage.valueAtPoint_(new Point(x, y));
				imageFeature[x + (y * width)] = (float) StColorValue._GetLuminance(color);
			}
		}
		return imageFeature;
	}

	/**
	 * Answer the luminance of the image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return float
	 * @category computing
	 */
	protected float imageLuminance_(StImage anImage) {
		float[] imageFeature = this.imageFeature_(anImage);
		float totalValue = 0;
		for (int i = 0; i < imageFeature.length; i++) {
			totalValue += imageFeature[i];
		}
		float imageLuminance = totalValue / imageFeature.length;
		return imageLuminance;
	}

	/**
	 * Answer the array of the printable characters.
	 * 
	 * @return char[]
	 * @category computing
	 */
	protected char[] printableCharacters() {
		boolean[] table = new boolean[256];
		for (int i = 0; i < 256; i++) {
			table[i] = false;
		}
		table[' '] = true;
		table['!'] = true;
		table['"'] = true;
		table['#'] = true;
		table['$'] = true;
		table['%'] = true;
		table['&'] = true;
		table['\''] = true;
		table['('] = true;
		table[')'] = true;
		table['*'] = true;
		table['+'] = true;
		table[','] = true;
		table['-'] = true;
		table['.'] = true;
		table['/'] = true;
		table[':'] = true;
		table[';'] = true;
		// table['<'] = true;
		table['='] = true;
		// table['>'] = true;
		table['?'] = true;
		table['@'] = true;
		table['['] = true;
		// table['\\'] = true;
		table[']'] = true;
		table['^'] = true;
		table['_'] = true;
		table['`'] = true;
		table['{'] = true;
		table['|'] = true;
		table['}'] = true;
		table['~'] = true;

		int count = 0;
		char[] collection = new char[256];
		for (int i = 0; i < 256; i++) {
			if (('0' <= i && i <= '9') || ('A' <= i && i <= 'Z') || ('a' <= i && i <= 'z')) {
				table[i] = true;
			}
			if (table[i] == true) {
				collection[count] = (char) i;
				count++;
			}
		}

		char[] result = new char[count];
		System.arraycopy(collection, 0, result, 0, count);
		return result;
	}

	/**
	 * Answer an appropriate index of the specified number in the sorted numbers.
	 * 
	 * @param aNumber float
	 * @param sortedNumbers float[]
	 * @return int
	 * @category computing
	 */
	protected int search_in_(float aNumber, float[] sortedNumbers) {
		int min = 0;
		int max = sortedNumbers.length - 1;
		int index = (min + max) / 2;
		float it = sortedNumbers[index];
		while ((aNumber != it) && (min <= max)) {
			if (aNumber < it) {
				max = index - 1;
			} else {
				min = index + 2;
			}
			index = (min + max) / 2;
			it = sortedNumbers[index];
		}

		if (aNumber != it) {
			index = Math.max(1, Math.min(Math.max(min, max), sortedNumbers.length - 1));
		}
		return index;
	}

	/**
	 * Convert the image to ascii characters.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return java.lang.String
	 * @category converting
	 */
	public String convert_(StImage anImage) {
		this.progress_(0.0f);

		int divisionWidth = this.divisionPoint().width;
		int divisionHeight = this.divisionPoint().height;
		int imageWidth = this.asciiLength() * divisionWidth;
		float aspect = (anImage.height() * imageWidth) / (float) (anImage.width() * divisionHeight);
		int imageHeight = Math.round(aspect) * divisionHeight;
		JunImageAdjuster anAdjuster = new JunImageAdjuster();
		anAdjuster.compute_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunImageToAscii.this.progress_(((Number) anObject).floatValue() / 2);
				return null;
			}
		});

		StImage theImage = anAdjuster.adjust_extent_(anImage, new Dimension(imageWidth, imageHeight));
		Map.Entry[] characterTable = (Map.Entry[]) this.characterTable();
		float[] luminanceCollection = new float[characterTable.length];
		for (int i = 0; i < luminanceCollection.length; i++) {
			luminanceCollection[i] = ((Number) ((Map.Entry) characterTable[i]).getKey()).floatValue();
		}

		this.progress_(0.5f);

		StringWriter stringWriter = new StringWriter(1024);
		BufferedWriter writer = new BufferedWriter(stringWriter);
		try {
			for (int y = 0; y < theImage.height(); y += divisionHeight) {
				for (int x = 0; x < theImage.width(); x += divisionWidth) {
					StImage smallImage = new StImage(divisionWidth, divisionHeight);
					smallImage.copy_from_in_rule_(smallImage.bounds(), new Point(x, y), theImage, StImage.Over);

					float imageLuminance = this.imageLuminance_(smallImage);
					Map.Entry entry = characterTable[this.search_in_(imageLuminance, luminanceCollection)];
					Object[] candidateCollection = (Object[]) entry.getValue();
					float[] imageFeature = this.imageFeature_(smallImage);

					TreeMap aTreeMap = new TreeMap(new Comparator() {
						public int compare(Object o1, Object o2) {
							float f1 = ((Number) o1).floatValue();
							float f2 = ((Number) o2).floatValue();
							return (f1 < f2) ? -1 : (f2 < f1) ? 1 : 0;
						}
					});
					for (int i = 0; i < candidateCollection.length; i++) {
						Object[] anArray = (Object[]) candidateCollection[i];
						Character aCharacter = (Character) anArray[0];
						float[] feature = (float[]) anArray[1];
						float coefficientOfCorrelation = (float) (new JunCorrelation(imageFeature, feature)).coefficient();
						aTreeMap.put(new Float(coefficientOfCorrelation), aCharacter);
					}

					char ch = ((Character) aTreeMap.get(aTreeMap.lastKey())).charValue();
					writer.write(ch);
				}

				writer.newLine();
				writer.flush();

				this.progress_(0.5f + ((float) y / (theImage.height() - 1) / 2));
			}
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				writer.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		this.progress_(1.0f);

		return stringWriter.toString();
	}

	/**
	 * Do something with the progress value.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category progress
	 */
	public void compute_(StBlockClosure aBlock) {
		this.progressValue().compute_(aBlock);
	}

	/**
	 * Answer the progress value.
	 * 
	 * @return float
	 * @category progress
	 */
	protected float progress() {
		return this.progressValue()._floatValue();
	}

	/**
	 * Set the progress value.
	 * 
	 * @param normalizedNumber float
	 * @category progress
	 */
	protected void progress_(float normalizedNumber) {
		if (0 <= normalizedNumber && normalizedNumber <= 1) {
			float truncatedValue = Math.round(normalizedNumber / 0.005) * 0.005f;

			if (this.progressValue()._floatValue() != truncatedValue) {
				this.progressValue().value_(truncatedValue);
			}
		}
	}

	/**
	 * Answer my current StValueHolder for the progress value.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category progress
	 */
	protected StValueHolder progressValue() {
		return progressValue;
	}

	/**
	 * Set my new value holder for the progress value.
	 * 
	 * @param aValueHolder jp.co.sra.smalltalk.StValueHolder
	 * @category progress
	 */
	protected void progressValue_(StValueHolder aValueHolder) {
		progressValue = aValueHolder;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunImageToAsciiViewAwt(this);
		} else {
			return new JunImageToAsciiViewSwing(this);
		}
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Image to Ascii Picture");
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();

			StMenu fileMenu = new StMenu($String("File"));
			fileMenu.add(new StMenuItem($String("Open") + "...", new MenuPerformer(this, "openImageToAscii")));
			fileMenu.addSeparator();
			fileMenu.add(new StMenuItem($String("Save") + "...", new MenuPerformer(this, "saveImageToAscii")));
			fileMenu.addSeparator();
			fileMenu.add(new StMenuItem($String("Quit"), new MenuPerformer(this, "quitImageToAscii")));
			_menuBar.add(fileMenu);

			StMenu miscMenu = new StMenu($String("Misc"));
			miscMenu.add(new StMenuItem($String("About <1p>", null, "ImageToAscii") + "...", new MenuPerformer(this, "aboutImageToAscii")));
			_menuBar.add(miscMenu);
		}
		return _menuBar;
	}

	/**
	 * Open an image file and convert it to an ascii string.
	 * 
	 * @return java.lang.String
	 * @throws java.io.FileNotFoundException 
	 * @throws java.io.IOException 
	 * @category menu messages
	 */
	public String openImageToAscii() throws FileNotFoundException, IOException {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "JPEG") + " (*.jpg)", new String[] { "*.jpg", "*.JPG", "*.jpeg", "*.JPEG" }) };
		File file = JunFileRequesterDialog.Request($String("Select a <1p> file.", null, "JPEG"), fileTypes, fileTypes[0]);
		if (file == null) {
			return null;
		}

		StImage image = null;
		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			JunImageStream stream = null;
			try {
				stream = JunJpegImageStream.On_(new FileInputStream(file));
				image = stream.nextImage();
			} finally {
				if (stream != null) {
					stream.close();
				}
			}
		} finally {
			cursor._restore();
		}

		if (image == null) {
			System.err.println("Failed to open a JPEG file - " + file);
			return null;
		}

		// Convert the image to ascii characters.
		this.lengthImageToAscii();

		String aString = null;
		cursor = new JunCursors(JunCursors.ExecuteCursor());
		try {
			cursor._show();

			_progress = new JunProgress();
			this.compute_(new StBlockClosure() {
				public Object value_(Object anObject) {
					_progress.value_(((Number) anObject).floatValue());
					return null;
				}
			});
			_progress.message_($String("converting <1p>...", null, "JPEG"));

			final StImage _image = image;
			aString = (String) _progress.do_(new StBlockClosure() {
				public Object value() {
					return convert_(_image);
				}
			});
		} finally {
			cursor._restore();
		}

		this.textModel().value_(aString);

		// Change the title.
		Window[] windows = this.builder().windows();
		if (windows != null) {
			baseName = file.getName();
			int index = baseName.lastIndexOf('.');
			if (index >= 0) {
				baseName = baseName.substring(0, index);
			}

			for (int i = 0; i < windows.length; i++) {
				((Frame) windows[i]).setTitle(baseName);
			}
		}

		return aString;
	}

	/**
	 * Save the current ascii string to a file.
	 * 
	 * @exception java.io.IOException
	 * @category menu messages
	 */
	public void saveImageToAscii() throws IOException {
		String aString = this._getText();
		if ((aString == null) || (aString.length() == 0)) {
			return;
		}

		File aFile = new File((baseName == null) ? "filename.txt" : baseName + ".txt");
		aFile = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, $String("Text")), aFile);
		if (aFile == null) {
			return;
		}

		FileWriter aWriter = null;
		try {
			aWriter = new FileWriter(aFile);
			aWriter.write(aString);
		} finally {
			if (aWriter != null) {
				aWriter.flush();
				aWriter.close();
			}
		}
	}

	/**
	 * Quit the application.
	 * 
	 * @category menu messages
	 */
	public void quitImageToAscii() {
		this.closeRequest();
	}

	/**
	 * Show the about dialog.
	 * 
	 * @category menu messages
	 */
	public void aboutImageToAscii() {
		About();
	}

	/**
	 * Set the ascii length.
	 * 
	 * @category menu messages
	 */
	public void lengthImageToAscii() {
		String aString = String.valueOf(this.asciiLength());
		aString = JunDialog.Request_($String("Ascii Length?"), aString);
		if (aString == null || aString.length() == 0) {
			return;
		}

		try {
			int aNumber = Integer.parseInt(aString);
			this.asciiLength_(Math.max(10, Math.min(aNumber, 1000)));
		} catch (NumberFormatException e) {
			// just ignore.
		}
	}

	/**
	 * Convert a image to white-black palette.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @return jp.co.sra.smalltalk.StImage
	 * @category private
	 */
	protected StImage convertToWhiteBlackPalette_(StImage anImage) {
		int width = anImage.width();
		int height = anImage.height();
		StImage whiteBlackImage = new StImage(width, height);
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				Point point = new Point(x, y);
				Color color = anImage.valueAtPoint_(point);
				if (StColorValue._GetLuminance(color) > 0.5) {
					whiteBlackImage.valueAtPoint_put_(point, Color.white);
				} else {
					whiteBlackImage.valueAtPoint_put_(point, Color.black);
				}
			}
		}
		return whiteBlackImage;
	}

}