package jp.co.sra.jun.goodies.prolog;

import java.awt.Cursor;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Hashtable;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StReadStream;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunPrologInterpreter class
 * 
 *  @author    kondo
 *  @created   1999/09/09 (by kondo)
 *  @updated   2003/04/24 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun490 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPrologInterpreter.java,v 8.11 2008/02/20 06:32:02 nisinaka Exp $
 */
public class JunPrologInterpreter extends JunApplicationModel {

	protected static Hashtable SystemPredicates;

	protected Hashtable systemPredicates;
	protected Hashtable userPredicates;
	protected StSymbol status;
	protected JunPrologList definition;
	protected JunPrologList question;
	protected JunPrologList questionEnv;
	protected JunPrologList clause;
	protected JunPrologList clauseEnv;
	protected JunPrologList queue;
	protected JunPrologList queueEnv;
	protected int envCounter;
	protected JunPrologList valueEnv;
	protected JunPrologList unbindList;
	protected JunPrologList cutBack;
	protected JunPrologList backTrack;
	protected StBlockClosure resolveAction;
	protected JunPrologList definitionStack;
	protected Hashtable tracePredicates;
	protected int traceCounter;
	protected PrintStream textCollector;
	protected boolean verbose;

	protected StValueHolder textValue;

	// textValue;
	// listValue;
	// predicate;

	/**
	 * Flush the hashtable for the system predicates.
	 * 
	 * @category Class initialization
	 */
	protected static void FlushSystemPredicates() {
		SystemPredicates = null;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		systemPredicates = new Hashtable();
		userPredicates = new Hashtable();
		tracePredicates = new Hashtable();
		textCollector = new PrintStream(new ByteArrayOutputStream());
		verbose = false;
		this.makeSystemPredicates();
	}

	/**
	 * Answer my current text collector.
	 *
	 * @return java.io.PrintStream
	 * @category accessing
	 */
	public PrintStream textCollector() {
		return textCollector;
	}

	/**
	 * Answer my text collector.
	 *
	 * @param newTextCollector java.io.PrintStream
	 * @category accessing
	 */
	public void textCollector_(PrintStream newTextCollector) {
		textCollector = newTextCollector;
	}

	/**
	 * Do the confirmation.
	 *
	 * @param messageString java.lang.String
	 * @return boolean
	 * @category public accessing
	 */
	public boolean confirm_(String messageString) {
		return JunDialog.Confirm_(messageString, false);
	}

	/**
	 * Refute the string and do the action.
	 *
	 * @param aString java.lang.String
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return boolean
	 * @category public accessing
	 */
	public boolean refute_action_(String aString, StBlockClosure aBlock) {
		StReadStream stream = new StReadStream(aString);
		return this.consult_action_(stream, aBlock);
	}

	/**
	 * Answer my text value.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category adaptor
	 */
	public StValueHolder textValue() {
		if (textValue == null) {
			textValue = new StValueHolder(new String());
		}
		return textValue;
	}

	/**
	 * Bind two JunPrologEntities.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param xEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @param y jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param yEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @category binding and unbinding
	 */
	protected void bind_env_and_env_(JunPrologEntity x, JunPrologList xEnv, JunPrologEntity y, JunPrologList yEnv) {
		JunPrologEntity list = new JunPrologList(this.fetchValue_env_(y, yEnv), valueEnv);
		list = x.cons_(list).cons_(xEnv.cdr());
		xEnv.cdr_(list);
		unbindList = xEnv.cons_(unbindList);
	}

	/**
	 * Answer the binding of the JunPrologEntity in the specified environment.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param xEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category binding and unbinding
	 */
	protected JunPrologEntity binding_env_(JunPrologEntity x, JunPrologList xEnv) {
		JunPrologList assocList = (JunPrologList) xEnv.cdr();
		if (assocList == null) {
			return null;
		}
		return assocList.assoc_(x);
	}

	/**
	 * Fetch the value of the JunPrologEntity in the specified environement.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param xEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category binding and unbinding
	 */
	protected JunPrologEntity fetchValue_env_(JunPrologEntity x, JunPrologList xEnv) {
		JunPrologEntity xx = x;
		JunPrologList xxEnv = xEnv;
		JunPrologEntity assoc;

		while (true) {
			valueEnv = (JunPrologList) xxEnv;
			if (xx != null && xx.isJunPrologVariable()) {
				assoc = this.binding_env_(xx, xxEnv);
				if (assoc == null) {
					return xx;
				}
				assoc = assoc.cdr();
				xx = assoc.car();
				xxEnv = (JunPrologList) assoc.cdr();
			} else {
				return xx;
			}
		}
	}

	/**
	 * Check whether the JunPrologEntity has no value in the specified environment.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param xEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @return boolean
	 * @category binding and unbinding
	 */
	protected boolean noValue_env_(JunPrologEntity x, JunPrologList xEnv) {
		if (x.isJunPrologVariable()) {
			JunPrologEntity assoc = this.binding_env_(x, xEnv);
			if (assoc == null) {
				return true;
			}
			assoc = assoc.cdr();
			return this.noValue_env_(assoc.car(), (JunPrologList) assoc.cdr());
		}
		return false;
	}

	/**
	 * Answer the null env.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @category binding and unbinding
	 */
	protected JunPrologList nullEnv() {
		envCounter++;
		return new JunPrologList(new JunPrologNumber(envCounter), null);
	}

	/**
	 * Unbind the list.
	 *
	 * @param start jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @param end jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @category binding and unbinding
	 */
	protected void unbindFrom_to_(JunPrologList start, JunPrologList end) {
		JunPrologEntity list = start;
		while (list != end) {
			JunPrologEntity env = list.car();
			JunPrologEntity assocList = env.cdr();
			assocList.car().cdr().cdr_(null);
			env.cdr_(assocList.cdr());
			list = list.cdr();
		}
	}

	/**
	 * Kernel predicates: Process the built-in predicate, Cut.
	 * 
	 * @category kernel predicates
	 */
	protected void builtInCut() {
		backTrack = (JunPrologList) cutBack.car();
		clause = (JunPrologList) clause.cdr();
		status = $("loop");
	}

	/**
	 * Kernel predicates: Process the built-in predicate, Fail.
	 * 
	 * @category kernel predicates
	 */
	protected void builtInFail() {
		status = $("back");
	}

	/**
	 * Kernel predicates: Process the built-in predicate.
	 *
	 * @param functor jp.co.sra.jun.goodies.prolog.JunPrologSymbol
	 * @category kernel predicates
	 */
	protected void builtInPredicate_(JunPrologSymbol functor) {
		if (functor.isJunPrologCut()) {
			this.builtInCut();
		} else if (functor.isJunPrologTrue()) {
			this.builtInTrue();
		} else if (functor.isJunPrologFail()) {
			this.builtInFail();
		} else if (functor.isJunPrologSend()) {
			this.builtInSend();
		} else if (functor.isJunPrologVar()) {
			this.builtInVar();
		} else {
			throw SmalltalkException.Error("Unexpected built-in predicate");
		}
	}

	/**
	 * Kernel predicates: Process the built-in predicate, Send.
	 * 
	 * @category kernel predicates
	 */
	protected void builtInSend() {
		JunPrologEntity list = this.expression_env_(clause.car().cdr(), clauseEnv);
		JunPrologEntity receiverSymbol = list.car();
		list = list.cdr();
		JunPrologSymbol selectorSymbol = (JunPrologSymbol) list.car();
		list = list.cdr();
		JunPrologList argumentList = (JunPrologList) list.car();
		list = list.cdr();

		JunPrologEntity unifyTerm = null;
		if (list != null) {
			if (list.cdr() == null) {
				unifyTerm = ((JunPrologList) clause.car().cdr()).nth_(4);
			} else {
				status = $("back");
				return;
			}
		}

		JunPrologEntity result = this.receiver_selector_arguments_(receiverSymbol, selectorSymbol, argumentList);
		if (result.equals(JunPrologSymbol.Cut())) {
			this.builtInCut();
			status = $("back");
			return;
		}

		if (unifyTerm == null) {
			if (result != JunPrologBoolean.FALSE) {
				result = JunPrologBoolean.TRUE;
			}
			if (result == JunPrologBoolean.TRUE) {
				clause = (JunPrologClause) clause.cdr();
				status = $("loop");
				return;
			} else {
				status = $("back");
				return;
			}
		} else {
			if (this.unify_env_and_env_(unifyTerm, clauseEnv, result, this.nullEnv())) {
				clause = (JunPrologClause) clause.cdr();
				status = $("loop");
				return;
			} else {
				status = $("back");
				return;
			}
		}
	}

	/**
	 * Kernel predicates: Process the built-in predicate, True.
	 * 
	 * @category kernel predicates
	 */
	protected void builtInTrue() {
		clause = (JunPrologClause) clause.cdr();
		status = $("loop");
	}

	/**
	 * Kernel predicates: Process the built-in predicate, Var.
	 * 
	 * @category kernel predicates
	 */
	protected void builtInVar() {
		JunPrologEntity var = clause.car().cdr().car();
		if (this.noValue_env_(var, clauseEnv)) {
			clause = (JunPrologList) clause.cdr();
			status = $("loop");
		} else {
			status = $("back");
		}
	}

	/**
	 * Kernel predicates: Call the JunPrologStructure.
	 *
	 * @param aJunPrologStructure jp.co.sra.jun.goodies.prolog.JunPrologStructure
	 * @category kernel predicates
	 */
	protected void call_(JunPrologStructure aJunPrologStructure) {
		if (aJunPrologStructure.cdr() == null) {
			this.callVariable_((JunPrologVariable) aJunPrologStructure.car());
			return;
		}

		JunPrologStructure structure = (JunPrologStructure) this.expression_env_(aJunPrologStructure, clauseEnv);
		if (this.unify_env_and_env_(aJunPrologStructure, clauseEnv, structure, clauseEnv) == false) {
			throw SmalltalkException.Error("Unexpected unify structure");
		}

		JunPrologEntity functor = structure.car();
		if (functor.isJunPrologVariable()) {
			status = $("back");
			return;
		}

		clause = (JunPrologClause) clause.cdr();
		clause = (JunPrologClause) structure.cons_(clause);
		status = $("loop");
	}

	/**
	 * Kernel predicates: Call the JunPrologVariable.
	 *
	 * @param aJunPrologVariable jp.co.sra.jun.goodies.prolog.JunPrologVariable
	 * @category kernel predicates
	 */
	protected void callVariable_(JunPrologVariable aJunPrologVariable) {
		JunPrologEntity horn = this.expression_env_(aJunPrologVariable, clauseEnv);
		if (this.unify_env_and_env_(aJunPrologVariable, clauseEnv, horn, clauseEnv) == false) {
			throw SmalltalkException.Error("Unexpected unify forn clause");
		}

		if (horn.consp() == false) {
			horn = new JunPrologStructure((JunPrologString) horn, null);
		}
		if (horn.car().consp() == false) {
			horn = new JunPrologClause((JunPrologStructure) horn, null);
		}
		horn = new JunPrologClause((JunPrologStructure) horn.car(), JunPrologBody.FromList_((JunPrologList) horn.cdr()));
		JunPrologEntity functor = horn.car().car();
		if (functor.isJunPrologVariable()) {
			status = $("back");
			return;
		}

		clause = (JunPrologClause) clause.cdr();
		horn = ((JunPrologList) horn).reverse();
		while (horn.consp()) {
			clause = (JunPrologClause) horn.car().cons_(clause);
			horn = horn.cdr();
		}
		status = $("loop");
	}

	/**
	 * Kernel predicates: Send the message to the receiver.
	 *
	 * @param receiverSymbol jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param selectorSymbol jp.co.sra.jun.goodies.prolog.JunPrologSymbol
	 * @param argumentList jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category kernel predicates
	 */
	protected JunPrologEntity receiver_selector_arguments_(JunPrologEntity receiverSymbol, JunPrologSymbol selectorSymbol, JunPrologList argumentList) {
		StObject receiver = receiverSymbol;
		if (receiverSymbol instanceof JunPrologSymbol && ((JunPrologSymbol) receiverSymbol).isJunPrologSelf()) {
			receiver = this;
		}

		/* 
		 * The following is OK in Smalltalk, but not applicable to Java.
		 * Better to override the perform_() method in each JunPrologObject.
		 */
		// if (receiver instanceof JunPrologObject) {
		//     receiver = (StObject) ((JunPrologObject) receiver).object();
		// }
		String selector = selectorSymbol.string();
		JunPrologList list = argumentList;

		Object result = null;
		if (list == null) {
			try {
				result = receiver.perform_(selector);
			} catch (Exception e) {
				throw new SmalltalkException(e);
			}
		} else {
			int size = list.length();
			Object[] arguments = new Object[size];
			for (int i = 0; i < size; i++) {
				arguments[i] = (list.car() instanceof JunPrologObject) ? ((JunPrologObject) list.car()).object() : list.car();
				list = (JunPrologList) list.cdr();
			}
			try {
				result = receiver.perform_withArguments_(selector, arguments);
			} catch (Exception e) {
				throw new SmalltalkException(e);
			}
		}

		JunPrologEntity resultEntity = null;
		if (result instanceof JunPrologEntity) {
			resultEntity = (JunPrologEntity) result;
		} else if (result instanceof Boolean) {
			resultEntity = ((Boolean) result).booleanValue() ? JunPrologBoolean.TRUE : JunPrologBoolean.FALSE;
		} else if (result instanceof Number) {
			resultEntity = new JunPrologNumber((Number) result);
		} else {
			throw new SmalltalkException("not implemented yet");
		}

		return resultEntity;
	}

	/**
	 * Output the answer to the text collector.
	 *
	 * @param aDictionary java.util.Hashtable
	 * @category outputing
	 */
	protected void outputAnswer_(Hashtable aDictionary) {
		Object[] keys = aDictionary.keySet().toArray();
		Arrays.sort(keys);
		for (int i = 0; i < keys.length; i++) {
			Object s = keys[i];
			if (i > 0) {
				textCollector.println();
			}
			textCollector.print(s);
			textCollector.print(" = ");
			textCollector.print(((JunPrologEntity) aDictionary.get(s)).printJunPrologString());
			textCollector.print(' ');
		}
	}

	/**
	 * Output the time to the text collector.
	 *
	 * @param msec long
	 * @category outputing
	 */
	protected void outputTime_(long msec) {
		if (verbose) {
			int goal = envCounter - 1;
			String string = "<";
			string += String.valueOf(msec) + " milliseconds, ";
			string += String.valueOf(goal) + " goals";
			textCollector.println(string);
		}
	}

	/**
	 * Output the variables to the text collector and ask users to confirm them.
	 *
	 * @param aDictionary java.util.Hashtable
	 * @return boolean
	 * @category outputing
	 */
	protected boolean outputVariables_(Hashtable aDictionary) {
		this.outputAnswer_(aDictionary);
		boolean bool = this.confirm_(JunSystem.$String("All right?"));
		if (bool) {
			textCollector.println();
		} else {
			textCollector.println(";");
		}
		return bool;
	}

	/**
	 * Collect the variables.
	 *
	 * @return java.util.Hashtable
	 * @category representation
	 */
	protected Hashtable collectVariables() {
		Hashtable dict = new Hashtable();
		this.collectVariables_to_(question, dict);
		return dict;
	}

	/**
	 * Collect variables to the hashtable.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param dict java.util.Hashtable
	 * @category representation
	 */
	protected void collectVariables_to_(JunPrologEntity x, Hashtable dict) {
		if (x != null && x.isJunPrologVariable() && (((JunPrologString) x).string()).charAt(0) != '~') {
			Object key = new String(x.printJunPrologString());
			if (dict.get(key) != null) {
				return;
			} else {
				Object value = this.represent_env_(x, questionEnv);
				if (value == null) {
					// Hashtable does not accept null as a value.
					value = new JunPrologObject(null);
				}
				dict.put(key, value);
			}
		}

		if (x == null || x.consp() == false) {
			return;
		}

		this.collectVariables_to_(x.car(), dict);
		this.collectVariables_to_(x.cdr(), dict);
	}

	/**
	 * Answer the expression of the JunPrologEntity in the specified environment.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param xEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category representation
	 */
	protected JunPrologEntity expression_env_(JunPrologEntity x, JunPrologList xEnv) {
		if (x != null && x.isJunPrologVariable()) {
			JunPrologEntity assoc = this.binding_env_(x, xEnv);
			if (assoc == null) {
				return x;
			}
			assoc = assoc.cdr();
			return this.expression_env_(assoc.car(), (JunPrologList) assoc.cdr());
		}

		if (x == null || x.consp() == false) {
			return x;
		}

		JunPrologEntity a = this.expression_env_(x.car(), xEnv);
		JunPrologEntity d = this.expression_env_(x.cdr(), xEnv);
		return ((JunPrologList) x)._newInstanceWith(a, d);
	}

	/**
	 * Represent the JunPrologEntity in the specified environment.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param xEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category representation
	 */
	protected JunPrologEntity represent_env_(JunPrologEntity x, JunPrologList xEnv) {
		if (x != null && x.isJunPrologVariable()) {
			JunPrologEntity assoc = this.binding_env_(x, xEnv);
			if (assoc == null) {
				return this.variableRepresent_env_((JunPrologVariable) x, xEnv);
			}
			assoc = assoc.cdr();
			return this.represent_env_(assoc.car(), (JunPrologList) assoc.cdr());
		}

		if (x == null || x.consp() == false) {
			return x;
		}

		JunPrologEntity a = this.represent_env_(x.car(), xEnv);
		JunPrologEntity d = this.represent_env_(x.cdr(), xEnv);
		return ((JunPrologList) x)._newInstanceWith(a, d);
	}

	/**
	 * Answer the variable representation of the JunPrologEntity in the specified environment.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param xEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologVariable
	 * @category representation
	 */
	protected JunPrologVariable variableRepresent_env_(JunPrologVariable x, JunPrologList xEnv) {
		return JunPrologVariable.Install_(x.string() + xEnv.car().printString());
	}

	/**
	 * Resolve the goal.
	 *
	 * @param goal jp.co.sra.jun.goodies.prolog.JunPrologClause
	 * @return boolean
	 * @category resolution
	 */
	protected boolean resolve_(JunPrologClause goal) {
		return this.resolve_action_(goal, null);
	}

	/**
	 * Resolve the goal and do the action.
	 *
	 * @param goal jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @param actionBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return boolean
	 * @category resolution
	 */
	protected boolean resolve_action_(JunPrologList goal, StBlockClosure actionBlock) {
		question = clause = goal;
		if (actionBlock == null) {
			resolveAction = new StBlockClosure() {
				public Object value_(Object dict) {
					return new Boolean(outputVariables_((Hashtable) dict));
				}
			};
		} else {
			resolveAction = actionBlock;
		}

		JunCursors cursor = new JunCursors(JunCursors.ExecuteCursor());
		try {
			cursor._show();

			this.resolveInitialize();
			this.resolveLoop();
			return this.resolveTerminate();

		} finally {
			cursor._restore();
		}
	}

	/**
	 * Initialize for the resolution.
	 * 
	 * @category resolution
	 */
	protected void resolveInitialize() {
		status = $("loop");
		envCounter = 0;
		questionEnv = clauseEnv = this.nullEnv();
		queue = queueEnv = valueEnv = null;
		unbindList = cutBack = backTrack = null;
		definitionStack = null;
		traceCounter = 0;
	}

	/**
	 * Do the loop for resolution.
	 *
	 * @return boolean
	 * @category resolution
	 */
	protected boolean resolveLoop() {
		long totalTime = 0;
		long time = System.currentTimeMillis();
		while (true) {
			if (status == $("loop")) {
				this.loop();
			}
			if (status == $("next")) {
				this.next();
			}
			if (status == $("back")) {
				this.back();
			}
			if (status == $("succ")) {
				time = System.currentTimeMillis() - time;
				totalTime += time;
				this.outputTime_(totalTime);
				Hashtable answer = this.collectVariables();
				if (answer.isEmpty()) {
					return true;
				}
				if (((Boolean) resolveAction.value_(answer)).booleanValue()) {
					return true;
				} else {
					status = $("back");
					time = System.currentTimeMillis();
				}
			}
			if (status == $("fail")) {
				time = System.currentTimeMillis() - time;
				totalTime += time;
				this.outputTime_(totalTime);
				return false;
			}
		}
	}

	/**
	 * Terminate the resolution.
	 *
	 * @return boolean
	 * @category resolution
	 */
	protected boolean resolveTerminate() {
		JunCursors cursor = new JunCursors(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try {
			cursor._show();
			this.deallocateEnv_(questionEnv);
		} finally {
			cursor._restore();
		}

		if (status == $("succ")) {
			textCollector.println("yes");
			return true;
		} else if (status == $("fail")) {
			textCollector.println("no");
			return false;
		}

		throw SmalltalkException.Error("Unexpected status");
	}

	/**
	 * Process the loop status.
	 * 
	 * @category resolve modules
	 */
	protected void loop() {
		if (clause == null) {
			if (queue == null) {
				status = $("succ");
				return;
			}

			clause = (JunPrologList) queue.car();
			clauseEnv = (JunPrologList) queueEnv.car();
			queue = (JunPrologList) queue.cdr();
			queueEnv = (JunPrologList) queueEnv.cdr();
			cutBack = (JunPrologList) cutBack.cdr();
			status = $("loop");
			return;
		}

		JunPrologEntity structure = clause.car();
		if (structure.consp() == false) {
			structure = new JunPrologStructure((JunPrologString) structure, null);
			clause.car_(structure);
		}
		JunPrologEntity functor = structure.car();
		if (functor.isJunPrologVariable()) {
			this.call_((JunPrologStructure) clause.car());
			return;
		}
		if (((JunPrologSymbol) functor).isBuiltInPredicate()) {
			this.builtInPredicate_((JunPrologSymbol) functor);
			return;
		}

		definition = (JunPrologList) userPredicates.get(functor);
		if (definition == null) {
			definition = (JunPrologList) systemPredicates.get(functor);
			if (definition == null) {
				status = $("back");
				return;
			}
		}
		status = $("next");
	}

	/**
	 * Process the next status.
	 * 
	 * @category resolve modules
	 */
	protected void next() {
		JunPrologList definitionEnv = this.nullEnv();
		if (definition.cdr() == null) {
			if (this.unify_env_and_env_(clause.car().cdr(), clauseEnv, definition.car().car().cdr(), definitionEnv) == false) {
				status = $("back");
				return;
			}

			queue = new JunPrologList(clause.cdr(), queue);
			queueEnv = clauseEnv.cons_(queueEnv);
			clause = (JunPrologList) definition.car().cdr();
			clauseEnv = definitionEnv;
			cutBack = new JunPrologList(backTrack, cutBack);
			status = $("loop");
			return;
		}

		JunPrologList saveBackTrack = backTrack;
		Object[] array = new Object[7];
		array[0] = clause;
		array[1] = clauseEnv;
		array[2] = queue;
		array[3] = queueEnv;
		array[4] = cutBack;
		array[5] = definition.cdr();
		array[6] = unbindList;

		backTrack = new JunPrologList(new JunPrologObject(array), saveBackTrack);
		if (this.unify_env_and_env_(clause.car().cdr(), clauseEnv, definition.car().car().cdr(), definitionEnv) == false) {
			status = $("back");
			return;
		}

		queue = new JunPrologList(clause.cdr(), queue);
		queueEnv = clauseEnv.cons_(queueEnv);
		clause = (JunPrologList) definition.car().cdr();
		clauseEnv = definitionEnv;
		cutBack = new JunPrologList(saveBackTrack, cutBack);
		status = $("loop");
	}

	/**
	 * Process the back status.
	 * 
	 * @category resolve modules
	 */
	protected void back() {
		if (backTrack == null) {
			status = $("fail");
			return;
		}

		Object[] array = (Object[]) ((JunPrologObject) backTrack.car()).object();
		backTrack = (JunPrologList) backTrack.cdr();
		clause = (JunPrologList) array[0];
		clauseEnv = (JunPrologList) array[1];
		queue = (JunPrologList) array[2];
		queueEnv = (JunPrologList) array[3];
		cutBack = (JunPrologList) array[4];
		definition = (JunPrologList) array[5];
		JunPrologList aList = (JunPrologList) array[6];
		this.unbindFrom_to_(unbindList, aList);
		unbindList = aList;
		status = $("next");
	}

	/**
	 * System predicate: nl
	 *
	 * @return boolean
	 * @category system predicates
	 */
	public boolean nl() {
		textCollector.println();
		return true;
	}

	/**
	 * System predicate: remove
	 *
	 * @return boolean
	 * @category system predicates
	 */
	public synchronized boolean remove() {
		Enumeration keys = userPredicates.keys();
		while (keys.hasMoreElements()) {
			Object aJunPrologSymbol = keys.nextElement();
			userPredicates.remove(aJunPrologSymbol);
			tracePredicates.remove(aJunPrologSymbol);
		}
		return true;
	}

	/**
	 * System predicate: write_
	 *
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @category system predicates
	 */
	public boolean write_(Object anObject) {
		if (anObject instanceof JunPrologEntity) {
			textCollector.print(((JunPrologEntity) anObject).printJunPrologString());
		} else {
			textCollector.print(anObject.toString());
		}
		return true;
	}

	/**
	 * Consult the stream and do the action.
	 *
	 * @param readStream jp.co.sra.smalltalk.StReadStream
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return boolean
	 * @category system support
	 */
	protected boolean consult_action_(StReadStream readStream, StBlockClosure aBlock) {
		Object[] condition = this.getCondition();
		try {

			JunPrologParser parser = new JunPrologParser();
			boolean result = true;
			Object node;
			while ((node = parser.parse_(readStream)) != $("eof")) {
				JunPrologList list = (JunPrologList) node;
				JunPrologClause hornClause = (JunPrologClause) list.cdr();
				if (list.car() == JunPrologBoolean.TRUE) {
					StringWriter writer = new StringWriter(20);
					hornClause.questionPrintJunPrologOn_(writer);
					textCollector.println(writer.toString());
					result = this.resolve_action_(hornClause, aBlock);
				} else {
					JunPrologEntity functor = hornClause.car().car();
					JunPrologDefinition program = (JunPrologDefinition) tracePredicates.get(functor);
					if (program == null) {
						program = (JunPrologDefinition) userPredicates.get(functor);
					}
					if (program == null) {
						program = JunPrologDefinition.FromList_(new JunPrologList(hornClause, null));
						userPredicates.put(functor, program);
					} else {
						program.nconc_(new JunPrologList(hornClause, null));
					}
				}
			}
			return result;

		} catch (IOException e) {
			return false;
		} finally {
			this.putCondition_(condition);
		}
	}

	/**
	 * System support for defining the system predicates.
	 *
	 * @return boolean
	 * @param readStream jp.co.sra.smalltalk.StReadStream
	 * @category system support
	 */
	protected boolean systemReconsult_(StReadStream readStream) {
		Object[] condition = this.getCondition();
		try {

			Hashtable newPredicates = new Hashtable();
			JunPrologParser parser = new JunPrologParser();
			Object node;
			while ((node = parser.parse_(readStream)) != $("eof")) {
				JunPrologList list = (JunPrologList) node;
				JunPrologClause hornClause = (JunPrologClause) list.cdr();
				if (list.car() == JunPrologBoolean.TRUE) {
					this.resolve_(hornClause);
				} else {
					JunPrologEntity functor = hornClause.car().car();
					JunPrologDefinition program = (JunPrologDefinition) newPredicates.get(functor);
					if (program == null) {
						program = JunPrologDefinition.FromList_(new JunPrologList(hornClause, null));
						newPredicates.put(functor, program);
					} else {
						program.nconc_(new JunPrologList(hornClause, null));
					}
					systemPredicates.put(functor, program);
				}
			}

			Enumeration keys = newPredicates.keys();
			while (keys.hasMoreElements()) {
				Object eachPredicate = keys.nextElement();
				systemPredicates.put(eachPredicate, newPredicates.get(eachPredicate));
			}

			return true;

		} finally {
			this.putCondition_(condition);
		}
	}

	/**
	 * Try to unify two JunPrologEntities and retrun true if succeeded.
	 *
	 * @param x jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param xEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @param y jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param yEnv jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @return boolean
	 * @category unification
	 */
	protected boolean unify_env_and_env_(JunPrologEntity x, JunPrologList xEnv, JunPrologEntity y, JunPrologList yEnv) {
		JunPrologEntity xx = x;
		JunPrologEntity yy = y;
		JunPrologEntity assoc;

		while (true) {
			if (xEnv == yEnv) {
				if (x.equals(y)) {
					return true;
				}
			}

			if (xx != null && xx.isJunPrologVariable()) {
				assoc = this.binding_env_(xx, xEnv);
				if (assoc != null) {
					valueEnv = xEnv;
					assoc = assoc.cdr();
					xx = this.fetchValue_env_(assoc.car(), (JunPrologList) assoc.cdr());
					return this.unify_env_and_env_(xx, valueEnv, yy, yEnv);
				} else {
					if (yy != null && yy.isJunPrologVariable()) {
						assoc = this.binding_env_(yy, yEnv);
						if (assoc != null) {
							valueEnv = yEnv;
							assoc = assoc.cdr();
							yy = this.fetchValue_env_(assoc.car(), (JunPrologList) assoc.cdr());
							return this.unify_env_and_env_(xx, xEnv, yy, valueEnv);
						}
					}
					this.bind_env_and_env_(xx, xEnv, yy, yEnv);
					return true;
				}
			}

			if (yy != null && yy.isJunPrologVariable()) {
				assoc = this.binding_env_(yy, yEnv);
				if (assoc != null) {
					valueEnv = yEnv;
					assoc = assoc.cdr();
					yy = this.fetchValue_env_(assoc.car(), (JunPrologList) assoc.cdr());
					return this.unify_env_and_env_(xx, xEnv, yy, valueEnv);
				}
				this.bind_env_and_env_(yy, yEnv, xx, xEnv);
				return true;
			}

			if (xx == null) {
				return (yy == null);
			} else if (xx.consp() == false) {
				return xx.equals(yy);
			}

			if (yy == null) {
				return (xx == null);
			} else if (yy.consp() == false) {
				return yy.equals(xx);
			}

			if (this.unify_env_and_env_(xx.car(), xEnv, yy.car(), yEnv) == false) {
				return false;
			}

			xx = xx.cdr();
			yy = yy.cdr();
		}
	}

	/**
	 * Menu message: do it.
	 *
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @category menu messages
	 */
	public boolean doIt(Object anObject) {
		JunPrologInterpreterView aView = (JunPrologInterpreterView) anObject;
		StReadStream stream = new StReadStream(aView.getSelectedText());
		return this.consult_action_(stream, null);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunPrologInterpreterViewAwt(this);
		} else {
			return new JunPrologInterpreterViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Prolog");
	}

	/**
	 * Deallocate the env values.
	 *
	 * @param env jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category private
	 */
	protected void deallocateEnv_(JunPrologEntity env) {
		JunPrologEntity assocList = env.cdr();
		while (assocList != null && assocList.consp()) {
			JunPrologEntity assoc = assocList.car();
			JunPrologEntity nextEnv = assoc.cdr().cdr();
			assoc.cdr().cdr_(null);
			if (nextEnv != null && nextEnv.consp()) {
				this.deallocateEnv_(nextEnv);
			}
			assocList = assocList.cdr();
		}
	}

	/**
	 * Answer the current condition as an array of objects.
	 *
	 * @return java.lang.Object[]
	 * @category private
	 */
	protected Object[] getCondition() {
		Object[] condition = new Object[14];
		condition[0] = status;
		condition[1] = definition;
		condition[2] = question;
		condition[3] = questionEnv;
		condition[4] = clause;
		condition[5] = clauseEnv;
		condition[6] = queue;
		condition[7] = queueEnv;
		condition[8] = valueEnv;
		condition[9] = unbindList;
		condition[10] = cutBack;
		condition[11] = backTrack;
		condition[12] = resolveAction;
		condition[13] = definitionStack;
		return condition;
	}

	/**
	 * Restore to the current condition.
	 *
	 * @param condition java.lang.Object[]
	 * @category private
	 */
	protected void putCondition_(Object[] condition) {
		status = (StSymbol) condition[0];
		definition = (JunPrologList) condition[1];
		question = (JunPrologClause) condition[2];
		questionEnv = (JunPrologList) condition[3];
		clause = (JunPrologClause) condition[4];
		clauseEnv = (JunPrologList) condition[5];
		queue = (JunPrologList) condition[6];
		queueEnv = (JunPrologList) condition[7];
		valueEnv = (JunPrologList) condition[8];
		unbindList = (JunPrologList) condition[9];
		cutBack = (JunPrologList) condition[10];
		backTrack = (JunPrologList) condition[11];
		resolveAction = (StBlockClosure) condition[12];
		definitionStack = (JunPrologList) condition[13];
	}

	/**
	 * Make the hashtable for the system predicates.
	 * 
	 * @category private
	 */
	protected void makeSystemPredicates() {
		if (SystemPredicates == null) {
			this.systemPredicatesNo0();
			this.systemPredicatesNo1();
			this.systemPredicatesNo2();
			this.systemPredicatesNo3();
			this.systemPredicatesNo4();
			this.systemPredicatesNo5();
			this.systemPredicatesNo6();
			this.systemPredicatesNo7();
			this.systemPredicatesNo8();
			this.systemPredicatesNo9();
			SystemPredicates = systemPredicates;
		} else {
			systemPredicates = SystemPredicates;
		}
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo0() {
		String string = "";
		string += "! :- builtin.";
		string += "true :- builtin.";
		string += "fail :- builtin.";
		string += "ver(X) :- builtin.";
		string += "send(X,Y,Z) :- builtin.";
		string += "send(X,Y,Z,A) :- builtin.";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo1() {
		String string = "";
		string += "repeat.";
		string += "repeat :- repeat.";
		string += "nonvar(X) :- var(X), !, fail.";
		string += "nonvar(X).";
		string += "integer(X) :- send(self,integer_,[X]).";
		string += "float(X) :- send(self,float_,[X]).";
		string += "double(X) :- send(self,double_,[X]).";
		// string += "fraction(X) :- send(self,fraction_,[X]).";
		string += "number(X) :- send(self,number_,[X]).";
		string += "symbol(X) :- send(self,symbol_,[X]).";
		string += "string(X) :- send(self,string:,[X]).";
		string += "list(X) :- send(self,list_,[X]).";
		string += "dotp(X) :- send(self,dotp_,[X]).";
		string += "atom(X) :- symbol(X).";
		string += "atom(X) :- nonvar(X), =(X,[]).";
		string += "atom(X) :- string(X).";
		string += "atomic(X) :- atom(X).";
		string += "atomic(X) :- number(X).";
		string += "structure(X) :- nonvar(X), not(atomic(X)).";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo2() {
		String string = "";
		string += "==(X,Y) :- send(X,=,[Y]).";
		string += "\\==(X,Y) :- ==(X,Y), !, fail.";
		string += "\\==(X,Y).";
		string += "=(X,X).";
		string += "\\=(X,Y) :- =(X,Y), !, fail.";
		string += "\\=(X,Y).";
		string += ">(X,Y) :- send(X,>,[Y]).";
		string += ">=(X,Y) :- send(X,>=,[Y]).";
		string += "<(X,Y) :- send(X,<,[Y]).";
		string += "=<(X,Y) :- send(X,<=,[Y]).";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo3() {
		String string = "";
		string += "+(X,Y,Z) :- send(X,+,[Y],Z).";
		string += "-(X,Y,Z) :- send(X,-,[Y],Z).";
		string += "*(X,Y,Z) :- send(X,*,[Y],Z).";
		string += "//(X,Y,Z) :- send(X,//,[Y],Z).";
		string += "/(X,Y,Z) :- send(X,/,[Y],Z).";
		string += "\\\\(X,Y,Z) :- send(X,\\\\,[Y],Z).";
		string += "is(Z,+(X,Y)) :- nonvar(Z), nonvar(X), nonvar(Y), send(X,+,[Y],Z).";
		string += "is(Z,+(X,Y)) :- var(Z), nonvar(X), nonvar(Y), send(X,+,[Y],Z).";
		string += "is(Z,+(X,Y)) :- var(X), nonvar(Y), nonvar(Z), send(Z,-,[Y],X).";
		string += "is(Z,+(X,Y)) :- var(Y), nonvar(Z), nonvar(X), send(Z,-,[X],Y).";
		string += "is(Z,-(X,Y)) :- nonvar(Z), nonvar(X), nonvar(Y), send(X,-,[Y],Z).";
		string += "is(Z,-(X,Y)) :- var(Z), nonvar(X), nonvar(Y), send(X,-,[Y],Z).";
		string += "is(Z,-(X,Y)) :- var(X), nonvar(Y), nonvar(Z), send(Z,+,[Y],X).";
		string += "is(Z,-(X,Y)) :- var(Y), nonvar(Z), nonvar(X), send(X,-,[Z],Y).";
		string += "is(Z,*(X,Y)) :- nonvar(Z), nonvar(X), nonvar(Y), send(X,*,[Y],Z).";
		string += "is(Z,*(X,Y)) :- var(Z), nonvar(X), nonvar(Y), send(X,*,[Y],Z).";
		string += "is(Z,*(X,Y)) :- var(X), nonvar(Y), nonvar(Z), send(Z,/,[Y],X).";
		string += "is(Z,*(X,Y)) :- var(Y), nonvar(Z), nonvar(X), send(Z,/,[X],Y).";
		string += "is(Z,/(X,Y)) :- nonvar(Z), nonvar(X), nonvar(Y), send(X,/,[Y],Z).";
		string += "is(Z,/(X,Y)) :- var(Z), nonvar(X), nonvar(Y), send(X,/,[Y],Z).";
		string += "is(Z,/(X,Y)) :- var(X), nonvar(Y), nonvar(Z), send(Z,*,[Y],X).";
		string += "is(Z,/(X,Y)) :- var(Y), nonvar(Z), nonvar(X), send(X,/,[Z],Y).";
		string += "is(X,Y) :- =(X,Y).";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo4() {
		String string = "";
		string += "listing :- send(self,userListing,[]).";
		string += "listing(X) :- send(self,userListing:,[X]).";
		string += "systemListing :- send(self,systemListing,[]).";
		string += "systemListing(X) :- send(self,systemListing:,[X]).";
		string += "editing :- send(self,editting,[]).";
		string += "consult :- send(self,consult,[]).";
		string += "consult(X) :- nonvar(X), send(self,consultFile:,[X]).";
		string += "reconsult :- send(self,reconsult,[]).";
		string += "reconsult(X) :- nonvar(X), send(self,reconsultFile:,[X]).";
		string += "saving :- send(self,saving,[]).";
		string += "saving(X) :- send(self,saving:,[X]).";
		string += "userPredicates(X) :- send(self,userPredicates,[],X).";
		string += "systemPredicates(X) :- send(self,systemPredicates,[],X).";
		string += "predicates([X|Y]) :- userPredicates(X), systemPredicates(Y).";
		string += "functor(T,F,A) :- nonvar(T), !, send(self,functorArityOf:,[T],[F|A]).";
		string += "functor(T,F,A) :- number(F), !, =(0,A),=(T,F).";
		string += "functor(T,F,A) :- atom(F), =<(0,A), '~addvar'([F],A,L), =..(T,L).";
		string += "'~addvar'(L,0,M) :- !, =(L,M).";
		string += "'~addvar'(L,NVars,M) :- -(NVars, 1, N), append(L,[FreeV],LV), '~addvar'(LV,N,M).";
		string += "arg(Nth,S,T) :- integer(Nth), <(0,Nth), structure(S), =..(S,[F|L]), nth(L,Nth,T).";
		string += "=..(X,Y) :- send(self,univ:,[['X'|'Y']]).";
		string += "name(X,Y) :- atomic(X), list(Y), !, send(self,symToList:,[X],Y).";
		string += "name(X,Y) :- atomic(X), !, send(self,symToStr:,[X],Y).";
		string += "name(X,Y) :- var(X), !, nonvar(Y), send(self, strToSym:,[Y],X).";
		string += "remove :- send(self,remove,[]).";
		string += "remove(X) :- send(self,remove:,[X]).";
		string += "clause(X) :- send(self,clauseSet:,[X]), repeat, send(self,clause:,[X],X).";
		string += "asserta(X) :- send(self,asserta:,[X]).";
		string += "assert(X) :- send(self,assertz:,[X]).";
		string += "assertz(X) :- send(self,assertz:,[X]).";
		string += "retract(X) :- repeat, send(self,retract:,[X],X).";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo5() {
		String string = "";
		string += "call(G) :- G.";
		string += "not(G) :- G, !, fail.";
		string += "not(G).";
		string += "or(X,Y) :- call(X).";
		string += "or(X,Y) :- call(Y).";
		string += "and(X,Y) :- call(X), call(Y).";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo6() {
		String string = "";
		string += "read(X) :- send(self,read,[],X).";
		string += "read(X,M) :- send(self,read_,[M],X).";
		string += "write(X) :- send(self,write_,[X]).";
		string += "nl :- send(self,nl,[]).";
		string += "tab(X) :- number(X), send(self,tab:,[X]).";
		string += "clear :- send(self,clear,[]).";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo7() {
		String string = "";
		// string += "clock(X) :- send({Time},millisecondClockValue,[],X).";
		string += "verbose(X) :- send(self,verbose:,[X]).";
		string += "gc :- send(self,gc,[]).";
		string += "inspect(X) :- send(X,inspect,[]).";
		string += "spy(X) :- send(self,spy:,[X]).";
		string += "nospy(X) :- send(self,nospy:,[X]).";
		string += "trace :- send(self,trace,[]).";
		string += "notrace :- send(self,notrace,[]).";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo8() {
		String string = "";
		string += "append([],X,X).";
		string += "append([A|X],Y,[A|Z]) :- append(X,Y,Z).";
		string += "member(X,[X|Y]).";
		string += "member(X,[Y|Z]) :- member(X,Z).";
		string += "reverse([],[]).";
		string += "reverse([H|T],L) :- reverse(T,Z), append(Z,[H],L).";
		string += "length(X,Y) :- send(self,length:,[X],Y).";
		string += "nth([X|Y],1,Z) :- !, =(X,Z).";
		string += "nth([X|Y],N,Z) :- -(N,1,PN), nth(Y,PN,Z).";
		string += "printlist(L) :- send(self,listPrint:,[L]).";
		string += "lispAppend(X,Y,Z) :- send(X,append:,[Y],Z).";
		string += "lispReverse(X,Y) :- send(X,reverse,[],Y).";
		string += "lispMember(X,Y) :- send(Y,member:,[X],A), \\=(A,[]).";
		string += "lispMember(X,Y,Z) :- send(Y,member:,[X],Z).";
		string += "lispAssoc(X,Y) :- send(Y,assoc:,[X],A), \\=(A,[]).";
		string += "lispAssoc(X,Y,Z) :- send(Y,assoc:,[X],Z).";
		string += "lispNconc(X,Y,Z) :- send(X,nconc:,[Y],Z).";

		this.systemReconsult_(new StReadStream(string));
	}

	/**
	 * Define some system predicates.
	 * 
	 * @category private
	 */
	private void systemPredicatesNo9() {
		String string = "";
		string += "% User System Predicates";

		this.systemReconsult_(new StReadStream(string));
	}

}