package jp.co.sra.jun.goodies.soroban;

import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;

import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.SystemResourceSupport;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.goodies.image.support.JunImageProcessor;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunSorobanModel class
 * 
 *  @author    m-asada
 *  @created   2006/03/13 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun674 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSorobanModel.java,v 8.16 2008/02/20 06:32:03 nisinaka Exp $
 */
public class JunSorobanModel extends JunApplicationModel implements ClipboardOwner {
	protected List byteArray;
	protected transient StPopupMenu _popupMenu;

	protected static String[] KomaPatterns;
	protected static StDisplayable[] ReedImages;
	protected static StDisplayable SanImageWithMark;
	protected static int KomaHeight;
	protected static int KushiHeight;

	static {
		FlushImages();
		KomaHeight = -1;
		KushiHeight = -1;
		KomaPatterns = null;
	}

	/**
	 * Answer the koma image height.
	 * 
	 * @return int
	 * @category Accessing
	 */
	public static int KomaHeight() {
		if (KomaHeight < 0) {
			KomaHeight = KomaImage().asImage().height();
		}
		return KomaHeight;
	}

	/**
	 * Answer the kushi image height.
	 * 
	 * @return int
	 * @category Accessing
	 */
	public static int KushiHeight() {
		if (KushiHeight < 0) {
			KushiHeight = KushiImage().asImage().height();
		}
		return KushiHeight;
	}

	/**
	 * Answer the mark image.
	 * 
	 * @return jp.co.sra.smalltalk.StDisplayable
	 * @category Accessing
	 */
	public static StDisplayable MarkImage() {
		if (SanImageWithMark == null) {
			InitializeImages();
		}
		return SanImageWithMark;
	}

	/**
	 * Answer the reed image height.
	 * 
	 * @return int
	 * @category Accessing
	 */
	public static int ReedHeight() {
		return ReedImages()[0].asImage().height();
	}

	/**
	 * Answer the reed images.
	 * 
	 * @return jp.co.sra.smalltalk.StDisplayable[]
	 * @category Accessing
	 */
	public static StDisplayable[] ReedImages() {
		if (ReedImages == null) {
			InitializeImages();
		}
		return ReedImages;
	}

	/**
	 * Answer the reed image width.
	 * 
	 * @return int
	 * @category Accessing
	 */
	public static int ReedWidth() {
		return ReedImages()[0].asImage().width();
	}

	/**
	 * Flush the receiver.
	 * 
	 * @category Class initialization
	 */
	protected static void FlushImages() {
		ReedImages = null;
		SanImageWithMark = null;
	}

	/**
	 * Initialize the receiver's images.
	 * 
	 * @category Class initialization
	 */
	protected static void InitializeImages() {
		StImage komaImage = KomaImage();
		StImage kushiImage = KushiImage();
		StImage sanImage = SanImage();
		SanImageWithMark = SanImageWithMark();
		ReedImages = new StDisplayable[10];
		String[] strings = JunSorobanModel.Patterns();
		for (int i = 0; i < strings.length; i++) {
			StImage rowImage = new StImage(24, 102, JunImageProcessor._WhiteBlackPalette());
			int offset = 0;
			String patterns = strings[i];
			for (int j = 0; j < patterns.length(); j++) {
				StImage image = (patterns.charAt(j) == '+') ? komaImage : ((patterns.charAt(j) == '-') ? kushiImage : sanImage);
				rowImage.copy_from_in_rule_(new Rectangle(0, offset, image.bounds().width, image.bounds().height), new Point(0, 0), image, StImage.Over);
				offset += image.height();
			}
			ReedImages[i] = rowImage;
		}
	}

	/**
	 * Answer the receiver's koma pattern strings.
	 * 
	 * @return java.lang.String[]
	 * @category Constants access
	 */
	public static String[] Patterns() {
		if (KomaPatterns == null) {
			KomaPatterns = new String[10];
			KomaPatterns[0] = "+-*-++++";
			KomaPatterns[1] = "+-*+-+++";
			KomaPatterns[2] = "+-*++-++";
			KomaPatterns[3] = "+-*+++-+";
			KomaPatterns[4] = "+-*++++-";
			KomaPatterns[5] = "-+*-++++";
			KomaPatterns[6] = "-+*+-+++";
			KomaPatterns[7] = "-+*++-++";
			KomaPatterns[8] = "-+*+++-+";
			KomaPatterns[9] = "-+*++++-";
		}
		return KomaPatterns;
	}

	/**
	 * Answer the koma image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Private
	 */
	protected static final StImage KomaImage() {
		return new StImage(SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/soroban/komaImage.gif"));
	}

	/**
	 * Answer the kushi image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Private
	 */
	protected static final StImage KushiImage() {
		return new StImage(SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/soroban/kushiImage.gif"));
	}

	/**
	 * Answer the kusi image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Private
	 */
	protected static final StImage KusiImage() {
		return new StImage(SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/soroban/kusiImage.gif"));
	}

	/**
	 * Answer the san image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Private
	 */
	protected static final StImage SanImage() {
		return new StImage(SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/soroban/sanImage.gif"));
	}

	/**
	 * Answer the san image with mark.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Private
	 */
	protected static final StImage SanImageWithMark() {
		return new StImage(SystemResourceSupport.createImage("/jp/co/sra/jun/goodies/soroban/sanImageWithMark.gif"));
	}

	/**
	 * Answer the receiver's position.
	 * 
	 * @return int
	 * @category Constants access
	 */
	public static int Position() {
		return 12;
	}

	/**
	 * Answer the receiver's reeds.
	 * 
	 * @return int
	 * @category Constants access
	 */
	public static int Reeds() {
		return 20;
	}

	/**
	 * Answer the unity value.
	 * 
	 * @return double
	 * @category Constants access
	 */
	public static double Unity() {
		return 1.00000000d;
	}

	/**
	 * Answer the zero value.
	 * 
	 * @return double
	 * @category Constants access
	 */
	public static double Zero() {
		return 0.00000000d;
	}

	/**
	 * Create a new instance of <code>JunSorobanModel</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunSorobanModel() {
		super();
	}

	/**
	 * Answer the receiver's byte array at the specified position.
	 * 
	 * @param index int
	 * @return double
	 * @category accessing 
	 */
	public int at_(int index) {
		return ((Number) byteArray.get(index)).intValue();
	}

	/**
	 * Set the specified pattern string value to the receiver's byte array at the specified position.
	 * 
	 * @param rowNumber int
	 * @param patternString java.lang.String
	 * @category accessing 
	 */
	public void at_pattern_(int rowNumber, String patternString) {
		int index = -1;
		String[] patterns = JunSorobanModel.Patterns();
		for (int i = 0; i < patterns.length; i++) {
			if (patterns[i].equals(patternString)) {
				index = i;
			}
		}
		byteArray.set(rowNumber, new Integer(index));
		this.changed_with_($("value"), new Integer(rowNumber));
		this.updateWindowTitle();
	}

	/**
	 * Set the specified value to the receiver's byte array at the specified position.
	 * 
	 * @param rowNumber int
	 * @param anInteger int
	 * @category accessing 
	 */
	public void at_put_(int rowNumber, int anInteger) {
		byteArray.set(rowNumber, new Integer(Math.max(0, Math.min(anInteger, 9))));
		this.changed_with_($("value"), new Integer(rowNumber));
		this.updateWindowTitle();
	}

	/**
	 * Answer the pattern string with the specified position.
	 * 
	 * @param rowNumber int
	 * @return java.lang.String
	 * @category accessing
	 */
	public String patternAt_(int rowNumber) {
		return new String(JunSorobanModel.Patterns()[((Number) byteArray.get(rowNumber)).intValue()]);
	}

	/**
	 * Notifies this object that it is no longer the owner of
	 * the contents of the clipboard.
	 *
	 * @param clipboard the clipboard that is no longer owned
	 * @param contents the contents which this owner had placed on the clipboard
	 * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard, java.awt.datatransfer.Transferable)
	 * @category clipboards
	 */
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
	}

	/**
	 * Answer the receiver's position.
	 * 
	 * @return int
	 * @category constants access
	 */
	public int position() {
		return JunSorobanModel.Position();
	}

	/**
	 * Answer the receiver's reeds.
	 * 
	 * @return int
	 * @category constants access
	 */
	public int reeds() {
		return JunSorobanModel.Reeds();
	}

	/**
	 * Answer the receiver's value as double.
	 * 
	 * @return double
	 * @category converting
	 */
	public double asDouble() {
		return (double) this.value();
	}

	/**
	 * Answer the receiver's value as float.
	 * 
	 * @return float
	 * @category converting
	 */
	public float asFloat() {
		return (float) this.value();
	}

	/**
	 * Answer the receiver's value as int.
	 * 
	 * @return int
	 * @category converting
	 */
	public int asInteger() {
		return (int) this.value();
	}

	/**
	 * Answer the receiver's value as Number.
	 * 
	 * @return java.lang.Number
	 * @category converting
	 */
	public Number asNumber() {
		return new Double(this.value());
	}

	/**
	 * Answer the receiver's value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double value() {
		double total = 0.0d;
		for (int index = 0; index < this.position(); index++) {
			double value = this.at_(index) * Math.pow(10, this.position() - (index + 1));
			total += value;
		}
		for (int index = this.position(); index < this.reeds(); index++) {
			double value = this.at_(index) * Math.pow(10, 0 - ((index + 1) - this.position()));
			total += value;
		}
		return total;
	}

	/**
	 * Set the receiver's value.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category accessing
	 */
	public JunSorobanModel value_(double aNumber) {
		String string = new DecimalFormat("000000000000.00000000").format(aNumber);
		string = string.replaceFirst("\\.", "");
		string = string.replaceFirst("-", "");
		for (int index = this.reeds() - 1; index >= 0; index--) {
			this.at_put_(index, Integer.parseInt(Character.toString(string.charAt(index))));
		}
		return this;
	}

	/**
	 * Set the receiver's value.
	 * 
	 * @param aSoroban jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category accessing
	 */
	public JunSorobanModel value_(JunSorobanModel aSoroban) {
		return this.value_(aSoroban.valueString());
	}

	/**
	 * Set the receiver's value.
	 * 
	 * @param aString java.lang.String
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category accessing
	 */
	public JunSorobanModel value_(String aString) {
		try {
			new DecimalFormat("000000000000.00000000").format(Double.parseDouble(aString));
		} catch (NumberFormatException e) {
			return this;
		}

		while (aString.startsWith("-")) {
			aString = aString.substring(1, aString.length());
		}
		if (aString.length() > this.reeds() + 1) {
			return this;
		}
		int decimalPointIndex = aString.indexOf('.');
		if (decimalPointIndex > this.position() || (aString.length() - decimalPointIndex - 1 > this.reeds() - this.position()) || (decimalPointIndex == -1 && aString.length() > this.position())) {
			return this;
		}
		String integral = null;
		String decimal = null;
		if (decimalPointIndex == -1) {
			integral = new DecimalFormat("000000000000").format(Long.parseLong(aString));
			decimal = new DecimalFormat(".00000000").format(0.0d);
		} else {
			integral = new DecimalFormat("000000000000").format(Long.parseLong(aString.substring(0, decimalPointIndex)));
			decimal = new DecimalFormat(".00000000").format(Double.parseDouble(aString.substring(decimalPointIndex, aString.length())));
		}

		String numberString = integral + decimal.substring(1, decimal.length());
		for (int index = this.reeds() - 1; index >= 0; index--) {
			this.at_put_(index, Integer.parseInt(Character.toString(numberString.charAt(index))));
		}
		return this;
	}

	/**
	 * Answer the receiver's value as String.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String valueString() {
		StringWriter aWriter = new StringWriter();
		for (int index = 0; index < this.position(); index++) {
			aWriter.write(Integer.toString(this.at_(index)));
		}
		aWriter.write(".");
		for (int index = this.position(); index < this.reeds(); index++) {
			aWriter.write(Integer.toString(this.at_(index)));
		}
		aWriter.flush();
		String aString = aWriter.toString();

		for (int i = 0; i < this.position() - 1; i++) {
			if (aString.startsWith("0")) {
				aString = aString.substring(1, aString.length());
			} else {
				break;
			}
		}
		while (true) {
			if (aString.endsWith("0")) {
				aString = aString.substring(0, aString.length() - 1);
			} else if (aString.endsWith(".")) {
				aString = aString.substring(0, aString.length() - 1);
				break;
			} else {
				break;
			}
		}
		return aString;
	}

	/**
	 * Answer the sum of the receiver and aNumber.
	 * This method corresponds to the methd '+' in Smalltalk.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category arithmetic
	 */
	public JunSorobanModel plus_(double aNumber) {
		return this.value_(this.value() + aNumber);
	}

	/**
	 * Answer the sum of the receiver and aNumber.
	 * This method corresponds to the methd '+' in Smalltalk.
	 * 
	 * @param aNumber jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category arithmetic
	 */
	public JunSorobanModel plus_(JunSorobanModel aNumber) {
		return this.plus_(aNumber.value());
	}

	/**
	 * Answer the difference between the receiver and aNumber.
	 * This method corresponds to the methd '-' in Smalltalk.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category arithmetic
	 */
	public JunSorobanModel minus_(double aNumber) {
		return this.value_(this.value() - aNumber);
	}

	/**
	 * Answer the difference between the receiver and aNumber.
	 * This method corresponds to the methd '-' in Smalltalk.
	 * 
	 * @param aNumber jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category arithmetic
	 */
	public JunSorobanModel minus_(JunSorobanModel aNumber) {
		return this.minus_(aNumber.value());
	}

	/**
	 * Answer the result of multiplying the receiver by aNumber.
	 * This method corresponds to the methd '*' in Smalltalk.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category arithmetic
	 */
	public JunSorobanModel multipliedBy_(double aNumber) {
		return this.value_(this.value() * aNumber);
	}

	/**
	 * Answer the result of multiplying the receiver by aNumber.
	 * This method corresponds to the methd '*' in Smalltalk.
	 * 
	 * @param aNumber jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category arithmetic
	 */
	public JunSorobanModel multipliedBy_(JunSorobanModel aNumber) {
		return this.multipliedBy_(aNumber.value());
	}

	/**
	 * Answer the result of dividing the receiver by aNumber.
	 * This method corresponds to the methd '/' in Smalltalk.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category arithmetic
	 */
	public JunSorobanModel dividedBy_(double aNumber) {
		return this.value_(this.value() / aNumber);
	}

	/**
	 * Answer the result of dividing the receiver by aNumber.
	 * This method corresponds to the methd '/' in Smalltalk.
	 * 
	 * @param aNumber jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @return jp.co.sra.jun.goodies.soroban.JunSorobanModel
	 * @category arithmetic
	 */
	public JunSorobanModel dividedBy_(JunSorobanModel aNumber) {
		return this.dividedBy_(aNumber.value());
	}

	/**
	 * Answer the absolute number.
	 * 
	 * @return double
	 * @category arithmetic
	 */
	public double abs() {
		return this.negative() ? Math.abs(this.value()) : this.value();
	}

	/**
	 * Answer the negation of the receiver.
	 * 
	 * @return double
	 * @category arithmetic
	 */
	public double negated() {
		return JunSorobanModel.Zero() - this.value();
	}

	/**
	 * Answer 1 divided by the receiver.
	 * Provide an error notification if the receiver is 0.
	 * 
	 * @return double
	 * @category arithmetic
	 */
	public double reciprocal() {
		return JunSorobanModel.Unity() / this.value();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		byteArray = new ArrayList();
		for (int i = 0; i < JunSorobanModel.Reeds(); i++) {
			byteArray.add(new Integer(0));
		}
		_popupMenu = null;
	}

	/**
	 * Answer the default view of the model.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunSorobanViewAwt(this);
		} else {
			return new JunSorobanViewSwing(this);
		}
	}

	/**
	 * Answer the receiver's window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return this.printString();
	}

	/**
	 * Copy the receiver's value to clipboard.
	 * 
	 * @category menu messages
	 */
	public void copyValue() {
		Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
		StringSelection contents = new StringSelection(this.valueString());
		clipboard.setContents(contents, this);
	}

	/**
	 * Paset the receiver's value from clipboard.
	 * 
	 * @category menu messages
	 */
	public void pasteValue() {
		Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
		Transferable contents = clipboard.getContents(this);
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new StringReader((String) contents.getTransferData(DataFlavor.stringFlavor)));
			String line = reader.readLine();
			if (line == null || line.length() == 0) {
				return;
			}
			this.value_(line);
		} catch (IOException e) {
		} catch (UnsupportedFlavorException e) {
		} finally {
			if (reader != null) {
				try {
					reader.close();
					reader = null;
				} catch (IOException e) {
				}
			}
		}
	}

	/**
	 * Reset the receiver's value.
	 * 
	 * @category menu messages
	 */
	public void resetValue() {
		for (int i = 0; i < this.reeds(); i++) {
			this.at_put_(i, 5);
		}
		try {
			Thread.sleep(150);
		} catch (InterruptedException e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		}

		for (int i = 0; i < this.reeds(); i++) {
			this.at_put_(i, 0);
			try {
				Thread.sleep(15);
			} catch (InterruptedException e) {
				System.err.println(e.getMessage());
				e.printStackTrace();
			}
		}
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write($String("Soroban"));
		aWriter.write(" [");
		for (int index = 0; index < this.position(); index++) {
			aWriter.write(Integer.toString(this.at_(index)));
		}
		aWriter.write(".");
		for (int index = this.position(); index < this.reeds(); index++) {
			aWriter.write(Integer.toString(this.at_(index)));
		}
		aWriter.write("]");
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			_popupMenu = new StPopupMenu();
			_popupMenu.add(new StMenuItem($String("Gohasan"), $("resetValue"), new MenuPerformer(this, "resetValue")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem($String("Copy"), $("copyValue"), new MenuPerformer(this, "copyValue")));
			_popupMenu.add(new StMenuItem($String("Paste"), $("pasteValue"), new MenuPerformer(this, "pasteValue")));
		}
		return _popupMenu;
	}

	/**
	 * Answer true if the receiver is ZERO, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isZero() {
		return this.value() == JunSorobanModel.Zero();
	}

	/**
	 * Answer true if the receiver is negative, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean negative() {
		return this.value() < JunSorobanModel.Zero();
	}

	/**
	 * Answer true if the receiver is positive, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean positive() {
		return this.value() >= JunSorobanModel.Zero();
	}
}
