package jp.co.sra.jun.goodies.spirodesign;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.smalltalk.StColorValue;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;

/**
 * JunSpiroPen class
 * 
 *  @author    m-asada
 *  @created   2006/03/28 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun676 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSpiroPen.java,v 8.11 2008/02/20 06:32:03 nisinaka Exp $
 */
public class JunSpiroPen extends JunSpiroObject {
	protected Collection penPoints;
	protected boolean isRainbow;

	public static int DefaultTickAngle = 6;

	/**
	 * Create a new instance of <code>JunSpiroPen</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunSpiroPen() {
		super();
	}

	/**
	 * Create a new instance of <code>JunSpiroPen</code> and initialize it.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public JunSpiroPen(Jun2dPoint aPoint) {
		super();
		this.point_(aPoint);
	}

	/**
	 * Create a new instance of <code>JunSpiroPen</code> and initialize it.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aColor java.awt.Color
	 * @param aWidth int
	 * @category Instance creation
	 */
	public JunSpiroPen(Jun2dPoint aPoint, Color aColor, int aWidth) {
		super();
		this.point_(aPoint);
		this.color_(aColor);
		this.width_(aWidth);
	}

	/**
	 * Create a new instance of <code>JunSpiroPen</code> and initialize it.
	 * 
	 * @param pointCollection java.util.Collection
	 * @category Instance creation
	 */
	public JunSpiroPen(Collection pointCollection) {
		super();
		this.points_(pointCollection);
	}

	/**
	 * Create a new instance of <code>JunSpiroPen</code> and initialize it.
	 * 
	 * @param pointCollection java.util.Collection
	 * @param aColor java.awt.Color
	 * @param aWidth int
	 * @category Instance creation
	 */
	public JunSpiroPen(Collection pointCollection, Color aColor, int aWidth) {
		super();
		this.points_(pointCollection);
		this.color_(aColor);
		this.width_(aWidth);
	}

	/**
	 * Initialize the receiver when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		penPoints = null;
		isRainbow = false;
	}

	/**
	 * Answer the receiver's mark.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#mark()
	 * @category accessing
	 */
	public Rectangle mark() {
		Rectangle mark = new Rectangle(0, 0, Math.max(this.defaultMarkExtent().width, this.width()), Math.max(this.defaultMarkExtent().height, this.width()));
		mark.translate((int) Math.round(this.point().x() - mark.getCenterX()), (int) Math.round(this.point().y() - mark.getCenterY()));
		return mark;
	}

	/**
	 * Answer the receiver's marks.
	 * 
	 * @return java.awt.Rectangle[]
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#marks()
	 * @category accessing
	 */
	public Rectangle[] marks() {
		return new Rectangle[] { this.mark() };
	}

	/**
	 * Anser the receiver's point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint point() {
		return this.isEmpty() ? new Jun2dPoint(0, 0) : this.points()[this.points().length - 1];
	}

	/**
	 * Set the receiver's point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void point_(Jun2dPoint aPoint) {
		Collection newPoints = new ArrayList(1);
		newPoints.add(aPoint);
		this.points_(newPoints);
	}

	/**
	 * Answer the receiver's pen points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#points()
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		if (penPoints == null) {
			penPoints = new ArrayList();
		}
		return (Jun2dPoint[]) penPoints.toArray(new Jun2dPoint[penPoints.size()]);
	}

	/**
	 * Set the receiver's pen points.
	 * 
	 * @param aCollection java.util.Collection
	 * @category accessing
	 */
	public void points_(Collection aCollection) {
		penPoints = aCollection;
	}

	/**
	 * Answer true if the receiver's color is rainbow, otherwise false.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean rainbow() {
		return isRainbow;
	}

	/**
	 * Set the receiver's rainbow color.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void rainbow_(boolean aBoolean) {
		isRainbow = aBoolean;
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#boundingBox()
	 * @category bounds accessing
	 */
	public Rectangle boundingBox() {
		if (this.isEmpty()) {
			return super.boundingBox();
		} else {
			Jun2dPoint[] points = this.points();
			Rectangle box = new Rectangle(points[0]._toPoint(), new Dimension(0, 0));
			for (int i = 1; i < points.length; i++) {
				box.add(points[i]._toPoint());
			}
			return box;
		}
	}

	/**
	 * Answer the default color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#defaultColor()
	 * @category defaults
	 */
	public Color defaultColor() {
		return Color.getHSBColor(1f / 12f, 3f / 4f, 9f / 10f);
	}

	/**
	 * Answer the default mark extent.
	 * 
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#defaultMarkExtent()
	 * @category defaults
	 */
	public Dimension defaultMarkExtent() {
		return new Dimension(7, 7);
	}

	/**
	 * Answer the default tick angle.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultTickAngle() {
		return DefaultTickAngle;
	}

	/**
	 * Answer the defalut rainbow number.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultReinbowNumber() {
		return Math.round(360f / Math.max(this.defaultTickAngle(), 1f) / 3f);
	}

	/**
	 * Answer the default width.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#defaultWidth()
	 * @category defaults
	 */
	public int defaultWidth() {
		return 2;
	}

	/**
	 * Display the receiver with marks on the graphics at the specified point.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @param displayPoint java.awt.Point
	 * @param aBoolean boolean
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#displayOn_at_with_Marks_(java.awt.Graphics, java.awt.Point, boolean)
	 * @category displaying
	 */
	public void displayOn_at_with_Marks_(Graphics graphicsContext, Point displayPoint, boolean aBoolean) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();
		try {
			if (aBoolean) {
				aGraphics.translate(displayPoint.x, displayPoint.y);
				try {
					Rectangle[] marks = this.marks();
					for (int i = 0; i < marks.length; i++) {
						Rectangle box = marks[i];
						aGraphics.setColor(Color.black);
						aGraphics.fillRect(box.x, box.y, box.width, box.height);
						aGraphics.setColor(Color.white);
						Rectangle insetBox = new Rectangle(box);
						insetBox.grow(-1, -1);
						aGraphics.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
						aGraphics.setColor(this.color());
						insetBox = new Rectangle(box);
						insetBox.grow(-2, -2);
						aGraphics.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
					}
				} finally {
					aGraphics.translate(-displayPoint.x, -displayPoint.y);
				}
			} else {
				if (this.isEmpty() == false) {
					this.applyAttributesToGraphicsContext_(aGraphics);
					if (this.isRainbow()) {
						Jun2dPoint[] points = this.points();
						for (int i = 0; i < points.length - 1; i++) {
							aGraphics.setColor(Color.getHSBColor((i % this.defaultReinbowNumber()) / (float) this.defaultReinbowNumber(), StColorValue._GetSaturation(this.color()), StColorValue._GetBrightness(this.color())));
							aGraphics.drawLine((int) Math.round(points[i].x() + displayPoint.x), (int) Math.round(points[i].y() + displayPoint.y), (int) Math.round(points[i + 1].x() + displayPoint.x), (int) Math.round(points[i + 1].y() + displayPoint.y));
						}
					} else {
						Jun2dPoint[] points = this.points();
						int[] xPoints = new int[points.length];
						int[] yPoints = new int[points.length];
						for (int i = 0; i < points.length; i++) {
							xPoints[i] = (int) Math.round(points[i].x()) + displayPoint.x;
							yPoints[i] = (int) Math.round(points[i].y()) + displayPoint.y;
						}
						aGraphics.drawPolyline(xPoints, yPoints, xPoints.length);
					}
				}
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
	}

	/**
	 * Anser true if the receiver is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.points().length == 0;
	}

	/**
	 * Answer true if the receiver's color is rainbow, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isRainbow() {
		return this.rainbow();
	}
}
