package jp.co.sra.jun.goodies.track;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunAbstractController;

/**
 * JunTrackSliderController class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/11/19 (by MATSUDA Ryouichi)
 *  @updated   1999/12/10 (by MATSUDA Ryouichi)
 *  @updated   2002/10/28 (by nisinaka)
 *  @updated   2003/03/24 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun473 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTrackSliderController.java,v 8.10 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunTrackSliderController extends JunAbstractController implements MouseListener, MouseMotionListener {
	protected StSymbol _tracking;
	protected Cursor _savedCursor;
	protected Point _previousPoint;
	protected Point _offsetPoint;
	protected boolean _redisplayBoolean;
	protected Graphics _graphicsContext;
	protected double _savedValue;

	/**
	 * add listeners.
	 * 
	 * @param newView sra.mvctest.StView
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category initialize-release
	 */
	protected void buildListener(StView newView) {
		final Component aView = (Component) newView;
		aView.addMouseListener(this);
		aView.addMouseMotionListener(this);
	}

	/**
	 * Answer my model as a JunTrackSliderModel.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @category model accessing
	 */
	public JunTrackSliderModel getTrackSliderModel() {
		return (JunTrackSliderModel) this.model();
	}

	/**
	 * Invoked when a mouse button has been pressed on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent e) {
		if (e.isPopupTrigger()) {
			super.mousePressed(e);
			return;
		}

		if (e.isMetaDown()) {
			return;
		}

		Point startPoint = e.getPoint();
		_tracking = this._trackingState(startPoint);
		if (_tracking == $("slider")) {
			if (this.getTrackSliderView().sliderRectangle().containsPoint_(startPoint)) {
				_offsetPoint = this.getTrackSliderView().sliderRectangle().center();
				_offsetPoint = new Point(_offsetPoint.x - startPoint.x, _offsetPoint.y - startPoint.y);
			} else {
				_offsetPoint = new Point(0, 0);
			}
			_previousPoint = null;
			_redisplayBoolean = false;
			this._trackSlider(startPoint);
		} else if (_tracking == $("firstMarker")) {
			if (this.getTrackSliderView().firstMarkerRectangle().containsPoint_(startPoint)) {
				_offsetPoint = this.getTrackSliderView().firstMarkerRectangle().rightCenter();
				_offsetPoint = new Point(_offsetPoint.x - startPoint.x, _offsetPoint.y - startPoint.y);
			} else {
				_offsetPoint = new Point(0, 0);
			}
			this.getTrackSliderView().notDisplaySlider_(true);
			_savedValue = this.getTrackSliderModel().doubleValue();
			_previousPoint = null;
			_redisplayBoolean = false;
			this._trackFirstMarker(startPoint);
		} else if (_tracking == $("lastMarker")) {
			if (this.getTrackSliderView().lastMarkerRectangle().containsPoint_(startPoint)) {
				_offsetPoint = this.getTrackSliderView().lastMarkerRectangle().leftCenter();
				_offsetPoint = new Point(_offsetPoint.x - startPoint.x, _offsetPoint.y - startPoint.y);
			} else {
				_offsetPoint = new Point(0, 0);
			}
			this.getTrackSliderView().notDisplaySlider_(true);
			_savedValue = this.getTrackSliderModel().doubleValue();
			_previousPoint = null;
			_redisplayBoolean = false;
			this._trackLastMarker(startPoint);
		} else if (_tracking == $("interval")) {
			if (this.getTrackSliderView().anotherIntervalRectangle().containsPoint_(startPoint)) {
				_offsetPoint = this.getTrackSliderView().anotherIntervalRectangle().center();
				_offsetPoint = new Point(_offsetPoint.x - startPoint.x, _offsetPoint.y - startPoint.y);
			} else {
				_offsetPoint = new Point(0, 0);
			}
			this.getTrackSliderView().notDisplaySlider_(true);
			_savedValue = this.getTrackSliderModel().doubleValue();
			_previousPoint = null;
			_redisplayBoolean = false;
			this._trackInterval(startPoint);
		}
	}

	/**
	 * Invoked when a mouse button has been released on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent e) {
		if (e.isPopupTrigger()) {
			super.mouseReleased(e);
			return;
		}

		if (_tracking == $("slider")) {
			if (_redisplayBoolean) {
				this.getTrackSliderView().notDisplaySlider_(false); // For Swing.
				this.getTrackSliderView().displayOn_(this._graphicsContext());
			}
		} else if (_tracking == $("firstMarker")) {
			this.getTrackSliderView().notDisplaySlider_(false);
			this.getTrackSliderView().displayOn_(this._graphicsContext());
		} else if (_tracking == $("lastMarker")) {
			this.getTrackSliderView().notDisplaySlider_(false);
			this.getTrackSliderView().displayOn_(this._graphicsContext());
		} else if (_tracking == $("interval")) {
			this.getTrackSliderView().notDisplaySlider_(false);
			this.getTrackSliderView().displayOn_(this._graphicsContext());
		}

		_tracking = null;
		_offsetPoint = null;
		_previousPoint = null;
		_redisplayBoolean = false;
		if (_graphicsContext != null) {
			_graphicsContext.dispose();
			_graphicsContext = null;
		}
	}

	/**
	 * Invoked when a mouse button is pressed on a component and then dragged.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent e) {
		if (_tracking == $("slider")) {
			this._trackSlider(e.getPoint());
		} else if (_tracking == $("firstMarker")) {
			this._trackFirstMarker(e.getPoint());
		} else if (_tracking == $("lastMarker")) {
			this._trackLastMarker(e.getPoint());
		} else if (_tracking == $("interval")) {
			this._trackInterval(e.getPoint());
		}
	}

	/**
	 * Invoked when the mouse button has been moved on a component (with no buttons no down).
	 *
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseMoved(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseMoved(MouseEvent e) {
		StSymbol symbol = this._trackingState(e.getPoint());
		if (symbol == $("slider")) {
			this.getComponent().setCursor(JunCursors.CrossCursor());
		} else if (symbol == $("firstMarker")) {
			this.getComponent().setCursor(JunCursors.Corner3Cursor());
		} else if (symbol == $("lastMarker")) {
			this.getComponent().setCursor(JunCursors.Corner1Cursor());
		} else if (symbol == $("interval")) {
			this.getComponent().setCursor(JunCursors.BetweenCursor());
		} else {
			this.getComponent().setCursor(Cursor.getDefaultCursor());
		}
	}

	/**
	 * Do the first marker tracking.
	 * 
	 * @param aPoint java.awt.Point
	 * @category tracking
	 */
	protected void _trackFirstMarker(Point aPoint) {
		if ((new StRectangle(this.getComponent().getSize())).expandedBy_(16, 16).containsPoint_(aPoint)) {
			if (aPoint.equals(_previousPoint) == false) {
				int x = aPoint.x + _offsetPoint.x;
				int y = aPoint.y + _offsetPoint.y;
				this.getTrackSliderView().setModelFirstMarker_(x, y);
				this.getTrackSliderView().displaySliderForValue_on_withColor_(_savedValue, this._graphicsContext(), Color.blue);
				_previousPoint = aPoint;
			}
			_redisplayBoolean = true;
		}
	}

	/**
	 * Answer the tracking state at the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category tracking
	 */
	protected StSymbol _trackingState(Point aPoint) {
		StSymbol symbol = null;
		if (this.getTrackSliderView().sliderArea().containsPoint_(aPoint)) {
			symbol = $("slider");
		}
		if (this.getTrackSliderView().sliderRectangle().containsPoint_(aPoint)) {
			symbol = $("slider");
		}
		if ((this.getTrackSliderModel().parentTracker() == null) && this.getTrackSliderModel().areMarkersActive()) {
			if (this.getTrackSliderView().firstMarkerRectangle().expandedBy_(new StRectangle(1, -2, 2, 2)).containsPoint_(aPoint)) {
				symbol = $("firstMarker");
			}
			if (this.getTrackSliderView().lastMarkerRectangle().expandedBy_(new StRectangle(1, -2, 2, 2)).containsPoint_(aPoint)) {
				symbol = $("lastMarker");
			}
			if (this.getTrackSliderView().anotherIntervalRectangle().containsPoint_(aPoint)) {
				symbol = $("interval");
			}
		}
		return symbol;
	}

	/**
	 * Do the interval tracking.
	 * 
	 * @param aPoint java.awt.Point
	 * @category tracking
	 */
	protected void _trackInterval(Point aPoint) {
		if ((new StRectangle(this.getComponent().getSize())).expandedBy_(16, 16).containsPoint_(aPoint)) {
			if (aPoint.equals(_previousPoint) == false) {
				int x = aPoint.x + _offsetPoint.x;
				int y = aPoint.y + _offsetPoint.y;
				this.getTrackSliderView().setModelInterval_(x, y);
				this.getTrackSliderView().displaySliderForValue_on_withColor_(_savedValue, this._graphicsContext(), Color.green);
				_previousPoint = aPoint;
			}
			_redisplayBoolean = true;
		}
	}

	/**
	 * Do the last marker tracking.
	 * 
	 * @param aPoint java.awt.Point
	 * @category tracking
	 */
	protected void _trackLastMarker(Point aPoint) {
		if ((new StRectangle(this.getComponent().getSize())).expandedBy_(16, 16).containsPoint_(aPoint)) {
			if (aPoint.equals(_previousPoint) == false) {
				int x = aPoint.x + _offsetPoint.x;
				int y = aPoint.y + _offsetPoint.y;
				this.getTrackSliderView().setModelLastMarker_(x, y);
				this.getTrackSliderView().displaySliderForValue_on_withColor_(_savedValue, this._graphicsContext(), Color.red);
				_previousPoint = aPoint;
			}
			_redisplayBoolean = true;
		}
	}

	/**
	 * Do the slider tracking.
	 * 
	 * @param aPoint java.awt.Point
	 * @category tracking
	 */
	protected void _trackSlider(Point aPoint) {
		if ((new StRectangle(this.getComponent().getSize())).expandedBy_(16, 16).containsPoint_(aPoint)) {
			if (aPoint.equals(_previousPoint) == false) {
				this.getTrackSliderView().setModelValue_(aPoint.x + _offsetPoint.x, aPoint.y + _offsetPoint.y);
				this.getTrackSliderView().displaySliderOn_withColor_(this._graphicsContext(), JunTrackSliderView.Gray40);
				_previousPoint = aPoint;
			}
			_redisplayBoolean = true;
		}
	}

	/**
	 * Method _graphicsContext.
	 * 
	 * @return java.awt.Graphics
	 * @category view accessing
	 */
	protected Graphics _graphicsContext() {
		if (_graphicsContext == null) {
			_graphicsContext = this.getComponent().getGraphics();
		}
		return _graphicsContext;
	}

	/**
	 * Answer my view as a Component.
	 * 
	 * @return java.awt.Component
	 * @category view accessing
	 */
	protected Component getComponent() {
		return (Component) this.view();
	}

	/**
	 * Answer my view as a JunTrackSliderView.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackSliderView
	 * @category view accessing
	 */
	public JunTrackSliderView getTrackSliderView() {
		return (JunTrackSliderView) this.view();
	}

	/**
	 * Answer my view as a JunTrackSliderView.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackSliderView
	 * @deprecated since Jun379, use getTrackSliderView()
	 * @category view accessing
	 */
	public JunTrackSliderView getView() {
		return (JunTrackSliderView) this.view();
	}
}
