package jp.co.sra.jun.octree.editor;

import java.awt.Color;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Window;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.StringTokenizer;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.surfaces.JunSphereSurface;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.metaball.abstracts.JunMetaball;
import jp.co.sra.jun.metaball.atoms.JunMetaSphere;
import jp.co.sra.jun.metaball.solid.JunMetaballSolid;
import jp.co.sra.jun.octree.basic.JunOctree;
import jp.co.sra.jun.octree.nodes.JunOctreeNode;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.lights.JunOpenGLLight;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.picking.JunOpenGLObjectPicker;
import jp.co.sra.jun.opengl.picking.JunOpenGLPickingRenderingContext;
import jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjector;
import jp.co.sra.jun.opengl.support.JunOpenGLDrawable;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderer;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOctreeEditModel class
 * 
 *  @author    nisinaka
 *  @created   1999/12/28 (by nisinaka)
 *  @updated   2005/03/02 (by nisinaka)
 *  @updated   2007/08/24 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun525 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOctreeEditModel.java,v 8.12 2008/02/20 06:32:17 nisinaka Exp $
 */
public class JunOctreeEditModel extends JunOpenGLDisplayModel {

	public static final double ThetaAccuracy = 0.001;

	/** The octrees to be edited. */
	protected JunOctree[] octrees = null;

	/** Selected octree nodes. */
	protected Vector selection = null;

	protected JunOpenGLDisplayLight[] displayLights = null;
	protected JunOpenGLProjector displayProjector = null;
	protected Jun3dPoint defaultEyePoint = null;
	protected Jun3dPoint defaultSightPoint = null;
	protected Jun3dPoint defaultUpVector = null;
	protected Double defaultViewFactor = null;

	protected JunButtonModel pickButton = null;
	protected JunButtonModel grabButton = null;
	protected JunButtonModel dragButton = null;
	protected JunButtonModel dollyButton = null;
	protected JunButtonModel focusButton = null;

	/**
	 * Create a new instance of <code>JunOctreeEditModel</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOctreeEditModel() {
		super();
		initialize();
	}

	/**
	 * Create a new instance of <code>JunOctreeEditModel</code> and initialize it.
	 * 
	 * @param aJunOctree jp.co.sra.jun.octree.basic.JunOctree
	 * @category Instance creation
	 */
	public JunOctreeEditModel(JunOctree aJunOctree) {
		this();

		Vector octrees = new Vector(1);
		octrees.addElement(aJunOctree);
		this.setOctrees_(octrees);
	}

	/**
	 * Create a new instance of JunOctreeEditModel with the specified JunOctree and open it.
	 * 
	 * @param aJunOctree jp.co.sra.jun.octree.basic.JunOctree
	 * @return jp.co.sra.jun.octree.editor.JunOctreeEditModel
	 * @category Instance creation
	 */
	public static JunOctreeEditModel Edit_(JunOctree aJunOctree) {
		JunOctreeEditModel editModel = new JunOctreeEditModel(aJunOctree);
		editModel.open();

		return editModel;
	}

	/**
	 * Create a new instance of JunOctreeEditModel with the specified JunOctree.
	 * 
	 * @param aJunOctree jp.co.sra.jun.octree.basic.JunOctree
	 * @return jp.co.sra.jun.octree.editor.JunOctreeEditModel
	 * @category Instance creation
	 */
	public static JunOctreeEditModel On_(JunOctree aJunOctree) {
		return new JunOctreeEditModel(aJunOctree);
	}

	/**
	 * Add a selected node.
	 * 
	 * @param aJunOctreeNode jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @category selection
	 */
	public void addSelection_(JunOctreeNode aJunOctreeNode) {
		if (selection.contains(aJunOctreeNode) == false) {
			selection.addElement(aJunOctreeNode);
		}
	}

	/**
	 * Called when a node is picked with the alt key down.
	 * 
	 * @param pickedNode jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @category manipulating
	 */
	public void altClick_(JunOctreeNode pickedNode) {
		// do nothing.
	}

	/**
	 * Answer the bounding box of the octrees.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#boundingBox()
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		if (this.hasOctree() == false) {
			return Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 0, 0));
		} else {
			Jun3dBoundingBox bounds = octrees[0].bounds();

			for (int i = 1; i < octrees.length; i++) {
				bounds = (Jun3dBoundingBox) bounds.merge_(octrees[i].bounds());
			}

			return bounds;
		}
	}

	/**
	 * Answer the current button status.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#buttonState()
	 * @category buttons
	 */
	public StSymbol buttonState() {
		if (pickButton.value() == true) {
			return $("pick");
		} else if (grabButton.value() == true) {
			return $("grab");
		} else if (dragButton.value() == true) {
			return $("drag");
		} else if (focusButton.value() == true) {
			return $("focus");
		}

		return $("none");
	}

	/**
	 * Change the eye point.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeEyePoint()
	 * @category menu messages
	 */
	public void changeEyePoint() {
		Jun3dPoint point = this.displayProjector().eyePoint();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String textValue = JunDialog.Request_(JunSystem.$String("Input a new eye point.") + " (x, y, z).", "(" + x + "," + y + "," + z + ")");

		if (textValue != null) {
			try {
				StringTokenizer st = new StringTokenizer(textValue, "(), ");
				x = Float.valueOf(st.nextToken()).floatValue();
				y = Float.valueOf(st.nextToken()).floatValue();
				z = Float.valueOf(st.nextToken()).floatValue();
				this.eyePoint_(new Jun3dPoint(x, y, z));
			} catch (Exception e) {
				JunDialog.Warn_(textValue + JunSystem.$String(" is invalid point."));
			}
		}
	}

	/**
	 * Menu message: Change the sight point.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeSightPoint()
	 * @category menu messages
	 */
	public void changeSightPoint() {
		Jun3dPoint point = this.displayProjector().sightPoint();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String textValue = JunDialog.Request_(JunSystem.$String("Input a new sight point.") + " (x, y, z).", "(" + x + "," + y + "," + z + ")");

		if (textValue != null) {
			try {
				StringTokenizer st = new StringTokenizer(textValue, "(), ");
				x = Float.valueOf(st.nextToken()).floatValue();
				y = Float.valueOf(st.nextToken()).floatValue();
				z = Float.valueOf(st.nextToken()).floatValue();
				this.sightPoint_(new Jun3dPoint(x, y, z));
			} catch (Exception e) {
				JunDialog.Warn_(textValue + JunSystem.$String(" is invalid point."));
			}
		}
	}

	/**
	 * Menu message: Change the up vector.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeUpVector()
	 * @category menu messages
	 */
	public void changeUpVector() {
		Jun3dPoint point = this.displayProjector().upVector();
		float x = (float) point.x();
		float y = (float) point.y();
		float z = (float) point.z();
		String textValue = JunDialog.Request_(JunSystem.$String("Input a new up vector.") + " (x, y, z).", "(" + x + "," + y + "," + z + ")");

		if (textValue != null) {
			try {
				StringTokenizer st = new StringTokenizer(textValue, "(), ");
				x = Float.valueOf(st.nextToken()).floatValue();
				y = Float.valueOf(st.nextToken()).floatValue();
				z = Float.valueOf(st.nextToken()).floatValue();
				this.upVector_(new Jun3dPoint(x, y, z));
			} catch (Exception e) {
				JunDialog.Warn_(textValue + JunSystem.$String(" is invalid point."));
			}
		}
	}

	/**
	 * Menu message: Change the view factor.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#changeViewFactor()
	 * @category menu messages
	 */
	public void changeViewFactor() {
		double value = this.displayProjector().viewFactor();
		String textValue = JunDialog.Request_(JunSystem.$String("Input a new view factor."), (new Double(value)).toString());

		if (textValue != null) {
			try {
				value = Double.valueOf(textValue).doubleValue();
				this.viewFactor_(value);
			} catch (Exception e) {
				JunDialog.Warn_(textValue + JunSystem.$String(" is invalid value."));
			}
		}
	}

	/**
	 * Clear the selected nodes.
	 * 
	 * @category selection
	 */
	public void clearSelection() {
		selection = new Vector();
	}

	/**
	 * Menu message: Close up.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#closeUp()
	 * @category menu messages
	 */
	public void closeUp() {
		JunOpenGLProjection aProjection = this.displayProjection();
		double aRadius = aProjection.regularHeight() / 2;
		double aDistance = aProjection.sightPoint().distance_(aProjection.eyePoint());
		Jun3dLine aLine = new Jun3dLine(aProjection.sightPoint(), aProjection.eyePoint());
		Jun3dPoint aPoint = (Jun3dPoint) aLine.atT_(aRadius / aDistance);
		aProjection.eyePoint_(aPoint);
		aProjection.zoomHeight_(this.computeZoomHeight());
		this.displayProjection_(aProjection);
		this.changed_($("projection"));
	}

	/**
	 * Called when a node is picked with the control key down.
	 * 
	 * @param pickedNode jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @category manipulating
	 */
	public void ctrlClick_(JunOctreeNode pickedNode) {
		// do nothing.
	}

	/**
	 * Answer the default eye point.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultEyePoint()
	 * @category defaults
	 */
	public Jun3dPoint defaultEyePoint() {
		if (this.hasOctree() == false) {
			return new Jun3dPoint(10000, 10000, 10000);
		}

		if (defaultEyePoint == null) {
			Jun3dBoundingBox box = this.boundingBox();
			double distance = box.origin().distance_(box.corner());
			distance = distance * 2;

			return new Jun3dPoint(distance, distance, distance);
		}

		return defaultEyePoint;
	}

	/**
	 * Set my new default eye point.
	 * 
	 * @param eyePoint jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultEyePoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category defaults
	 */
	public void defaultEyePoint_(Jun3dPoint eyePoint) {
		defaultEyePoint = eyePoint;
	}

	/**
	 * Answer the default light color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultLightColor()
	 * @category defaults
	 */
	public Color defaultLightColor() {
		return Color.gray;
	}

	/**
	 * Answer the default light point.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultLightPoint()
	 * @category defaults
	 */
	public Jun3dPoint defaultLightPoint() {
		return new Jun3dPoint(0, 0, 1);
	}

	/**
	 * Answer the default sight point.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultSightPoint()
	 * @category defaults
	 */
	public Jun3dPoint defaultSightPoint() {
		if (this.hasOctree() == false) {
			return new Jun3dPoint(0, 0, 0);
		}

		if (defaultSightPoint == null) {
			return (Jun3dPoint) this.boundingBox().center();
		}

		return defaultSightPoint;
	}

	/**
	 * Set my new default sight point.
	 * 
	 * @param sightPoint jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultSightPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category defaults
	 */
	public void defaultSightPoint_(Jun3dPoint sightPoint) {
		defaultSightPoint = sightPoint;
	}

	/**
	 * Answer my current default up vector.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultUpVector()
	 * @category defaults
	 */
	public Jun3dPoint defaultUpVector() {
		if (defaultUpVector == null) {
			return new Jun3dPoint(-1, -1, 1);
		}

		return defaultUpVector;
	}

	/**
	 * Set my new default up vector.
	 * 
	 * @param upVector jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultUpVector_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category defaults
	 */
	public void defaultUpVector_(Jun3dPoint upVector) {
		defaultUpVector = upVector;
	}

	/**
	 * Answer my current default view factor.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultViewFactor()
	 * @category defaults
	 */
	public double defaultViewFactor() {
		if (defaultViewFactor == null) {
			return 10.0;
		}

		return defaultViewFactor.doubleValue();
	}

	/**
	 * Set my new default view factor.
	 * 
	 * @param viewFactor jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultViewFactor_(double)
	 * @category defaults
	 */
	public void defaultViewFactor_(double viewFactor) {
		defaultViewFactor = new Double(viewFactor);
	}

	/**
	 * Answer the first display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLight1()
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight1() {
		return this.displayLights()[0];
	}

	/**
	 * Answer the second display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLight2()
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight2() {
		return this.displayLights()[1];
	}

	/**
	 * Answer the third display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLight3()
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight3() {
		return this.displayLights()[2];
	}

	/**
	 * Answer the fourth display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLight4()
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight4() {
		return this.displayLights()[3];
	}

	/**
	 * Answer the fifth display light.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLight5()
	 * @category lighting
	 */
	public JunOpenGLDisplayLight displayLight5() {
		return this.displayLights()[4];
	}

	/**
	 * Answer the array of lights which are on.
	 * 
	 * @return jp.co.sra.jun.opengl.lights.JunOpenGLLight[]
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLightCollection()
	 * @category lighting
	 */
	public JunOpenGLLight[] displayLightCollection() {
		Vector lightCollection = new Vector();
		Jun3dTransformation t = (Jun3dTransformation) this.displayProjection().asEyeTransformation().inverse();

		for (int i = 0; i < this.displayLights().length; i++) {
			JunOpenGLDisplayLight light = this.displayLights()[i];

			if (light.isOn()) {
				lightCollection.addElement(light.light().transform_(t));
			}
		}

		int size = lightCollection.size();
		JunOpenGLLight[] lights = new JunOpenGLLight[size];
		lightCollection.copyInto(lights);

		return lights;
	}

	/**
	 * Answer the array of all display lights.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight[]
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLights()
	 * @category lighting
	 */
	public JunOpenGLDisplayLight[] displayLights() {
		if (displayLights == null) {
			displayLights = new JunOpenGLDisplayLight[5];
			displayLights[0] = JunOpenGLDisplayLight.ParallelLight_color_position_(true, this.defaultLightColor(), this.defaultLightPoint());
			displayLights[1] = new JunOpenGLDisplayLight();
			displayLights[2] = new JunOpenGLDisplayLight();
			displayLights[3] = new JunOpenGLDisplayLight();
			displayLights[4] = JunOpenGLDisplayLight.AmbientLight_color_(true, this.defaultLightColor());

			final JunOctreeEditModel self = this;

			for (int i = 0; i < displayLights.length; i++) {
				displayLights[i].compute_(new StBlockClosure() {
					public Object value() {
						self.updateLightMenuIndication();
						self.changed_($("light"));

						return null;
					}
				});
			}
		}

		return displayLights;
	}

	/**
	 * Answer the current projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayProjection()
	 * @category accessing
	 */
	public JunOpenGLProjection displayProjection() {
		return this.displayProjector().projection();
	}

	/**
	 * Set the projection.
	 * 
	 * @param aProjection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayProjection_(jp.co.sra.jun.opengl.projection.JunOpenGLProjection)
	 * @category accessing
	 */
	public void displayProjection_(JunOpenGLProjection aProjection) {
		this.displayProjector().projection_(aProjection);
	}

	/**
	 * Answer the current projector.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayProjector()
	 * @category accessing
	 */
	public JunOpenGLProjector displayProjector() {
		if (displayProjector == null) {
			displayProjector = new JunOpenGLProjector();
			displayProjector.eyePoint_(this.defaultEyePoint());
			displayProjector.sightPoint_(this.defaultSightPoint());
			displayProjector.upVector_(this.defaultUpVector());
			displayProjector.viewFactor_(this.defaultViewFactor());
		}

		return displayProjector;
	}

	/**
	 * Divide the selected nodes.
	 * 
	 * @category menu messages
	 */
	public void divideSelection() {
		if (selection.isEmpty() == false) {
			JunOctreeNode[] nodes = new JunOctreeNode[selection.size()];
			selection.copyInto(nodes);

			Vector newSelection = new Vector(nodes.length * 8);

			for (int i = 0; i < nodes.length; i++) {
				nodes[i].beDivided();

				JunOctreeNode[] moreNodes = (JunOctreeNode[]) nodes[i].contents();

				for (int j = 0; j < moreNodes.length; j++) {
					if (newSelection.contains(moreNodes[j]) == false) {
						newSelection.addElement(moreNodes[j]);
					}
				}
			}

			selection = newSelection;
			this.changed_($("object"));
		}
	}

	/**
	 * Change the dolly and update the projector.
	 * 
	 * @param factor double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#dolly_(double)
	 * @category projection
	 */
	public void dolly_(double factor) {
		double dollyFactor;
		Jun3dLine aLine;
		Jun3dPoint eyePoint;
		Jun3dPoint sightPoint;
		JunOpenGLProjector thisDisplayProjector = this.displayProjector();

		//if (StController._ShiftDown()) {
		//	dollyFactor = factor / 20;
		//} else {
		dollyFactor = factor;

		//}
		aLine = new Jun3dLine(thisDisplayProjector.eyePoint(), thisDisplayProjector.sightPoint());
		eyePoint = (Jun3dPoint) aLine.atT_(0.0 + dollyFactor);
		sightPoint = (Jun3dPoint) aLine.atT_(1.0 + dollyFactor);
		thisDisplayProjector.eyePoint_(eyePoint);
		thisDisplayProjector.sightPoint_(sightPoint);
		this.changed_($("projection"));
	}

	/**
	 * Initialize the dolly button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#dollyButton()
	 * @category buttons
	 */
	public JunButtonModel dollyButton() {
		if (dollyButton == null) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.DollyCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());
					return model;
				}
			});
			dollyButton = button;
		}

		return dollyButton;
	}

	/**
	 * Answer my drag button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#dragButton()
	 * @category buttons
	 */
	public JunButtonModel dragButton() {
		if (dragButton == null) {
			final JunOctreeEditModel self = this;
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.QuartersCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());
					if (model.value()) {
						self.focusButton().value_(false);
						self.grabButton().value_(false);
						self.pickButton().value_(false);
					}
					self.changed_($("state"));
					return model;
				}
			});
			dragButton = button;
		}

		return dragButton;
	}

	/**
	 * Erase the selected nodes.
	 * 
	 * @category menu messages
	 */
	public void eraseSelection() {
		if (selection.isEmpty() == false) {
			JunOctreeNode[] nodes = new JunOctreeNode[selection.size()];
			selection.copyInto(nodes);

			for (int i = 0; i < nodes.length; i++) {
				nodes[i].beEmpty();
			}

			this.changed_($("object"));
		}
	}

	/**
	 * Change the eye point and update the projector.
	 * 
	 * @param a3dPoint jp.co.sra.jun.grometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#eyePoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category projection
	 */
	public void eyePoint_(Jun3dPoint a3dPoint) {
		this.displayProjector().eyePoint_(a3dPoint);
		this.changed_($("projection"));
	}

	/**
	 * Fill the selected nodes.
	 * 
	 * @category menu messages
	 */
	public void fillSelection() {
		if (selection.isEmpty() == false) {
			JunOctreeNode[] nodes = new JunOctreeNode[selection.size()];
			selection.copyInto(nodes);

			for (int i = 0; i < nodes.length; i++) {
				nodes[i].beFilled();
			}

			this.changed_($("object"));
		}
	}

	/**
	 * Fit the projector to display the entire objects.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#fit()
	 * @category projection
	 */
	public void fit() {
		if (this.hasOctree()) {
			this.fitSilently();
			this.changed_($("projection"));
		}
	}

	/**
	 * Fit the sight point to the current view.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#fitSight()
	 * @category projection
	 */
	public void fitSight() {
		if (this.hasOctree()) {
			Jun3dPoint sightPoint = this.computeSightPoint();
			this.sightPoint_(sightPoint);
		}
	}

	/**
	 * Fit the 3d object to the current view.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#fitSilently()
	 * @category projection
	 */
	public void fitSilently() {
		if (this.hasOctree()) {
			Jun3dPoint sightPoint = this.computeSightPoint();
			this.displayProjector().sightPoint_(sightPoint);

			double zoomHeight = this.computeZoomHeight();
			this.displayProjector().zoomHeight_(zoomHeight);
		}
	}

	/**
	 * Fit the zoom height.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#fitZoom()
	 * @category projection
	 */
	public void fitZoom() {
		if (this.hasOctree()) {
			double zoomHeight = this.computeZoomHeight();
			this.zoomHeight_(zoomHeight);
		}
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#flushProjector()
	 * @category flushing
	 */
	public void flushProjector() {
		displayProjector = null;
	}

	/**
	 * Change the focus area.
	 * 
	 * @param originalRectangle jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @param newRectangle jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#focus_to_(jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox, jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox)
	 * @category manipulating
	 */
	public void focus_to_(Jun2dBoundingBox originalRectangle, Jun2dBoundingBox newRectangle) {
		JunOpenGLProjection aProjection = this.displayProjection();
		Jun3dPoint originalCenter = aProjection.translateTo3dPointFromPoint_((Jun2dPoint) originalRectangle.origin().plus_(originalRectangle.corner()).dividedBy_(2));
		Jun3dPoint newCenter = aProjection.translateTo3dPointFromPoint_((Jun2dPoint) newRectangle.origin().plus_(newRectangle.corner()).dividedBy_(2));
		aProjection.sightPoint_((Jun3dPoint) aProjection.sightPoint().plus_(newCenter.minus_(originalCenter)));

		Jun3dPoint originalTop = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(0, originalRectangle.origin().y()));
		Jun3dPoint originalBottom = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(0, originalRectangle.corner().y()));
		Jun3dPoint originalLeft = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(originalRectangle.origin().x(), 0));
		Jun3dPoint originalRight = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(originalRectangle.corner().x(), 0));
		double originalHeight = originalTop.minus_(originalBottom).length();
		double originalWidth = originalLeft.minus_(originalRight).length();
		Jun3dPoint newTop = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(0, newRectangle.origin().y()));
		Jun3dPoint newBottom = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(0, newRectangle.corner().y()));
		Jun3dPoint newLeft = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(newRectangle.origin().x(), 0));
		Jun3dPoint newRight = aProjection.translateTo3dPointFromPoint_(new Jun2dPoint(newRectangle.corner().x(), 0));
		double newHeight = newTop.minus_(newBottom).length();
		double newWidth = newLeft.minus_(newRight).length();
		double newViewHeight = aProjection.regularHeight() * newHeight / originalHeight;
		double newViewWidth = newViewHeight * newWidth / newHeight;
		double newZoomHeight = newViewHeight;
		if (originalWidth / originalHeight * newViewHeight < newViewWidth) {
			double tmpViewHeight = originalWidth * newViewHeight / (originalHeight * newViewWidth);
			newZoomHeight = newViewHeight * tmpViewHeight;
		}

		if (((originalHeight * 0.01d) < newHeight) && ((newHeight * 0.01d) < originalHeight)) {
			aProjection.zoomHeight_(newZoomHeight);
		}

		this.displayProjection_(aProjection);
		this.changed_($("projection"));
	}

	/**
	 * Answer my focus button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#focusButton()
	 * @category buttons
	 */
	public JunButtonModel focusButton() {
		if (focusButton == null) {
			final JunOctreeEditModel self = this;
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.GlassCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());

					if (model.value()) {
						self.grabButton().value_(false);
						self.dragButton().value_(false);
						self.pickButton().value_(false);
					}

					self.changed_($("state"));

					return model;
				}
			});
			focusButton = button;
		}

		return focusButton;
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param deltaPoint jp.co.sra.jun.grometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grab_(jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category manipulating
	 */
	public void grab_(Jun2dPoint deltaPoint) {
		this.grab_(deltaPoint, false);
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param deltaPoint jp.co.sra.jun.grometry.basic.Jun2dPoint
	 * @param shiftDown boolean
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grab_(jp.co.sra.jun.geometry.basic.Jun2dPoint, boolean)
	 * @category manipulating
	 */
	public void grab_(Jun2dPoint deltaPoint, boolean shiftDown) {
		if (this.hasOctree()) {
			Jun2dPoint fromPoint = new Jun2dPoint(0, 0);
			Jun2dPoint toPoint;

			if (shiftDown) {
				toPoint = new Jun2dPoint(deltaPoint.x() / 10, deltaPoint.y() / 10);
			} else {
				toPoint = new Jun2dPoint(deltaPoint.x(), deltaPoint.y());
			}

			this.grab_xy_(fromPoint, toPoint);
		}
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param from2dPoint jp.co.sra.jun.grometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.grometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grab_xy_(jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category manipulating
	 */
	public void grab_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.hasOctree()) {
			JunOpenGLProjection aProjection = this.displayProjection();
			JunSphereSurface grabSphere = this.grabSphere();
			Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
			Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
			JunAngle rotationAngle = (new Jun3dLine(grabSphere.center(), toPoint)).angleWithLine_(new Jun3dLine(grabSphere.center(), fromPoint));

			if (Math.abs(rotationAngle.rad()) > ThetaAccuracy) {
				Jun3dLine rotationAxis = new Jun3dLine(grabSphere.center(), (Jun3dPoint) grabSphere.center().minus_(((Jun3dPoint) fromPoint.minus_(grabSphere.center())).product_((Jun3dPoint) toPoint.minus_(grabSphere.center()))));
				Jun3dTransformation transformation = Jun3dTransformation.Rotate_around_(rotationAngle, rotationAxis);
				Jun3dTransformation transformationInv = Jun3dTransformation.Rotate_around_(rotationAngle.mul_(-1), rotationAxis);
				Jun3dPoint upPoint = (Jun3dPoint) aProjection.sightPoint().plus_(aProjection.unitUpVector());
				aProjection.eyePoint_(transformationInv.applyTo_(aProjection.eyePoint()));
				aProjection.upVector_((Jun3dPoint) transformation.applyTo_(upPoint).minus_(aProjection.sightPoint()));
				this.displayProjection_(aProjection);
				this.changed_($("projection"));
			}
		}
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @param aPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grab3dPoint_(jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category private
	 */
	public Jun3dPoint grab3dPoint_(Jun2dPoint aPoint) {
		JunOpenGLProjection projection = this.displayProjection();
		JunSphereSurface handleSphere = this.grabSphere();
		JunOpenGLParallelProjection prj = (JunOpenGLParallelProjection) projection.asParallelProjection();
		Jun3dLine line = new Jun3dLine(prj.translateTo3dPointInNearBoundaryPlaneFromPoint_(aPoint), prj.translateTo3dPointInFarBoundaryPlaneFromPoint_(aPoint));
		Jun3dPoint[] candidates = handleSphere.crossPointsWithLine_(line);

		if (candidates.length == 0) {
			return prj.translateTo3dPointFromPoint_(aPoint);
		} else {
			return candidates[0];
		}
	}

	/**
	 * Answer my grab button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#grabButton()
	 * @category buttons
	 */
	public JunButtonModel grabButton() {
		if (grabButton == null) {
			final JunOctreeEditModel self = this;
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.HandCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());

					if (model.value()) {
						self.focusButton().value_(false);
						self.dragButton().value_(false);
						self.pickButton().value_(false);
					}

					self.changed_($("state"));

					return model;
				}
			});
			grabButton = button;
		}

		return grabButton;
	}

	/**
	 * Create a sphere for grabbing.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunSphereSurface
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grabSphere()
	 * @category private
	 */
	public JunSphereSurface grabSphere() {
		JunSphereSurface grabSphere;
		JunOpenGLProjection projection = this.displayProjection();
		grabSphere = new JunSphereSurface(projection.sightPoint(), projection.regularHeight() / 2);

		return grabSphere;
	}

	/**
	 * Answer true if the receiver has octrees, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasOctree() {
		return (octrees != null && octrees.length > 0);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	public void initialize() {
		octrees = new JunOctree[0];
		displayLights = null;
		displayProjector = null;
		selection = new Vector();
		dollyButton = null;
		pickButton = null;
		grabButton = null;
		dragButton = null;
	}

	/**
	 * Focus on the specified area.
	 * 
	 * @param from2dPoint jp.co.sra.jun.grometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.grometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#look_xy_(jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category manipulating
	 */
	public void look_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.hasOctree()) {
			JunOpenGLProjection aProjection = this.displayProjection();
			Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
			Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
			aProjection.eyePoint_((Jun3dPoint) aProjection.eyePoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
			this.displayProjection_(aProjection);
			this.changed_($("projection"));
		}
	}

	/**
	 * Specify an accuracy and compute a melted octree.
	 * 
	 * @category menu messages
	 */
	public void meltWithAccuracy() {
		if (this.hasOctree()) {
			String accuracyString = JunDialog.Request_(JunSystem.$String("Accuracy?"), "1.0d");
			if (accuracyString == null || accuracyString.length() == 0) {
				return;
			}

			double accuracy = Double.valueOf(accuracyString).doubleValue();
			if (accuracy < (1.0d - 8)) {
				return;
			}

			this.meltWithAccuracy_(accuracy);
			this.changed_($("object"));
		}
	}

	/**
	 * Specify a depth and compute a melted octree.
	 * 
	 * @category menu messages
	 */
	public void meltWithDepth() {
		if (this.hasOctree()) {
			String depthString = JunDialog.Request_(JunSystem.$String("Depth?"), "1");
			if (depthString == null || depthString.length() == 0) {
				return;
			}

			int depth = Math.round(Float.valueOf(depthString).floatValue());
			if (depth < 1) {
				return;
			}

			this.meltWithDepth_(depth);
			this.changed_($("object"));
		}
	}

	/**
	 * Create new model and open it.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#newModel()
	 * @category menu messages
	 */
	public JunApplicationModel newModel() {
		try {
			JunOctree newOctree = (JunOctree) _PerformWith(octrees[0].getClass(), "Bounds_", Jun3dBoundingBox.Origin_extent_(Jun3dPoint.Zero(), Jun3dPoint.Unity()));
			return JunOctreeEditModel.Edit_(newOctree);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Create an application window of the default view and open it.
	 * 
	 * @return java.awt.Frame
	 * @see jp.co.sra.smalltalk.StApplicationModel#open()
	 * @category interface opening
	 */
	public Frame open() {
		this.flushProjector();
		this.fitSilently();
		this.updateMenuIndication();

		return super.open();
	}

	/**
	 * Menu message: Open the light #1.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openLight1()
	 * @category menu messages
	 */
	public void openLight1() {
		JunOpenGLDisplayLight displayLight = this.displayLight1();

		Window[] windows = displayLight.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			if (windows[i].isShowing()) {
				windows[i].toFront();
			} else {
				windows[i].setVisible(true);
			}
		}

		if (windows.length == 0) {
			displayLight.openPositionLight();
		}
	}

	/**
	 * Menu message: Open the light #2.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openLight2()
	 * @category menu messages
	 */
	public void openLight2() {
		JunOpenGLDisplayLight displayLight = this.displayLight2();

		Window[] windows = displayLight.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			if (windows[i].isShowing()) {
				windows[i].toFront();
			} else {
				windows[i].setVisible(true);
			}
		}

		if (windows.length == 0) {
			displayLight.openPositionLight();
		}
	}

	/**
	 * Menu message: Open the light #3.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openLight3()
	 * @category menu messages
	 */
	public void openLight3() {
		JunOpenGLDisplayLight displayLight = this.displayLight3();

		Window[] windows = displayLight.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			if (windows[i].isShowing()) {
				windows[i].toFront();
			} else {
				windows[i].setVisible(true);
			}
		}

		if (windows.length == 0) {
			displayLight.openPositionLight();
		}
	}

	/**
	 * Menu message: Open the light #4.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openLight4()
	 * @category menu messages
	 */
	public void openLight4() {
		JunOpenGLDisplayLight displayLight = this.displayLight4();

		Window[] windows = displayLight.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			if (windows[i].isShowing()) {
				windows[i].toFront();
			} else {
				windows[i].setVisible(true);
			}
		}

		if (windows.length == 0) {
			displayLight.openPositionLight();
		}
	}

	/**
	 * Menu message: Open the light #5.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#openLight5()
	 * @category menu messages
	 */
	public void openLight5() {
		JunOpenGLDisplayLight displayLight = this.displayLight5();

		Window[] windows = displayLight.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			if (windows[i].isShowing()) {
				windows[i].toFront();
			} else {
				windows[i].setVisible(true);
			}
		}

		if (windows.length == 0) {
			displayLight.openAmbientLight();
		}
	}

	/**
	 * Pan the camera.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#pan()
	 * @category projection
	 */
	public void pan() {
		this.zoom_(0.5);
	}

	/**
	 * Pick a JunOctreeNode at the specified point on the view.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint,MouseEvent
	 * @param event DOCUMENT ME!
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#pick_with_(jp.co.sra.jun.geometry.basic.Jun2dPoint, java.awt.event.MouseEvent)
	 * @category manipulating
	 */
	public JunOpenGL3dObject pick_with_(Jun2dPoint aPoint, MouseEvent event) {
		if (this.hasOctree()) {
			JunOctreeNode pickedTip = this.pickTipAt_(aPoint);

			if (pickedTip == null) {
				if (selection.isEmpty() == false) {
					this.clearSelection();
					this.changed_($("selection"));
				}
			} else {
				if (event.isShiftDown()) {
					if (selection.contains(pickedTip)) {
						this.removeSelection_(pickedTip);
						this.changed_($("selection"));
					} else {
						this.addSelection_(pickedTip);
						this.changed_($("selection"));
					}
				} else {
					if (event.isControlDown()) {
						this.ctrlClick_(pickedTip);
					} else if (event.isAltDown()) {
						this.altClick_(pickedTip);
					} else {
						this.clearSelection();
						this.addSelection_(pickedTip);
						this.changed_($("selection"));
					}
				}
			}
		}

		return null;
	}

	/**
	 * Answer my pick button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#pickButton()
	 * @category buttons
	 */
	public JunButtonModel pickButton() {
		if (pickButton == null) {
			final JunOctreeEditModel self = this;
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.NormalCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					model.value_(!model.value());
					if (model.value()) {
						self.focusButton().value_(false);
						self.dragButton().value_(false);
						self.grabButton().value_(false);
					}
					self.changed_($("state"));
					return model;
				}
			});
			pickButton = button;
		}

		return pickButton;
	}

	/**
	 * Called when quiting.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#quitDoing()
	 * @category menu messages
	 */
	public void quitDoing() {
		Window[] windows = this.builder().windows();
		for (int i = 0; i < windows.length; i++) {
			windows[i].dispose();
		}
	}

	/**
	 * Close the display model.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		if (this.displayLights() != null) {
			for (int i = 0; i < this.displayLights().length; i++) {
				//JunOpenGLDisplayLight light = this.displayLights()[i];
				//if (light.builder() != null) {
				//	light.builder().window().setVisible(false);
				//	super._close(light.builder().window());
				//}
			}
		}
		super.release();
	}

	/**
	 * Remove the node from the currently selected nodes.
	 * 
	 * @param aJunOctreeNode jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @category selection
	 */
	public void removeSelection_(JunOctreeNode aJunOctreeNode) {
		selection.removeElement(aJunOctreeNode);
	}

	/**
	 * Render the octrees on the rendering context.
	 * 
	 * @param renderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#renderOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category displaying
	 */
	public void renderOn_(final JunOpenGLRenderingContext renderingContext) {
		renderingContext.clear();
		renderingContext.cullFaceBack();

		JunOpenGLLight[] lights = this.displayLightCollection();

		if (this.hasOctree() == false) {
			this.displayProjector().projectOn_(renderingContext);
		} else {
			this.displayProjector().project_withLights_on_(null, lights, renderingContext);
			renderingContext.paint_(Color.black);
			renderingContext.lineWidth_(4);

			for (int i = 0; i < octrees.length; i++) {
				octrees[i].originExtentNodesDo_(new StBlockClosure() {
					public Object value_value_value_(Object o1, Object o2, Object o3) {
						Jun3dPoint origin = (Jun3dPoint) o1;
						Jun3dPoint extent = (Jun3dPoint) o2;
						final JunOctreeNode node = (JunOctreeNode) o3;

						if (selection.contains(node)) {
							renderingContext.productTransformation_while_((Jun3dTransformation) Jun3dTransformation.Scale_(extent).product_(Jun3dTransformation.Translate_(origin)), new StBlockClosure() {
								public Object value() {
									node.renderWireframeOn_(renderingContext);

									return null;
								}
							});
						}

						return null;
					}
				});
			}

			renderingContext.pointSize_(5);

			for (int i = 0; i < octrees.length; i++) {
				octrees[i].originExtentTipsDo_(new StBlockClosure() {
					public Object value_value_value_(Object o1, Object o2, Object o3) {
						Jun3dPoint origin = (Jun3dPoint) o1;
						Jun3dPoint extent = (Jun3dPoint) o2;
						final JunOctreeNode tip = (JunOctreeNode) o3;
						renderingContext.productTransformation_while_((Jun3dTransformation) Jun3dTransformation.Scale_(extent).product_(Jun3dTransformation.Translate_(origin)), new StBlockClosure() {
							public Object value() {
								if (tip.isEmpty() == false) {
									renderingContext.paint_(Color.white);
									tip.renderPolygonOn_(renderingContext);
								} else {
									renderingContext.paint_(Color.black);
									tip.renderVertexOn_(renderingContext);
								}

								return null;
							}
						});

						return null;
					}
				});
			}

			renderingContext.flush();
		}
	}

	/**
	 * Menu message: Reset the view.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#resetView()
	 * @category menu messages
	 */
	public void resetView() {
		this.flushProjector();
		this.fit();
		this.updateMenuIndication();
	}

	/**
	 * Select all nodes.
	 * 
	 * @category menu messages
	 */
	public void selectAll() {
		final JunOctreeEditModel self = this;
		this.clearSelection();

		for (int i = 0; i < octrees.length; i++) {
			octrees[i].originExtentTipsDo_(new StBlockClosure() {
				public Object value_value_value_(Object origin, Object extent, Object tip) {
					self.addSelection_((JunOctreeNode) tip);

					return null;
				}
			});
		}

		this.changed_($("selection"));
		this.updateMenuIndication();
	}

	/**
	 * Change the sight point and update the projector.
	 * 
	 * @param a3dPoint jp.co.sra.jun.grometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#sightPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category projection
	 */
	public void sightPoint_(Jun3dPoint a3dPoint) {
		this.displayProjector().sightPoint_(a3dPoint);
		this.changed_($("projection"));
	}

	/**
	 * Simplify the entire octree.
	 * 
	 * @category menu messages
	 */
	public void simplify() {
		if (this.hasOctree()) {
			for (int i = 0; i < octrees.length; i++) {
				octrees[i].beCompact();
			}
		}

		this.clearSelection();
		this.changed_($("object"));
	}

	/**
	 * Slide the projection.
	 * 
	 * @param from2dPoint jp.co.sra.jun.grometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.grometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#slide_xy_(jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category manipulating
	 */
	public void slide_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.hasOctree()) {
			JunOpenGLProjection aProjection = this.displayProjection();
			Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
			Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
			aProjection.sightPoint_((Jun3dPoint) aProjection.sightPoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
			aProjection.eyePoint_((Jun3dPoint) aProjection.eyePoint().minus_((Jun3dPoint) toPoint.minus_(fromPoint)));
			this.displayProjection_(aProjection);
			this.changed_($("projection"));
		}
	}

	/**
	 * Spawning the JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#spawningObject()
	 * @category menu messages
	 */
	public JunOpenGL3dObject spawningObject() {
		if (this.hasOctree()) {
			Vector objects = new Vector();

			for (int i = 0; i < octrees.length; i++) {
				objects.addElement(octrees[i].asJunOpenGL3dObject());
			}

			return new JunOpenGL3dCompoundObject(objects);
		} else {
			return null;
		}
	}

	/**
	 * Spawn as an OpenGL object.
	 * 
	 * @category menu messages
	 */
	public void spawnOpenGLObject() {
		JunOpenGLDisplayModel displayModel = this.spawningObject().show();
		displayModel.displayProjection_((JunOpenGLProjection) this.displayProjection().copy());
		displayModel.changed_($("object"));
	}

	/**
	 * Spawn the selected nodes.
	 * 
	 * @category menu messages
	 */
	public void spawnSelection() {
		if (selection.isEmpty() == false) {
			final Vector spawningOctrees = new Vector(selection.size());

			for (int i = 0; i < octrees.length; i++) {
				octrees[i].originExtentNodesDo_(new StBlockClosure() {
					public Object value_value_value_(Object origin, Object extent, Object node) {
						if (selection.contains(node)) {
							Jun3dBoundingBox bbox = Jun3dBoundingBox.Origin_extent_((Jun3dPoint) origin, (Jun3dPoint) extent);
							JunOctree newOctree = JunOctree.Bounds_(bbox);
							spawningOctrees.addElement(newOctree);
						}

						return null;
					}
				});
			}

			JunOctreeEditModel editModel = new JunOctreeEditModel();
			editModel.open();
			editModel.displayProjection_((JunOpenGLProjection) this.displayProjection().copy());
			editModel.setOctrees_(spawningOctrees);
			editModel.changed_($("object"));
		}
	}

	/**
	 * Unselect all nodes.
	 * 
	 * @category menu messages
	 */
	public void unselectAll() {
		this.clearSelection();
		this.changed_($("selection"));
	}

	/**
	 * Change the up vector and update the projector.
	 * 
	 * @param a3dPoint jp.co.sra.jun.grometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#upVector_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @category projection
	 */
	public void upVector_(Jun3dPoint a3dPoint) {
		this.displayProjector().upVector_(a3dPoint);
		this.changed_($("projection"));
	}

	/**
	 * Change the view factor and update the projector.
	 * 
	 * @param factor double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#viewFactor_(double)
	 * @category projection
	 */
	public void viewFactor_(double factor) {
		this.displayProjector().viewFactor_(factor);
		this.changed_($("projection"));
	}

	/**
	 * Look closer.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#zoom()
	 * @category projection
	 */
	public void zoom() {
		this.zoom_(2.0);
	}

	/**
	 * Change the zoom and update the projector.
	 * 
	 * @param factor double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#zoom_(double)
	 * @category projection
	 */
	public void zoom_(double factor) {
		double zoomFactor;

		//if (StController._ShiftDown()) {
		//	if (factor >= 1.0) {
		//		zoomFactor = 1.0 + (factor - 1.0) * 0.1;
		//	} else {
		//		zoomFactor = 1.0 - (1.0 - factor) * 0.1;
		//	}
		//} else {
		zoomFactor = factor;

		//}
		this.displayProjector().zoom_(zoomFactor);
		this.changed_($("projection"));
	}

	/**
	 * Change the zoom height and update the projector.
	 * 
	 * @param aNumber double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#zoomHeight_(double)
	 * @category projection
	 */
	public void zoomHeight_(double aNumber) {
		this.displayProjector().zoomHeight_(aNumber);
		this.changed_($("projection"));
	}

	/**
	 * Compute the current sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#computeSightPoint()
	 * @category private
	 */
	protected Jun3dPoint computeSightPoint() {
		return (Jun3dPoint) this.boundingBox().center();
	}

	/**
	 * Compute the current zoom height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#computeZoomHeight()
	 * @category private
	 */
	protected double computeZoomHeight() {
		double maxHeight;
		double maxOffsetX = 0;
		double maxOffsetY = 0;
		double depth = this.displayProjection().distance();
		Jun3dPoint up = this.displayProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(0, -1), depth);
		Jun3dPoint down = this.displayProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(0, 1), depth);
		Jun3dPoint right = this.displayProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(1, 0), depth);
		Jun3dPoint left = this.displayProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(-1, 0), depth);
		final JunPlane horizontal = new JunPlane(this.displayProjection().eyePoint(), left, right);
		final JunPlane vertical = new JunPlane(this.displayProjection().eyePoint(), up, down);

		if (this.hasOctree()) {
			for (int i = 0; i < octrees.length; i++) {
				JunOctree octree = octrees[i];
				Jun3dPoint origin = octree.origin();
				Jun3dPoint corner = octree.corner();
				Jun3dPoint[] points = new Jun3dPoint[8];
				points[0] = new Jun3dPoint(origin.x(), origin.y(), origin.z());
				points[1] = new Jun3dPoint(corner.x(), origin.y(), origin.z());
				points[2] = new Jun3dPoint(origin.x(), corner.y(), origin.z());
				points[3] = new Jun3dPoint(corner.x(), corner.y(), origin.z());
				points[4] = new Jun3dPoint(origin.x(), origin.y(), corner.z());
				points[5] = new Jun3dPoint(corner.x(), origin.y(), corner.z());
				points[6] = new Jun3dPoint(origin.x(), corner.y(), corner.z());
				points[7] = new Jun3dPoint(corner.x(), corner.y(), corner.z());

				for (int j = 0; j < 8; j++) {
					double offsetX = vertical.distanceFromPoint_(points[j]);
					double offsetY = horizontal.distanceFromPoint_(points[j]);

					if (offsetX > maxOffsetX) {
						maxOffsetX = offsetX;
					}

					if (offsetY > maxOffsetY) {
						maxOffsetY = offsetY;
					}
				}
			}
		}

		double factor = 2.5;

		if (maxOffsetX > maxOffsetY) {
			maxHeight = maxOffsetX * factor;
		} else {
			maxHeight = maxOffsetY * factor;
		}

		return maxHeight;
	}

	/**
	 * Specify an accuracy and compute a melted octree.
	 * 
	 * @param accuracy double
	 * @category private-functions
	 */
	protected void meltWithAccuracy_(final double accuracy) {
		final Vector meltNodes = (Vector) selection.clone();
		final JunMetaballSolid metaballSolid = this.metaballSolidOrder_withoutAll_(4, meltNodes);

		for (int i = 0; i < octrees.length; i++) {
			octrees[i].originExtentNodesDo_(new StBlockClosure() {
				public Object value_value_value_(Object origin, Object extent, Object node) {
					if (meltNodes.contains(node)) {
						((JunOctreeNode) node).origin_extent_accuracy_block_((Jun3dPoint) origin, (Jun3dPoint) extent, accuracy, new StBlockClosure() {
							public Object value_(Object point) {
								metaballSolid.valueAt_((Jun3dPoint) point);

								return null;
							}
						});
					}

					return null;
				}
			});
		}
	}

	/**
	 * Specify a depth and compute a melted octree.
	 * 
	 * @param depth double
	 * @category private-functions
	 */
	protected void meltWithDepth_(final int depth) {
		final Vector meltNodes = (Vector) selection.clone();
		final JunMetaballSolid metaballSolid = this.metaballSolidOrder_withoutAll_(3, meltNodes);

		for (int i = 0; i < octrees.length; i++) {
			octrees[i].originExtentNodesDo_(new StBlockClosure() {
				public Object value_value_value_(Object origin, Object extent, Object node) {
					if (meltNodes.contains(node)) {
						((JunOctreeNode) node).origin_extent_depth_block_((Jun3dPoint) origin, (Jun3dPoint) extent, depth + 1, new StBlockClosure() {
							public Object value_(Object point) {
								metaballSolid.valueAt_((Jun3dPoint) point);

								return null;
							}
						});
					}

					return null;
				}
			});
		}
	}

	/**
	 * Create a new JunMetaballSolid.
	 * 
	 * @param order int
	 * @param aCollectionOfJunOctreeNode java.util.Vector
	 * @return jp.co.sra.jun.metaball.solid.JunMetaballSolid
	 * @category private-functions
	 */
	protected JunMetaballSolid metaballSolidOrder_withoutAll_(final int order, Vector aCollectionOfJunOctreeNode) {
		final JunMetaballSolid metaballSolid = new JunMetaballSolid();

		for (int i = 0; i < octrees.length; i++) {
			octrees[i].originExtentNodesWithoutAll_do_(aCollectionOfJunOctreeNode, new StBlockClosure() {
				public Object value_value_value_(Object o1, Object o2, Object o3) {
					// Jun3dPoint origin = (Jun3dPoint) o1;
					Jun3dPoint extent = (Jun3dPoint) o2;
					JunOctreeNode node = (JunOctreeNode) o3;

					if (node.isTip()) {
						double r = (extent.x() * extent.y() * extent.z() * 3.0d) / 4.0d / Math.PI;
						r = Math.pow(r, 1.0d / 3.0d);

						double sign = node.isFilled() ? 1.0d : -1.0d;
						double weight = r * sign;
						JunMetaball metaball = JunMetaSphere.Center_order_weight_(extent, order, weight);
						metaballSolid.add_(metaball);
					}

					return null;
				}
			});
		}

		return metaballSolid;
	}

	/**
	 * Pick a JunOctreeNode at the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @category private
	 */
	protected JunOctreeNode pickTipAt_(Jun2dPoint aPoint) {
		Jun3dPoint sightPoint = this.displayProjection().translateTo3dPointFromPoint_(aPoint);
		double zoomHeight = sightPoint.minus_(this.displayProjection().translateTo3dPointFromPoint_((Jun2dPoint) aPoint.plus_(new Jun2dPoint(0, -0.01d)))).length();
		JunOpenGLProjection projection = (JunOpenGLProjection) this.displayProjection().copy();
		projection.sightPoint_(sightPoint);

		if (this.displayProjection().isParallel()) {
			projection.eyePoint_((Jun3dPoint) projection.eyePoint().plus_(sightPoint).minus_(this.displayProjection().sightPoint()));
		}

		projection.zoomHeight_(zoomHeight * 5);

		return this.pickTipProjection_(projection);
	}

	/**
	 * Pick a tip with the projection.
	 * 
	 * @param projection jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @return jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @category private
	 */
	protected JunOctreeNode pickTipProjection_(final JunOpenGLProjection projection) {
		final int[] pickedIndex = new int[2];
		final ArrayList tips = new ArrayList();

		JunOpenGLObjectPicker objectPicker = new JunOpenGLObjectPicker();
		objectPicker.setRenderer(new JunOpenGLRenderer() {
			public void renderOn_(JunOpenGLRenderingContext renderingContext) {
				final JunOpenGLPickingRenderingContext pickingRenderingContext = (JunOpenGLPickingRenderingContext) renderingContext;
				pickingRenderingContext.projection_(projection);

				pickedIndex[1] = -1;
				pickedIndex[0] = pickingRenderingContext.selectNearestObjectWhile_(new StBlockClosure() {
					public Object value() {
						for (int i = 0; i < octrees.length; i++) {
							octrees[i].originExtentTipsDo_(new StBlockClosure() {
								public Object value_value_value_(Object o1, Object o2, Object o3) {
									Jun3dPoint origin = (Jun3dPoint) o1;
									Jun3dPoint extent = (Jun3dPoint) o2;
									final JunOctreeNode tip = (JunOctreeNode) o3;
									pickedIndex[1]++;
									tips.add(tip);
									pickingRenderingContext.setId_(pickedIndex[1]);
									pickingRenderingContext.productTransformation_while_((Jun3dTransformation) Jun3dTransformation.Scale_(extent).product_(Jun3dTransformation.Translate_(origin)), new StBlockClosure() {
										public Object value() {
											if (tip.isFilled()) {
												tip.renderPolygonOn_(pickingRenderingContext);
											} else {
												tip.renderVertexOn_(pickingRenderingContext);
											}
											return null;
										}
									});
									return null;
								}
							});
						}
						return null;
					}
				});
			}

			public void superimposeOn_(Graphics aGraphics, JunOpenGLDrawable aDrawable) {
			}
		});

		return (pickedIndex[0] >= 0) ? (JunOctreeNode) tips.get(pickedIndex[0]) : null;
	}

	/**
	 * Set the octrees.
	 * 
	 * @param aCollectionOfJunOctree java.util.Vector
	 * @category private
	 */
	protected void setOctrees_(Vector aCollectionOfJunOctree) {
		if ((aCollectionOfJunOctree == null) || aCollectionOfJunOctree.isEmpty()) {
			octrees = new JunOctree[0];
		} else {
			octrees = new JunOctree[aCollectionOfJunOctree.size()];
			aCollectionOfJunOctree.copyInto(octrees);
		}

		this.clearSelection();
	}

	/**
	 * Answer the title string.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Octree Editor");
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createEditMenu());
			_menuBar.add(this._createManipulateMenu());
			_menuBar.add(this._createLightMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "File" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createFileMenu()
	 * @category resources
	 */
	protected StMenu _createFileMenu() {
		StMenu fileMenu = new StMenu(JunSystem.$String("File"), $("fileMenu"));
		fileMenu.add(new StMenuItem(JunSystem.$String("New"), new MenuPerformer(this, "newModel")));
		fileMenu.addSeparator();
		fileMenu.add(new StMenuItem(JunSystem.$String("Quit"), new MenuPerformer(this, "quitDoing")));
		return fileMenu;
	}

	/**
	 * Create an "Edit" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createEditMenu()
	 * @category resources
	 */
	protected StMenu _createEditMenu() {
		StMenu editMenu = new StMenu(JunSystem.$String("Edit"), $("editMenu"));
		editMenu.add(new StMenuItem(JunSystem.$String("Select all"), new MenuPerformer(this, "selectAll")));
		editMenu.add(new StMenuItem(JunSystem.$String("Unselect all"), new MenuPerformer(this, "unselectAll")));
		return editMenu;
	}

	/**
	 * Create a "Manipulate" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createManipulateMenu() {
		StMenu manipulateMenu = new StMenu(JunSystem.$String("Manipulate"), $("manipulateMenu"));

		manipulateMenu.add(new StMenuItem(JunSystem.$String("Divide"), new MenuPerformer(this, "divideSelection")));
		manipulateMenu.add(new StMenuItem(JunSystem.$String("Erase"), new MenuPerformer(this, "eraseSelection")));
		manipulateMenu.add(new StMenuItem(JunSystem.$String("Fill"), new MenuPerformer(this, "fillSelection")));
		StMenu meltMenu = new StMenu(JunSystem.$String("Melt"));
		meltMenu.add(new StMenuItem(JunSystem.$String("Accuracy") + "...", new MenuPerformer(this, "meltWithAccuracy")));
		meltMenu.add(new StMenuItem(JunSystem.$String("Depth") + "...", new MenuPerformer(this, "meltWithDepth")));
		manipulateMenu.add(meltMenu);
		manipulateMenu.add(new StMenuItem(JunSystem.$String("Simplify"), new MenuPerformer(this, "simplify")));
		manipulateMenu.addSeparator();
		manipulateMenu.add(new StMenuItem(JunSystem.$String("Spawn"), new MenuPerformer(this, "spawnSelection")));
		return manipulateMenu;
	}

	/**
	 * Create a "Misc" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createMiscMenu()
	 * @category resources
	 */
	protected StMenu _createMiscMenu() {
		StMenu miscMenu = new StMenu(JunSystem.$String("Misc"), $("miscMenu"));
		miscMenu.add(new StMenuItem(JunSystem.$String("Viewfinder"), new MenuPerformer(this, "spawnOpenGLObject")));
		return miscMenu;
	}

}
