package jp.co.sra.jun.opengl.chart;

import java.awt.Color;
import java.awt.Font;
import java.awt.event.MouseEvent;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.goodies.font.JunFontModel;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.texture.JunOpenGLTexture;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunChartAbstract class
 * 
 *  @author    nisinaka
 *  @created   1998/11/19 (by nisinaka)
 *  @updated   2005/03/02 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun642 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunChartAbstract.java,v 8.15 2008/02/20 06:32:17 nisinaka Exp $
 */
public abstract class JunChartAbstract extends JunOpenGLDisplayModel {

	protected JunChartData data;
	protected HashMap visualObjectDictionary;
	protected JunOpenGL3dObject xAxis;
	protected JunOpenGL3dObject yAxis;
	protected JunOpenGL3dObject zAxis;
	protected Color axisColor;
	protected Color labelForegroundColor;
	protected Color labelBackgroundColor;
	protected Color[] colors;
	protected int colorIndex;
	protected double thickness;
	protected double intervalBetweenCharts;
	protected ArrayList extra3dObjects;
	protected double labelHeight;
	protected boolean useScalableLabels;

	/**
	 * Answer the class for the chart data of this chart.
	 * 
	 * @return java.lang.Class
	 * @category Accessing
	 */
	public static Class ChartDataClass() {
		return JunChartData.class;
	}

	/**
	 * Create a new instance of <code>JunChartAbstract</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunChartAbstract() {
		super();
	}

	/**
	 * Create a new instance of a Chart and initialize it with the collection
	 * of sample. This method corresponds to the class method "samples:" of
	 * JunChartAbstract in Smalltalk.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @category Instance creation
	 */
	protected JunChartAbstract(Vector aCollectionOfSample) {
		this();
		this.data_(new JunChartData(aCollectionOfSample));
	}

	/**
	 * Create a new instance of a Chart and initialize it with the collection
	 * of sample and a number of keys. This method corresponds to the class
	 * method "samples:numberOfKeys:" of JunChartAbstract in Smalltalk.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @param numberOfKeys int
	 * @category Instance creation
	 */
	protected JunChartAbstract(Vector aCollectionOfSample, int numberOfKeys) {
		this();
		this.data_(new JunChartData(aCollectionOfSample, numberOfKeys));
	}

	/**
	 * Create a new instance of a Chart and initialize it with the
	 * JunChartData. This method corresponds to the class method "data:" of
	 * JunChartAbstract in Smalltalk.
	 * 
	 * @param aChartData jp.co.sra.jun.opengl.chart.JunChartData
	 * @category Instance creation
	 */
	protected JunChartAbstract(JunChartData aChartData) {
		this();
		this.data_(aChartData);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		//this.initializeLights();
		this.initializeProjector();
		data = null;
		this.flushDisplayObject();
		this.flushAxes();
		axisColor = null;
		labelForegroundColor = null;
		labelBackgroundColor = null;
		colors = null;
		thickness = 0.1;
		intervalBetweenCharts = 0.3;
		extra3dObjects = null;
		labelHeight = 0;
		useScalableLabels = false;
	}

	/**
	 * Initialize the lights to be off.
	 * 
	 * @category initialize-release
	 */
	protected void initializeLights() {
		for (int i = 0; i < this.displayLights().length; i++) {
			this.displayLights()[i].beOff();
		}
	}

	/**
	 * Initialize the default projector.
	 * 
	 * @category initialize-release
	 */
	protected void initializeProjector() {
		this.defaultEyePoint_(new Jun3dPoint(0.5, 0.5, 100.0));
		this.defaultSightPoint_(new Jun3dPoint(0.5, 0.5, 0.0));
		this.defaultUpVector_(new Jun3dPoint(0, 1, 0));
	}

	/**
	 * Answer the chart data of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.chart.JunChartData
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public JunChartData data() {
		try {
			if (data == null) {
				Class aChartDataClass = (Class) _Perform(this.getClass(), "ChartDataClass");
				data = (JunChartData) _New(aChartDataClass);
			}

			return data;
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Set the chart data of the receiver.
	 * 
	 * @param aChartData jp.co.sra.jun.opengl.chart.JunChartData
	 * @category accessing
	 */
	public void data_(JunChartData aChartData) {
		data = aChartData;
	}

	/**
	 * Answer the JunOpenGL3dObject to show.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayObject()
	 * @category accessing
	 */
	public JunOpenGL3dObject displayObject() {
		if (super.displayObject() == null) {
			super.displayObject_(this.asJunOpenGL3dObject());
		}
		return super.displayObject();
	}

	/**
	 * Answer the collection of extra 3d objects.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing
	 */
	public ArrayList extra3dObjects() {
		return extra3dObjects;
	}

	/**
	 * Set the collection of extra 3d objects.
	 * 
	 * @param aCollectionOf3dObject java.util.ArrayList
	 * @category accessing
	 */
	public void extra3dObjects_(ArrayList aCollectionOf3dObject) {
		extra3dObjects = aCollectionOf3dObject;
	}

	/**
	 * Set the collection of extra 3d objects.
	 * 
	 * @param anArrayOf3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject[]
	 * @category accessing
	 */
	public void extra3dObjects_(JunOpenGL3dObject[] anArrayOf3dObject) {
		ArrayList aList = null;
		if (anArrayOf3dObject != null) {
			aList = new ArrayList(anArrayOf3dObject.length);
			for (int i = 0; i < anArrayOf3dObject.length; i++) {
				aList.add(anArrayOf3dObject[i]);
			}
		}
		this.extra3dObjects_(aList);
	}

	/**
	 * Answer the interval between charts of the receiver.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double intervalBetweenCharts() {
		return intervalBetweenCharts;
	}

	/**
	 * Set the inteval between charts of the receiver.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void intervalBetweenCharts_(double aNumber) {
		intervalBetweenCharts = aNumber;
	}

	/**
	 * Register the <code>JunOpenGL3dObject</code> as a visual object of the receiver.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aValue java.lang.Object
	 * @category accessing
	 */
	public void registerVisualObject_withValue_(JunOpenGL3dObject a3dObject, Object aValue) {
		a3dObject.name_(JunOpenGL3dObject.UniqueId());
		this.visualObjectDictionary().put(a3dObject.name(), aValue);
	}

	/**
	 * Answer the current samples of the chart.
	 * 
	 * @return java.lang.Object[][][]
	 * @category accessing
	 */
	public Object[][][] samples() {
		return this.data().samples();
	}

	/**
	 * Set the samples of the chart.
	 * 
	 * @param aCollectionOfSample java.util.Vector
	 * @category accessing
	 */
	public void samples_(Vector aCollectionOfSample) {
		this.data().samples_(aCollectionOfSample);
		this.flushDisplayObject();
	}

	/**
	 * Answer the thickness of the chart element of the receiver.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double thickness() {
		return thickness;
	}

	/**
	 * Set the thickness of the chart element of the receiver.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void thickness_(double aNumber) {
		thickness = aNumber;
	}

	/**
	 * Answer the Object which corresponds to the <code>JunOpenGL3dObject</code>.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object valueForVisualObject_(JunOpenGL3dObject a3dObject) {
		if (a3dObject == null) {
			return null;
		}

		return this.visualObjectDictionary.get(a3dObject);
	}

	/**
	 * Set the information about the value range.
	 * 
	 * @param minimumNumber double
	 * @param maximumNumber double
	 * @category accessing
	 */
	public void valueRangeFrom_to_(double minimumNumber, double maximumNumber) {
		this.data().valueRangeFrom_to_(minimumNumber, maximumNumber);
	}

	/**
	 * Set the information about the value range.
	 * 
	 * @param minimumNumber double
	 * @param maximumNumber double
	 * @param interval double
	 * @category accessing
	 */
	public void valueRangeFrom_to_(double minimumNumber, double maximumNumber, double interval) {
		this.data().valueRangeFrom_to_by_(minimumNumber, maximumNumber, interval);
	}

	/**
	 * Answer the hastable for all visual objects.
	 * 
	 * @return java.util.HashMap
	 * @category accessing
	 */
	public HashMap visualObjectDictionary() {
		if (visualObjectDictionary == null) {
			visualObjectDictionary = new HashMap();
		}
		return visualObjectDictionary;
	}

	/**
	 * Answer the color for axis.
	 * 
	 * @return java.awt.Color
	 * @category axes accessing
	 */
	public Color axisColor() {
		if (axisColor == null) {
			axisColor = this.defaultAxisColor();
		}

		return axisColor;
	}

	/**
	 * Set the color for axis.
	 * 
	 * @param aColor java.awt.Color
	 * @category axes accessing
	 */
	public void axisColor_(Color aColor) {
		axisColor = aColor;
	}

	/**
	 * Hide the x-axis.
	 * 
	 * @category axes accessing
	 */
	public void hideXAxis() {
		xAxis = null;
		this.updateAxesMenuIndication();
		this.flushDisplayObject();
		this.changed_($("presentation"));
	}

	/**
	 * Hide the y-axis.
	 * 
	 * @category axes accessing
	 */
	public void hideYAxis() {
		yAxis = null;
		this.updateAxesMenuIndication();
		this.flushDisplayObject();
		this.changed_($("presentation"));
	}

	/**
	 * Hide the z-axis.
	 * 
	 * @category axes accessing
	 */
	public void hideZAxis() {
		zAxis = null;
		this.updateAxesMenuIndication();
		this.flushDisplayObject();
		this.changed_($("presentation"));
	}

	/**
	 * Answer the background color for a label.
	 * 
	 * @return java.awt.Color
	 * @category axes accessing
	 */
	public Color labelBackgroundColor() {
		if (labelBackgroundColor == null) {
			labelBackgroundColor = this.defaultLabelBackgroundColor();
		}

		return labelBackgroundColor;
	}

	/**
	 * Set the background color for a labe.
	 * 
	 * @param aColor java.awt.Color
	 * @category axes accessing
	 */
	public void labelBackgroundColor_(Color aColor) {
		labelBackgroundColor = aColor;
	}

	/**
	 * Answer the foreground color for a label.
	 * 
	 * @return java.awt.Color
	 * @category axes accessing
	 */
	public Color labelForegroundColor() {
		if (labelForegroundColor == null) {
			labelForegroundColor = this.defaultLabelForegroundColor();
		}

		return labelForegroundColor;
	}

	/**
	 * Set the foreground color for a labe.
	 * 
	 * @param aColor java.awt.Color
	 * @category axes accessing
	 */
	public void labelForegroundColor_(Color aColor) {
		labelForegroundColor = aColor;
	}

	/**
	 * Answer my current label height.
	 * 
	 * @return double
	 * @category axes accessing
	 */
	public double labelHeight() {
		if (labelHeight <= 0) {
			labelHeight = this.defaultLabelHeight();
		}
		return labelHeight;
	}

	/**
	 * Set my new label height.
	 * 
	 * @param height double
	 * @category axes accessing
	 */
	public void labelHeight_(double height) {
		labelHeight = Math.max(0.01, Math.min(height, 1));

		if (this.hasXAxis()) {
			this.showXAxisWithLabels();
		}
		if (this.hasXAxis()) {
			this.showYAxisWithLabels();
		}
		if (this.hasZAxis()) {
			//
		}
	}

	/**
	 * Show the x-axis.
	 * 
	 * @category axes accessing
	 */
	public void showXAxis() {
		xAxis = this.createXAxis();
		this.updateAxesMenuIndication();
		this.flushDisplayObject();
		this.changed_($("presentation"));
	}

	/**
	 * Show the x-axis with labels.
	 * 
	 * @category axes accessing
	 */
	public void showXAxisWithLabels() {
		xAxis = new JunOpenGL3dCompoundObject(this.createXAxis(), this.createXAxisLabels());
		this.updateAxesMenuIndication();
		this.flushDisplayObject();
		this.changed_($("presentation"));
	}

	/**
	 * Show the y-axis.
	 * 
	 * @category axes accessing
	 */
	public void showYAxis() {
		yAxis = this.createYAxis();
		this.updateAxesMenuIndication();
		this.flushDisplayObject();
		this.changed_($("presentation"));
	}

	/**
	 * Show the y-axis with labels.
	 * 
	 * @category axes accessing
	 */
	public void showYAxisWithLabels() {
		yAxis = new JunOpenGL3dCompoundObject(this.createYAxis(), this.createYAxisLabels());
		this.updateAxesMenuIndication();
		this.flushDisplayObject();
		this.changed_($("presentation"));
	}

	/**
	 * Show the z-axis.
	 * 
	 * @category axes accessing
	 */
	public void showZAxis() {
		zAxis = this.createZAxis();
		this.updateAxesMenuIndication();
		this.flushDisplayObject();
		this.changed_($("presentation"));
	}

	/**
	 * Toggle show or hide x-axis.
	 * 
	 * @category axes accessing
	 */
	public void toggleXAxis() {
		if (this.hasXAxis()) {
			this.hideXAxis();
		} else {
			this.showXAxisWithLabels();
		}
	}

	/**
	 * Toggle show or hide y-axis.
	 * 
	 * @category axes accessing
	 */
	public void toggleYAxis() {
		if (this.hasYAxis()) {
			this.hideYAxis();
		} else {
			this.showYAxisWithLabels();
		}
	}

	/**
	 * Toggle show or hide z-axis.
	 * 
	 * @category axes accessing
	 */
	public void toggleZAxis() {
		if (this.hasZAxis()) {
			this.hideZAxis();
		} else {
			this.showZAxis();
		}
	}

	/**
	 * Makes the receiver to use the scalable labels.
	 * 
	 * @category axes accessing
	 */
	public void useScalableLabels() {
		useScalableLabels = true;
		if (this.hasXAxis()) {
			this.showXAxisWithLabels();
		}
		if (this.hasYAxis()) {
			this.showYAxisWithLabels();
		}
	}

	/**
	 * Makes the receiver to use the texture labels.
	 * 
	 * @category axes accessing
	 */
	public void useTextureLabels() {
		useScalableLabels = false;
		if (this.hasXAxis()) {
			this.showXAxisWithLabels();
		}
		if (this.hasYAxis()) {
			this.showYAxisWithLabels();
		}
	}

	/**
	 * Answer the x-axis of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category axes accessing
	 */
	public JunOpenGL3dObject xAxis() {
		return xAxis;
	}

	/**
	 * Answer the array of x-axis labels.
	 * 
	 * @return java.lang.String[]
	 * @category axes accessing
	 */
	public String[] xAxisLabels() {
		return this.data().keyLabels();
	}

	/**
	 * Set the array of x-axis labels.
	 * 
	 * @param anArray java.lang.String[]
	 * @category axes accessing
	 */
	public void xAxisLabels_(String[] anArray) {
		this.data().keyLabels_(anArray);
	}

	/**
	 * Answer the y-axis of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category axes accessing
	 */
	public JunOpenGL3dObject yAxis() {
		return yAxis;
	}

	/**
	 * Answer the z-axis of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category axes accessing
	 */
	public JunOpenGL3dObject zAxis() {
		return zAxis;
	}

	/**
	 * Answer true if the receiver has an x-axis, otherwise false.
	 * 
	 * @return boolean
	 * @category axes testing
	 */
	public boolean hasXAxis() {
		return (this.xAxis() != null);
	}

	/**
	 * Answer true if the receiver has a y-axis, otherwise false.
	 * 
	 * @return boolean
	 * @category axes testing
	 */
	public boolean hasYAxis() {
		return (this.yAxis() != null);
	}

	/**
	 * Answer true if the receiver has a z-axis, otherwise false.
	 * 
	 * @return boolean
	 * @category axes testing
	 */
	public boolean hasZAxis() {
		return (this.zAxis() != null);
	}

	/**
	 * Answer true if the receiver has scalable labels, otherwise false.
	 * 
	 * @return boolean
	 * @category axes testing
	 */
	public boolean hasScalableLabels() {
		return this.isScalableLabels() && (this.hasXAxis() || this.hasYAxis());
	}

	/**
	 * Answer true if the receiver has textured labels, otherwise false.
	 * 
	 * @return boolean
	 * @category axes testing
	 */
	public boolean hasTexturedLabels() {
		return this.isTexturedLabels() && (this.hasXAxis() || this.hasYAxis());
	}

	/**
	 * Answer true if the receiver is specified to use scalable labels.
	 * 
	 * @return boolean
	 * @category axes testing
	 */
	public boolean isScalableLabels() {
		return useScalableLabels;
	}

	/**
	 * Answer true if the receiver is specified to use scalable labels.
	 * 
	 * @return boolean
	 * @category axes testing
	 */
	public boolean isTexturedLabels() {
		return !this.isScalableLabels();
	}

	/**
	 * Answer the array of color. These colors are used to paint each chart
	 * elements.
	 * 
	 * @return java.awt.Color[]
	 * @category color accessing
	 */
	public Color[] colors() {
		if (colors == null) {
			colors = this.defaultColors();
		}

		return colors;
	}

	/**
	 * Set the colors.
	 * 
	 * @param anArray java.awt.Color[]
	 * @category color accessing
	 */
	public void colors_(Color[] anArray) {
		colors = anArray;
	}

	/**
	 * Answer another color.
	 * 
	 * @return java.awt.Color
	 * @category color accessing
	 */
	public Color nextColor() {
		Color[] colors = this.colors();

		if ((colors == null) || (colors.length == 0)) {
			return null;
		}

		if (colorIndex >= colors.length) {
			this.resetColorIndex();
		}

		return this.colors()[colorIndex++];
	}

	/**
	 * Reset the index of the colors.
	 * 
	 * @category color accessing
	 */
	private void resetColorIndex() {
		colorIndex = 0;
	}

	/**
	 * Convert the receiver as a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dCompoundObject chart3dObject = this.createChart3dObject();
		if (xAxis != null) {
			chart3dObject.add_(xAxis);
		}
		if (yAxis != null) {
			chart3dObject.add_(yAxis);
		}
		if (zAxis != null) {
			chart3dObject.add_(zAxis);
		}
		if (extra3dObjects != null) {
			chart3dObject.addAll_(extra3dObjects);
		}
		return chart3dObject;
	}

	/**
	 * Initialize the receiver with the attributes of the chart.
	 * 
	 * @param aChart jp.co.sra.jun.opengl.chart.JunChartAbstract
	 * @category copying
	 */
	protected void _copyAttributes(JunChartAbstract aChart) {
		aChart.initialize();

		JunChartData aData = aChart.data();
		aData.samples_(this.data().samples());
		aData.numberOfKeys_(this.data().numberOfKeys());
		if (axisColor != null) {
			aChart.axisColor_(axisColor);
		}
		if (labelForegroundColor != null) {
			aChart.labelForegroundColor_(labelForegroundColor);
		}
		if (labelBackgroundColor != null) {
			aChart.labelBackgroundColor_(labelBackgroundColor);
		}
		aChart.intervalBetweenCharts_(intervalBetweenCharts);
		aChart.thickness_(thickness);
		if (colors != null) {
			aChart.colors_(colors);
		}
		if (this.hasXAxis()) {
			aChart.showXAxisWithLabels();
		}
		if (this.hasYAxis()) {
			aChart.showYAxisWithLabels();
		}
		if (this.hasZAxis()) {
			aChart.showZAxis();
		}
	}

	/**
	 * Answer the block closure to evaluate which an object is picked.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category defaults
	 */
	protected StBlockClosure defaultActionForPickedObject() {
		return new StBlockClosure();
	}

	/**
	 * Answer the default color for axis.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultAxisColor() {
		return Color.blue;
	}

	/**
	 * Answer the default colors.
	 * 
	 * @return java.awt.Color[]
	 * @category defaults
	 */
	protected Color[] defaultColors() {
		if (this.samples() == null) {
			Color[] defaultColors = { Color.black };

			return defaultColors;
		}

		return this.defaultColors_(this.data().sheet().valueSize());
	}

	/**
	 * Answer the default colors.  The number of the colors are specified.
	 * 
	 * @param size int
	 * @return java.awt.Color[]
	 * @category defaults
	 */
	protected Color[] defaultColors_(int size) {
		Color[] defaultColors = new Color[size];

		for (int i = 0; i < size; i++) {
			defaultColors[i] = Color.getHSBColor((float) ((double) i / size), (float) 1, (float) 1);
		}

		return defaultColors;
	}

	/**
	 * Answer the default background color for a label.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultLabelBackgroundColor() {
		return Color.white;
	}

	/**
	 * Answer the default foreground color for a label.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	protected Color defaultLabelForegroundColor() {
		return Color.black;
	}

	/**
	 * Answer the default height for a label.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultLabelHeight() {
		return 0.05;
	}

	/**
	 * Flush the cached <code>JunOpenGL3dObjects</code> for axes.
	 * 
	 * @category flushing
	 */
	protected void flushAxes() {
		xAxis = null;
		yAxis = null;
		zAxis = null;
	}

	/**
	 * Flush the cache for display objects.
	 * 
	 * @category flushing
	 */
	protected void flushDisplayObject() {
		super.flushDisplayObject();
		visualObjectDictionary = null;
		this.resetColorIndex();
	}

	/**
	 * Answer the <code>JunOpenGL3dObject</code> at the specified point on the view.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param anEvent java.awt.event.MouseEvent
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category manipulating
	 */
	public JunOpenGL3dObject pick_with_(Jun2dPoint aPoint, MouseEvent anEvent) {
		JunOpenGL3dObject pickedObject = super.pick_with_(aPoint, anEvent);

		if (pickedObject != null) {
			this.pickedObject_(pickedObject);
		}

		return pickedObject;
	}

	/**
	 * The JunOpenGL3dObject is picked. Do something according to the definition.
	 * 
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category manipulating
	 */
	protected void pickedObject_(JunOpenGL3dObject anObject) {
		this.defaultActionForPickedObject().value_(anObject);
	}

	/**
	 * Print my string representation on the Writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		super.printOn_(aWriter);

		BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
		bw.write('(');
		bw.newLine();
		bw.write("     data :  ");
		if (data == null) {
			bw.write("null");
		} else {
			data.printOn_(bw);
		}
		bw.newLine();
		bw.write(')');
		bw.flush();
	}

	/**
	 * Answer the title string.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#windowTitle()
	 * @category interface opening
	 */
	public String windowTitle() {
		return $String("3D Chart");
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createEditMenu());
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createLightMenu());
			_menuBar.add(this._createAxesMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create an "Axes" menu.
	 * 
	 * @return jp.co.sra.smalltalk.StMenu
	 * @category resources
	 */
	protected StMenu _createAxesMenu() {
		StMenu axesMenu = new StMenu(JunSystem.$String("Axes"));
		axesMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Show X-Axis"), $("toggleXAxisMenu"), new MenuPerformer(this, "toggleXAxis")));
		axesMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Show Y-Axis"), $("toggleYAxisMenu"), new MenuPerformer(this, "toggleYAxis")));
		axesMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Show Z-Axis"), $("toggleZAxisMenu"), new MenuPerformer(this, "toggleZAxis")));
		return axesMenu;
	}

	/**
	 * Create a "Misc" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createMiscMenu()
	 * @category resources
	 */
	protected StMenu _createMiscMenu() {
		StMenu miscMenu = new StMenu(JunSystem.$String("Misc"), $("miscMenu"));

		// Texture
		StMenu textureMenu = new StMenu(JunSystem.$String("Texture"), $("textureMenu"));
		textureMenu.add(new StMenuItem(JunSystem.$String("From user..."), $("textureFromUserMenu"), new MenuPerformer(this, "textureFromUser")));
		textureMenu.add(new StMenuItem(JunSystem.$String("From file..."), $("textureFromFileMenu"), new MenuPerformer(this, "textureFromFile")));
		textureMenu.add(new StMenuItem(JunSystem.$String("Take away"), new MenuPerformer(this, "textureTakeAway")));
		textureMenu.addSeparator();
		textureMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Linear approx."), $("textureLinearMenu"), new MenuPerformer(this, "textureLinear")));
		textureMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Clamp"), $("textureClampMenu"), new MenuPerformer(this, "textureClamp")));
		textureMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Modulate"), $("textureModulateMenu"), new MenuPerformer(this, "textureModulate")));
		textureMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Mipmap"), $("textureMipmapMenu"), new MenuPerformer(this, "textureMipmap")));
		miscMenu.add(textureMenu);

		// Spawn
		StMenu spawnMenu = new StMenu(JunSystem.$String("Spawn"), $("spawnMenu"));
		spawnMenu.add(new StMenuItem(JunSystem.$String("Band Chart"), new MenuPerformer(this, "spawnBandChart")));
		spawnMenu.add(new StMenuItem(JunSystem.$String("Bar Chart"), new MenuPerformer(this, "spawnBarChart")));
		spawnMenu.add(new StMenuItem(JunSystem.$String("Pie Chart"), new MenuPerformer(this, "spawnPieChart")));
		spawnMenu.add(new StMenuItem(JunSystem.$String("Plot Chart"), new MenuPerformer(this, "spawnPlotChart")));
		spawnMenu.add(new StMenuItem(JunSystem.$String("Line Chart"), new MenuPerformer(this, "spawnLineChart")));
		spawnMenu.add(new StMenuItem(JunSystem.$String("Radar Chart"), new MenuPerformer(this, "spawnRadarChart")));
		miscMenu.add(spawnMenu);

		// Viewport
		miscMenu.add(new StMenuItem(JunSystem.$String("Viewport"), $("viewportMenu"), new MenuPerformer(this, "spawnViewport")));

		// Bounds
		miscMenu.add(new StMenuItem(JunSystem.$String("Bounds"), $("boundsMenu"), new MenuPerformer(this, "showBounds")));

		return miscMenu;
	}

	/**
	 * Update the menu indications.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category resources
	 */
	public void updateMenuIndication() {
		super.updateMenuIndication();
		this.updateAxesMenuIndication();
	}

	/**
	 * Update the check marks of the axes menu.
	 * 
	 * @category resources
	 */
	public void updateAxesMenuIndication() {
		StMenu axesMenu = (StMenu) this._menuBar().atNameKey_($("axesMenu"));
		if (axesMenu == null) {
			return;
		}

		((StCheckBoxMenuItem) axesMenu.atNameKey_($("toggleXAxisMenu"))).beSelected(this.hasXAxis());
		((StCheckBoxMenuItem) axesMenu.atNameKey_($("toggleYAxisMenu"))).beSelected(this.hasYAxis());
		((StCheckBoxMenuItem) axesMenu.atNameKey_($("toggleZAxisMenu"))).beSelected(this.hasZAxis());
	}

	/**
	 * Update the misc menu indication.
	 * 
	 * @category resources
	 */
	public void updateMiscMenuIndication() {
		super.updateMiscMenuIndication();

		StMenu miscMenu = (StMenu) this._menuBar().atNameKey_($("miscMenu"));
		if (miscMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();

		// Spawn
		menuItem = miscMenu.atNameKey_($("spawnMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}
	}

	/**
	 * Action for a menu : Spawn a band chart.
	 * 
	 * @category menu messages
	 */
	public void spawnBandChart() {
		JunChartBand aChart = new JunChartBand();
		this._copyAttributes(aChart);
		aChart.open();
	}

	/**
	 * Action for a menu : Spawn a bar chart.
	 * 
	 * @category menu messages
	 */
	public void spawnBarChart() {
		JunChartBar aChart = new JunChartBar();
		this._copyAttributes(aChart);
		aChart.open();
	}

	/**
	 * Action for a menu : Spawn a line chart.
	 * 
	 * @category menu messages
	 */
	public void spawnLineChart() {
		JunChartLine aChart = new JunChartLine();
		this._copyAttributes(aChart);
		aChart.open();
	}

	/**
	 * Action for a menu : Spawn a pie chart.
	 * 
	 * @category menu messages
	 */
	public void spawnPieChart() {
		JunChartPie aChart = new JunChartPie();
		this._copyAttributes(aChart);
		aChart.open();
	}

	/**
	 * Action for a menu : Spawn a plot chart.
	 * 
	 * @category menu messages
	 */
	public void spawnPlotChart() {
		JunChartPlot aChart = new JunChartPlot();
		this._copyAttributes(aChart);
		aChart.open();
	}

	/**
	 * Action for a menu : Spawn a radar chart.
	 * 
	 * @category menu messages
	 */
	public void spawnRadarChart() {
		JunChartRadar aChart = new JunChartRadar();
		this._copyAttributes(aChart);
		aChart.open();
	}

	/**
	 * Create a <code>JunOpenGL3dObject</code> which represents the chart.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category private
	 */
	protected abstract JunOpenGL3dCompoundObject createChart3dObject();

	/**
	 * Create a <code>JunOpenGL3dObject</code> of x-axis.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject createXAxis() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.name_("x axis");
		Jun3dPoint[] points1 = { new Jun3dPoint(0, 0, 0), new Jun3dPoint(1, 0, 0) };
		compoundObject.add_(new JunOpenGL3dPolyline(points1, this.axisColor()));
		Jun3dPoint[] points2 = { new Jun3dPoint(1, 0, 0), new Jun3dPoint(0.95, -0.02, 0), new Jun3dPoint(0.95, 0, 0), new Jun3dPoint(1, 0, 0) };
		compoundObject.add_(new JunOpenGL3dPolygon(points2, this.axisColor()));
		Jun3dPoint[] points3 = { new Jun3dPoint(1, 0, 0), new Jun3dPoint(0.95, 0, -0.02), new Jun3dPoint(0.95, 0, 0), new Jun3dPoint(1, 0, 0) };
		compoundObject.add_(new JunOpenGL3dPolygon(points3, this.axisColor()));
		compoundObject.paint_(this.axisColor());
		return compoundObject;
	}

	/**
	 * Create a <code>JunOpenGL3dObject</code> of x-axis labels.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected abstract JunOpenGL3dObject createXAxisLabels();

	/**
	 * Create a <code>JunOpenGL3dObject</code> of y-axis.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject createYAxis() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.name_("y axis");
		Jun3dPoint[] points1 = { new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 1, 0) };
		compoundObject.add_(new JunOpenGL3dPolyline(points1, this.axisColor()));
		Jun3dPoint[] points2 = { new Jun3dPoint(0, 1, 0), new Jun3dPoint(-0.02, 0.95, 0), new Jun3dPoint(0, 0.95, 0), new Jun3dPoint(0, 1, 0) };
		compoundObject.add_(new JunOpenGL3dPolygon(points2, this.axisColor()));
		Jun3dPoint[] points3 = { new Jun3dPoint(0, 1, 0), new Jun3dPoint(0, 0.95, -0.02), new Jun3dPoint(0, 0.95, 0), new Jun3dPoint(0, 1, 0) };
		compoundObject.add_(new JunOpenGL3dPolygon(points3, this.axisColor()));
		compoundObject.paint_(this.axisColor());
		return compoundObject;
	}

	/**
	 * Create a <code>JunOpenGL3dObject</code> of y-axis labels.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject createYAxisLabels() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		JunChartData data = this.data();
		double minimumNumber = data.minimumOfValueRange();
		double maximumNumber = data.maximumOfValueRange();
		double interval = data.intervalOfValueRange();
		for (double aNumber = (Math.floor(minimumNumber / interval) + 1) * interval; aNumber <= (Math.floor(maximumNumber / interval) * interval); aNumber += interval) {
			if ((aNumber == minimumNumber) || (aNumber == maximumNumber)) {
				continue;
			}
			double y = data.normalizeValue_(aNumber);
			Jun3dPoint[] points = { new Jun3dPoint(0, y, 0), new Jun3dPoint(-0.05, y, 0) };
			compoundObject.add_(new JunOpenGL3dPolyline(points, this.axisColor()));
			compoundObject.add_(this.label_rightCenterAt_(String.valueOf(aNumber), new Jun3dPoint(-0.05, y, 0)));
		}
		return compoundObject;
	}

	/**
	 * Create a <code>JunOpenGL3dObject</code> of z-axis.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject createZAxis() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.name_("z axis");
		Jun3dPoint[] points1 = { new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 0, 1) };
		compoundObject.add_(new JunOpenGL3dPolyline(points1, this.axisColor()));
		Jun3dPoint[] points2 = { new Jun3dPoint(0, 0, 1), new Jun3dPoint(-0.02, 0, 0.95), new Jun3dPoint(0, 0, 0.95), new Jun3dPoint(0, 0, 1) };
		compoundObject.add_(new JunOpenGL3dPolygon(points2, this.axisColor()));
		Jun3dPoint[] points3 = { new Jun3dPoint(0, 0, 1), new Jun3dPoint(0, -0.02, 0.95), new Jun3dPoint(0, 0, 0.95), new Jun3dPoint(0, 0, 1) };
		compoundObject.add_(new JunOpenGL3dPolygon(points3, this.axisColor()));
		compoundObject.paint_(this.axisColor());
		return compoundObject;
	}

	/**
	 * Create a label with the String at the right-center of the Point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject label_rightCenterAt_(String aString, Jun3dPoint aPoint) {
		if (this.isScalableLabels()) {
			return this.scalableLabel_rightCenterAt_(aString, aPoint);
		} else {
			return this.texturedLabel_rightCenterAt_(aString, aPoint);
		}
	}

	/**
	 * Create a label with the String at the top-center of the Point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject label_topCenterAt_(String aString, Jun3dPoint aPoint) {
		if (this.isScalableLabels()) {
			return this.scalableLabel_topCenterAt_(aString, aPoint);
		} else {
			return this.texturedLabel_topCenterAt_(aString, aPoint);
		}
	}

	/**
	 * Create a scalable label with the String at the right-center of the Point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject scalableLabel_rightCenterAt_(String aString, Jun3dPoint aPoint) {
		Font font = JunFontModel.TextStyle(100);
		JunOpenGL3dObject object = JunOpenGL3dObject.Text_font_(aString, font);
		Jun3dBoundingBox box = object.boundingBox();
		double scale = 1 / Math.max(box.height() / this.labelHeight(), 1);
		object = object.scaledBy_(new Jun3dPoint(scale, scale, 1));
		box = box.scaledBy_(new Jun3dPoint(scale, scale, 1));
		Jun3dPoint offset = new Jun3dPoint(box.width(), box.height() / 2, 0);
		Jun3dPoint margin = new Jun3dPoint(this.labelHeight() / 10 * -1, 0, 0);
		object = object.translatedBy_(aPoint.minus_(offset).plus_(margin));
		//box = box.translatedBy_(aPoint.minus_(offset).plus_(margin));
		object.paint_(this.labelForegroundColor());
		return object;
	}

	/**
	 * Create a scalable label with the String at the top-center of the Point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject scalableLabel_topCenterAt_(String aString, Jun3dPoint aPoint) {
		Font font = JunFontModel.TextStyle(100);
		JunOpenGL3dObject object = JunOpenGL3dObject.Text_font_(aString, font);
		Jun3dBoundingBox box = object.boundingBox();
		double scale = 1 / Math.max(box.height() / this.labelHeight(), 1);
		object = object.scaledBy_(new Jun3dPoint(scale, scale, 1));
		box = box.scaledBy_(new Jun3dPoint(scale, scale, 1));
		Jun3dPoint offset = new Jun3dPoint(box.width() / 2, box.height(), 0);
		Jun3dPoint margin = new Jun3dPoint(0, this.labelHeight() / 10, 0);
		object = object.translatedBy_(aPoint.minus_(offset).plus_(margin));
		//box = box.translatedBy_(aPoint.minus_(offset).plus_(margin));
		object.paint_(this.labelForegroundColor());
		return object;
	}

	/**
	 * Create a textured label with the String at the right-center of the Point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject texturedLabel_rightCenterAt_(String aString, Jun3dPoint aPoint) {
		JunOpenGLTexture texture = JunOpenGLTexture.TextureForString_foreColor_backColor_(aString, this.labelForegroundColor(), this.labelBackgroundColor());
		double height = this.defaultLabelHeight();
		double width = (height * texture.width()) / texture.height();
		double originX = aPoint.x() - width;
		double originY = aPoint.y() - (height / 2);
		double cornerX = aPoint.x();
		double cornerY = aPoint.y() + (height / 2);
		double z = aPoint.z();
		Jun3dPoint[] points = { new Jun3dPoint(originX, originY, z), new Jun3dPoint(cornerX, originY, z), new Jun3dPoint(cornerX, cornerY, z), new Jun3dPoint(originX, cornerY, z) };
		JunOpenGL3dObject label = new JunOpenGL3dPolygon(points);
		label.paint_(Color.white);
		label.texture_(texture);
		return label;
	}

	/**
	 * Create a textured label with the String at the top-center of the Point.
	 * 
	 * @param aString java.lang.String
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private
	 */
	protected JunOpenGL3dObject texturedLabel_topCenterAt_(String aString, Jun3dPoint aPoint) {
		JunOpenGLTexture texture = JunOpenGLTexture.TextureForString_foreColor_backColor_(aString, this.labelForegroundColor(), this.labelBackgroundColor());
		double height = this.defaultLabelHeight();
		double width = (height * texture.width()) / texture.height();
		double originX = aPoint.x() - (width / 2);
		double originY = aPoint.y() - height;
		double cornerX = aPoint.x() + (width / 2);
		double cornerY = aPoint.y();
		double z = aPoint.z();
		Jun3dPoint[] points = { new Jun3dPoint(originX, originY, z), new Jun3dPoint(cornerX, originY, z), new Jun3dPoint(cornerX, cornerY, z), new Jun3dPoint(originX, cornerY, z) };
		JunOpenGL3dObject label = new JunOpenGL3dPolygon(points);
		label.paint_(Color.white);
		label.texture_(texture);
		return label;
	}

}